#!perl 
#===============================================================================
#
# Copyright (C) 2000 - 2014 by SAP AG,  http://www.sap.com.
# All rights reserved.
#
# This software is the confidential and proprietary information of SAP AG.
# You shall not disclose such confidential information and shall use it only
# in accordance with the terms of the license agreement you entered into with SAP.
#
#       AUTHOR: i072928
#

=head1 NAME

HLMConfigurator

=head1 DESCRIPTION

Registers/Unregisters HLM.
=cut

#===============================================================================
package LCM::HLMConfigurator;

use strict;
use base qw(SDB::Install::Base Exporter);
use SDB::Install::SysVars qw($path_separator $isWin);
use LCM::ProcessExecutor;

#----------------------------------------------------------------------------
#
# Actions that can be executed on HDB components.
#
our $action_register   = 'action_register';
our $action_unregister = 'action_unregister';

my $ext = $isWin ? '.exe' : '';

our @EXPORT_OK = qw($action_register $action_unregister);

sub new {
	my ( $class, $logger, $doNotSetLogLocation, $specificHosts, $isAddHostScenario ) = @_;
	my $self = {};
	bless( $self, $class );
	$self->{_logger} = $logger;
	$self->{_doNotSetLogLocation} = $doNotSetLogLocation;
	$self->{_specificHosts} = $specificHosts;
	$self->{_isAddHostScenario} = $isAddHostScenario;
	return $self;
}

sub _getExecutablePath {
    my ($self, $action, $hlmInstallationDir, $sid) = @_;
    my $command = $self->_getHLMCommand ( $action );
    my $hlmPath = $self->_getHLMPath ( $hlmInstallationDir );
    my $executablePath = $hlmPath . $command;
    return $executablePath;
}

sub unregisterLocalHLMInstance {
    my ( $self, $hlmInstallationDir, $sid, $instanceNumber ) = @_;

    $self->{_logger}->getMsgLst ()->addMessage ( "Unregistering local HANA Lifecycle Manager instance..." );    

    my $executablePath = $self->_getExecutablePath($action_unregister, $hlmInstallationDir, $sid);
    my $rc = $self->_handleLocalHLMInstance ( $action_unregister, $sid, $instanceNumber, $executablePath );
    return undef unless $rc;
    return 1;
}

sub unregisterRemoteHLMInstance {
    my ( $self, $hlmInstallationDir, $sid, $instanceNumber, $remoteHosts, $executeOnlyOnHosts ) = @_;
    
    $self->{_logger}->getMsgLst ()->addMessage ( "Unregistering HANA Lifecycle Manager instance on remote hosts..." );

    my $executablePath = $self->_getExecutablePath($action_unregister, $hlmInstallationDir, $sid);
    my $rc = $self->_handleRemoteHLMInstance ( $action_unregister, $sid, $instanceNumber, $executablePath, $remoteHosts, $executeOnlyOnHosts );
    return undef unless $rc;
    return 1;
}

sub registerLocalHLMInstance {
    my ( $self, $hlmInstallationDir, $sid, $instanceNumber) = @_;

    $self->{_logger}->getMsgLst ()->addMessage ( "Registering local HANA Lifecycle Manager instance..." );    

    my $executablePath = $self->_getExecutablePath($action_register, $hlmInstallationDir, $sid);
    return $self->_handleLocalHLMInstance ( $action_register, $sid, $instanceNumber, $executablePath);
}

sub registerRemoteHLMInstance {
    my ( $self, $hlmInstallationDir, $sid, $instanceNumber, $remoteHosts, $executeOnlyOnHosts ) = @_;

    $self->{_logger}->getMsgLst ()->addMessage ( "Registering HANA Lifecycle Manager instance on remote hosts..." );

    my $executablePath = $self->_getExecutablePath($action_register, $hlmInstallationDir, $sid);
   	return $self->_handleRemoteHLMInstance ( $action_register, $sid, $instanceNumber, $executablePath, $remoteHosts, $executeOnlyOnHosts );
}

sub _handleLocalHLMInstance {
	my ( $self, $action, $sid, $instanceNumber, $executablePath) = @_;
	my $actionDescription = $self->_getScenarioMessage( $action );
	unless ( -e $executablePath ) {
		$self->{_logger}->getMsgLst ()
			->addWarning ( "Cannot find $executablePath executable. Probably the version of HANA Lifecycle Manager is old. Skipping "
				. lc( $actionDescription )
				. " of HANA Lifecycle Manager" );
		# fault tolerant, HLM may not be installed, return 2 to signal warning to the caller
		return 2;
	}

	my @args;
    push( @args, "-s $sid" );
    push( @args, "-n $instanceNumber" );
    
    
	my $exer = new LCM::ProcessExecutor ( $executablePath, \@args );
	$exer->quoteCmdLineArguments ( 0 );    # do not quote arguments as HLM bash scripts cannot handle them
	
	if($self->{_isAddHostScenario}) {
		return $self->_handleAddHostScenarioForLocalHLM($exer);
	}
	
	return $exer->execExtProgram (
		$self->{_logger},
		"$actionDescription of SAP HANA Lifecycle Manager...",
		"$actionDescription of SAP HANA Lifecycle Manager failed",
		$self->{_doNotSetLogLocation}
	);
}

sub _handleAddHostScenarioForLocalHLM {
	my ($self, $exer) = @_;
	
	my $exitCode = $exer->executeProgram ();
    $self->{_logger}->getMsgLst()->addMessage(undef, $exer->getMsgLst());
    if ($exitCode == 0) {
    	return 1;
    } else {
    	return undef;
    }
}

sub _handleRemoteHLMInstance {
	my ( $self, $action, $sid, $instanceNumber, $executablePath, $remoteHosts, $executeOnlyOnHosts ) = @_;
	my $actionDescription = $self->_getScenarioMessage( $action );		
	
	my @remoteHostnames = @{ $remoteHosts->getHostNames() };
	if( defined $executeOnlyOnHosts){
       @remoteHostnames = @{$executeOnlyOnHosts}; 
	}
	my $exitCode = 1;
	foreach my $currRemoteHost (@remoteHostnames) {
	    my $cmd = $executablePath . " -s $sid" . " -n $instanceNumber";
    	my $rc = $remoteHosts->executeParallel ( $cmd, undef, undef, undef, undef, undef, [$currRemoteHost] );
    	$self->{_logger}->getMsgLst ()->appendMsgLst ( $remoteHosts->getMsgLst());
    	if($rc){
    		$self->{_logger}->getMsgLst()->addWarning (
    				      "Error occurred while executing $executablePath on remote host $currRemoteHost. Skipping "
    					. lc( $actionDescription ) . " of HANA Lifecycle Manager" );
    					#	Return 2 to signal warning to the caller
    					$exitCode = 2;
    	}
		$remoteHosts->resetMsgLstContext();
    }
	# fault tolerant, HLM may not be installed on the host
	return $exitCode;
}

sub _getHLMCommand {
	my ( $self, $action ) = @_;

	return 'unregister-hlm.sh' if $action eq $action_unregister;
	return 'register-hlm.sh'          if $action eq $action_register;
	return 'unknown_argument';
}

sub _getScenarioMessage {
	my ( $self, $action ) = @_;

	return 'Unregistering' if $action eq $action_unregister;
	return 'Registering'   if $action eq $action_register;
	return 'Unknown action';
}

sub _getHLMPath {
	my ( $self, $hlmInstallationDir ) = @_;
	my $executable = $hlmInstallationDir . $path_separator;
	return $executable;
}

1;
