package LCM::InternalNetworkDetector;

use base qw(SDB::Install::Base);

use File::Spec;
use Scalar::Util;
use LCM::Manifests::GenericManifest;
use SDB::Install::SysVars qw($path_separator $isWin);
use strict;

my $CONFIG_FILE                                           = 'global.ini';
my $COMMUNICATION_PROPERTY_SECTION                        = 'communication';
my $INTERNAL_NETWORK_PROPERTY                             = 'internal_network';
my $LISTEN_INTERFACE_PROPERTY                             = 'listeninterface';
my $INITIAL_HDBREG_REVISION_WITH_INTERNAL_NETWORK_SUPPORT = '60';
my $HDB_RELEASE_ONE                                       = '1.00';

my $HDB_MANIFEST_PATH =
	$path_separator . join( $path_separator, 'usr', 'sap', '%s', 'SYS', 'exe', 'hdb', 'manifest' );
my $OLD_HDB_MANIFEST_PATH = $path_separator
	. join( $path_separator, 'usr', 'sap', '%s', 'SYS', 'exe', 'run', 'hdb', 'manifest' );

#----------------------------------------------------------------------------
# $_[0] - class
# $_[1] - HanaPropertiesHelper
# $_[2] - blessed ref to class capable of logging
#
sub new {
	my $class                = shift();
	my $hanaPropertieshelper = shift();
	my $logClass             = shift();
	my $self                 = $class->SUPER::new ( @_ );
	$self->{_HanaPropertiesHelper} = $hanaPropertieshelper;
	$self->{_logClass}             = $logClass;
	return $self;
}

sub isHdbSupportInternalNewtork
{
	my ( $self, $sid ) = @_;
	$self->getLogClass->getMsgLst()->addMessage ( 'Checking if HDB supports configuration of inter-service communication...' );

	my $manifestFilePath = $self->getHdbManifestFilePath ($sid);
	unless ( $manifestFilePath ) {
		$self->getLogClass->PushError ( 'Cannot detect HDB manifest file.' );
		return undef;
	}

	my $manifest = new LCM::Manifests::GenericManifest ( $manifestFilePath );
	if ( $manifest->errorState ) {
		$self->getLogClass->PushError ( 'Invalid manifest', $manifest->getErrMsgLst () );
		return undef;
	}
	
	my $isReleaseSupported = 0;
	$isReleaseSupported = $manifest->{data}->{release} eq $HDB_RELEASE_ONE;
	if ( $isReleaseSupported && ($manifest->{data}->{'rev-number'} < $INITIAL_HDBREG_REVISION_WITH_INTERNAL_NETWORK_SUPPORT) ) {
	   return 0;
	}

	return 1;
}

sub getAllAddresses {
	my ( $self, $remoteHosts, $isDistributed, $failOnError) = @_;
	my $localAdresses = $self->getAddressesFromLocalHost ($failOnError);
	
	return undef if !defined $localAdresses;
	
	my @addresses = ( @{$localAdresses} );

	if ( $isDistributed ) {
		my $remoteAdresses = $self->getAddressesFromRemoteHosts ( $remoteHosts, $failOnError);
		return undef if !defined $remoteAdresses;
		push @addresses, @{$remoteAdresses};
	}

	return \@addresses;
}

sub getAddressesFromRemoteHosts {
	my ( $self, $remoteHosts, $failOnError ) = @_;
	$remoteHosts->setMsgLstContext ( $self->getLogClass->getMsgLstContext () );

	my $installDir = $self->{_HanaPropertiesHelper}->{_globalTrexInstallDir};
	my $hdbregPath = File::Spec->catfile ( $installDir, 'bin', 'hdbreg' );
	my $cmd        = "$hdbregPath -b --internal_network=0.0.0.0/24 --check_only";
	my $rc         = $remoteHosts->executeParallel ( $cmd, undef, undef, undef, undef, undef, undef, 1 );
	$self->getLogClass->getMsgLst ()->appendMsgLst ( $remoteHosts->getMsgLst () );

	my $hasError;
	my $index = 0;
    foreach my $errorCode ( @{ $remoteHosts->{exit_codes} } ) {
		if ( $errorCode != 1 )    # hdbreg actually returns exit code 1
		{
			$hasError = 1;
			$self->getLogClass->PushError (
                'Error while trying to detect network address on the remote host ' . $remoteHosts->getHostNameByIndex($index));
		}
		$index++;
	}

	if ( $hasError ) {
		if ( $failOnError ) {
			return undef;
		}
	}

	my $outputFromAllHosts = $remoteHosts->getOutputBuffer ();
	my $addresses = [];
	for my $currentHostOutput ( @$outputFromAllHosts ) {
		my @lines = split( /\n/, $currentHostOutput );
		while ( defined $lines[0] && $lines[0] !~ /^[\s]*Available networks:/ ) {
			shift @lines;
		}
		for my $line ( @lines ) {
			if ( $line
				=~ m!(((2(5[0-5]|[0-4][0-9])|[01]?[0-9][0-9]?)\.){2,3}(2(5[0-5]|[0-4][0-9])|[01]?[0-9][0-9]?)(/(3[012]|[12]?[0-9])))!
				) {
				push( @$addresses, $1 );
			}
		}
	}
	return $addresses;
}

sub getAddressesFromLocalHost {
	my ($self, $failOnError) = @_;	
	my $installDir = $self->{_HanaPropertiesHelper}->{_globalTrexInstallDir};
	my $command    = File::Spec->catfile ( $installDir, 'bin', 'hdbreg' );
	my $arguments  = [ '-b', '--internal_network=0.0.0.0/24', '--check_only' ];
	my @addresses;
	my $executor = new LCM::ProcessExecutor ( $command, $arguments );
	$executor->setMsgLstContext ( $self->getLogClass->getMsgLstContext () );

	my $rc = $executor->executeProgram ();
	
	if ( $rc != 1 ) {    # hdbreg actually returns exit code 1
        $self->getLogClass->PushError (
            'Error while trying to detect local network address.' );
        if ( $failOnError ) {
            return undef;
        }
    }
	
	my $output = $executor->{outputLines};

	while ( defined $output->[0] && $output->[0] !~ /^[\s]*Available networks:/ ) {
		shift @$output;
	}
	for my $line ( @$output ) {
		if ( $line
			=~ m!(((2(5[0-5]|[0-4][0-9])|[01]?[0-9][0-9]?)\.){2,3}(2(5[0-5]|[0-4][0-9])|[01]?[0-9][0-9]?)(/(3[012]|[12]?[0-9])))!
			) {
			push @addresses, $1;
		}
	}

	return \@addresses;
}

sub isNostopAvailable {
	my ($self) = @_;
	if(!exists $self->{isNostopAvailable}){	
		$self->{isNostopAvailable} = 0;
#		my $installDir = $self->{_HanaPropertiesHelper}->{_globalTrexInstallDir};
#		my $command    = File::Spec->catfile ( $installDir, 'bin', 'hdbreg' );
#		my $arguments  = [ '--help' ];
#		my $executor = new LCM::ProcessExecutor ( $command, $arguments );
#		my $rc = $executor->executeProgram ();
#		my $output = $executor->{outputLines};
#		while (defined $output->[0]) {
#			if($output->[0] =~ /^.*--nostop.*$/){
#				$self->{isNostopAvailable} = 1;
#			}
#			shift @$output;
#		}
	}
	return $self->{isNostopAvailable};
}

sub detectNetworkListenInterface {
	my ( $self, $sid ) = @_;
	return $self->_getProperty ( $sid, $LISTEN_INTERFACE_PROPERTY );
}

sub detectInternalNetworkAddress {
	my ( $self, $sid ) = @_;
	return $self->_getProperty ( $sid, $INTERNAL_NETWORK_PROPERTY );
}

sub _getProperty {
	my ( $self, $sid, $property ) = @_;

	my $hanaPropertiesHelper = $self->{_HanaPropertiesHelper};
	my ($uid, $gid) = $hanaPropertiesHelper->getPermissions($sid);

	my $propertyValue =
		$hanaPropertiesHelper->getGlobalProperty ( $CONFIG_FILE, $COMMUNICATION_PROPERTY_SECTION,
		$property, $uid, $gid );

	$self->getLogClass->getMsgLst ()->addMessage ( undef, $hanaPropertiesHelper->getMsgLst () );
	$self->getLogClass->getErrMsgLst ()
		->addMessage ( undef, $hanaPropertiesHelper->getErrMsgLst () );

	return $propertyValue;
}

sub getLogClass {
	return shift()->{_logClass};
}

sub getHdbManifestFilePath {
	my ( $self, $sid ) = @_;
	my $path;

	$path = sprintf( $HDB_MANIFEST_PATH, $sid );
	return $path if $path;

	$path = sprintf( $OLD_HDB_MANIFEST_PATH, $sid );
	return $path if $path;

	return undef;
}

1;
