package LCM::Landscape::DescriptionXSD;

use strict;
use File::Spec;
use SDB::Install::User;
use SDB::Install::MsgLst;
use parent 'SDB::Install::Base';
use SDB::Install::SysVars qw($isWin);
use SDB::Common::Utils qw(createSysAdminUserName);
use LCM::FileUtils qw(writeToFile createDirectory);

my $LANDSCAPE_DESCRIPTION_XSD = <<END;
<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema" elementFormDefault="qualified" version="1.09">
	<xs:element name="applicationSystemInLandscape">
		<xs:annotation>
			<xs:documentation>Is a grouping element which represents the CIM Assocation "SAP_ApplicationSystemInLandscape"</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="databaseSystem" minOccurs="0" maxOccurs="unbounded"/>
				<xs:element ref="unspecificStandaloneApplicationSystem" minOccurs="0" maxOccurs="unbounded"/>
				<xs:element ref="bCHostController" minOccurs="0" maxOccurs="unbounded"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="bCHostController">
		<xs:annotation>
			<xs:documentation>the data on host, name and technology type can be used to compile a valied name for the CIM class "SAP_BCHostController"</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="sid"/>
				<xs:element ref="host"/>
				<xs:element ref="installationPath"/>
				<xs:element ref="installationNumber" minOccurs="0"/>
				<xs:element ref="installedProductOnApplicationSystem"/>
				<xs:element ref="installedSWComponentOnApplicationSystem" minOccurs="0"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="caption">
		<xs:simpleType>
			<xs:restriction base="xs:string"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="computerSystem">
		<xs:annotation>
			<xs:documentation>Represents SAP_ComputerSystem</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="host"/>
				<xs:element ref="sharedDirectory"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="computerSystemsInLandscape">
		<xs:annotation>
			<xs:documentation>Groups all computerSystem entities</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="computerSystem" maxOccurs="unbounded"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="databaseSystem">
		<xs:annotation>
			<xs:documentation>the data on host, name and technology type can be used to compile a valied name for the CIM class "SAP_DatabaseSystem"</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="sid">
					<xs:annotation>
						<xs:documentation>name of the system which is captured as SID System Identification</xs:documentation>
					</xs:annotation>
				</xs:element>
				<xs:element ref="dBName"/>
				<xs:element ref="dBTypeForSAP"/>
				<xs:element ref="host"/>
				<xs:element ref="installationPath"/>
				<xs:element ref="installationNumber"/>
				<xs:element ref="installedProductOnApplicationSystem">
					<xs:annotation>
						<xs:documentation>Represents CIM Association SAP_InstalledProductOnApplicationSystem</xs:documentation>
					</xs:annotation>
				</xs:element>
				<xs:element ref="installedSWComponentOnApplicationSystem" minOccurs="0"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="dBName">
		<xs:simpleType>
			<xs:restriction base="xs:string"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="dBTypeForSAP">
		<xs:simpleType>
			<xs:restriction base="xs:string">
				<xs:enumeration value="HDB"/>
				<xs:enumeration value="ORA"/>
				<xs:enumeration value="DB2"/>
				<xs:enumeration value="DB4"/>
				<xs:enumeration value="DB6"/>
			</xs:restriction>
		</xs:simpleType>
	</xs:element>
	<xs:element name="host" type="xs:token"/>
	<xs:element name="identifyingNumber" type="xs:token"/>
	<xs:complexType name="includeFileType">
		<xs:sequence>
			<xs:element name="path" type="xs:string"/>
		</xs:sequence>
	</xs:complexType>
	<xs:element name="includeCurrentStack" type="includeFileType"/>
	<xs:element name="includeVariables" type="includeFileType"/>
	<xs:element name="installationNumber">
		<xs:simpleType>
			<xs:restriction base="xs:string"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="installationPath">
		<xs:simpleType>
			<xs:restriction base="xs:string"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="installDateTime" type="myDateTimeType"/>
	<xs:element name="installedProductOnApplicationSystem">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="installedProductVersion" maxOccurs="unbounded">
					<xs:annotation>
						<xs:documentation>Represents CIM Class SAP_InstalledProductVersion</xs:documentation>
					</xs:annotation>
				</xs:element>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="installedProductVersion">
		<xs:annotation>
			<xs:documentation>Represents CIM class SAP_InstalledProduct</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="identifyingNumber"/>
				<xs:element ref="name"/>
				<xs:element ref="vendor"/>
				<xs:element ref="version"/>
				<xs:element ref="caption"/>
				<xs:element ref="installedSPStack" minOccurs="0"/>
				<xs:element ref="collectedSoftwareFeatures" minOccurs="0">
					<xs:annotation>
						<xs:documentation>Represents CIM Association SAP_CollectedSoftwareFeatures</xs:documentation>
					</xs:annotation>
				</xs:element>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="installedSoftwareComponent">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="identifyingNumber"/>
				<xs:element ref="name"/>
				<xs:element ref="vendor"/>
				<xs:element ref="version"/>
				<xs:element ref="installedSupportPackageSoftwareComponent" minOccurs="0"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="installedSPStack">
		<xs:annotation>
			<xs:documentation>Represents CIM Class SAP_InstalledSupportPackageStack. There is an 0..1, 0..1 association defined to SAP_InstalledProduct which is called SAP_InstalledSPStackProduct. We omitted adding this as another XML element, since there is no grouping imposed by this association. When transferring this data to CIM the association needs to be added of course.</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="identifyingNumber"/>
				<xs:element ref="caption"/>
				<xs:element ref="releaseDate"/>
				<xs:element ref="installDateTime"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="installedSupportPackage">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="sPLevel"/>
				<xs:element ref="patchLevel"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="installedSupportPackageSoftwareComponent">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="installedSupportPackage" maxOccurs="unbounded"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="installedSWComponentOnApplicationSystem">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="installedSoftwareComponent" maxOccurs="unbounded"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="installedSWFeature">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="caption"/>
				<xs:element ref="instanceKey"/>
				<xs:element ref="identifyingNumber"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="instanceKey" type="instkeytype"/>
	<xs:simpleType name="instkeytype">
		<xs:restriction base="xs:token">
			<xs:pattern value="[0-9]{1,3}"/>
		</xs:restriction>
	</xs:simpleType>
	<xs:element name="landscapeDescription">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="version"/>
				<xs:element ref="includeVariables"/>
				<xs:element ref="includeCurrentStack"/>
				<xs:element ref="productSystem"/>
				<xs:element ref="computerSystemsInLandscape"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="name">
		<xs:simpleType>
			<xs:restriction base="xs:string"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="patchLevel">
		<xs:simpleType>
			<xs:restriction base="xs:int"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="productSystem">
		<xs:annotation>
			<xs:documentation>the data on host, name and technology type can be used to compile a valied name for the CIM class "SAP_ProductSystemLandscape"  The product system itself is almost not materialized. In HANA however the landscape description file somehow materializes the product system. In this sense the computerSystem shall refer to the computerSystem where the landscapedescription file is stored.</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="sid">
					<xs:annotation>
						<xs:documentation>This is the name of the system. The corresponding field in CIM is called name.</xs:documentation>
					</xs:annotation>
				</xs:element>
				<xs:element ref="host"/>
				<xs:element ref="installationPath" minOccurs="0"/>
				<xs:element ref="installationNumber" minOccurs="0"/>
				<xs:element ref="productSystemLandscapeProduct"/>
				<xs:element ref="applicationSystemInLandscape"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="productSystemLandscapeProduct">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="installedProductVersion" maxOccurs="unbounded"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="releaseDate" type="myDateType"/>
	<xs:simpleType name="myDateType">
		<xs:union memberTypes="xs:date">
			<xs:simpleType>
				<xs:restriction base="xs:string">
					<xs:enumeration value="$(stackReleaseDate)"/>
				</xs:restriction>
			</xs:simpleType>
		</xs:union>
	</xs:simpleType>
	<xs:simpleType name="myDateTimeType">
		<xs:union memberTypes="xs:dateTime">
			<xs:simpleType>
				<xs:restriction base="xs:string">
					<xs:enumeration value="$(stackInstallDateTime)"/>
				</xs:restriction>
			</xs:simpleType>
		</xs:union>
	</xs:simpleType>
	<xs:element name="sid" type="xs:string"/>
	<xs:element name="sPLevel">
		<xs:simpleType>
			<xs:restriction base="xs:int"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="technologyType">
		<xs:simpleType>
			<xs:restriction base="xs:string">
				<xs:enumeration value="IMDBClient"/>
				<xs:enumeration value="IMDBStudioRepository"/>
				<xs:enumeration value="LoadController"/>
				<xs:enumeration value="SybaseReplicationServer"/>
			</xs:restriction>
		</xs:simpleType>
	</xs:element>
	<xs:element name="unspecificStandaloneApplicationSystem">
		<xs:annotation>
			<xs:documentation>the data on host, name and technology type can be used to compile a valied name for the CIM class "SAP_UnspecificStandaloneApplicationSystem"</xs:documentation>
		</xs:annotation>
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="sid"/>
				<xs:element ref="host"/>
				<xs:element ref="technologyType"/>
				<xs:element ref="installationPath"/>
				<xs:element ref="installationNumber" minOccurs="0"/>
				<xs:element ref="installedProductOnApplicationSystem"/>
				<xs:element ref="installedSWComponentOnApplicationSystem" minOccurs="0"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
	<xs:element name="vendor">
		<xs:simpleType>
			<xs:restriction base="xs:string"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="version">
		<xs:simpleType>
			<xs:restriction base="xs:string"/>
		</xs:simpleType>
	</xs:element>
	<xs:element name="sharedDirectory" type="xs:token"/>
	<xs:element name="collectedSoftwareFeatures">
		<xs:complexType>
			<xs:sequence>
				<xs:element ref="installedSWFeature" maxOccurs="unbounded"/>
			</xs:sequence>
		</xs:complexType>
	</xs:element>
</xs:schema>
END
my $LANDSCAPE_DESCRIPTION_XSD_FILE = 'landscapeDescription.xsd';
my $FILE_MODE = 0755;

sub new {
	my ($class, $messageListContext) = @_;
	my $self = $class->SUPER::new();
	$self->setMsgLstContext($messageListContext) if(defined($messageListContext));
	return $self;
}

sub store {
	my ($self, $sid, $targetDirectory) = @_;
	my ($uid, $gid) = (undef, undef);
	my $errorList = new SDB::Install::MsgLst();
	my $filePath = File::Spec->catfile($targetDirectory, $LANDSCAPE_DESCRIPTION_XSD_FILE);

	if(!$isWin){
		my $sidadmUserName = createSysAdminUserName($sid);
		my $sidadmUser = new SDB::Install::User($sidadmUserName);
		($uid, $gid) = ($sidadmUser->id(), $sidadmUser->gid());
	}
	if(!-d $targetDirectory && !createDirectory($targetDirectory, $FILE_MODE, $uid, $gid, undef, $errorList)){
		$self->getErrMsgLst()->addError("Failed to create directory '$targetDirectory'", $errorList);
		return undef;
	}
	if(!writeToFile($filePath, $LANDSCAPE_DESCRIPTION_XSD, $FILE_MODE, $uid, $gid, $errorList)){
		$self->getErrMsgLst()->addError("Failed to write file '$filePath' to the file system", $errorList);
		return undef;
	}
	return 1;
}

1;