package LCM::Landscape::SoftwareComponent;

use strict;
use File::Spec;
use LCM::Component;
use LCM::Landscape::Cockpit::ManifestStub;
use SDB::Install::Globals qw($gKeynameClient $gKeynameStudio $gKeynameXS2);

my $SCV_NAME_HANA_COCKPIT='SAP_HANA_COCKPIT';

sub new {
	require XML::LibXML;

	my ($class, $systemComponentManager, $scvElement) = @_;
	my $self = bless({}, $class);
	my $scvName = $self->_determineScvName($scvElement);
	my $scvPPMSId = $self->_determineScvPPMSId($scvElement);
	my $ownComponent = $self->_determineOwnComponent($scvPPMSId, $scvName, $systemComponentManager);
	$self->_setScvPPMSId($scvPPMSId);
	$self->_setOwnComponent($ownComponent);

	return $self;
}

sub isEmpty {
	my ($self) = @_;
	return !defined($self->_getOwnComponent());
}

sub createElement {
	my ($self) = @_;
	my $softwareComponentTag = XML::LibXML::Element->new('software-component');
	my $featureMap = $self->_getFeatureMap();

	for my $element (keys %$featureMap){
		my $tag = XML::LibXML::Element->new($element);
		my $value = $featureMap->{$element};
		$tag->appendTextNode($value);
		$softwareComponentTag->appendChild($tag);
	}
	return $softwareComponentTag;
}

sub _getName {
	my ($self) = @_;
	my $ownComponent = $self->_getOwnComponent();
	my $manifest = $ownComponent->getManifest();

	return $manifest->getValue('keyname');
}

sub _getVersion {
	my ($self) = @_;
	my $ownComponent = $self->_getOwnComponent();
	my $manifest = $ownComponent->getManifest();
	my $release = $manifest->getValue('release');
	my $hostAgentRelease = $manifest->getValue('hostagent release');

	return (defined $release ? $release : $hostAgentRelease);
}

sub _getCaption {
	my ($self) = @_;
	my $ownComponent = $self->_getOwnComponent();
	my $manifest = $ownComponent->getManifest();
	my $keyCaption = $manifest->getValue('keycaption');
	my $keyName = $manifest->getValue('keyname');

	return (defined $keyCaption ? $keyCaption : $keyName);
}

sub _getVendor {
	my ($self) = @_;
	my $ownComponent = $self->_getOwnComponent();
	my $manifest = $ownComponent->getManifest();

	return $manifest->getValue('keyvendor');
}

sub _getLevel {
	my ($self) = @_;
	my $ownComponent = $self->_getOwnComponent();
	my $manifest = $ownComponent->getManifest();
	my $spNumber = $manifest->getValue('sp-number');
	my $revNumber = $manifest->getValue('rev-number');
	my $hostAgentPatchNumber =  $manifest->getValue('hostagent patch number');

	return defined($spNumber) ? $spNumber : defined($revNumber) ? $revNumber : $hostAgentPatchNumber;
}

sub _getOsSet {
	my ($self) = @_;
	my $ownComponent = $self->_getOwnComponent();
	my $manifest = $ownComponent->getManifest();
	my $platform = $manifest->getValue('platform');

	if (!defined($platform)) {
		my $hostAgentOs = $manifest->getValue('hostagent os');
		if (defined $hostAgentOs) {
			$platform = $hostAgentOs;
		} else {
			$platform = 'linuxx86_64';
		}
	}
	return $platform;
}

sub _getPatchLevel {
	my ($self) = @_;
	my $ownComponent = $self->_getOwnComponent();
	my $manifest = $ownComponent->getManifest();
	my $spPatchLevel = $manifest->getValue('sp-patchlevel');
	my $revPatchLevel = $manifest->getValue('rev-patchlevel');
	my $hostAgentPatchNumber =  $manifest->getValue('hostagent patch number');

	return defined($spPatchLevel) ? $spPatchLevel : defined($revPatchLevel) ? $revPatchLevel : $hostAgentPatchNumber;
}

sub _getFeatureMap {
	my ($self) = @_;
	my $featureMap = {};
	my $caption = $self->_getCaption();
	my $ppmsNumber = $self->_getScvPPMSId();
	my $osSet = $self->_getOsSet();
	my $name = $self->_getName();
	my $vendor = $self->_getVendor();
	my $version = $self->_getVersion();
	my $spLevel = $self->_getLevel();
	my $patchLevel = $self->_getPatchLevel();
	my $revChangeList = $self->_getRevChangeList();
	
	$featureMap->{'caption'} = $caption if defined $caption;
	$featureMap->{'ppms-number'} = $ppmsNumber if defined $ppmsNumber;
	$featureMap->{'os-set'} = $osSet if defined $osSet;
	$featureMap->{'name'} = $name if defined $name;
	$featureMap->{'vendor'} = $vendor if defined $vendor;
	$featureMap->{'version'} = $version if defined $version;
	$featureMap->{'sp-level'} = $spLevel if defined $spLevel;
	$featureMap->{'patch-level'} = $patchLevel if defined $patchLevel;
	$featureMap->{'rev-changelist'} = $revChangeList if defined $revChangeList;
	

	return $featureMap;
}

sub _getScvPPMSId {
	my ($self) = @_;
	return $self->{_scvPPMSId};
}

sub _setScvPPMSId {
	my ($self, $scvPPMSId) = @_;
	$self->{_scvPPMSId} = $scvPPMSId; 
}

sub _getOwnComponent {
	my ($self) = @_;
	return $self->{_ownComponent};
}

sub _setOwnComponent {
	my ($self, $component) = @_;
	$self->{_ownComponent} = $component; 
}

sub _determineScvPPMSId {
	my ($self, $scvElement) = @_;
	my $scvPPMSId = undef;

	eval {
		my $scvKeyElement = $scvElement->getElementsByTagName('software-component-version-key')->[0];
		my $ppmsIdElement = $scvKeyElement->getElementsByTagName('PPMS-ID')->[0];
		$scvPPMSId = $ppmsIdElement->textContent();
	};
	return $scvPPMSId;
}

sub _determineScvName {
	my ($self, $scvElement) = @_;
	my $scvName = undef;

	eval {
		my $scvKeyElement = $scvElement->getElementsByTagName('software-component-version-key')->[0];
		my $ppmsIdElement = $scvKeyElement->getElementsByTagName('name')->[0];
		$scvName = $ppmsIdElement->textContent();
	};
	return $scvName;
}

sub _determineOwnComponent {
	my ($self, $scvPPMSId, $scvName, $systemComponentManager) = @_;
	my $detectedComponents = $systemComponentManager->getAllComponents();
	my ($ownComponent) = grep { $scvPPMSId eq $_->getComponentId() } @{$detectedComponents};
	my $ownKeyName = defined($ownComponent) ? $ownComponent->getComponentKeyName() : undef;

	if($ownKeyName eq $gKeynameClient || $ownKeyName eq $gKeynameStudio){
		# Generation of LM_STRUCTURE for client/studio based on PD.XML is not yet implemented.
		# Explicitly ignore the detected client/studio in order not to break the LM_STRUCTURE
		return undef;
	}
	if($scvName eq $SCV_NAME_HANA_COCKPIT){
		my $xsComponentRegistryPath = $self->_determineXSComponentRegistryPath($systemComponentManager);
		my $cockpitManifestStub = new LCM::Landscape::Cockpit::ManifestStub($systemComponentManager);

		return new LCM::Component($xsComponentRegistryPath, $cockpitManifestStub, $systemComponentManager);
	}
	return ($ownComponent);
}

sub _determineXSComponentRegistryPath {
	my ($self, $systemComponentManager) = @_;
	my $xsComponent = $systemComponentManager->getComponentByKeyName($gKeynameXS2);

	return defined($xsComponent) ? File::Spec->catfile($xsComponent->getPath(), 'cmp-registry', 'cmp_registry_sap.xml') : undef;
}

sub _getRevChangeList{
    my ($self) = @_;
    my $ownComponent = $self->_getOwnComponent();
    my $manifest = $ownComponent->getManifest();
    my $revChangeList = $manifest->getValue('rev-changelist');
    return defined $revChangeList ? $revChangeList : 0 ;
}

1;