package LCM::PersistencyUtils;

use strict;

use SDB::Install::MsgLst;
use SDB::Install::Globals qw ( $gProductNameEngine $gSAPLocation);
use SDB::Install::SAPSystem qw ( CollectSAPSystems getPendingInfoFromPersFile @STEP_NAMES @UPGRADE_STEP_NAMES );
use LCM::Task::TaskStatus qw ( FINISHED );
use LCM::Component qw ( PREPARE_PHASE OFFLINE_PHASE ONLINE_PHASE ALL );
use LCM::Persistence::XMLDetector;
use LCM::FileUtils qw(deleteFile);

use base qw ( Exporter );

our @EXPORT_OK = qw ( 	existsHdblcmPendingUpdate
						existsHdbupdPendingUpdate
						existsHdbinstPendingInstall
						existsHdblcmPendingInstall
						existsHdbinstPendingInstallOnly
						existsHdbupdPendingUpdateOnly
						existsPendingComponentsUpdate
						getPendingUpdateSIDs
						getAllPendingInstallInfo
						getAllPendingUpdateInfo
						getHdblcmPendingInfo
						getHdbupdPersistenceFilename
						getHdbinstPersistenceFilename
						getPersistenceEntryValue
						isPendingUpdate );

sub existsHdbupdPendingUpdateOnly {
	my ($sid) = @_;

	if (!defined $sid) {
		return undef;
	}

	return (existsHdbupdPendingUpdate($sid) && !existsHdblcmPendingUpdate($sid)) ? 1 : 0;
}

sub existsHdbinstPendingInstallOnly {
	my ($sid, $target) = @_;

	if (!defined $sid) {
		return undef;
	}

	return (existsHdbinstPendingInstall($sid, $target) && !existsHdblcmPendingInstall($sid, $target)) ? 1 : 0;
}

sub existsHdblcmPendingUpdate {
	my ($sid) = @_;

	if (!defined $sid) {
		return undef;
	}

	require LCM::Configuration::GenericStackUpdateConfiguration;
	my $hdblcmUpdateConfiguration  = new LCM::Configuration::GenericStackUpdateConfiguration();
	$hdblcmUpdateConfiguration->{current_sid} = $sid;

	return $hdblcmUpdateConfiguration->pers_exists( $hdblcmUpdateConfiguration->pers_filename() ) ? 1 : 0;
}

sub existsHdblcmPendingInstall {
	my ($sid, $target) = @_;

	if (!defined $sid) {
		return undef;
	}
	require LCM::Configuration::GenericStackInstallConfiguration;
	my $installConfiguration  = new LCM::Configuration::GenericStackInstallConfiguration();
	$installConfiguration->{current_sid} = $sid;
	$installConfiguration->{params}->{SID}->{value} = $sid;
	$installConfiguration->{params}->{Target}->{value} = $target;
	return $installConfiguration->pers_exists( $installConfiguration->pers_filename() ) ? 1 : 0;
}

sub existsHdbinstPendingInstall {
	my ($sid, $target) = @_;
	if (!defined $sid) {
		return undef;
	}

	my $newServerConfig = _createHdbinstConfiguration($sid, $target);
	my $hdbinstPersistenceFile = $newServerConfig->pers_filename();

	if ( ! $newServerConfig->pers_exists($hdbinstPersistenceFile) ) {
		my $oldHdbinstPersistenceFile = $newServerConfig->getOldLocalPersFilename ();

		if (defined $oldHdbinstPersistenceFile && defined $newServerConfig->validateOldLocalPersistencyFile ()){
			$hdbinstPersistenceFile = $oldHdbinstPersistenceFile;
			
		}
	}
	return $newServerConfig->pers_exists($hdbinstPersistenceFile) ? 1 : 0;
}

sub _createHdbinstConfiguration {
	my ($sid, $target) = @_;

	my $allSystems = CollectSAPSystems();
	my $sapSystem = $allSystems->{$sid};
	require SDB::Install::Configuration::NewServerConfig;
	my $newServerConfig = new SDB::Install::Configuration::NewServerConfig();

	$newServerConfig->{current_sid} = $sid;
	$newServerConfig->{sapSys}      = $sapSystem;
	$newServerConfig->{params}->{SID}->{value} = $sid;
	$newServerConfig->{params}->{Target}->{value} = $target;
	
	return $newServerConfig;
}

sub existsHdbupdPendingUpdate {
	my ($sid) = @_;

	if (!defined $sid) {
		return undef;
	}

	my $updConfig = _createHdbupdConfiguration($sid);
	my $hdbupdPersistenceFile = $updConfig->pers_filename();
	my $rc = $updConfig->pers_exists($hdbupdPersistenceFile);

	if (!$rc) {
		my $oldHdbupdPersistenceFile = $updConfig->getOldLocalPersFilename ();

		if (defined $oldHdbupdPersistenceFile && defined $updConfig->validateOldLocalPersistencyFile ()){
			$hdbupdPersistenceFile = $oldHdbupdPersistenceFile;
		}
	}

	return ($updConfig->pers_exists($hdbupdPersistenceFile)) ? 1 : 0;
}

sub getHdbupdPersistenceFilename {
	my ($sid) = @_;
	if (!defined $sid) {
		return undef;
	}
	my $updConfig = _createHdbupdConfiguration($sid);
	return $updConfig->pers_filename();
}

sub getHdbinstPersistenceFilename {
	my ($sid, $target) = @_;
	if (!defined $sid) {
		return undef;
	}
	my $newServerConfig = _createHdbinstConfiguration($sid, $target);
	return $newServerConfig->pers_filename();
}

sub existsPendingComponentsUpdate{
    my ($sid) = @_;
    my $persistencyFileList = LCM::Persistence::XMLDetector::detect( $sid );
    return %$persistencyFileList ? 1 : 0 ;
}

sub _createHdbupdConfiguration {
	my ($sid) = @_;

	if (!defined $sid) {
		return undef;
	}

	my $allSystems = CollectSAPSystems();
	my $sapSystem = $allSystems->{$sid};
	require SDB::Install::Configuration::Upgrade;
	my $updConfig = new SDB::Install::Configuration::Upgrade();

	$updConfig->{current_sid} = $sid;
	$updConfig->{sapSys}      = $sapSystem;

	return $updConfig;
}

sub isPendingUpdate {
	my ($sid, $pendingUpdateSystems) = @_;
	my @pendingUpdateSIDs = getPendingUpdateSIDs($pendingUpdateSystems);
	for my $pendingUpdateSID (@pendingUpdateSIDs) {
		if ($sid eq $pendingUpdateSID) {
			return 1;
		}
	}
	return 0;
}

sub getPendingUpdateSIDs {
	my ($pendingUpdateSystems) = @_;
	my @pendingSIDs = ();
	my $allSystems = CollectSAPSystems();

	foreach my $currSID (sort keys %{$allSystems}) {
		my $currSystem = $allSystems->{$currSID};
		if (existsHdbupdPendingUpdate($currSID) || existsPendingComponentsUpdate($currSID)) {
			push(@pendingSIDs, $currSID);
			$pendingUpdateSystems->{$currSID} = $currSystem;
		}
	}

	return @pendingSIDs;
}

sub getAllPendingInstallInfo {
    my ($instconfig, $twoLiner, $msgLst) = @_;
    my $hdinstPendingInfo;
    if (defined $instconfig->getSAPSystem()) {
        require SDB::Install::Configuration::NewDB;
        my $newDBConfig = new SDB::Install::Configuration::NewDB();
        $newDBConfig->{current_sid} = $instconfig->getSID();
        $newDBConfig->{sapSys}      = $instconfig->getSAPSystem();
        $hdinstPendingInfo = getPendingInfoFromPersFile('Installation', 'of', $newDBConfig, \@STEP_NAMES, '', $twoLiner, $msgLst);
    } else {
        $hdinstPendingInfo = getPendingInfoFromPersFile('Installation', 'of', $instconfig, \@STEP_NAMES, '', $twoLiner, $msgLst);
    }
	my $overallPendingInfo = '';
	$overallPendingInfo .= $gProductNameEngine.': '.$hdinstPendingInfo if (defined $hdinstPendingInfo);
	$overallPendingInfo .= getAllPendingComponentsInfo($instconfig->getSID(), '', '', $twoLiner, $msgLst);
    return $overallPendingInfo;
}

sub getAllPendingUpdateInfo {
	my ($sapSystem, $overallPrefix, $prefixForAdditionalComponentsInfo, $twoLiner, $msgLst, $isResident) = @_;

	require LCM::Configuration::GenericStackUpdateConfiguration;
	my $hdblcmUpdateConfiguration  = new LCM::Configuration::GenericStackUpdateConfiguration();
	my $sid = $sapSystem->get_sid();

	$hdblcmUpdateConfiguration->{current_sid} = $sid;

	my $overallPendingInfo = '';
	my $hdblcmPendingUpdInfo = getHdblcmPendingInfo('Update',
													$hdblcmUpdateConfiguration,
													$overallPrefix,
													$twoLiner,
                                                    $isResident);

	if (!defined $hdblcmPendingUpdInfo) {
		$prefixForAdditionalComponentsInfo = '';
	} else {
		$overallPendingInfo = $hdblcmPendingUpdInfo;
	}

	my $hdbupdPendingInfo = !$isResident ? _getHdbUpdPendingInfo($sapSystem, $sid, $twoLiner) : undef;

	if (defined $hdbupdPendingInfo) {
		$overallPendingInfo .= $overallPrefix.$prefixForAdditionalComponentsInfo.$gProductNameEngine.': '.$hdbupdPendingInfo;
	}

    if (defined $isResident && !$isResident && !$hdblcmPendingUpdInfo && !$hdbupdPendingInfo) {
        return undef;
    }

	$overallPendingInfo .= getAllPendingComponentsInfo($sid, $overallPrefix, $prefixForAdditionalComponentsInfo, $twoLiner, $msgLst);
	return $overallPendingInfo;
}

sub _getHdbUpdPendingInfo {
	my ($sapSystem, $sid, $twoLiner) = @_;

	require SDB::Install::Configuration::Upgrade;
	my $updConfig  = new SDB::Install::Configuration::Upgrade();

	$updConfig->{current_sid} = $sid;
	$updConfig->{sapSys}      = $sapSystem;

	return getPendingInfoFromPersFile('Update',
                                      'to',
                                      $updConfig,
                                      \@UPGRADE_STEP_NAMES,
                                      '',
                                      $twoLiner);
}

sub getHdblcmPendingInfo {
	my ($action,      				# e.g 'Update'
        $hdblcmConfiguration, 		# e.g. LCM::Configuration::GenericStackUpdateConfiguration
        $prefix,      				# e.g. for indentation
        $twoLiner,     				# if true, linebreak will be inserted before the "(", if applicable
        $isResident                 # if true, will return info only for action update_components. Ignored if set to undef
	) = @_;

	if (!$hdblcmConfiguration->pers_exists()) {
		return undef;
	}
	my $current_sid = $hdblcmConfiguration->{current_sid};
    if(!existsHdbupdPendingUpdate($current_sid) && !existsPendingComponentsUpdate($current_sid)){
    	return undef;
    }

	my $persistenceData = $hdblcmConfiguration->pers_load();
    if (defined $isResident) {
        return undef if $isResident  && $persistenceData->{'pers_action'} eq 'update';
        return undef if !$isResident && $persistenceData->{'pers_action'} eq 'update_components';
    }

	my $linebreak = $twoLiner ? "\n" : " ";
	my $phase = $persistenceData->{'Phase'};
	my $prepareUpdate = $persistenceData->{'PrepareUpdate'};
	my $isSuccessfullyPreparedUpdate = $persistenceData->{'pers_successfully_prepared_update'};
	my $date  = $hdblcmConfiguration->pers_date_string();
	my $info = $prefix.$action." is pending$linebreak";

	if (($phase eq PREPARE_PHASE) || $prepareUpdate || $isSuccessfullyPreparedUpdate) {
		$info .= "(prepared at ".$date.")";
	} else {
		my $stepName = $persistenceData->{'step_name'};
		$info .= "(since $date)";
		$info .= " at step \'$stepName\'" if (defined $stepName);
	}

	return $info . "\n";
}

sub getAllPendingComponentsInfo{
    my ($sid, $overallPrefix, $prefixForAdditionalComponentsInfo, $twoLiner, $msgLst) = @_;
    my $persistencyFileList = LCM::Persistence::XMLDetector::detect($sid, $msgLst);
    my $allComponentInfo = '';
    my $prepared = '';
    while (my ($componentKey, $persistency) = each %$persistencyFileList ){
       $allComponentInfo .= $overallPrefix. $prefixForAdditionalComponentsInfo;
       if (!defined $persistency) {
           $allComponentInfo .= $componentKey. ": Invalid persistence file";
           next;
       }
       my $pendingVersion = $persistency->getTargetVersionString();
       my $pendingDate = $persistency->getCreationDateString();
       my $nextPhase = $persistency->getNextPhaseString();
       my $componentName = $persistency->getComponentNameString();
       my $stepName = $persistency->getCurrentStepString();
       my $linebreak = $twoLiner ? "\n" : " ";
       my $isInstall = $persistency->getSourceVersionString() eq '<unknown>';
       my $action = $isInstall ?'Install of':'Update to';
       $allComponentInfo .= "$componentName: $action version $pendingVersion is pending" . $linebreak . "(since $pendingDate)";
       if (!defined $stepName || $stepName eq '<unknown>') {
           $stepName = 'Prepared' if (($nextPhase eq OFFLINE_PHASE));
           $stepName = $isInstall ? 'Finalize installation' : 'Finalize update' if (($nextPhase eq ONLINE_PHASE));
       }
       $allComponentInfo .= " at step \'$stepName\'\n";
    }
    return $allComponentInfo;
}

sub getPersistenceEntryValue {
	my ( $configuration, $entry ) = @_;
	return undef if ( ! $configuration->pers_exists() );
	my $persistenceData = $configuration->pers_load();
	return $persistenceData->{$entry};
}

1;
