package LCM::PostExecuteAction::RegenerateCertificates::RegenerateCertificatesOnLocalHost;

use strict;
use LCM::ProcessExecutor;
use base qw(LCM::PostExecuteAction::RegenerateCertificates);
use LCM::TraceLoggingEnvironment;
use LCM::HostsParser;
use SDB::Install::Tools qw(escapeCommandLineArgument);
use LCM::ExecutionWarningsObservable;

my $PROGRESS_MESSAGE = "Regenerating SSL certificates...";
my $END_MESSAGE_SUCCESS = "Finished regenerating SSL certificates.";
my $END_MESSAGE_FAILURE = "Failed regenerating SSL certificates.";

my $STATUS_OK = 1;
my $STATUS_ERROR = 0;

sub _initialize {
	my ($self) = @_;
	
	if( $self->_checkInstconfig() == $STATUS_ERROR || 
	    $self->_checkAndSetLocalHost() == $STATUS_ERROR 
	  ) {
		return $STATUS_ERROR;
	}
	
	return $STATUS_OK;
}

sub execute{
	my ($self) = @_;
	
	$self->_prepareMessageHandling($self->_getLogHandler(), $PROGRESS_MESSAGE);
	
	if($self->_initialize() == $STATUS_ERROR) {
		$self->_finalizeMessageHandling($STATUS_ERROR);
		return $STATUS_ERROR;
	}
	
	my ($localHost, $instconfig) = ($self->_getLocalHost(), $self->_getInstconfig());
	my $certificateHostmapParameter = $instconfig->{params}->{CertificatesHostmap};
	
	my $command = $instconfig->getResidentInstallerExecutable ();
    if (!defined $command){
    	$self->_addWarningMessage("Unable to get path to executable!");
        $self->_finalizeMessageHandling($STATUS_ERROR);
		return $STATUS_ERROR;
    }

    my $args = $self->_getArgs($localHost, $certificateHostmapParameter);
    my $input = $self->_getKeystorePassword();
	my $executor = new LCM::ProcessExecutor($command,$args, [$input]);
	$executor->setProcessEnvironment(LCM::TraceLoggingEnvironment::PrepareProcessExecutorHdbEnvironment($command));
	my $rc = $executor->executeProgram ();
	$self->_addOutputFromExternalProcess($executor, 0);
	my $exitCode = $STATUS_ERROR;				
	$exitCode = $STATUS_OK if (defined $rc && $rc == 0 );
	$self->_finalizeMessageHandling($exitCode);
	
	return $exitCode;
}

sub _getArgs {
	my ($self, $localhost, $certificateHostmapParameter) = @_;
	my $certificateHostmap = $certificateHostmapParameter->{value};
	my $localCertificateHostname = $certificateHostmap->{$localhost};

	if ((not defined $localCertificateHostname) and (defined $certificateHostmap)) {
		foreach my $host (keys (%{$certificateHostmap})) {
			if (LCM::HostsParser::IsLocalHost($host)) {
				$localCertificateHostname = $certificateHostmap->{$host};
				last;
			}
		}
	}

	return ['--main', 'LCM::RegenerateCertificates::main', '-hostname', $localCertificateHostname, '-restart_sha', 'true'];
}

sub _checkAndSetLocalHost {
	my ($self) = @_;

	my $localHost = $self->_getInstconfig()->getLocalHanaHost();
	if(!$localHost){
		$self->_addWarningMessage("Unable to determine hostname for localhost, aborting execution!");
		return $STATUS_ERROR;
	}
	
	$self->_setLocalHost($localHost);
	return $STATUS_OK;
}

sub _setLocalHost {
	$_[0]->{_localHost} = $_[1];
}

sub _getLocalHost {
	return $_[0]->{_localHost};
}

sub _checkInstconfig {
	my ($self) = @_;
	
	if(!$self->_getInstconfig()){
		$self->_addWarningMessage("Unable to get configuration object, aborting execution!");
		return $STATUS_ERROR;
	}
	
	return $STATUS_OK;
}

sub _getEndMessage{
	my ($self, $exitCode) = @_;
	return $exitCode == $STATUS_OK ? $END_MESSAGE_SUCCESS : $END_MESSAGE_FAILURE;
}

sub _finalizeMessageHandling {
    my ($self, $status) = @_;
    $self->SUPER::_finalizeMessageHandling($status);
    
    if ($status eq $STATUS_ERROR){
    	my $localHost = $self->_getInstconfig()->getLocalHanaHost();
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning("SAP Host Agent was not configured to work with hdblcm on host $localHost. Check your SAP Host Agent installation.");
    }
}

1;
