package LCM::RegenerateCertificates;

use strict;
use File::Spec;
use File::Copy;
use Getopt::Long;
use Term::ReadKey;
use SDB::Install::SysVars qw($isWin);
use LCM::ProcessExecutor;
use base qw (Exporter);
use SDB::Install::Tools qw(trim);

our @EXPORT = qw (
    ERROR_RC
    NO_NEED_REGENERATION
    NEED_REGENERATION
);

use constant ERROR_RC               => '010';
use constant NO_NEED_REGENERATION   => '011';
use constant NEED_REGENERATION      => '012';

my ($FORCE_OPTION, $FQHOSTNAME, $RESTART_SHA) = ('', '', '');

my ($UID, $GID);
if(!$isWin){
	(undef, undef, $UID, $GID) = getpwnam("sapadm");
} 

my $SHA_EXE_DIR = File::Spec->catdir('/usr', 'sap', 'hostctrl', 'exe');
my $SEC_DIR = File::Spec->catdir($SHA_EXE_DIR, 'sec');

my $LD_LIBRARY_PATH = $SHA_EXE_DIR . ($ENV{LD_LIBRARY_PATH} ? ":$ENV{LD_LIBRARY_PATH}" : '');

my $SHA_EXE = File::Spec->catdir($SHA_EXE_DIR, 'saphostexec');
my $SAPGENPSE = File::Spec->catfile($SHA_EXE_DIR, 'sapgenpse');

my $PSE = File::Spec->catfile($SEC_DIR, 'SAPSSLS.pse');
my $PSE_BACKUP = File::Spec->catfile($SEC_DIR, 'SAPSSLS.pse_backup');
my $CRED = File::Spec->catfile($SEC_DIR, 'cred_v2');
my $CRED_BACKUP = File::Spec->catfile($SEC_DIR, 'cred_v2_backup');

my $RESTART_SHA_CMD = "$SHA_EXE -restart 2>&1";
my $GET_SHA_STATUS_CMD = "$SHA_EXE -status 2>&1";

my ($passwd, $cn, $regenerate, $output) = ('', '', 'false', []);

sub isCorrectHostagent {
	return 1 if(-x $SAPGENPSE);
	push @{$output}, "Error: Incorrect version of SAP HostAgent installed - sapgenpse is not part of the distribution.";
	return 0;
}

sub createSecDir {
	if( -d $SEC_DIR || mkdir($SEC_DIR) ){
		if( _changeOwnerAndMode($UID, $GID, $SEC_DIR, 0750) ) {
			return 1;
		}
	}
	push @{$output}, "Error: Cannot create ${SEC_DIR}";
	return 0;
}

sub readMasterPassword {
	chomp($passwd = <STDIN>);
	return 1;
}

sub createPse {
    push @{$output}, "Info: Generating server PSE...";
    my $rc = 1;
    if(-e $PSE) {
        $rc = move($PSE, $PSE_BACKUP);
    }
    if($rc){
        my $command = $SAPGENPSE;
        my $args = ['gen_pse', '-p', 'SAPSSLS.pse', '-x', $passwd, $cn];
        my $executor = new LCM::ProcessExecutor($command, $args);
        my $exitCode = $executor->executeProgram();
        if( $exitCode == 0 ) {
            if( _changeOwnerAndMode($UID, $GID, $PSE, 0600) ) {
                push @{$output}, "Info: Done.";
                return 1;
            }
            push @{$output}, "Error: Failed to change owner/mode of server 'SAPSSLS.pse'.";
            return 0;
        }
        my $cmdOutput =$executor->getOutputLines();
        push @{$output}, @$cmdOutput;
        push @{$output}, "Error: Failed to generate server PSE. Exit code: '" . $exitCode . "'.";
        return 0;
    }
    push @{$output}, "Error: Failed to move existing 'SAPSSLS.pse' to 'SAPSSLS.pse_backup' .";
    return 0;
}

sub restartHostagent {
	push @{$output}, "Info: Restarting the SAP Host Agent (Command: '$RESTART_SHA_CMD')...";
	my @cmdOutput = `$RESTART_SHA_CMD`;
	if($? != 0){
		push @{$output}, @cmdOutput;
		my $exitCode = $?>>8;
		push @{$output}, "Error: Failed to restart the SAP Host Agent. Exit code : '" . $exitCode . "'.";
		return 0;
	}
	if(_waitForStartedSHA()){
		push @{$output}, "Info: Done.";
		return 1;
	}
	push @{$output}, "Error: The SAP Host Agent failed to start.";
	return 0;
}

sub grantAccess {
    push @{$output}, "Info: Granting PSE access to the sapadm user...";
    my $rc = 1;
    $rc = move($CRED, $CRED_BACKUP) if(-e $CRED);

    if($rc){
        my (undef, undef, $uid, $gid) = getpwnam("sapadm");
        my $command = $SAPGENPSE;
        my $args = ['seclogin', '-p', 'SAPSSLS.pse', '-x', $passwd, '-O', 'sapadm'];
        my $executor = new LCM::ProcessExecutor($command, $args, undef, undef, undef, $uid, $gid);
        my $exitCode = $executor->executeProgram();
        if($exitCode != 0){
            my $cmdOutput = $executor->getOutputLines();
            push @{$output}, @$cmdOutput;
            push @{$output}, "Error: Failed to grant PSE access to the sapadm user. Exit code : '" . $exitCode . "'.";
            return 0;
        } elsif( _changeOwnerAndMode($UID, $GID, $CRED, 0400) ) {
            push @{$output}, "Info: Done.";
            return 1;
        } else {
            push @{$output}, "Error: Failed to change owner/mode of 'cred_v2' file.";
            return 0;
        }
    }
    push @{$output}, "Error: Failed to move existing 'cred_v2' to 'cred_v2_backup'.";
    return 0;
}

sub needRegeneration {
	push @{$output}, "Info: Checking whether the server PSE needs to be regenerated...";

	if($FORCE_OPTION){
        push @{$output}, "Info: Force regeneration is specified.";
		$regenerate = 'true';
		return 1;
	}
	
	my $certOwner = getCertOwner();
	
	if($? == 0){
		if("x${cn}" eq "x${certOwner}"){
	        push @{$output}, "Info: No need of certificate generation.";
		} else {
	        push @{$output}, "Info: New certificate will be generated.";
			$regenerate = 'true';
		}
		return 1;
	}
	my $exitCode = $?>>8;
	push @{$output}, "Error: Failed to check if the server PSE needs to be regenerated. Exit code : '" . $exitCode . "'.";
	return 0;
}

sub getCertOwner {
	return '' if(!isCorrectHostagent());
	
	local %ENV = %ENV;
	$ENV{LD_LIBRARY_PATH} = $LD_LIBRARY_PATH;
	$ENV{LC_ALL}='en_US';
	$ENV{SECUDIR}=$SEC_DIR;
	$ENV{USER}='sapadm';

	#temporarily set uid & gid to sapadm's in order to execute certificates modification 
	my (undef, undef, $uid, $gid) = getpwnam("sapadm");
	local $> = $uid;
	local $) = $gid;
	my @outputOfCmd = `"${SAPGENPSE}" get_my_name -p SAPSSLS.pse -n Subject 2>&1 | grep '^Subject *: *' | sed 's/^Subject *: *//'`;
	if($? == 0 && defined $outputOfCmd[0]){
		chomp($outputOfCmd[0]);
		return $outputOfCmd[0];
	}
	return '';
}

sub main {
    my $newLine = $isWin ? "\r\n" : "\n";

	if($isWin){
		print "Info: Windows not supported.".$newLine;
		return 0;
	}
	
	GetOptions ('force' => \$FORCE_OPTION, 'restart_sha' => \$RESTART_SHA, 'hostname=s' => \$FQHOSTNAME);
	my $rc = 0;
	
	if(!$FQHOSTNAME) {
		push @{$output}, "Error: Hostname parameter is not set. Use -hostname option to set it.";
		$rc = 1;
	} else {
		$cn = "CN=${FQHOSTNAME}";
	    local %ENV = %ENV;
		$ENV{LD_LIBRARY_PATH} = $LD_LIBRARY_PATH;
		$ENV{LC_ALL}='en_US';
		$ENV{SECUDIR}=$SEC_DIR;
		
		$rc = _execute();
	}
	print $_ . $newLine for(@$output);
	return $rc;
}

sub _changeOwnerAndMode {
	my ($uid, $gid, $file, $mode) = @_;
	return 0 if(!chown($uid, $gid, $file));
	return 0 if(!chmod($mode, $file));
	return 1;
}

sub _waitForStartedSHA {
	for(1..20){
		push @{$output}, "Info: Waiting for the SAP Host Agent to start (Command: '$GET_SHA_STATUS_CMD')...";
		sleep(3);
		my @cmdOutput = `$GET_SHA_STATUS_CMD`;
		if($? == 0){
			push @{$output}, "Info: SAP Host Agent started successfully";
			return 1;
		}
	}
	return 0;
}

sub checkNeededRegenerationMain {
    my $newLine = $isWin ? "\r\n" : "\n";

    if ( $isWin ) {
        print "Info: Windows not supported." . $newLine;
        return ERROR_RC;
    }
    
    GetOptions ('hostname=s' => \$FQHOSTNAME);
    checkNeededRegeneration($FQHOSTNAME);
}

sub checkNeededRegeneration {
	($FQHOSTNAME) = @_;
	
	my $newLine = $isWin ? "\r\n" : "\n";
	if ( $isWin ) {
        print "Info: Windows not supported." . $newLine;
        return ERROR_RC;
    }
	
	if(!$FQHOSTNAME) {
        return ERROR_RC;
    }

    $cn = "CN=${FQHOSTNAME}";
        
	return ERROR_RC    if ( ! isCorrectHostagent() );
	return ERROR_RC    if ( ! needRegeneration() );
	    
	if ( $regenerate eq 'true' ) {
	   return NEED_REGENERATION;
	}

	return NO_NEED_REGENERATION;
}

sub _execute {
	
	return 1 if(!isCorrectHostagent());
	# INCORRECT VERSION OF SHA
	
	return 1 if(!createSecDir());
	# FAILED TO CREATE SECUDIR
	
	return 1 if (!needRegeneration());
	# FAILED TO CHECK IF REGENERATION IS NEEDED

	return 0 if($regenerate ne 'true');
	# SUCCESS NO NEED OF REGENERATION

	return 1 if(!readMasterPassword());

	return 1 if(!createPse());
	# FAILED TO CREATE PSE

	return 1 if(!grantAccess());
	# FAILED TO GRANT ACCESS
	
	return 1 if($RESTART_SHA && !restartHostagent());
	# FAILED TO RESTART SHA
	
	return 0;
	# SUCCESS
}

1;
