package LCM::SAPDSigner;
use parent 'SDB::Install::Base';
use warnings;
use strict;

use File::Spec;
use File::Basename;
use LCM::FileUtils qw(MyRealPath findInFile);
use LCM::ProcessExecutor;
use LCM::SapHostAgentFunctions;
use SDB::Install::SAPSystem qw(CollectSAPSystems);
use SDB::Install::Saphostagent qw( getSHAVersion getActiveSaphostexecDir getSaphostexecPath);
use LCM::App::ApplicationContext;
use LCM::VerificationError;
use File::stat;

my $SIGNATURE_NOT_FOUND_ERROR = "Cannot find signature file (SIGNATURE.SMF) in the component location.";
my $SIGNATURE_NOT_VALID_ERROR = "Either some files are corrupt or the signature has not been created with an official SAP certificate.";
my $SIGNATURE_DOES_NOT_MATCH_ERROR = "The found signature file (%s) seems to belong to another component.";
my $ERROR_MESSAGE_TEMPLATE = "Failed to authenticate component '%s'. %s.";

my $self = undef;

sub getInstance {
    return $self if (defined $self);

    my $isHostagentInstallationUsed = 0;
    my $sapdsignerPath = _getSAPDSignerFromHostangetInstallation();
    if (!$sapdsignerPath) {
        $sapdsignerPath = _getSAPDSignerFromInstalledSAPSystem();
    } else {
        $isHostagentInstallationUsed = 1;
    }

    if ($sapdsignerPath) {
        $self = bless({}, shift);
        $self->setIsHostagentInstallationUsed($isHostagentInstallationUsed);
        $self->setExecutablePath($sapdsignerPath);
    }

    return $self;
}

sub _getSAPDSignerFromHostangetInstallation {
    if (LCM::SapHostAgentFunctions->new()->isHostagentInstalled()) {
        my $sapdsignerPath = File::Spec->catfile(getActiveSaphostexecDir(), 'sapdsigner');
        my $sapdsignerExecutable = File::stat::stat($sapdsignerPath);
        if ($sapdsignerExecutable && -x $sapdsignerExecutable) {
            return $sapdsignerPath;
        }
    }
    return undef;
}

sub _getSAPDSignerFromInstalledSAPSystem {
    my $config = LCM::App::ApplicationContext->getInstance()->getConfiguration();
    my $sid = defined($config) && $config->can('getSID') ? $config->getSID() : undef;
    if ($sid) {
        my $systems = CollectSAPSystems(undef, 1);
        my $installedSystem = $systems->{$sid};
        return undef if !$installedSystem;

        my $sapdsignerPath = File::Spec->catfile($installedSystem->get_globalSAPHostAgentSetupDir(), 'sapdsigner');
        my $sapdsignerExecutable = File::stat::stat($sapdsignerPath);
        if ($sapdsignerExecutable && -x $sapdsignerExecutable) {
            return $sapdsignerPath;
        }
    }
    return undef;
}

sub authenticateComponentSet {
    my ($self, $components) = @_;
    $self->getMsgLst()->addProgressMessage("Verifying component authenticity...");
    $self->clearAuthenticationErrors();
    $self->clearVerifiedSignatures();
    $self->initProgressHandler($components);

    my $areAllComponentsAuthenticated = 1;
    for my $component (sort { $a->getComponentName() cmp $b->getComponentName() } @$components) {
        my $componentName = $component->getComponentName();
# This if handles the cockpit case when all cmps have a single manifest
        if ($self->isSignatureVerified($component)) {
            $self->setProgress((' ' x 2)."The signature file of component '$componentName' was already verified. Skipping authentication.");
            next;
        }
        $self->setProgress((' ' x 2)."Verifying '$componentName'...");

        if (!$self->authenticateComponent($component, \my $authenticationError)) {
            $self->addAuthenticationError(LCM::VerificationError->new($componentName, $authenticationError));
            $areAllComponentsAuthenticated = 0;
        }
    }
    return $areAllComponentsAuthenticated;
}

sub authenticateComponent {
    my ($self, $component, $authenticationError) = @_;
    my $signature = $component->getSignature();
    my $componentName = $component->getComponentName();
    if (! defined $signature) {
        $$authenticationError = $SIGNATURE_NOT_FOUND_ERROR;
        $self->getErrMsgLst()->addError(sprintf($ERROR_MESSAGE_TEMPLATE, $componentName, $$authenticationError));
        return 0;
    }
    my $signatureFile = MyRealPath($signature->getSignatureFilePath());
    my $rootVerifyPath = $self->_getRootSapdsignerVerifyPath($component);
    if (! defined $rootVerifyPath) {
        $$authenticationError = sprintf($SIGNATURE_DOES_NOT_MATCH_ERROR, $signatureFile);
        return 0;
    }

    my $args = ['-verify', MyRealPath($rootVerifyPath), '-manifest', $signatureFile, '-usemanifestfiles'];
    if (!$self->_executeSAPDSigner($args)) {
        $$authenticationError = $SIGNATURE_NOT_VALID_ERROR;
        $self->getErrMsgLst()->addError(sprintf($ERROR_MESSAGE_TEMPLATE, $componentName, $$authenticationError));
        return 0;
    }

    $signature->setIsValid(1);
    return 1;
}

sub _getRootSapdsignerVerifyPath {
    my ($self, $component) = @_;
    my $signature = $component->getSignature();
    my $signatureFile = $signature->getSignatureFilePath();
    my $componentPath = $component->getPath();
    my $componentName = $component->getComponentName();
    my $dirNameContainingComponentManifest = (grep {$_} File::Spec->splitdir($componentPath))[-1];
    my $manifestPathInSignatureLine = $signature->containsEntry(qr/$dirNameContainingComponentManifest\/+manifest$/, 1);
    my $manifestPathInSignature = (defined $manifestPathInSignatureLine && $manifestPathInSignatureLine =~ /(\S+\/+manifest$)/) ? $1 : undef;
    if (! defined $manifestPathInSignature) {
        $self->getErrMsgLst()->addError(sprintf($ERROR_MESSAGE_TEMPLATE, $componentName, "Could not find an entry for the component manifest in the file '$signatureFile'."));
        return undef;
    }
    my ($file, $signatureFileDir)  = fileparse(MyRealPath($signatureFile));
    my $dirContainingSignatureFile = ($signatureFileDir eq File::Spec->rootdir())
                                        ? $signatureFile
                                        : (grep {$_} File::Spec->splitdir($signatureFileDir))[-1];
    my $firstDirOfManifestPath     = (grep {$_} File::Spec->splitdir($manifestPathInSignature))[0];

# Layout 1: The path to the manifest file is relative to the directory of the SIGNATURE.SMF
# SIGNATURE.SMF located in   '<PATH>/'
# server manifest located in '<PATH>/SAP_HANA_DATABASE/server/manifest'
# SIGNATURE.SMF has path     'SAP_HANA_DATABASE/server/manifest'
#
# Exepcted root verify path: '<PATH>/SAP_HANA_DATABASE/'
#  - or -
# SIGNATURE.SMF located in   '<PATH>/SAP_HANA_DATABASE/'
# server manifest located in '<PATH>/SAP_HANA_DATABASE/server/manifest'
# SIGNATURE.SMF has path     'server/manifest'
#
# Exepcted root verify path: '<PATH>/SAP_HANA_DATABASE/'
    my $manifestPathRelativeToSignatureFileDir = File::stat::stat(File::Spec->catfile($signatureFileDir, $manifestPathInSignature));
    if (defined $manifestPathRelativeToSignatureFileDir) {
        return File::Spec->canonpath($signatureFileDir);
    }

# Layout 2: As in Layout 1 but the SIGNATURE.SMF has been moved inside the component directory (i.e. our 'Download and Extract' scenario does this for multiple components)
# SIGNATURE.SMF located in   '<PATH>/SAP_HANA_DATABASE/'
# server manifest located in '<PATH>/SAP_HANA_DATABASE/server/manifest'
# SIGNATURE.SMF has path     'SAP_HANA_DATABASE/server/manifest'
#
# Exepcted root verify path: '<PATH>' (i.e. One dir up from SIGNATURE.SMF location)
    if ($dirContainingSignatureFile eq $firstDirOfManifestPath) {
        my $candidateVerifyPath = dirname($signatureFileDir);
        if (File::stat::stat(File::Spec->catfile($candidateVerifyPath, $manifestPathInSignature))) {
            return File::Spec->canonpath($candidateVerifyPath);
        }
    }

# Layout 3: Exceptional case when SIGNATURE.SMF is expected to be inside the componend directory but has been moved one directory up.
# SIGNATURE.SMF located in   '<PATH>'
# server manifest located in '<PATH>/SAP_HANA_DATABASE/server/manifest'
# SIGNATURE.SMF has path     'server/manifest'
#
# Exepcted root verify path: '<PATH>/SAP_HANA_DATABASE/' (i.e. One dir up from manifest location)
    if ($dirNameContainingComponentManifest eq $firstDirOfManifestPath) {
        my $candidateVerifyPath = dirname($componentPath);
        if (File::stat::stat(File::Spec->catfile($candidateVerifyPath, $manifestPathInSignature))) {
            return File::Spec->canonpath($candidateVerifyPath);
        }
    }

    $self->getErrMsgLst()->addError(sprintf($ERROR_MESSAGE_TEMPLATE, $componentName, "Could not detect valid 'sapdsigner' verify path from '$signatureFile'"));
    return undef;
}

sub authenticateInstaller {
    my ($self, $installerComponent) = @_;
    return 0 if !$installerComponent;

    my $componentName = $installerComponent->getComponentName();
    my $signature = $installerComponent->getSignature();
    if (!$signature) {
        $self->getErrMsgLst()->addError(sprintf($ERROR_MESSAGE_TEMPLATE, $componentName, $SIGNATURE_NOT_FOUND_ERROR));
        return 0;
    }

    my $installPath   = MyRealPath(dirname($installerComponent->getPath()));
    my $signatureFile = MyRealPath($signature->getSignatureFilePath());
    my $fileListPath  = File::Spec->catfile($installPath, 'filelist.resident');
    my $args = [ '-verify', $installPath, '-manifest', $signatureFile, '-filelist', $fileListPath ];
    if (!$self->_executeSAPDSigner($args)) {
        $self->getErrMsgLst()->addError(sprintf($ERROR_MESSAGE_TEMPLATE, $componentName, $SIGNATURE_NOT_VALID_ERROR));
        return 0;
    }
    return 1;
}

sub _executeSAPDSigner {
    my ($self, $args) = @_;
    my $executablePath = $self->getExecutablePath();
    my $newWorkingDir = $self->isHostagentInstallationUsed() ? getActiveSaphostexecDir() : undef;
    my $processExecutor = LCM::ProcessExecutor->new($executablePath, $args, undef, $newWorkingDir);
    $processExecutor->setMsgLstContext($self->getMsgLstContext());
    return !$processExecutor->executeProgram();
}

sub setExecutablePath {
    my ($self, $path) = @_;
    $self->{executablePath} = $path;
}

sub getExecutablePath {
    my ($self) = @_;
    return $self->{executablePath};
}

sub setIsHostagentInstallationUsed {
    my ($self, $isHostagentInstallationUsed) = @_;
    $self->{isHostagentInstallationUsed} = $isHostagentInstallationUsed;
}

sub isHostagentInstallationUsed {
    my ($self) = @_;
    return $self->{isHostagentInstallationUsed};
}

sub getAuthenticationErrors {
    my ($self) = @_;
    return $self->{authenticationErrors} // [];
}

sub addAuthenticationError {
    my ($self, $errorObject) = @_;
    $self->{authenticationErrors} //= [];
    push(@{$self->{authenticationErrors}}, $errorObject);
}

sub clearAuthenticationErrors {
    my ($self) = @_;
    $self->{authenticationErrors} = [];
}

sub getVerifiedSignatures {
    my ($self) = @_;
    return $self->{verifiedSignatureSmfs} // [];
}

sub isSignatureVerified {
    my ($self, $component) = @_;
    my $signature = $component->getSignature();
    return $signature && $signature->isValid();
}

sub clearVerifiedSignatures {
    my ($self) = @_;
    $self->{verifiedSignatureSmfs} = [];
}

# setProgressHandler & setProgress are needed for GUI
# to update the SAPDSignerVerificationDialog guages
sub setProgressHandler {
    my ($self, $handler) = @_;
    $self->{progressHandler} = $handler;
}

sub getProgressHandler {
    my ($self) = @_;
    return $self->{progressHandler};
}

sub setProgress {
    my ($self, $message) = @_;
    $self->getMsgLst()->addProgressMessage($message);
    $self->{progressHandler}->addLine($message) if (defined $self->{progressHandler});
}

sub initProgressHandler {
    my ($self, $components) = @_;
    my $progressHandler = $self->getProgressHandler();
    return 1 if (!defined $progressHandler);
    return $progressHandler->InitProgress(scalar(@$components) + 1, 0);
}

1;
