#!perl 
#===============================================================================
#
# Copyright (C) 2000 - 2014 by SAP AG,  http://www.sap.com.
# All rights reserved.
#
# This software is the confidential and proprietary information of SAP AG.
# You shall not disclose such confidential information and shall use it only
# in accordance with the terms of the license agreement you entered into with SAP.
#
#       AUTHOR: i072928
#

=head1 NAME

ShellConfigurator

=head1 DESCRIPTION

Configures OS user shell files.
=cut

#===============================================================================
package LCM::ShellConfigurator;

use strict;
use base qw(SDB::Install::Base);
use SDB::Install::SysVars qw($path_separator);

sub new {
    my ($class, $osUserHomeDir) = @_;
    my $self = {};    
    bless ($self, $class);
    $self->{_osUserHomeDir} = $osUserHomeDir;
    return $self;
}

sub updateBashrcFile {
    my ( $self ) = @_;
    
    my $bashRcFile    = $self->{_osUserHomeDir} . $path_separator . '.bashrc';
    my $sourcePattern = 'source ~/.sapenv.sh';

    return undef unless $self->_updateShellConfigurationFile ( $bashRcFile, $sourcePattern );
    return 1;
}

sub updateCshrcFile {
    my ( $self ) = @_;
    
    my $cshRcFile     = $self->{_osUserHomeDir} . $path_separator . '.cshrc';
    my $sourcePattern = 'source ~/.sapenv.csh';

    return undef unless $self->_updateShellConfigurationFile ( $cshRcFile, $sourcePattern );
    return 1;
}

sub updateBashSapenvFile {
    my ( $self ) = @_;

    my $sapenvFile      = $self->{_osUserHomeDir} . $path_separator . '.sapenv.sh';
    my $hostnamePattern = "alias hostname='hostname -s'";
    my $pathPattern     = 'export PATH=${PATH}:/usr/sap/tools/bin';

    return undef unless $self->_updateShellConfigurationFile ( $sapenvFile, $hostnamePattern );
    return undef unless $self->_updateShellConfigurationFile ( $sapenvFile, $pathPattern );
    return 1;
}

sub updateCshSapenvFile {
    my ( $self ) = @_;
    
    my $sapenvFile      = $self->{_osUserHomeDir} . $path_separator . '.sapenv.csh';
    my $hostnamePattern = "alias hostname 'hostname -s'";
    my $pathPattern     = 'setenv PATH ${PATH}:/usr/sap/tools/bin';

    return undef unless $self->_updateShellConfigurationFile ( $sapenvFile, $hostnamePattern );
    return undef unless $self->_updateShellConfigurationFile ( $sapenvFile, $pathPattern );
    return 1;
}

sub _updateShellConfigurationFile {
    my ( $self, $confFile, $propertyPattern ) = @_;
    #
    # Check if the property is contained in the shell file.
    #
    my $rc = $self->_isShellFileConfigured ( $confFile, '^' . quotemeta( $propertyPattern ) . '$' );
    return $rc if not defined $rc;

    if ( $rc ) {
        return 1;
    }

    unless ( open( FH, '>>', $confFile ) ) {
        $self->setErrorMessage ( "Cannot open the $confFile file." );
        return undef;
    }
    print FH $propertyPattern . "\n";    # append pattern to the file
    close FH;

    return 1;
}

sub _isShellFileConfigured {
    my ( $self, $confFile, $propertyPattern ) = @_;

    unless ( open( FH, '<', $confFile ) ) {
        $self->setErrorMessage ( "Cannot open the $confFile file." );
        return undef;
    }

    my $propertyPatternFound = 0;
    while ( my $line = <FH> ) {
        if ( $line =~ /$propertyPattern/ ) {
            $propertyPatternFound = 1;
        }
    }
    close FH;
    return $propertyPatternFound;
}

1;