package LCM::SignatureFileChecker;

use strict;
use File::Spec;
use LCM::FileUtils qw(readFile);
use SDB::Install::System qw(isAdmin);
use SDB::Install::Globals qw($gSignatureManifestName $gProductNameEngine determineSignedManifestPath);
use parent 'SDB::Install::BaseLegacy';

my $MESSAGE_SAP_NOTE = "See SAP note 2078425, section 'How to prepare SAP HANA packages for installation or update' for instructions how to prepare packages for update or implications of ignoring this error.";
my $MESSAGE_OPTION_IGNORE = "You can ignore this error and continue with the execution by passing the ignore option 'check_signature_file' to hdblcm, started as root.";
my $MESSAGE_ERROR_MISSING_FILE = "File '$gSignatureManifestName' cannot be found in the $gProductNameEngine installation kit. To include the signature when extracting the installation kit from a .SAR archive, use the SAPCAR option: -manifest SIGNATURE.SMF";
my $TEMPLATE_ERROR_NON_READABLE = "File '%s' cannot be read from the effective user (%s). Check file permissions.";
my $TEMPLATE_ERROR_INVALID_FILE = "File '%s' does not belong to a $gProductNameEngine component.";

my $SEARCH_PATTERNS = [
	'hdblcm',
	'operations.d',
	'adapters.d',
	'descriptors.d',
	'instruntime',
];

sub new {
	my ($class, $candidateDirectory, $sapEnvironment) = @_;
	my $self = $class->SUPER::new();
	$self->_setCandidateDirectory($candidateDirectory);
    $self->_setSapEnvironment($sapEnvironment);
	return $self;
}

sub _isFaultTolerant {
    my ($self) = @_;
    return $self->_isSapEnvironment() && isAdmin();
}

sub checkFileExistence {
	my ($self) = @_;
	my $filePath = $self->_getFilePath();

	return 1 if(defined($filePath));
    
    my $message = $MESSAGE_ERROR_MISSING_FILE;
    if ($self->_isFaultTolerant()) {
        $message.=" $MESSAGE_SAP_NOTE";
        $self->getMsgLst()->AddWarning($message);
    } else {
        $message.=" $MESSAGE_OPTION_IGNORE $MESSAGE_SAP_NOTE";
        $self->PushError($message);
    }
    return 0;    
}

sub checkFilePermissions {
	require SDB::Install::NewDBUser;

	my ($self) = @_;
	my $filePath = $self->_getFilePath();
	my $currentUser = SDB::Install::User->new()->getname();

	return 1 if(defined($filePath) && (-r $filePath));

	$self->PushError(sprintf($TEMPLATE_ERROR_NON_READABLE, $filePath, $currentUser));
	return 0;
}

sub checkFileContent {
	my ($self) = @_;
	my $filePath = $self->_getFilePath();

	return 0 if(!defined($filePath));

	my $fileContents = readFile($filePath) // [];
	my @patterns = @{$SEARCH_PATTERNS};

	for my $line (@{$fileContents}){
		for(my $index = 0; $index < scalar(@patterns); ++$index){
			my $pattern = $patterns[$index];
			if($line =~ /$pattern/){
				splice(@patterns, $index, 1);
				last;
			}
		}
		return 1 if(scalar(@patterns) == 0);
	}

    my $message = sprintf($TEMPLATE_ERROR_INVALID_FILE, $filePath);
    if ($self->_isFaultTolerant()) {
        $message.=" $MESSAGE_SAP_NOTE";
        $self->getMsgLst()->AddWarning($message);
    } else {
        $message.=" $MESSAGE_OPTION_IGNORE $MESSAGE_SAP_NOTE";
        $self->PushError($message);
    }
    return 0;    
    
}

sub checkFile {
	my ($self) = @_;

    if (! $self->checkFileExistence()) {
        return $self->_isFaultTolerant()? 1 : 0;
    }
	return 0 if(! $self->checkFilePermissions());
    if (! $self->checkFileContent()) {
        return $self->_isFaultTolerant()? 1 : 0;
    }	
	return 1;
}

sub _setCandidateDirectory {
	my ($self, $directory) = @_;
	$self->{_candidateDirectory} = $directory;
}

sub _getCandidateDirectory {
	my ($self) = @_;
	return $self->{_candidateDirectory};
}

sub _setSapEnvironment {
    my ($self, $sapEnvironment) = @_;
    $self->{_sapEnvironment} = $sapEnvironment;
}

sub _isSapEnvironment {
    my ($self) = @_;
    return $self->{_sapEnvironment};
}

sub _getFilePath {
	my ($self) = @_;
	my $candidateDirectory = File::Spec->catfile($self->_getCandidateDirectory(), '..');
	my $signatureFilePath = determineSignedManifestPath($candidateDirectory);

	return (-f $signatureFilePath) ? $signatureFilePath : undef;
}

1;