package LCM::SlAnalytics::SLAMonitor;

use strict;
use File::Spec;
use LCM::Installer;
use LCM::FileUtils;
use SDB::Install::MsgLst;
use SDB::Install::SysInfo;
use SDB::Install::Manifest;
use LCM::Utils::CommonUtils;
use SDB::Install::NewDBUser;
use LCM::App::ApplicationContext;
use SAPDB::Install::Hostname qw(hostname);
use SDB::Install::System qw(loadSSLRequiringPackage);
use SDB::Install::SysVars qw($isWin $path_separator);
use LCM::HostsUtility qw(GetInstanceDirOnHostNotPartOfSystem);
use SDB::Install::Globals qw($gKeynameEngine $gLogDir $gProductNameInstaller $gTmpDir);

my %LMPROCESS_BY_CONFIGURATION = (
	"LCM::Configuration::GenericStackInstallConfiguration" => 'HDBLCM_INSTALL_SYSTEM',
	"LCM::Configuration::GenericStackUpdateConfiguration" => 'HDBLCM_UPDATE_SYSTEM', # If the server is not selected, it'll be 'HDBLCM_MANAGE_ADDITIONAL_COMPONENTS'
	"LCM::Configuration::GenericStackUninstallation" => 'HDBLCM_UNINSTALL_SYSTEM',
	"LCM::Configuration::GenericStackUpdateResidentConfiguration" => 'HDBLCM_MANAGE_ADDITIONAL_COMPONENTS_RESIDENT',
	"LCM::Configuration::SLD::RegistrationConfiguration" => 'HDBLCM_CONFIGURE_SLD',
	"LCM::Configuration::Hosts::AddHosts::AddRemoteHostsConfiguration" => 'HDBLCM_ADD_REMOTE_HOSTS',
	"LCM::Configuration::Hosts::AddHosts::AddLocalAndRemoteHostsConfiguration" => 'HDBLCM_ADD_LOCAL_AND_REMOTE_HOSTS',
	"LCM::Configuration::Hosts::RemoveHosts::RemoveHostsConfiguration" => 'HDBLCM_REMOVE_HOSTS',
	"LCM::Configuration::Hosts::RemoveHosts::Web::RemoveHostsConfiguration" => 'HDBLCM_REMOVE_HOSTS',
	"LCM::Configuration::RenameSystemConfiguration" => 'HDBLCM_RENAME_SYSTEM',
	"LCM::Configuration::InternalNetworkConfiguration" => 'HDBLCM_CONFIGURE_ISC',
	"LCM::Configuration::UpdateComponentListConfiguration" => 'HDBLCM_UPDATE_LM_STRUCTURE',
	"LCM::Configuration::PrintComponentListConfiguration" => 'HDBLCM_LIST_INSTALLED_COMPONENTS',
	"LCM::Configuration::UnregisterConfiguration" => 'HDBLCM_UNREGISTER_SYSTEM',
	"LCM::Configuration::AddHostRolesConfiguration" => 'HDBLCM_ADD_HOST_ROLES',
	# Only update host is not presented, coz it's not real scenario

	# Added for the WebUI scenarios
	"LCM::Configuration::Web::UpdateSystemConfiguration" => 'HDBLCM_UPDATE_SYSTEM',
	"LCM::Configuration::Web::UpdateComponentsConfiguration" => 'HDBLCM_MANAGE_ADDITIONAL_COMPONENTS_RESIDENT',
	"LCM::Configuration::Web::InternalNetworkConfiguration" => 'HDBLCM_CONFIGURE_ISC',
	"LCM::Configuration::Web::GenericStackUninstallation" => 'HDBLCM_UNINSTALL_ADDITIONAL_COMPONENTS',
	"LCM::Configuration::Web::AddHostRolesConfiguration" => 'HDBLCM_ADD_HOST_ROLES',
	);

my $slaMonitoringHasFinished;

my $usageType;
my $parameters;
my $startDate;
my $endDate;
my $lmProcess;
my $status;
my $statusMessage;

my $sid;
my $instanceNumber;
my $host;
my $hanaShared;
my $hdbServerVersion;
my $dbMode;
my $isMultiDb;

my $subtasks;

my $getTrexInstance = sub {
	my ($configuration) = @_;
    my $instance = $configuration->getOwnInstance();
    if(defined $instance){
    	return $instance;
    }
	# in case of add host & register rename
	my $target = $configuration->getDefault('Target');
	my $sid = $configuration->getDefault('SID');
	if (!defined $sid){
		return undef;
	}
	my $sapSidDir = $target . $path_separator . $sid;
    return undef if (!-d $sapSidDir);

	my $instanceDir = LCM::HostsUtility::GetInstanceDirOnHostNotPartOfSystem($configuration, $sapSidDir );
	if (defined $instanceDir){
		return new SDB::Install::SAPInstance::TrexInstance( $instanceDir, $sid );
	}
	return undef;
};

my $fillHdbSystemProperties = sub {
	my ($configuration) = @_;
    my $instance = $getTrexInstance->($configuration);
	if(defined $instance){
		# any resident or update scenario
		$host = $instance->get_host();
		$sid = $instance->get_sid();
		$instanceNumber = $instance->get_nr();
		$hdbServerVersion = $instance->get_version();
		$hanaShared = $instance->get_globalSidDir();
		$dbMode = $instance->getDbMode();
	}else{
		# in case of install scenario
		$host = $configuration->getValue('HostName');
		$sid = $configuration->getValue('SID');
		$instanceNumber = $configuration->getValue('InstanceNumber');
		$hanaShared = $configuration->getValue('Target') . $path_separator . $sid;
		$dbMode = $configuration->getValue('DBMode');

		my $mediumCmpMgr = $configuration->getComponentManager();
		my $serverCmp = $mediumCmpMgr->getComponentByKeyName($gKeynameEngine);
		$hdbServerVersion = $serverCmp->getVersion() if(defined $serverCmp);
	}

	$isMultiDb = (defined $dbMode && $dbMode eq 'multidb');

	if( $hdbServerVersion =~ m/^\w+\.\w+\.\w+\.\w+\.\w+$/ ){
		$hdbServerVersion =~ s/\.[^.]+$//g;
	}
	$host = hostname() unless(defined $host);
};

my $fillExecutionCallParameters = sub {
	my ($configuration) = @_;
	my $params = $configuration->{params};
	my $action = $configuration->getAction();
	my $result = "action=$action ";
	for my $paramId (keys %$params) {
		my $param = $configuration->{params}->{$paramId};
		my $paramType = $configuration->getType($paramId);
		my $value = $configuration->getValueOrBatchValue($paramId);
 #filter dummy params, hidden params, and passwords
		next if (!$value || ! exists $param->{opt} || $paramType =~ /passwd/ || $param->{hidden});
		my $currDefault = $configuration->getDefault($paramId);
		next if (defined $currDefault && ($value eq $currDefault));
		my $paramName = $param->{opt};
#special handling for params with map values
		if ($paramType eq 'mapping_list') {
			for my $key (@{$param->{origin_values}}) {
				my $mapValue = $value->{$key};
				my $defaultMapValue = $param->{default_map}->{$key};
				$result .= "$paramName=$key=$mapValue " if ($mapValue ne $defaultMapValue);
			}
			next;
		}
		$result .= "$paramName=$value ";
	}
	$result =~ s/\s+$//;
	$parameters = $result;
};

my $hasValidSlaData = sub {
	if(!defined $startDate || !defined $endDate || !defined $lmProcess || !defined $status || !defined $statusMessage){
		return 0;
	}
	if(!defined $sid || !defined $instanceNumber || !defined $host || !defined $hdbServerVersion || !defined $hanaShared){
		return 0;
	}
	return  1;
};

my $createElement = sub {
	my ($tag, $text) = @_;
	my $xmlElement = XML::LibXML::Element->new($tag);
	$xmlElement->appendTextNode($text) if defined $text;
	return $xmlElement;
};

my $createCDATAElement = sub {
	my ($tag, $text) = @_;
	my $xmlElement = XML::LibXML::Element->new($tag);
	my $data = XML::LibXML::CDATASection->new($text);
	$xmlElement->appendChild($data) if $text;
	return $xmlElement;
};

my $createApplicationSystemElement = sub {
	my $sysinfo = new SDB::Install::SysInfo ();

	my $applicationSystem = $createElement->('application_system');	

	my $applicationSystemItem = $createElement->('application_system_item');

	my $extSidName = $createElement->('extsidname', "$host:$sid" );	
	$applicationSystemItem->appendChild($extSidName);	

	my $computerSystem = $createElement->('computer_system');	

	my $computerSystemItem = $createElement->('computer_system_item');	

	if($isWin){
		my $version = $sysinfo->{'version'};
		if ($version =~ /NT\ 5\.0/){
			$version = '2000';
		} elsif ($version =~ /NT\ 5\.1/){
			$version = 'XP';
		} elsif ($version =~ /NT\ 5\.2/){
			$version = 'Server 2003';
		} elsif ($version =~ /NT\ 6\.0/){
			$version = 'Vista / Server 2008';
		} elsif ($version =~ /NT\ 6\.1/){
			$version = 'Windows 7 / Server 2008 R2';
		}	
		$computerSystemItem->appendChild($createElement->('opsys', $version));
		$computerSystemItem->appendChild($createElement->('opsysrelease', $sysinfo->{'subversion'}));		
	}else{
		my $linuxDistribution = $sysinfo->findLinuxDistribution();
		$computerSystemItem->appendChild($createElement->('opsys', $linuxDistribution->{'distri'}));	
		$computerSystemItem->appendChild($createElement->('opsysrelease', $linuxDistribution->{'revision'}));	
	}

	my $applicationSystemInstance = $createElement->('application_system_instance');

	my $applicationSystemInstanceItem = $createElement->('application_system_instance_item');

	$applicationSystemInstanceItem->appendChild($createElement->('instancenumber', $instanceNumber));	
	$applicationSystemInstanceItem->appendChild($createElement->('hostname', $host));	
	$applicationSystemInstanceItem->appendChild($createElement->('sapsystemname', $sid));	

	$applicationSystemInstance->appendChild($applicationSystemInstanceItem);

	$computerSystemItem->appendChild($applicationSystemInstance);	

	$computerSystem->appendChild($computerSystemItem);	

	$applicationSystemItem->appendChild($computerSystem);

	my $databaseSystem = $createElement->('database_system');	

	my $databaseSystemItem = $createElement->('database_system_item');	
	$databaseSystemItem->appendChild($createElement->('systemhome', $hanaShared));	
	$databaseSystemItem->appendChild($createElement->('dbtypeforsap', 'HDB'));	
	if(defined $hdbServerVersion){
		$databaseSystemItem->appendChild($createElement->('release', $hdbServerVersion));	
		$databaseSystemItem->appendChild($createElement->('patchlevel', $hdbServerVersion));	
	}
	$databaseSystemItem->appendChild($createElement->('dbname', 'SAP HANA database'));	

	$databaseSystem->appendChild($databaseSystemItem);	

	$applicationSystemItem->appendChild($databaseSystem);

	$applicationSystem->appendChild($applicationSystemItem);	 

	return $applicationSystem;
};

my $createStepElement = sub {
	my $subtask = shift();
	my $step = $createElement->('step');
	my $startedAt  = $subtask->getStartedAt();
	my $finishedAt = $subtask->getFinishedAt();
	my $runtime    = $subtask->getRuntime();
	$step->appendChild($createElement->('stepname', $subtask->getName()));
	$step->appendChild($createElement->('startdate',$startedAt))  if (defined $startedAt);
	$step->appendChild($createElement->('enddate',  $finishedAt)) if (defined $finishedAt);
	$step->appendChild($createElement->('runtime',  $runtime)) 	  if (defined $runtime);
	$step->appendChild($createElement->('stepstatus', $subtask->getStatus()->getState()));
	$step->appendChild($createElement->('modules'));

	return $step;
};

my $createSteps = sub {
	my $steps = $createElement->('steps');
	for my $subtask (@{$subtasks}) {
		$steps->appendChild($createStepElement->($subtask));
	}
	return $steps;
};

my $createGeneralProcessDetailsElement = sub {
	my $generalScenarioDetails = $createElement->('general_scenario_details');
	$generalScenarioDetails->appendChild($createElement->('lmprocess', $lmProcess));
	$generalScenarioDetails->appendChild($createElement->('startdate', $startDate));
	$generalScenarioDetails->appendChild($createElement->('enddate', $endDate));

	my $generalProcessDetails = $createElement->('general_process');
	my $run = $createElement->('process_item');	
	$run->appendChild($createElement->('runtime', $endDate - $startDate));	
	$run->appendChild($createElement->('processname', $lmProcess));	# Write WEB, CLI, GUI
	$run->appendChild($createElement->('processtype', "HDBLCM"));	# Write WEB, CLI, GUI
	$run->appendChild($createElement->('startdate', $startDate));	
	$run->appendChild($createElement->('enddate', $endDate));

	$run->appendChild($createElement->('runstatus', $status));# REMOVE IT

	my $specific_process_details = $createElement->('specific_process_details');
	my $specific_process_details_item = $createElement->('specific_process_details_item');
	$specific_process_details_item->appendChild($createElement->('runstatus', $status));
	$specific_process_details_item->appendChild($createElement->('runstatusmsg', $statusMessage));
	if($usageType){
		$specific_process_details_item->appendChild($createElement->('usagetype', $usageType));
	}else{
		$specific_process_details_item->appendChild($createElement->('usagetype', "INTERNAL"));
	}
	my $executionMode = LCM::App::ApplicationContext::getInstance()->getMode();
	$specific_process_details_item->appendChild($createElement->('execution_mode', $executionMode));
	$specific_process_details_item->appendChild($createElement->('multidb', ($isMultiDb == 1) ? 'true' : 'false'));
	$specific_process_details_item->appendChild($createCDATAElement->('parameters', $parameters));

	$specific_process_details->appendChild($specific_process_details_item);
	$run->appendChild($specific_process_details);
	if(defined $subtasks) {
		$run->appendChild($createSteps->());
	}

	$generalProcessDetails->appendChild($run);

	$generalScenarioDetails->appendChild($generalProcessDetails);
	return $generalScenarioDetails;
};

my $buildSlaXml = sub {
	# create SLA XML file
	my $analysis = $createElement->('analysis'); # root element
	$analysis->appendChild($createElement->('title','SAP HANA Database Lifecycle Management via HDBLCM'));
	$analysis->appendChild($createApplicationSystemElement->());
	$analysis->appendChild($createGeneralProcessDetailsElement->());
	$analysis->appendChild($createElement->('VERSION','1.0'));

	my $document = XML::LibXML::Document->new('1.0', 'utf-8');
	$document->setDocumentElement($analysis);

	return $document->toString(1);
};

my $generateFeedbackForm = sub {
	my $slaXmlAsString = shift();
	my $preparedSlaXmlAsString = $slaXmlAsString;
	if($preparedSlaXmlAsString !~ m/\n$/){
		$preparedSlaXmlAsString = $preparedSlaXmlAsString."\n";
	}
	$preparedSlaXmlAsString =~ s/[\r\n]+/\\n\\\n/g;
	$preparedSlaXmlAsString =~ s/'/\\'/g;

	my $feedbackFormTemplateFilePath = File::Spec->catfile($hanaShared, 'hdblcm', 'resources', 'templates', 'feedback.template');
	my $feedbackFormTemplateString = join('', @{LCM::FileUtils::readFile($feedbackFormTemplateFilePath)});
	my $feedbackFormHtmlContent = $feedbackFormTemplateString;
	$feedbackFormHtmlContent =~ s/\Q<![CDATA[!INCLUDE!]]>\E/$preparedSlaXmlAsString/;

	my $feedbackFormFilePath = File::Spec->catfile($hanaShared, 'hdblcm', 'resources', 'feedback', 'feedback.html');
	open(my $fh, '>', $feedbackFormFilePath) or die "Failed to open file '$feedbackFormFilePath' $!";
	if(!$isWin) {
		my $user = new SDB::Install::NewDBUser($sid);
		chown($user->uid(), $user->gid(), $feedbackFormFilePath);
		chmod(0700, $feedbackFormFilePath);
	}
	print $fh $feedbackFormHtmlContent;
	close $fh;
};

my $sendReport = sub {
	# create SLA xml report
	my $slaXmlAsString = $buildSlaXml->(); # root element

	# write it to file
	my $localSlaFilename = $sid . "_" . $host . "_" . $instanceNumber . "_" . $startDate . "_" . "analysis.xml";
	my $slaReportFilepath = File::Spec->catfile ($gLogDir, $localSlaFilename);
	open (my $FILE_DECRIPTOR, '>', $slaReportFilepath); 
	print $FILE_DECRIPTOR $slaXmlAsString;
	$FILE_DECRIPTOR->flush();
	close ($FILE_DECRIPTOR);

	# Internal usage
	if(LCM::Utils::CommonUtils::isSAPEnvironment()){
		# copy the file via FTP
		my $ftpShareTargetDirectory =  "LM_ANALYTICS_HDBLCM_INTERNAL"; # Internal usage if there is not env variable
		if(defined $usageType){
			if($usageType eq "OQ_DELTA" || $usageType eq "DELTA_OQ" || $usageType eq "OQ_ATOM" || $usageType eq "ATOM_OQ" || $usageType eq "OQ"){
				$ftpShareTargetDirectory =  "LM_ANALYTICS_HDBLCM_OQ";
			}else{
				$ftpShareTargetDirectory =  "LM_ANALYTICS_HDBLCM_OQ_DEV";
				if($usageType eq ""){
					$usageType = "DEV";
				}
			}
		}

		my $ftp = Net::FTP->new("SLANALYTICSFTP.wdf.sap.corp", Port=>21, Timeout=>30);
		$ftp->login('anonymous', '');
		$ftp->put($slaReportFilepath, sprintf("/%s/%s", $ftpShareTargetDirectory, $localSlaFilename));
		$ftp->quit();
	} else {
		# Customer usecase -> generate feedback form by the provided template
		$generateFeedbackForm->($slaXmlAsString);
	}
};

# Should never fail
sub start{
	eval {
		require XML::LibXML;
		$slaMonitoringHasFinished = 0;
		my ($configuration) = @_;
		if(!defined $configuration){
			return;
		}
		$lmProcess = $LMPROCESS_BY_CONFIGURATION{ref($configuration)};
		if(!defined $lmProcess){
			return;
		}
		if($lmProcess eq 'HDBLCM_UPDATE_SYSTEM' && !$configuration->{componentManager}->isHDBServerComponentSelected()){
			$lmProcess = 'HDBLCM_MANAGE_ADDITIONAL_COMPONENTS';
		}
		if($lmProcess eq 'HDBLCM_UPDATE_SYSTEM' && $configuration->isBootstrapFromResident()) {
			$lmProcess = 'HDBLCM_UPDATE_SYSTEM_RESIDENT'; # Different handling when the update is started via bootstrap from resident
		}
		$fillHdbSystemProperties->($configuration);
		$fillExecutionCallParameters->($configuration);
	    $usageType = $ENV{"HDBLCM_LM_ANALYTICS_USAGE"};
	    $startDate = time(); # The other detections should not affect the timestamp
		$endDate = undef; # reinitialize

		LCM::App::ApplicationContext::getInstance()->getApplication()->clearFeedbackForm();
		
        #load the Net::Ftp module for the reporting 
        if(LCM::Utils::CommonUtils::isSAPEnvironment()){
            my $messageList = new SDB::Install::MsgLst();
            if(!loadSSLRequiringPackage('Net::FTP', $messageList, $messageList)){
                die(${$messageList->getMsgLstString()} . "\n");
            }
        }
	};
	if($@){
		#do not fail
	}
}

# Should never fail
sub stop{
	eval {
		my ($executionExitCode, $executionStatusMessage, $tasks) = @_;
		$subtasks = $tasks;
		if(!defined $startDate || defined $endDate){
			# either the 'start' was called with invalid parameters (or not called at all)
			# or the stop has been called already 
			return;
		}
	    $endDate = time(); #convert to int
		# only 0 and undef is considered success,i.e. anything else is error
		if(!defined $executionExitCode || $executionExitCode == 0){
			$status = "SUCCESS";
			$statusMessage = "Successful execution";
		}else {
			$status = "ERROR";
			$statusMessage = "Failed execution";
		}
		$statusMessage = $executionStatusMessage if defined $executionStatusMessage;

		if($hasValidSlaData->()){
			# IMPORTANT: Generates SL Analytic report if there is enough system data
			$sendReport->();
	    }
	    $slaMonitoringHasFinished = 1;
	};
	if($@){
		#do not fail
	}
}

sub hasSLAMonitorFinished{
	return $slaMonitoringHasFinished;
}

1;
