package LCM::Slpp::ExecutionStep;

use strict;
use integer; # important due to the timestamps
use Time::HiRes qw(time);
use LCM::Slpp::ExecutionState qw(STATE_INITIAL STATE_FINISHED STATE_ERROR STATE_RUNNING);

sub new {
	my ($class, $name, $resourcesLogFilename, $executor) = @_;

	my $self = {
		name => $name,
		resourcesLogFilename => $resourcesLogFilename,
		executor => $executor,
		progress => 0,
		state => STATE_INITIAL
	};

	return bless($self, $class);
}

sub init {
	my ($self, $scenarioExecutor) = @_;
	$self->{scenarioExecutor} = $scenarioExecutor;

	my $executionStepInformationHash = {};
	$executionStepInformationHash->{resourcesLogFilename} = $self->{resourcesLogFilename};
	$executionStepInformationHash->{warnings} = [];
	$executionStepInformationHash->{progress} = $self->getProgress();
	$executionStepInformationHash->{state} = $self->getState();
	$executionStepInformationHash->{startedAt} = '';
	$executionStepInformationHash->{finishedAt} = '';

	$scenarioExecutor->sendTaskInfo($self->getName(), $executionStepInformationHash);
}

sub getScenarioExecutor {
	my ($self) = @_;
	return $self->{scenarioExecutor};
}

sub setProgress {
	my ($self, $newProgress) = @_;
	my $scenarioExecutor = $self->getScenarioExecutor();
	$self->{progress} = $newProgress;
	$scenarioExecutor->sendTaskInfo($self->getName(), $self->_updateExecutionStepsHash());
}

sub getProgress { return $_[0]->{progress}; }

sub setState {
	my ($self, $newState) = @_;
	$self->{state} = $newState;

	my $informationHash = $self->_updateExecutionStepsHash();
	if ($self->{state} eq STATE_FINISHED || $self->{state} eq STATE_ERROR){
		$informationHash->{logLocation} = $self->getLogLocation();
		$informationHash->{finishedAt} = int(time());
		for my $warningMessage (@{$self->getWarnings()}){
			push(@{$informationHash->{warnings}}, $warningMessage);
		}
	}
	$informationHash->{startedAt} = int(time()) if($self->{state} eq STATE_RUNNING);
	$self->getScenarioExecutor()->sendTaskInfo($self->getName(), $informationHash);
}

sub getLogLocation {
	my ($self) = @_;

	return undef if(! $self->{executor}->can("getLogLocation"));

	my $executorLogLocation = $_[0]->{executor}->getLogLocation();
#	sub parseLogFileLocation substitutes .log with .msg for the native GUI
	$executorLogLocation =~ s/msg$/log/g;
	return $executorLogLocation;
}

sub getState { return $_[0]->{state}; }

sub getName { return $_[0]->{name}; }

sub getWarnings {
	my ($self) = @_;
	return [] if(!$self->{executor}->can('getWarnings'));
	return $self->{executor}->getWarnings();
}

sub _updateExecutionStepsHash {
	my ($self) = @_;
	my $informationHash = $self->getScenarioExecutor()->{taskInformation}->{$self->getName()};
	$informationHash->{progress} = $self->getProgress();
	$informationHash->{state} = $self->getState();
	return $informationHash;
}

sub getNumberOfExpectedOutputLines {
	my ($self) = @_;
	return 5 if(!$self->{executor}->can('getNumberOfExpectedOutputLines')); # Consider 5 as the default
	return $self->{executor}->getNumberOfExpectedOutputLines();
}

1;