package LCM::Task;

use strict;
use parent qw (SDB::Install::Base Exporter);

use LCM::Task::TaskType;
use LCM::Task::TaskStatus;
use LCM::App::ApplicationContext;
use LCM::Task::TaskProgressHandler;
use SDB::Install::Globals qw ($gKeynameEngine getFlavourProductName);
use SDB::Install::Tools qw(formatSecondsIntoHhMmSs);

our $gFlavourProductName = 'FLAVOUR_PRODUCT_NAME';
our $gActionScope        = 'ACTIONSCOPE';
our @EXPORT = qw ($gFlavourProductName $gActionScope);

sub new{
	my ($class, $configuration) = @_;
	my $self = bless({}, $class);
	$self->_setConfiguration($configuration);
	$self->_setStatus(new LCM::Task::TaskStatus($self));
	$self->setSkipped(0);
	return $self;
}

sub getId{
	...
}

sub getName{
	...
}

sub getExecutionName {
	...
}

sub getDescription{
	return '';
}

sub getStepName(){
    my $self = shift;
    return $self->getId();
}

sub getStatus{
	return $_[0]->{_status};
}

sub getType{
	return LCM::Task::TaskType->STEP;
}

sub getStartedAt{
	return $_[0]->{_startedAt};
}

sub getFinishedAt{
	return $_[0]->{_finishedAt};
}

sub getRuntime {
	my $self = shift;
	my $startAt = $self->getStartedAt();
	my $finishAt = $self->getFinishedAt();
	if ((defined $startAt && defined $finishAt) && int($finishAt) >= int($startAt)) {
		return $finishAt - $startAt;
	}
	return undef;
}

sub getRuntimeInHhMmSs {
	my $self = shift;
	my $runtime = $self->getRuntime();
	return 'n/a' if (! defined $runtime);
	return formatSecondsIntoHhMmSs($runtime);
}

sub getLogLocation{
    my $self = shift;
    return $self->{_logLocation};
}

sub getProgress{
	return int($_[0]->{_progress});
}

sub getLastProgressMessage{
	my $self = shift();
	my $messages = $self->_getProgressMessages();
	my $messagesCount = scalar (@{$messages});
	if($messagesCount == 0) {
		return undef;
	}

	return $messages->[-1];
}

sub getWarningMessages{
	my $self = shift();
	if(!defined $self->{_warningMessages}) {
		$self->{_warningMessages} = [];
	}
	return $self->{_warningMessages};
}

sub _formatFlavourMessage {
    my($self, $message) = @_;
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    $message =~ s/\Q$gFlavourProductName\E/$flavourProductName/g;
    return $message;
}

sub _formatMessage {
    my ( $self, $message ) = @_;
    my $actionscope;
    my $configuration = $self->_getConfiguration();
    my $scope = $configuration->getValue('Scope');
    $actionscope = ! defined $scope ? 'System' : ucfirst($scope);
    $message =~ s/$gActionScope/$actionscope/g;
    return $self->_formatFlavourMessage($message);
}

sub hasWarningMessages{
	my $warningMessagesCount = scalar @{$_[0]->getWarningMessages()};
	return $warningMessagesCount != 0;
}

sub addListener{
	my ($self, $listener) = @_;
	my $listenersArray = $self->_getListeners();
	push(@{$listenersArray}, $listener);
}

sub isCritical{
	return 1;
}

sub hasParentTask{
	return defined $_[0]->{_parentTask};
}

sub getParentTask{
	return $_[0]->{_parentTask};
}

sub execute{
	my $self = shift();
	$self->_setStartedAt(time());
	$self->getStatus()->_setRunningState();
	$self->_executeInternal();
	$self->_setFinishedAt(time());
}

sub _executeInternal{
	...
}

sub isHidden{
	return 0;
}

sub shouldBeShownOnSummaryPage {
	return 1;
}

sub setSkipped{
    my ($self, $skip) = @_;
    $self->{_skip} = $skip;
}

sub isSkipped{
    my $self = shift;
    return $self->{_skip};
}

sub isResumable{
	return 0;
}

sub isComponentTask {
	return 0;
}

sub _getNumberOfExpectedOutputLines{
	...
}

sub _setStatus{
	my ($self, $status) = @_;
	$self->{_status} = $status;
	$self->_notifyTaskChanged();
}

sub _setStartedAt{
	my ($self, $startedAt) = @_;
	$self->{_startedAt} = $startedAt;
	$self->_notifyTaskChanged();
}

sub _setFinishedAt{
	my ($self, $finishedAt) = @_;
	$self->{_finishedAt} = $finishedAt;
	$self->_notifyTaskChanged();
}

sub _setProgress{
	my ($self, $progress) = @_;
	$self->{_progress} = $progress;
	$self->_notifyTaskChanged();
}

sub _setLogLocation{
    my ($self,$logLocation) = @_;
    $self->{_logLocation} = $logLocation;
}

sub _getConfiguration{
	return $_[0]->{_configuration};
}

sub _setConfiguration{
	my ($self, $configuration) = @_;
	$self->{_configuration} = $configuration;
}

sub _getInstalledServerComponent {
    my $self = shift();
    my $componentManager = $self->_getConfiguration()->getComponentManager();
    my $systemComponentManager = $componentManager->getSystemComponentManager();

    return $systemComponentManager->getComponentByKeyName($gKeynameEngine);
}

sub _getProgressMessages{
	my $self = shift();
	if(!defined $self->{_progressMessages}) {
		$self->{_progressMessages} = [];
	}
	return $self->{_progressMessages};
}

sub _addProgressMessage{
	my ($self, $message) = @_;
	if($self->hasParentTask()){
		$self->getParentTask()->_addProgressMessage($message);
	}
	my $messages = $self->_getProgressMessages();
	push(@{$messages}, $message);
	$self->_increaseProgress();
	$self->_notifyTaskChanged();
}

sub getFinalInfoMessage{
    return undef;
}

sub getFailedExecutionMessage {
	my ($self) = @_;
	return sprintf('%s failed', $self->getExecutionName());
}

sub _increaseProgress{
	my $self = shift();
	my $numberOfExpectedOutputLines = $self->_getNumberOfExpectedOutputLines();
	if ( $numberOfExpectedOutputLines == 0 ) {
		return;
	}
	
	my $newProgress = $self->{_progress} + 100 / $numberOfExpectedOutputLines;
	if($newProgress > 99) {
		$newProgress = 99;
	}
	
	$self->_setProgress($newProgress);
}

sub _addWarningMessage{
	my ($self, $message) = @_;
	my $messages = $self->getWarningMessages();
	push(@{$messages}, $message);
	$self->_notifyTaskChanged();
}

sub _getListeners{
	my $self = shift();
	if(!defined $self->{_listeners}) {
		$self->{_listeners} = [];
	}
	return $self->{_listeners};
}

sub _setParentTask{
	my ($self, $parentTask) = @_;
	$self->{_parentTask} = $parentTask;
}

sub _notifyTaskChanged{
	my $self = shift();
	for my $listener (@{$self->_getListeners()}){
		$listener->notifyTaskChanged($self);
	}
}

sub _parseLogFileLocation {
    my ($self, $outputLines) = @_;
    if (!defined $outputLines){
        return undef;
    }
    my $log_location;
    my $pattern = "Log file written to \'(.*?)\'";
    foreach my $line (reverse @$outputLines){
        ( $log_location ) = $line =~ /$pattern/i;
        
        if (defined $log_location) {
            $log_location =~ s/\.log/\.msg/g;
            return $log_location;
        }
    }
}

sub getSlppLogFileName{
	return undef;
}

sub initProgressHandler{
	my ($self) = @_;
	$self->getMsgLst()->setProgressHandler(new LCM::Task::TaskProgressHandler($self));
}

sub shouldPersistStatus {
	return 1;
}

1;
