package LCM::Task::CheckInstallationTask::CheckInstallationTask;

use strict;
use parent 'LCM::Task::SHASupportedTask';

use LCM::Task qw($gFlavourProductName);
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::System qw (isSidadmin);
use SDB::Install::RemoteHostctrlHosts;
use LCM::Task::TaskProgressHandler::CheckInstallationTaskProgressHandler;

my $TASK_NAME = "Check $gFlavourProductName System Installation";
my $PROGRESS_MESSAGE = "Checking $gFlavourProductName System installation ...";
my $SUCCESS_END_MESSAGE = "$gFlavourProductName System checked.";
my $FAIL_END_MESSAGE = "$gFlavourProductName System check failed.";
my $OPERATION = "hdblcm_check_system_v1";

sub getId {
    return 'check_installation';
}

sub getName {
    return $_[0]->_formatMessage($TASK_NAME);
}

sub getExecutionName {
    return $_[0]->_formatMessage($TASK_NAME);
}

sub _executeInternal {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    my $message = $self->getMsgLst()->addProgressMessage($self->_formatMessage($PROGRESS_MESSAGE));
    my $oldDepth = $progressHandler->getIntendationDepth();
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
    $progressHandler->setIntendationDepth($oldDepth + 1);

    if($isWin){
        $self->setErrorMessage ('Windows currently not supported.', undef);
        $self->getStatus()->_setErrorState();
    }

    my $rc = isSidadmin($self->getSID()) ? $self->_executeSHAOperation() : $self->_executeAsRoot();
    if($rc) {
        $self->getStatus()->_setFinishedState();
        $message->endMessage (undef, $self->_formatMessage($SUCCESS_END_MESSAGE));
    } else {
        $self->getStatus()->_setErrorState();
        $message->endMessage (undef, $self->_formatMessage($FAIL_END_MESSAGE));
    }

    $progressHandler->setIntendationDepth($oldDepth);
    $self->setMsgLstContext($saveContext);
}

sub _executeSHAOperation {
    my($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $optionsMap = $self->buildSHAOptionsMap();
    my $passwordKeys = ['Password'];

    my $localHost = $self->_createRemoteHostsHostctrlObject();
    my $exitCode = $localHost->executeHostctrlParallel($OPERATION, $configuration, undef, $passwordKeys, undef, undef, undef,
                                                        undef, undef, $optionsMap, undef, undef, undef, 1);
    my $outputBuffer = $localHost->getOutputBuffer();
    my $executionOutput = [ split('\n', $outputBuffer->[0]) ];
    $self->_setLogLocation($self->_parseLogFileLocation($executionOutput));

    if (!defined $exitCode || ($exitCode != 0)) {
        $self->setErrorMessage($FAIL_END_MESSAGE, $localHost->getErrMsgLst());
        $exitCode = undef;
    }

    return defined($exitCode) ? 1 : undef;
}

sub _createRemoteHostsHostctrlObject {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my @localHost = ($configuration->getLocalHanaHost());
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts(@localHost);

    $localExecutionHost->setOutputHandler($localHost[0], $self->getMsgLst()->getProgressHandler());
    $localExecutionHost->setMsgLstContext($self->getMsgLstContext());

    return $localExecutionHost;
}

sub _executeAsRoot {
    my ($self) = @_;
    my $executor = $self->_createProcessExecutor();
    my $saveCntxt = $self->getMsgLstContext();
    $executor->setMsgLstContext( $self->getMsgLstContext() );
    my $rc = $executor->executeProgram( 1 );
    $self->_setLogLocation($self->_parseLogFileLocation($executor->getOutputLines()));
    $self->setMsgLstContext( $saveCntxt );

    return 1 if (defined $rc && $rc == 0);

    my $errorMsg = "Error occurred while executing hdbcheck";
    $self->getErrMsgLst()->addError($errorMsg);
    return undef;
}

sub _createProcessExecutor {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $command = $self->_getExecutable();
    my $args = $self->buildArgs();
    my $executor;
    if ( $self->_isPasswordArgsExists() ){
        my $xmlPasswordStream = $configuration->getXmlPasswordStream();
        $executor = new LCM::ProcessExecutor( $command, $args, $xmlPasswordStream );
    } else {
        $executor = LCM::ProcessExecutor( $command, $args );
    }
    $executor->setOutputHandler( $self->getMsgLst()->getProgressHandler() );
    $executor->setProcessEnvironment (LCM::TraceLoggingEnvironment::PrepareProcessExecutorHdbEnvironment($self->_getExecutable()));
    return $executor;
}

sub buildSHAOptionsMap{
    my ($self) = @_;
    my $optionsMap = $self->SUPER::buildSHAOptionsMap();
    $optionsMap->{SID} = $self->getSID();

    return $optionsMap;
}

sub buildArgs{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $paramIdsToSkip = ['InstallSSHKey', 'PropertyFile', 'RemoteExecution', 'RootUser', 'Target', 'ScopeCheckTool', 'SID'];

    my $args = $configuration->getCmdLineArgsFromInstconfig(new SDB::Install::Configuration::HdbCheck(),
        $paramIdsToSkip, 'Server', 'hdbcheck');

    push(@{$args}, "-b");
    push(@{$args}, "--read_password_from_stdin=xml");

    my $propertyFilePath = $configuration->getValue('PropertyFile');
    if ($propertyFilePath) {
        push(@{$args}, "--property_file=$propertyFilePath");
    }
    my $installSshKeys = $configuration->getValue('InstallSSHKey');
    if ($installSshKeys) {
        push(@{$args}, "--install_ssh_key=$installSshKeys");
    }
    my $remoteExecution = $configuration->getValue('RemoteExecution');
    if ($remoteExecution) {
        push(@{$args}, "--remote_execution=$remoteExecution");
    }
    my $rootUser = $configuration->getValue('RootUser');
    if ($rootUser) {
        push(@{$args}, "--root_user=$rootUser");
    }
    my $scope = $configuration->getValue('ScopeCheckTool');
    if ($scope) {
        push(@{$args}, "--scope=$scope");
    }

    return $args;
}

sub _isPasswordArgsExists{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    return 1 if($configuration->getValue("Password"));
    return 1 if($configuration->getValue("RootPassword"));

    return 0;
}

sub _getExecutable {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $hdbExecutable =
           join($path_separator,
                $configuration->getSAPSystem()->get_globalTrexInstallDir(),
                'bin',
                'hdbcheck');

    return $hdbExecutable;
}

sub _getNumberOfExpectedOutputLines{
    return 100;
}

sub initProgressHandler {
    my ($self) = @_;
    $self->getMsgLst()->setProgressHandler(new LCM::Task::TaskProgressHandler::CheckInstallationTaskProgressHandler($self));
}

sub getParameterToOptionMapping {
    my($self) = @_;
    return {
        'ScopeCheckTool'    => 'SCOPE',
        'PropertyFile'      => 'PROP_FILE'
    };
}

1;