package LCM::Task::CommonTask::UpdateLocalHostTask;

use strict;
use parent qw (LCM::Task::SHASupportedTask Exporter);
use LCM::Task qw($gFlavourProductName);
use SDB::Install::System qw (isAdmin);
use LCM::Configuration::Hosts::UpdateHosts::UpdateLocalHostConfiguration qw($PARAMETER_TO_OPTION_MAPPING);
use SAPDB::Install::Hostname;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::SapHostAgentFunctions;
use File::Spec;
use LCM::HostsUtility;
use LCM::FileUtils;
use LCM::Utils::CommonUtils qw(getHdblcmUploadDirectory);
use SDB::Install::Saphostagent qw (setDatabaseProperties);
use SDB::Install::System qw (getSAPDrive);
use LCM::Task::TaskProgressHandler::UpdateHostProgressHandler;

our @EXPORT = qw ($TASK_ID_UPDATE_LOCAL_HOST);
our $TASK_ID_UPDATE_LOCAL_HOST = 'update_local_host';

my $SEC_DIR = File::Spec->catdir('/usr', 'sap', 'hostctrl', 'exe', 'sec');
my $PSE = File::Spec->catfile($SEC_DIR, 'SAPSSLS.pse');


sub new{
    my($class, $hostconfig) = @_;
    my $self = $class->SUPER::new($hostconfig);
	$self->_initTaskMessages();
	return $self;
}

sub initProgressHandler {
    my ($self) = @_;
    my $progressHandler = LCM::Task::TaskProgressHandler::UpdateHostProgressHandler->new($self);
    $self->getMsgLst()->setProgressHandler($progressHandler);
}

sub getId {
	return $TASK_ID_UPDATE_LOCAL_HOST;
}

sub getName {
	my $self = shift; 
    return $self->{_name};
}

sub getExecutionName {
	my $self = shift; 
	return $self->{_executionName};
}

sub _executeInternal {
    my $self = shift;
    my $taskProgressHandler = $self->getMsgLst()->getProgressHandler();
    my $msg;
    if(! isAdmin()){
        $msg = $self->getMsgLst()->addMessage($self->getExecutionName(). "...");
    } else {
        $msg = $self->getMsgLst()->addProgressMessage($self->getExecutionName(). "...");
    }
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst ()]);
     
    $taskProgressHandler->setIntendationDepth($taskProgressHandler->getIntendationDepth()+1);
    my $configuration = $self->_getConfiguration();
    my $exitCode = 1;
    $configuration->setMsgLstContext ([$self->getMsgLst ()]);

    if (! isAdmin()) {
       $exitCode = $self->_executeUpdateHostThroughSHAOnLocalHost();
    } else {
        my $rc;
        $rc = $self->executeComponentsRegistration($configuration);
        $exitCode = $exitCode && $rc;
        $rc = $self->updateSLDInforamation($configuration);
        $exitCode = $exitCode && $rc;
        $rc = $self->createHdblcmFileUploadRoot($configuration);
        $exitCode = $exitCode && $rc;


        if ($self->_shallRegenerateCertificates($configuration)) {
            $rc = $self->regenerateCertificates($configuration);
            $exitCode = $exitCode && $rc;
        } elsif (!$self->_getSHAHelper()->isHostagentInstalled()) {
        	$self->getMsgLst()->addMessage("Skipping regeneration of certificates. SAP Host Agent is not installed.");
        }

        if ($self->_shallDeployConfigurations($configuration)) {
            $rc = $self->deploySHAConfigurations($configuration);
            $exitCode = $exitCode && $rc;
        } elsif (!$self->_getSHAHelper()->isHostagentInstalled()) {
        	$self->getMsgLst()->addMessage("Skipping deployment of configurations. SAP Host Agent is not installed.");
        }

        if ($self->_shallSetSHADatabaseProperties($configuration)){
            $self->setSHADatabaseProperties ($configuration);
        }
        else{
            $self->getMsgLst()->addMessage("Skipping set of SAP Host Agents database properties.");
        }
    }

    if ($exitCode) {
    	$configuration->deletePersistenceFileIfNeeded();
    	$self->getStatus()->_setFinishedState();
        $self->_addUpdateHostWarningIfNeeded($configuration);
    } else {
    	$self->getStatus()->_setErrorState();
    	$self->setErrorMessage( $self->{_taskFailedMessage}, $self->getErrMsgLst());
    }
    $msg->endMessage (undef, $self->getStatus()->getMessage());
    $taskProgressHandler->setIntendationDepth($taskProgressHandler->getIntendationDepth()-1);
    $self->setMsgLstContext($saveCntxt);

    $self->persistStep();
}

sub _addUpdateHostWarningIfNeeded {
    my ($self, $configuration) = @_;
    my $persistenceFileName = $configuration->{persistenceFileName};
    
    my $isRenameConfiguration = $configuration->isa('LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationInRename');
    my $instance = ($isRenameConfiguration ? $configuration->getOwnInstanceAfterRename() : $configuration->getOwnInstance());
    my $allHostNames = $instance->get_allhosts();

    my @remainingHosts = ();
    for my $hostName (@$allHostNames) {
        my $hostDirPath = $instance->get_hostNameDir ($hostName);
        my $errlst = new SDB::Install::MsgLst();
        my $contents = LCM::FileUtils::listDirectory($hostDirPath, $errlst);
        if (! defined $contents) {
            my $errorMessage = $errlst->getMsgLstString();
            $self->_addWarningMessage("Could not scan for remaining status files: $$errorMessage", undef);
            next;
        }
        push(@remainingHosts, $hostName) if (grep(/$persistenceFileName/, @$contents));
    }
    my $numberOfRemainingHosts = scalar(@remainingHosts);
    if ($numberOfRemainingHosts) {
        my $hostsQuantity = ($numberOfRemainingHosts > 1) ? "hosts" : "host";
        my $hosts = join(', ', @remainingHosts);
        my $warning = "To finish the update of the $gFlavourProductName System, execute 'hdblcm --action=update_host' on $hostsQuantity $hosts.";
        $self->_addWarningMessage($self->_formatMessage($warning));
    }
}

sub _getNumberOfExpectedOutputLines{
	return 30;
}

sub _initTaskMessages{
    my($self) = @_;
    $self->{_executionName} = $self->_formatMessage("Updating $gFlavourProductName Instance Integration on Local Host");
    $self->{_name} = $self->_formatMessage("Update $gFlavourProductName Instance Integration on Local Host");
    $self->{_taskFailedMessage} = $self->_formatMessage("Update of $gFlavourProductName instance integration failed.");
    my $configuration = $self->_getConfiguration();
    if($configuration->shallRegisterComponents() && !$self->_shallRegenerateCertificates($configuration) 
         && !$configuration->shallUpdateSLD() && !$self->_shallDeployConfigurations($configuration)){
            my $componentsRegistration = $configuration->getValue('ComponentsRegistration');
            if($componentsRegistration =~ /^register/){
                $self->{_executionName} = $self->_formatMessage("Registering $gFlavourProductName Components on Local Host");
                $self->{_name} = $self->_formatMessage("Register $gFlavourProductName Components on Local Host");
                $self->{_taskFailedMessage} = $self->_formatMessage("Registering $gFlavourProductName components failed on local host");
            } else {
                $self->{_executionName} = $self->_formatMessage("Unregistering $gFlavourProductName Components on Local Host");
                $self->{_name} = $self->_formatMessage("Unregister $gFlavourProductName Components on Local Host");
                $self->{_taskFailedMessage} = $self->_formatMessage("Unregistering of $gFlavourProductName components failed on local host");
                
            }
    }
    $self->getStatus()->_setActionDoneMessage($self->{_name}." finished successfully");
    $self->getStatus()->_setActionFailedMessage($self->{_taskFailedMessage});
    $self->getStatus()->_setActionAbortedMessage($self->{_name}." aborted");
        
}

sub _shallRegenerateCertificates{
    my($self, $configuration) = @_;
    if (!$configuration->shallRegenerateCertificates()) {
       return 0;
    } 
    return $self->_getSHAHelper()->isHostagentInstalled();
}

sub _shallDeployConfigurations{
    my($self, $configuration) = @_;
    if (!$configuration->shallDeployConfigurations()) {
       return 0;
    } 
    return $self->_getSHAHelper()->isHostagentInstalled();
}

sub _shallSetSHADatabaseProperties{
    my($self, $configuration) = @_;
    if (!$configuration->shallSetDatabaseProperty() || !defined $configuration->getValue('SQLSysPasswd')){
        return 0;
    }
    return $self->_getSHAHelper()->isHostagentInstalled();
}


sub setSHADatabaseProperties{
    my($self, $configuration) = @_;
    my $isRenameConfiguration = $configuration->isa('LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationInRename');
    my $instance = ($isRenameConfiguration ? $configuration->getOwnInstanceAfterRename() : $configuration->getOwnInstance());
    setDatabaseProperties ($self->getMsgLst (), $instance->get_sid(), $instance->get_nr(), $instance->getSqlHost(), $configuration->getValue('SQLSysPasswd'), $configuration->getValue('SystemUser'),undef,undef,undef,1);
    return 1;
}

sub _executeUpdateHostThroughSHAOnLocalHost {
    my ($self) = @_;
    
    my @localHost = (hostname());
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts(@localHost);
    $localExecutionHost->resetMsgLstContext ();
    my $saveCntxt = $localExecutionHost->setMsgLstContext ($self->getMsgLstContext());
    $localExecutionHost->setOutputHandler($localHost[0],$self->getMsgLst()->getProgressHandler());
    my $rc =  $self->_executeUpdateHostThroughSHAOperation($localExecutionHost);
    $localExecutionHost->setMsgLstContext ($saveCntxt);
    return $rc;        
}

sub _executeUpdateHostThroughSHAOperation {
    my ($self, $localHost) = @_;

    my $configuration = $self->_getConfiguration();
    my $password = $configuration->getValue('Password');
    if (defined $password) {
        $localHost->useSidadm();
    }
    else{
        $localHost->useHostagentUser(); 
    }
    my $hostctrlOperation = 'hdblcm_update_host_v3';
    my $rc = 1;
    my $optionsMap = $self->buildSHAOptionsMap();
    my $exitCode = $localHost->executeHostctrlParallel($hostctrlOperation,
                                                    $configuration,
                                                    undef, # param IDs
                                                    ['Password','HostagentPassword','SQLSysPasswd'], # password IDs
                                                    undef, # remote ignore
                                                    undef, # remote timeout
                                                    "Updating local host...",
                                                    "Local host updated.",
                                                    "Updating local host failed!",
                                                    $optionsMap,
                                                    undef, # host option map
#                                                    [$currHost], # only on hosts
                                                    undef,                    
                                                    undef, # do not fail
                                                    0, # Suppress Console msgs
                                                    );

    $self->_setLogLocation($self->_parseLogFileLocation($localHost->getOutputBuffer()));
    if (!defined $exitCode || ($exitCode != 0)) {
      $self->setErrorMessage( "Updating host through SAP Host Agent failed.", $localHost->getErrMsgLst());
      $rc = undef;
    }
    return $rc;
}

sub executeComponentsRegistration {
    my ($self, $config) = @_;
    return 1 if(!$config->shallRegisterComponents());
    my $componentsRegistration = $config->getValue('ComponentsRegistration');
    my $installer = new LCM::Installer();
    my $sharedDir;
    my $sid;
    if($installer->isInstalled() && defined $installer->getOwnSapSystem()){
       $sharedDir = $installer->getOwnSapSystem()->getHanaInstPath();
       $sid = $installer->getSid();
    }
    else{
        if ($isWin){
            my $drive = getSAPDrive ($self->getErrMsgLst ());
            if (!defined $drive){
                return undef;
            }
            $sharedDir = join ($path_separator, $drive, 'usr', 'sap');
        }
        else{
            $sharedDir = $config->getValue('Target');
        }
        $sid = $config->getValue('SID');
        $config->detectSystemComponents();
    }

    if (!defined $sharedDir && defined $sid){
        # In SID rename case LCM::Installer class
        # cannot find its own installation after server rename.
        # Use getCollectSAPSystems($no_cache) as fallback
        # to get the renamed system.
        my $systems = $config->getCollectSAPSystems (1);
        if (defined $systems->{$sid}){
            $sharedDir = $systems->{$sid}->get_target ();
        }
    }

    my $skippedComponentString = $config->getValue('SkippedComponents');
    my $skippedComponents;
    if(defined $skippedComponentString){
        if(index($skippedComponentString,',') > 0){
            @$skippedComponents = split(',', $skippedComponentString);  
        }
        else{
            $skippedComponents = [$skippedComponentString];
        } 
    }
    if($componentsRegistration eq 'register'){
        require LCM::PostExecuteAction::RegisterComponents;
        my $registerAction = new LCM::PostExecuteAction::RegisterComponents($self,$config,1,$sharedDir,$sid,$skippedComponents);
        return $registerAction->execute();
    }
    if($componentsRegistration eq 'unregister'){
       require LCM::PostExecuteAction::UnregisterComponents;
       my $unregisterAction = new LCM::PostExecuteAction::UnregisterComponents($self,$config,1,$sharedDir,$sid,$skippedComponents);
       return $unregisterAction->execute();

    }
}

sub updateSLDInforamation {
    my ($self, $config) = @_;
    
    return 1 if(!$config->shallUpdateSLD());
    
    my $msg = $self->getMsgLst()->addProgressMessage("Updating component list location...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    my $rc = $self->_updateSLDInforamationIfNeeded($config);
    $msg->endMessage( undef, "Update of component list location" );
    $self->setMsgLstContext($saveCntxt);
    return $rc;
}

sub createHdblcmFileUploadRoot {
    my ($self, $configuration) = @_;

    return 1 if($isWin);

    my $installer = new LCM::Installer();
    my $sid = $configuration->getSID() || $installer->getSid();
    my $uploadDirectory = getHdblcmUploadDirectory($sid);

    return 1 if(-d $uploadDirectory);

    my $gid = getgrnam('sapsys');
    my $uid = getpwnam(sprintf('%sadm', lc($sid)));

    return createDirectory($uploadDirectory, 0750, $uid, $gid, 1, $self->getErrMsgLst());
}

sub _updateSLDInforamationIfNeeded {
    my ($self, $instconfig) = @_;
    my $hdbInstance = $instconfig->getOwnInstance();

    if(!defined $hdbInstance){
        $self->getErrMsgLst()->addMessage($self->_formatMessage("Unable to get $gFlavourProductName System Instance"));
        return undef;
    }
    
    my $installer = new LCM::Installer();
    my $pythonExecutor = new LCM::Python::HanaPropertiesHelper($hdbInstance, 0, $self->getMsgLst()->getProgressHandler());
    my ($uid, $gid) = $pythonExecutor->getPermissions($installer->getSid());
    my $isSldEnabled = $self->_checkForEnabledSLD($pythonExecutor, $uid, $gid);
    
    return 1 if(!$isSldEnabled);
    
    my $lmStructurePath = File::Spec->catdir($installer->getOwnSapSystem()->get_globalSidDir(), "lm_structure");
    my $isUpdateNeeded = $self->_checkIfUpdateOfLmStructurePathIsNeeded($pythonExecutor, $uid, $gid, $lmStructurePath);
    
    return 1 if(!$isUpdateNeeded);
    
    return $self->_updateLmStructurePath($pythonExecutor, $uid, $gid, $lmStructurePath);
}

sub _checkForEnabledSLD {
    my ($self, $pythonExecutor, $uid, $gid) = @_;
    my $msg = $self->getMsgLst()->addMessage("Checking for enabled SLD registration...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    $pythonExecutor->setMsgLstContext($self->getMsgLstContext());
    my $sldEnableValue = $pythonExecutor->getGlobalProperty('nameserver.ini', 'sld', 'enable', $uid, $gid);
    my $isSldEnabled = ($sldEnableValue =~ m/$bool_true_pattern/);
    
    if($isSldEnabled){
        $self->getMsgLst()->addMessage("SLD registration is enabled. Proceed with checking the value of 'lmStructurePath'");
    } else {
        $self->getMsgLst()->addMessage("SLD registration is not enabled. No need of changes");
    }
    
    $self->setMsgLstContext($saveCntxt);
    return $isSldEnabled;
}

sub _checkIfUpdateOfLmStructurePathIsNeeded {
    my ($self, $pythonExecutor, $uid, $gid, $realLmStructurePath) = @_;
    my $msg = $self->getMsgLst()->addMessage("Check if value of property 'lmStructurePath' needs update...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    $pythonExecutor->setMsgLstContext($self->getMsgLstContext());
    my $lmStructurePathValue = $pythonExecutor->getGlobalProperty('nameserver.ini', 'sld', 'lmStructurePath', $uid, $gid);
    my $isLmStructurePathChanged = ($realLmStructurePath ne $lmStructurePathValue);

    if($isLmStructurePathChanged){
        $self->getMsgLst()->addMessage("Current value '$lmStructurePathValue' differs from expected '$realLmStructurePath'");
        $self->getMsgLst()->addMessage("Value of property 'lmStructurePath' needs update");
    } else {
        $self->getMsgLst()->addMessage("Value of property 'lmStructurePath' does not need update");
    }
    
    $self->setMsgLstContext($saveCntxt);
    return $isLmStructurePathChanged;
}

sub _updateLmStructurePath {
    my ($self, $pythonExecutor, $uid, $gid, $newValue) = @_;
    my $msg = $self->getMsgLst()->addMessage("Updating value of property 'lmStructurePath'...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    $pythonExecutor->setMsgLstContext($self->getMsgLstContext());
    my $returnCode = $pythonExecutor->setGlobalProperty('nameserver.ini', 'sld', 'lmStructurePath', "$newValue", $uid, $gid);
    $self->setMsgLstContext($saveCntxt);
    return $returnCode;
}

sub deploySHAConfigurations {
    my ($self, $config) = @_;
    return 1 if(!$config->shallDeployConfigurations());
    my $hdbInstance = $config->getHdbOwnInstance();
    my $username = lc($hdbInstance->get_sid()) . 'adm';
    my $password = $config->getValue('Password');
    my $hosts = [$hdbInstance->get_host()];

    require LCM::SHA::OperationsDeployer;
    require LCM::SHA::DeploymentStrategy::DetectManagedObjects;

    my $deployer = new LCM::SHA::OperationsDeployer();
    $deployer->SetProgressHandler($self->getMsgLst()->getProgressHandler());
    my $strategy = new LCM::SHA::DeploymentStrategy::DetectManagedObjects($username, $password, $config->getValue('UseHttp'), $config->getValue ('SSOCertificate'));
    my $pathToSignatureManifest = File::Spec->catfile($hdbInstance->get_globalSidDir(), 'hdblcm', 'SIGNATURE.SMF');
    $strategy->setPathToSignatureManifest($pathToSignatureManifest);
    $deployer->setMsgLstContext([$self->getMsgLst()]);
    my $returnCode = $deployer->deployConfigurations($hosts, $strategy, 0, 0, 1);
    if(!defined($returnCode) || $returnCode){
        $self->_addWarningMessage('Deployment of SAP Host Agent configurations finished with errors');
        # remain fault tollerant, but show warnings
        return 1;
    }
    return 1;
}

sub regenerateCertificates {
    my ($self, $config) = @_;   
    my $certificatesHostmap = $config->getValue('CertificatesHostmap');
    my $localHostname = $config->{localHostname};    
    if(!defined $localHostname){
       $localHostname = hostname();
    }        
    my $localHostCertificate = $certificatesHostmap->{$localHostname};
    
    if ( ! defined $localHostCertificate ) {
        if ($self->_existsCertificatesOnLocalHost()) {
           $self->getMsgLst()->addMessage("Certificate hostname for local host is not defined. Skipping regeneration of local host certificate.");
           return 1;
        } else {
           $config->{params}->{CertificatesHostmap}->{value}->{$localHostname} = $localHostname;
        }
    }
    
    require LCM::PostExecuteAction::RegenerateCertificates::RegenerateCertificatesOnLocalHost;
    my $executor = new LCM::PostExecuteAction::RegenerateCertificates::RegenerateCertificatesOnLocalHost($self, $config, $config->getValue('Password'), 1);
    return $executor->execute(); 
}

sub _existsCertificatesOnLocalHost {
    if(-e $PSE) {
       return 1;
    } else {
       return 0;
    }
}

sub getParameterToOptionMapping{
    return $PARAMETER_TO_OPTION_MAPPING;
}

sub buildSHAOptionsMap{
	 my ($self) = @_;
	 my $optionsMap = $self->SUPER::buildSHAOptionsMap();
	 my $installer = new LCM::Installer();
	 my $sid;
	 if($installer->isInstalled()){
	 	$sid = $installer->getSid();
	 } else {
	 	$sid = $self->_getConfiguration()->getSID();
	 }
	 $optionsMap->{'SID'} = $sid;
	 return $optionsMap;
}

sub getSlppLogFileName {
	return 'update_localhost.log';
}

sub shouldBeShownOnSummaryPage {
	return 0;
}

sub _getSHAHelper {
	my ($self) = @_;
	if (!defined $self->{_shaHelper}) {
		$self->{_shaHelper} = new LCM::SapHostAgentFunctions(undef);
	}
	return $self->{_shaHelper};
}

sub shouldPersistFinishedStep {
    return 1;
}

1;
