package LCM::Task::ConfigureInternalNetwork::BaseInternalNetworkSubtask;

use strict;
use File::Spec;
use parent 'LCM::Task';
use LCM::ProcessExecutor;
use SDB::Install::SysVars qw($isWin);

my $CONFIGURATION_NAME = 'Inter-Service Communication';
my $PROGRESS_MESSAGE = sprintf("Configuring %s...", $CONFIGURATION_NAME);
my $ACTION_DONE_MESSAGE = sprintf("%s configured", $CONFIGURATION_NAME);
my $ACTION_DONE_WITH_WARNINGS_MESSAGE = sprintf('%s configured with warnings', $CONFIGURATION_NAME);
my $ACTION_FAILED_MESSAGE = sprintf("Configuration of %s failed", $CONFIGURATION_NAME);
my $ACTION_ABORTED_MESSAGE = sprintf("Configuration of %s aborted", $CONFIGURATION_NAME);
my $UNABLE_TO_GET_HDB = 'Unable to get HDB Instance';

sub new {
	my ($class, $configuration) = @_;
	my $self = $class->SUPER::new($configuration);
	my $status = $self->getStatus();

	$status->_setActionDoneMessage($ACTION_DONE_MESSAGE);
	$status->_setActionDoneWithWarningsMessage($ACTION_DONE_WITH_WARNINGS_MESSAGE);
	$status->_setActionFailedMessage($ACTION_FAILED_MESSAGE);
	$status->_setActionAbortedMessage($ACTION_ABORTED_MESSAGE);

	return $self;
}

sub getName {
	return sprintf('Configure %s', $CONFIGURATION_NAME);
}

sub getId {
	...
}

sub getExecutionName {
	return sprintf('Configuring %s', $CONFIGURATION_NAME);
}

# Override
sub getSlppLogFileName {
	return 'configure_internal_network.log';
}

# Override
sub _executeInternal {
	my $self = shift();
	my $configuration = $self->_getConfiguration();
	my $message = $self->getMsgLst()->addProgressMessage($PROGRESS_MESSAGE);
	my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
	my $progressHandler = $self->getMsgLst()->getProgressHandler();
	my $currentDepth = $progressHandler->getIntendationDepth();

	$progressHandler->setIntendationDepth($currentDepth + 1);

	if (!defined $self->_configure($configuration)) {
		$self->getStatus()->_setErrorState();
	} else {
		$self->getStatus()->_setFinishedState();
	}

	$progressHandler->setIntendationDepth($currentDepth);
	$message->endMessage (undef, $self->getStatus()->getMessage($self->_shallCreatePersistenceFile()));
	$self->setMsgLstContext($saveContext);
}

# Windows is currently not supported
sub _configure {
	my ($self, $configuration) = @_;

	if($isWin){
		$self->setErrorMessage('Windows currently not supported.');
		return undef;
	}
	my $hdbInstance = $configuration->getOwnInstance();
	if(!defined $hdbInstance){
		$self->setErrorMessage($UNABLE_TO_GET_HDB);
		return undef;
	}

	my $executor = $self->_createProcessExecutor($configuration, $hdbInstance);
	my $returnCode = $executor->executeProgram();

	$self->_setLogLocation($self->LCM::Component::parseLogFileLocation($executor->getOutputLines()));

	if (!defined $returnCode || $returnCode){
		my $hdbToolName = $self->_getHdbToolName();
		my $message = $self->setErrorMessage ("Execution of $hdbToolName failed");
		return undef;
	}
	return 1;
}

sub _createProcessExecutor {
	my ($self, $configuration, $hdbInstance) = @_;
	my $command = File::Spec->catfile($hdbInstance->get_globalTrexInstallDir(), 'bin', $self->_getHdbToolName());
	my $arguments = $self->_createHdbToolArgs($configuration);
	my $input = $configuration->getXmlPasswordStream($self->_getPasswordKeys());

	push(@{$arguments}, '--read_password_from_stdin=xml') if (defined $input);

	my $executor = new LCM::ProcessExecutor($command,$arguments, $input);
	$executor->setMsgLstContext($self->getMsgLstContext());
	$executor->setOutputHandler($self->getMsgLst()->getProgressHandler());

	return $executor;
}

sub _shallCreatePersistenceFile {
	...
}

sub _getHdbToolName {
	...
}

sub _createHdbToolArgs {
	...
}

sub _getPasswordKeys {
	...
}

sub _getNumberOfExpectedOutputLines {
	...
}

1;