package LCM::Task::DeployXSAComponentsTask;

use strict;
use parent 'LCM::Task::GenericStackTask::ComponentTask';
use File::Basename qw (dirname);
use SDB::Install::MsgLst;
use SDB::Install::NewDBUser;
use SDB::Install::System qw(isAdmin);
use SDB::Install::SysVars qw($isWin);
use LCM::Utils::ComponentsDependencySort qw(sortComponentsTopologically);
use SDB::Install::Globals qw($gShortProductNameXS2 $gLogDir $gFlavourPlatform);

my $PROGRESS_MESSAGE = 'Installing XS Advanced Components';

our @EXPORT = qw ($PROGRESS_MESSAGE);

sub new{
    my($class,$configuration,$component) = @_;
    my $self = shift->SUPER::new($configuration);
    $self->_addListeners();
    return $self;
}

sub getId {
    return 'install_xsa_components_task';
}

sub getName {
    return "Install XS Advanced Components";
}

sub getExecutionName {
    return $PROGRESS_MESSAGE;
}

sub getSlppLogFileName {
    return 'xs_install_components.log';
}

sub _getNumberOfExpectedOutputLines {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $componentManager = $configuration->getComponentManager();
    my @xs2Components = grep { $_->isComponentSelected() } @{$componentManager->getXs2ApplicationComponents()};

    return 30 * scalar(@xs2Components);
}

sub _executeInternal {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName() . '...');
    my $savedIsIntended = $progressHandler->setIntendationDepth(1);
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);

    LCM::ExecutionWarningsObservable->getInstance()->registerListener($self);

    if(! $self->_executeXsLogin($configuration)) {
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning("Installation of XS Advanced Components failed");
        $message->endMessage(undef, "Login into $gShortProductNameXS2 failed");
        $progressHandler->setIntendationDepth($savedIsIntended);
        $self->getStatus()->_setErrorState();
        $self->setMsgLstContext($saveContext);
        LCM::ExecutionWarningsObservable->getInstance()->unregisterListener($self);
        return;
    }

    my $failedDeployment = undef;
    my $deployedXs2Components;
    eval{
        ($failedDeployment, $deployedXs2Components) = $self->_deployXs2Components($configuration);
        $self->_generateProductInstallerLogFile($configuration);
    };

    if(! $self->_executeXsLogout($configuration)) {
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning("Failed to logout from $gShortProductNameXS2");
    }

    if($@ || defined $failedDeployment) {
        my $middle = length($@) > 0 ? '' : ' the following';
        my $suffix = length($@) > 0 ? "$@" : $failedDeployment->getComponentName();
        my $endMessage = sprintf('Installation of%s XS Advanced Components failed: %s', $middle, $suffix);
        LCM::ExecutionWarningsObservable->getInstance()->notifyWarning($endMessage);
        $message->endMessage(undef, $endMessage);
        if ( defined $failedDeployment ){
            my $topologyOrderedXs2Components = $self->getTopologyOrderedXs2Components($configuration, $deployedXs2Components);
            my $deployedXs2Applications = join (',', map { $_->getComponentBatchKey() } @{$topologyOrderedXs2Components});
            $configuration->setDeployedXs2Applications($deployedXs2Applications);
        }
        $self->getStatus()->_setErrorState();
    } else {
        $configuration->setDeployedXs2Applications();
        $self->getStatus()->_setFinishedState();
        $message->endMessage(undef, sprintf("%s finished", $self->getName()));
    }
    $self->persistStep();

    LCM::ExecutionWarningsObservable->getInstance()->unregisterListener($self);
    $progressHandler->setIntendationDepth($savedIsIntended);
    $self->setMsgLstContext($saveContext);
}

# Override
sub persistFinishedStep {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    $configuration->addToPersistedFinishedSteps('xs_components');
}

# Override
sub isSkipped {
    my $self = shift;
    my $configuration = $self->_getConfiguration();
    return 0 if ! defined $configuration;

    return 1 if $self->SUPER::isSkipped();
    return 0 if ! $configuration->can('getPersistedSteps');

    my $persistedSteps = $configuration->getPersistedSteps();
    if( defined $persistedSteps ) {
        return 1 if ( grep( /^xs_components/, split(',', $persistedSteps) ) );
    }

    return 0;
}

sub _executeXsLogin {
    my ($self, $configuration) = @_;
    my $apiUrl = $configuration->getApiUrl();
    my $userName = $configuration->getValue('OrgManagerUser');
    my $organisation = $configuration->getValue('OrgName');
    my $arguments = ['login', '-a', $apiUrl,'-u', $userName, '--stdin', '-o', $organisation, '-s', 'SAP'];

    if($apiUrl =~ /^https/){
        my $sid = $configuration->getSID();
        my $targetDirectory = $configuration->getValue('Target');
        my $certificateFilePath = File::Spec->catfile($targetDirectory, $sid, 'xs', 'controller_data', 'controller', 'ssl-pub', 'router', 'default.root.crt.pem');
        push(@{$arguments}, '--cacert');
        push(@{$arguments}, $certificateFilePath);
    }

    my $passwordInput = [ $configuration->getValue('OrgManagerPassword') ];
    my $executor = $self->_createProcessExecutor($configuration, $arguments, $passwordInput);

    $executor->setOutputHandler($self->getMsgLst()->getProgressHandler());
    my $exitCode = $executor->executeProgram();
    $self->getMsgLst ()->addMessage(undef, $executor->getMsgLst());

    if (!defined $exitCode || $exitCode) {
        my $errMsgLst = $executor->getErrMsgLst();
        my $errorMessage = "Cannot login into $gShortProductNameXS2";
        $self->appendErrorMessage($errorMessage, ($errMsgLst->isEmpty() ? undef : $errMsgLst));
        return undef;
    }
    return 1;
}

sub _generateProductInstallerLogFile {
    my ($self, $configuration) = @_;
    my $sid = $configuration->getSID();
    my $targetDirectory = $configuration->getValue('Target');
    my $xsPath = File::Spec->catfile($targetDirectory, $sid, 'xs', 'bin', 'xs');
    my $logFileName = File::Spec->catfile($gLogDir, $self->getSlppLogFileName());
    my ($uid, $gid) = (undef, undef);

    if (!$isWin && isAdmin()) {
        my $user = new SDB::Install::NewDBUser($sid);
        ($uid, $gid) = ($user->uid(), $user->gid());
    }

    my $message = $self->getMsgLst()->addMessage('Getting Product Installer logs...');
    my $arguments = ['-c', "'$xsPath' logs product-installer --last 1000 > '$logFileName'"];
    my $executor = new LCM::ProcessExecutor('/bin/bash', $arguments, undef, undef, undef, $uid, $gid);

    $executor->setMsgLstContext([$message->getSubMsgLst(), undef]);

    my $exitCode = $executor->executeProgram();
    if (!defined($exitCode) || $exitCode) {
        my $errMsgLst = $executor->getErrMsgLst();
        $self->getMsgLst()->addWarning('Failed to get the Product Installer logs', ($errMsgLst->isEmpty() ? undef : $errMsgLst));
        return;
    }
    $self->_setLogLocation($logFileName);
    $self->getMsgLst()->addMessage("Log file written to '$logFileName'");
}

sub _executeXsLogout {
    my ($self, $configuration) = @_;
    my $executor = $self->_createProcessExecutor($configuration, ['logout'], undef);

    $executor->setOutputHandler($self->getMsgLst()->getProgressHandler());
    my $exitCode = $executor->executeProgram();
    $self->getMsgLst ()->addMessage(undef, $executor->getMsgLst());

    if (!defined $exitCode || $exitCode) {
        my $errMsgLst = $executor->getErrMsgLst();
        my $errorMessage = "Cannot logout from $gShortProductNameXS2";
        $self->appendErrorMessage($errorMessage, ($errMsgLst->isEmpty() ? undef : $errMsgLst));
        return undef;
    }
    return 1;
}

sub _createProcessExecutor {
    my ($self, $configuration, $arguments, $stdin) = @_;
    my $sid = $configuration->getSID();
    my $targetDirectory = $configuration->getValue('Target');
    my $command = File::Spec->catfile($targetDirectory, $sid, 'xs', 'bin', 'xs');
    my ($uid, $gid) = (undef, undef);

    if (!$isWin && isAdmin()) {
        my $user = new SDB::Install::NewDBUser($sid);
        ($uid, $gid) = ($user->uid(), $user->gid());
    }
    return new LCM::ProcessExecutor($command, $arguments, $stdin, dirname($command), undef, $uid, $gid);
}

sub getTopologyOrderedXs2Components{
    my ($self, $configuration, $xs2ApplicationComponents) = @_;
    my $componentManager = $configuration->getComponentManager();
    my $xs2Components = $xs2ApplicationComponents;
    if( ! defined $xs2Components) {
        $xs2Components = [ grep { $_->isComponentSelected() } @{$componentManager->getXs2ApplicationComponents()} ];
    }

    return sortComponentsTopologically($componentManager, $xs2Components);
}

sub _deployXs2Components {
    my ($self, $configuration) = @_;
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    my $identationDepth = $progressHandler->getIntendationDepth();
    my $errorList = new SDB::Install::MsgLst();

    my @deployedXs2Components = ();
    my $persDeployedXs2Components = $configuration->getDeployedXs2Applications();
    for my $component (@{$self->getTopologyOrderedXs2Components($configuration)}){
        $component->setMsgLstContext([$self->getMsgLst(), $errorList]);
        $component->setProgressHandler($self->getMsgLst()->getProgressHandler());

        my $componentKey = $component->getComponentBatchKey();
        if( $persDeployedXs2Components =~ /$componentKey/){
            push @deployedXs2Components, $component;
            next;
        }
        
        my $executionResult = 1;
        if ($component->isUpdate()) {
            $executionResult = $component->updateComponent($configuration);
        } else {
            $executionResult = $component->installComponent($configuration);
        }
        $progressHandler->setIntendationDepth($identationDepth);
        if(!$executionResult){
            $self->getMsgLst()->addProgressMessage(sprintf("Installation of %s failed ", $component->getComponentName()));
            my $message = 'Installation of XS Advanced components failed';
            $self->appendErrorMessage($message, $errorList->isEmpty() ? undef : $errorList);
             return ($component, \@deployedXs2Components);
        }
        push @deployedXs2Components, $component;
        my $topologyOrderedXs2Components = $self->getTopologyOrderedXs2Components($configuration, \@deployedXs2Components);
        my $deployedXs2Applications = join (',', map { $_->getComponentBatchKey() } @{$topologyOrderedXs2Components});
        $configuration->setDeployedXs2Applications($deployedXs2Applications);
        $configuration->setStep(1);
    }
    return (undef, \@deployedXs2Components);
}

#implementing required interface by LCM::ExecutionWarningsObservable
sub onWarning {
	my ($self,$message) = @_;
	$self->_addWarningMessage($message);
}

sub isServerComponentTask {
    return 0;
}

sub getComponentKeyName{
    my $self = shift();
    return "XSApplication";
}

sub shouldPersistFinishedStep { return 0; }
sub getLogLocation{ return undef; }
sub getPhase{ return undef; }

1;