package LCM::Task::GenericStackProcessTask;

use strict;
use parent qw ( LCM::Task::SLProcessTask );

use LCM::Task::GenericStackTask::RemoveOutdatedStatusFiles;
use LCM::Task::GenericStackTask::PersistScopeInstanceValuesTask;
use LCM::Task::GenericStackTask::CollectHDBServerPlugins;
use LCM::Task::GenericStackTask::CollectReferenceDatas;
use LCM::Task::GenericStackTask::AddHostsTask;
use LCM::Task::GenericStackTask::ComponentTask::UninstallTask;
use LCM::Task::CommonTask::UpdateLocalHostTask;
use LCM::Task::CommonTask::UpdateRemoteHostsTask;
use LCM::Task::GenericStackTask::ServerUpdateTaskFailedHandler;
use LCM::Task::CommonTask::AssignAdditionalLocalRolesTask;
use LCM::Task::CommonTask::AssignAdditionalRemoteRolesTask;
use LCM::Task::DeployXSAComponentsTask;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::SkippedComponentsManager qw ( $SCENARIO_INSTALL );
use LCM::Component qw (PREPARE_PHASE);
use LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory;
use LCM::Task::GenericStackTask::ComponentTaskListFactory;
use SDB::Install::Globals qw ($gKeynameInstaller $gKeynameEngine $gSpecificHostsRolesRegex $gHostRoleXS2Worker $gFlavourPlatform);
use LCM::Task qw($gActionScope);
use experimental qw (smartmatch);

sub getComponentManager {
    my $self = shift();
    return $self->_getConfiguration()->getComponentManager();
}

sub getSelectedComponents {
    my $self = shift();
    return $self->getComponentManager()->getSelectedComponents();
}

sub isUpdate {
    ...
}

# Override
sub getSubtasks{
    my ($self) = @_;
    if(!defined $self->{_subtask}){
        $self->_createSubtasks();
    }
    return $self->SUPER::getSubtasks();
}

#
# Only protected/private subs bellow this line
#

sub _createSubtasks {
    my $self = shift();
    my $configuration = $self->_getConfiguration();

    if ($self->isUpdate()){
        $self->addSubtask(new LCM::Task::GenericStackTask::RemoveOutdatedStatusFiles($configuration));
    }

    my $taskList = $self->_createUninstallDeprecatedComponentsTasklist();
    for my $task (@{$taskList}){
        $self->addSubtask($task);
    }

    if ($self->isUpdate() && $configuration->isDistributedSystem() && $configuration->isScopeInstance()) {
        $self->addSubtask(new LCM::Task::GenericStackTask::PersistScopeInstanceValuesTask($configuration));
    }
    $self->addSubtask(new LCM::Task::GenericStackTask::CollectHDBServerPlugins($configuration));
    $self->addSubtask(new LCM::Task::GenericStackTask::CollectReferenceDatas($configuration));

    $taskList = createComponentTasklist($configuration,$self->getComponentManager()->getSelectedComponents());
    for my $task (@{$taskList}){
        $self->addSubtask($task);
    }

    $taskList = $self->_createRolesAndHostsTasklist();
    for my $task (@{$taskList}){
        $self->addSubtask($task);
    }

    $self->_addUpdateSystemHostsSteps();

    my $components = $self->getComponentManager()->getSelectedComponents();
    for my $component (@{$components}) {
        if($component->getComponentName() eq 'Component List') {
            $self->addSubtask(new LCM::Task::GenericStackTask::ComponentTask::ComponentListComponentTask($configuration, $component)) if($configuration->getPhase() ne PREPARE_PHASE);
        }        
    }

    if($self->_shallAddDeployXSAComponentsTask()){
        $self->addSubtask(new LCM::Task::DeployXSAComponentsTask($configuration));
    }
}

sub _createUninstallDeprecatedComponentsTasklist {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $taskList = [];
    return $taskList if $configuration->isPrepareUpdate(); 
    my $mcm = $self->getComponentManager();
	my $serverComponent = $mcm->getComponentByKeyName($gKeynameEngine);
	return $taskList if (!defined $serverComponent || !$serverComponent->isComponentSelected() || !$serverComponent->isUpdate());
	for my $component (@{$configuration->getDeprecatedComponentsForUninstall()}) {
		my $task = new LCM::Task::GenericStackTask::ComponentTask::UninstallTask($configuration, $component, 0);
		$task->setIndentationOffset(0);
		push (@{$taskList}, $task);
	}
	return $taskList;
}

sub _createRolesAndHostsTasklist {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $localRolesTask = $self->_createLocalRolesTask();
    my $remoteRolesTasklist = $self->_createRemoteRolesTasklist();
    my $addHostsTasklist = $self->_createAddHostsTasklist();

    my $taskList = [];

    my $xsControllerHost = $configuration->getXsControllerHostname();
    if ((! defined $xsControllerHost) || ($self->_isXsControllerLocalHost)) {
        if (defined $localRolesTask) {
            push (@{$taskList}, $localRolesTask);
        }
        for my $task (@{$remoteRolesTasklist}){
            push (@{$taskList}, $task);
        }
        for my $task (@{$addHostsTasklist}){
            push (@{$taskList}, $task);
        }
    } elsif ($self->_isXsControllerExistingRemoteHost()) {
        for my $task (@{$remoteRolesTasklist}){
            push (@{$taskList}, $task);
        }
        if (defined $localRolesTask) {
            push (@{$taskList}, $localRolesTask);
        }
        for my $task (@{$addHostsTasklist}){
            push (@{$taskList}, $task);
        }
    } elsif ($self->_isXsControllerHostNewHost()) {
        for my $task (@{$addHostsTasklist}){
            push (@{$taskList}, $task);
        }
        if (defined $localRolesTask) {
            push (@{$taskList}, $localRolesTask);
        }
        for my $task (@{$remoteRolesTasklist}){
            push (@{$taskList}, $task);
        }
    }

    return $taskList;
}

sub _createLocalRolesTask {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    if($self->_shallAssignAdditionalLocalRoles()) {
        my $task = new LCM::Task::CommonTask::AssignAdditionalLocalRolesTask($configuration);
        $task->setLocalHostRolesString($self->_getFilteredLocalRoles());
        return $task;
    } else {
        return undef;
    }
}

sub _filterRolesMapByHost {
    my ($self, $rolesMap, $host) = @_;
    my %filteredRolesMap;
    my @filtered_keys = [ grep {$_ ne $host} (keys %$rolesMap) ];
    @filteredRolesMap{@filtered_keys} = @$rolesMap{@filtered_keys};
    return \%filteredRolesMap;
}

sub _createHostsDetector {
    my ($self, $configuration) = @_;

    my $hdbInstance = $configuration->getOwnInstance();
    return undef if (!defined $hdbInstance);

    my $hostDetector = LCM::Hosts::HostsDetector->new(undef, $configuration->getMsgLstContext(), $hdbInstance);
    $hostDetector->_retrieveHostRolesFromIni($hdbInstance);

    return $hostDetector;
}

sub _createRemoteRolesTasklist {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $taskList = [];

    if($self->_shallAssignAdditionalRemoteRoles()) {
        my $autoAssignedRolesMap = $self->_getAutoAsignedRolesMap();
        my $rolesMap = $configuration->getValue('AddRoles') || {};
        my $hostsDetector = $self->_createHostsDetector($configuration);
        if(defined $hostsDetector){
            $autoAssignedRolesMap = $hostsDetector->filterExistingRoles($autoAssignedRolesMap);
            $rolesMap = $hostsDetector->filterExistingRolesInHostsAndRoles($self->_getAddHosts(), $rolesMap);
        }
        my @hosts = keys(%$rolesMap);
        if ($self->_shallSeparateRemoteRolesTask()) {
            my $xsControllerHost = $configuration->getXsControllerHostname();
            if ( exists $autoAssignedRolesMap->{$xsControllerHost} || exists $rolesMap->{$xsControllerHost} ){
                my $task1 = new LCM::Task::CommonTask::AssignAdditionalRemoteRolesTask($configuration);
                $task1->setXsControllerHostTask(1);
                $task1->setHosts([$xsControllerHost]);
                $task1->setAutoAsignedRolesMap($autoAssignedRolesMap || {});
                $task1->setRemoteRolesMap($rolesMap || {});
                push (@{$taskList}, $task1);
            }
            if( scalar($self->_filterRolesMapByHost($autoAssignedRolesMap, $xsControllerHost)) > 0 ||
                scalar($self->_filterRolesMapByHost($rolesMap, $xsControllerHost)) > 0 ){
                my $task2 = new LCM::Task::CommonTask::AssignAdditionalRemoteRolesTask($configuration);
                my $otherHosts = [ grep { (($_ ne $xsControllerHost) && (! LCM::HostsParser::IsLocalHost($_)) && ($rolesMap->{$_})) } @hosts ];
                $task2->setHosts($otherHosts);
                $task2->setAutoAsignedRolesMap($autoAssignedRolesMap || {});
                $task2->setRemoteRolesMap($rolesMap || {});
                push (@{$taskList}, $task2);
            }
        } else {
            my $remoteHosts = [ grep { ! LCM::HostsParser::IsLocalHost($_) } @hosts ];
            my $task = new LCM::Task::CommonTask::AssignAdditionalRemoteRolesTask($configuration);
            $task->setHosts($remoteHosts);
            $task->setAutoAsignedRolesMap($autoAssignedRolesMap || {});
            $task->setRemoteRolesMap($rolesMap || {});
            push (@{$taskList}, $task);
        }
    }

    return $taskList;
}

sub _shallSeparateRemoteRolesTask {
    my ($self) = @_;
    return $self->_isXsControllerExistingRemoteHost() && $self->_moreThanOneExistingRemoteHostsXsWorker();
}

sub _isXsControllerExistingRemoteHost {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $xsControllerHost = $configuration->getXsControllerHostname();
    return 0 if ((! defined $xsControllerHost) || LCM::HostsParser::IsLocalHost($xsControllerHost));

    my $isAutoAssignEnabled = $configuration->getValue('AutoAddXS2Roles') && ! $configuration->isSkipped('AutoAddXS2Roles');
    my $autoAssignedRolesMap = $isAutoAssignEnabled ? $configuration->getAutoAssignXs2RolesMap() : {};
    my $autoAssignedRole = $autoAssignedRolesMap->{$xsControllerHost};
    return 1 if (index($autoAssignedRole, $gHostRoleXS2Worker) != -1);

    my $rolesParameter = $configuration->{params}->{AddRoles};
    my $mapValues = $rolesParameter->{value};

    return 1 if (defined($mapValues->{$xsControllerHost}) && (index($mapValues->{$xsControllerHost}, $gHostRoleXS2Worker) != -1));

    return 0;
}

sub _moreThanOneExistingRemoteHostsXsWorker {
    my ($self) = @_;

    my $configuration = $self->_getConfiguration();
    my $xsControllerHost = $configuration->getXsControllerHostname();
    return 0 if (! defined $xsControllerHost);

    my $isAutoAssignEnabled = $configuration->getValue('AutoAddXS2Roles') && ! $configuration->isSkipped('AutoAddXS2Roles');
    my $autoAssignedRolesMap = $isAutoAssignEnabled ? $configuration->getAutoAssignXs2RolesMap() : {};
    foreach my $key (keys %$autoAssignedRolesMap) {
        return 1 if ($key ne $xsControllerHost);
    }

    my $rolesParameter = $configuration->{params}->{AddRoles};
    my $mapValues = $rolesParameter->{value};
    foreach my $key (keys %$mapValues) {
        next if LCM::HostsParser::IsLocalHost($key);
        return 1 if ($key ne $xsControllerHost);
    }

    return 0;
}

sub _createAddHostsTasklist {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $hostsDetector = $self->_createHostsDetector($configuration);
    my $allHosts = $self->_getAddHosts();
    my $taskList = [];

    if($self->_shallAddHosts()) {
        if ($self->_shallSeparateAddHostTask()) {
            my $xsControllerHost = $configuration->getXsControllerHostname();
            $allHosts = $hostsDetector->filterExistingHosts($allHosts) if(defined $hostsDetector);
            my @xsControllerHost = grep { (index($_, $xsControllerHost)!=-1) } @$allHosts;
            if (scalar(@xsControllerHost) > 0) {
                my $task1 = new LCM::Task::GenericStackTask::AddHostsTask($configuration);
                $task1->setXsControllerHostTask(1);
                $task1->setHosts(\@xsControllerHost);
                push (@{$taskList}, $task1);
            }
            my $task2 = new LCM::Task::GenericStackTask::AddHostsTask($configuration);
            my @otherHosts = grep { (index($_, $xsControllerHost)==-1) } @$allHosts;
            $task2->setHosts(\@otherHosts);
            push (@{$taskList}, $task2);
        } else {
            my $task = new LCM::Task::GenericStackTask::AddHostsTask($configuration);
            $allHosts = $hostsDetector->filterExistingHosts($allHosts) if(defined $hostsDetector);
            $task->setHosts($allHosts);
            push (@{$taskList}, $task);
        }
    }

    return $taskList;
}

sub _shallSeparateAddHostTask {
    my ($self) = @_;
    return $self->_isXsControllerHostNewHost() && $self->_moreThanOneNewXsWorkerHosts();
}

sub _isXsControllerHostNewHost {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $xsControllerHost = $configuration->getXsControllerHostname();
    return 0 if (! defined $xsControllerHost);
    my $addHosts = $configuration->getValue('AddHosts');
    my $isAddingHosts = defined($addHosts) && !$configuration->isSkipped('AddHosts');
    return 0 if(!$isAddingHosts);

    my $isAutoAssignEnabled = $configuration->getValue('AutoAddXS2Roles') && ! $configuration->isSkipped('AutoAddXS2Roles');
    my $filter = ".*";
    if (! $isAutoAssignEnabled) {
    	$filter = "role=($gHostRoleXS2Worker)";
    }

    my @hostsForAddition = $self->_filterListString($addHosts, $filter);

    foreach (@hostsForAddition) {
        return 1 if (index($_, $xsControllerHost)!=-1);
    }

    return 0;
}

sub _moreThanOneNewXsWorkerHosts {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $addHosts = $configuration->getValue('AddHosts');
    my $isAddingHosts = defined($addHosts) && !$configuration->isSkipped('AddHosts');
    return 0 if(!$isAddingHosts);

    my @hostsForAddition = $self->_filterListString($addHosts, "role=($gHostRoleXS2Worker)");
    return scalar(@hostsForAddition) > 1 ? 1 : 0;
}

sub _isXsControllerLocalHost {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $xsControllerHost = $configuration->getXsControllerHostname();
    return 0 if (! defined $xsControllerHost);
    return LCM::HostsParser::IsLocalHost($xsControllerHost);
}

sub _shallAddDeployXSAComponentsTask {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $componentManager = $configuration->getComponentManager();
    my @selectedXs2Components = grep { $_->isComponentSelected() } @{$componentManager->getXs2ApplicationComponents()};

    return ($configuration->getPhase() ne PREPARE_PHASE) && (scalar(@selectedXs2Components) > 0);
}

# Override
sub _executeInternal {
    my $self = shift() ;
    my $configuration = $self->_getConfiguration();
    $self->_initActionMessages();

    if($configuration->isCheckOnly()){
        $self->getStatus()->_setFinishedState();
        return;
    }

    if (!defined $self->{_subtask}) {
        $self->_createSubtasks();
    }

    $self->_addProgressMsg();
    $self->SUPER::_executeInternal();
}

sub _addUpdateSystemHostsSteps {
	my $self = shift;
	my $configuration = $self->_getConfiguration();
    $self->_initComponentsRegistrationManagers();
    
    if($self->_shallExecuteUpdateHostOnLocalHost()){
        $self->addSubtask(new LCM::Task::CommonTask::UpdateLocalHostTask($self->_createUpdateLocalHostConfiguration()));
    }
    if($self->_shallExecuteUpdateHostOnRemoteHosts()){
        $self->addSubtask(new LCM::Task::CommonTask::UpdateRemoteHostsTask($self->_createUpdateRemoteHostConfiguration()));
    }
}

sub _initComponentsRegistrationManagers {
    my($self) = @_;
    my $installedComponents;
    my $systemComponentManager = $self->_getConfiguration()->getSystemComponentManager();
    if(defined $systemComponentManager){
        $installedComponents = $systemComponentManager->getAllComponents();
    }
    my @components = @{$self->getSelectedComponents()};
    $self->{_localComponentRegistrationManager} = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager(\@components,$installedComponents,$SCENARIO_INSTALL);
    $self->{_remoteComponentRegistrationManager} = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager(\@components,$installedComponents,$SCENARIO_INSTALL);
}

sub _initActionMessages {
	...
}

sub _addProgressMsg {
	my ($self) = @_;

	my $selectedComponents = $self->getSelectedComponents();
	my $configuration = $self->_getConfiguration();
	my $existsComponentInPendingState;

	foreach my $cmp (@{$selectedComponents}) {
		if ($cmp->isInPendingState($configuration)) {
			$existsComponentInPendingState = 1;
			last;
		}
	}

	my $isUpdate = $self->isUpdate();
	my $progressMsg;

	if ($existsComponentInPendingState && !$configuration->getIgnore('check_pending_upgrade')) {
		my $action = ($isUpdate) ? 'update' : 'install';
		$progressMsg = 'Resuming ' . $action .  ' of components...';
	} else {
		$progressMsg =  ($isUpdate ? 'Updating' : 'Installing') . ' components...';
	}

	my $msg = $self->getMsgLst()->addProgressMessage($progressMsg);

	return 1;
}

sub _formatMessage {
    my ( $self, $message ) = @_;
    my $actionscope;
    my $isInstallingServer = $self->_isInstallingServer();
    if( !$self->_isInstallingServer() ){
        $actionscope = 'components';
        $message =~ s/$gActionScope/$actionscope/g;
    }
    return $self->SUPER::_formatMessage($message);
}

sub _isInstallingServer {
    my($self) = @_;
    my $serverComponent = $self->getComponentManager()->getComponentByKeyName($gKeynameEngine);
    return 0 unless(defined($serverComponent));
    return $serverComponent->isComponentSelected() && !$serverComponent->isUpdate();
}

sub _shallExecuteUpdateHostOnLocalHost {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    
    return 1 if($self->_shallDeploySHAConfigurations());
    return 0 if($self->_isOnlyHdblcmCmpSelected());
    return 1 if(!$configuration->isSkipped('CertificatesHostmap'));
    if(!$self->_shallSkipComponentsLocalRegistration()){
        return 1;
    }
    
    return 0;
}

sub _shallExecuteUpdateHostOnRemoteHosts {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    return 0 if($configuration->isScopeInstance());
    return 0 if(!$configuration->isDistributedSystem());
    return 1 if($self->_shallDeploySHAConfigurations(1)); # Perform check for the remote hosts
    return 0 if($self->_isOnlyHdblcmCmpSelected());

    return 1 if(!$configuration->isSkipped('CertificatesHostmap'));
    if(!$self->_shallSkipComponentsRemoteRegistration()){
        return 1;
    }

    return 0;
}

sub _shallDeploySHAConfigurations {
    require LCM::SapHostAgentFunctions;

    my ($self, $checkDeployOnRemoteHosts) = @_;
    my $configuration = $self->_getConfiguration();

    return 0 if ($configuration->shouldSkipHostagentCalls());

    my $shaHelper = new LCM::SapHostAgentFunctions(undef);

    my $isSHAForInstall = $configuration->getValue("InstallHostagent");
    my $isSHAInstalledOnLocalHost = $shaHelper->isHostagentInstalled();
    my $isHDBLCMSelected = scalar(grep {$_->getComponentKeyName() eq $gKeynameInstaller} @{$self->getSelectedComponents()});
    my $willSHABeInstalled = $isSHAInstalledOnLocalHost || $isSHAForInstall;

    if($checkDeployOnRemoteHosts){
        my $isAddingNewHosts = (! $configuration->isSkipped('AddHosts')) && $configuration->getValue('AddHosts');
        return ( ($isAddingNewHosts || $isHDBLCMSelected) && $willSHABeInstalled );
    }
    return ( $isHDBLCMSelected && $willSHABeInstalled );
}

sub _isOnlyHdblcmCmpSelected {
    my ($self) = @_;
    my @components = @{ $self->getSelectedComponents() };
    if ((scalar (@components) == 1) and ($components[0]->getComponentKeyName() eq $gKeynameInstaller)) {
        return 1;
    }
    return 0;
}

sub _shallSkipComponentsRemoteRegistration {
     my($self) = @_;
     return ! $self->_getRemoteComponentRegistrationManager()->existRegistrableComponents();
}

sub _shallSkipComponentsLocalRegistration {
     my($self) = @_;
     return ! $self->_getLocalComponentRegistrationManager()->existRegistrableComponents();
}

sub _getRemoteComponentRegistrationManager {
	my $self = shift;
	return $self->{_remoteComponentRegistrationManager};
}

sub _getLocalComponentRegistrationManager {
    my $self = shift;
    return $self->{_localComponentRegistrationManager};
}

sub _createUpdateLocalHostConfiguration {
    my($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $shallRegenerateCertificates = 0;
    if(!$configuration->isSkipped('CertificatesHostmap') && !$configuration->shouldSkipHostagentCalls()){
        $shallRegenerateCertificates = 1;
    }
    my $componentRegistration;
    if(!$self->_shallSkipComponentsLocalRegistration()){
       $componentRegistration = 'register'; 
       $self->_prepareSkippedComponentsForLocalRegistration();
    }
    my $skippedComponents = $self->_getSkippedComponentsForLocalRegistration();
    my $shallDeployConfigurations = $self->_shallDeploySHAConfigurations();

    my $paramHash = {
        ComponentsRegistration => $componentRegistration,
        SkippedComponents => $skippedComponents,
        GenerateCertificates => $shallRegenerateCertificates,
        DeployShaConfigurations => $shallDeployConfigurations,
        UpdateComponentListLocation => 0,
    };
    return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createLocalConfiguration($configuration, $paramHash);
}

sub _createUpdateRemoteHostConfiguration {
    my($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $shallRegenerateCertificates = 0;
    if(!$configuration->isSkipped('CertificatesHostmap') && !$configuration->shouldSkipHostagentCalls()){
        $shallRegenerateCertificates = 1;
    }
    my $componentRegistration ;
    if(!$self->_shallSkipComponentsRemoteRegistration()){
       $componentRegistration = 'register'; 
       $self->_prepareSkippedComponentsForRemoteRegistration();
    }
    my $skippedComponents = $self->_getSkippedComponentsForRemoteRegistration();
    my $shallDeployConfigurations = $self->_shallDeploySHAConfigurations(1); # Perform check for the remote hosts
    my $remoteHostsObjects = $configuration->getRemoteHostsObjects();

    my $paramHash = {
        ComponentsRegistration => $componentRegistration,
        SkippedComponents => $skippedComponents,
        GenerateCertificates => $shallRegenerateCertificates,
        DeployShaConfigurations => $shallDeployConfigurations,
        UpdateComponentListLocation => 0,
    };
    return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createRemoteConfiguration($configuration, $remoteHostsObjects, $paramHash);
}

sub _prepareSkippedComponentsForLocalRegistration {
    my($self) = @_;
    $self->_getLocalComponentRegistrationManager()->detectComponents();
}

sub _prepareSkippedComponentsForRemoteRegistration {
    my($self) = @_;
    $self->_getRemoteComponentRegistrationManager()->detectComponents();
}

sub _getSkippedComponentsForLocalRegistration {
    my($self) = @_;
    return $self->_getLocalComponentRegistrationManager()->getSkippedComponents();
}

sub _getSkippedComponentsForRemoteRegistration {
    my($self) = @_;
    return $self->_getRemoteComponentRegistrationManager()->getSkippedComponents();
}

sub _shallAssignAdditionalLocalRoles {
    return $_[0]->_getFilteredLocalRoles() ? 1 : 0;
}

sub _getFilteredLocalRoles {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $localRolesString = $configuration->getAddLocalRolesString();
    return undef if ! defined $localRolesString || $localRolesString eq '';

    my $hostsDetector = $self->_createHostsDetector($configuration);
    return $localRolesString if ! defined $hostsDetector;

    return $hostsDetector->filterExistingLocalRoles($localRolesString);
}

sub _getAutoAsignedRolesMap {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $isAutoAssignEnabled = $configuration->getValue('AutoAddXS2Roles') && ! $configuration->isSkipped('AutoAddXS2Roles');
    return $isAutoAssignEnabled ? $configuration->getAutoAssignXs2RolesMap() : {};
}

sub _shallAssignAdditionalRemoteRoles {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    return 0 if($configuration->isScopeInstance());
    return 0 if(!$configuration->isDistributedSystem());

    my $autoAssignedRolesMap = $self->_getAutoAsignedRolesMap();
    my $rolesMap = $configuration->getValue('AddRoles') || {};

    my $hostsDetector = $self->_createHostsDetector($configuration);
    if(defined $hostsDetector){
        $autoAssignedRolesMap = $hostsDetector->filterExistingRoles($autoAssignedRolesMap);
        $rolesMap = $hostsDetector->filterExistingRolesInHostsAndRoles($self->_getAddHosts(), $rolesMap);
    }

    for my $host (keys %$rolesMap) {
        my $isLocalHost = LCM::HostsParser::IsLocalHost($host);
        return 1 if($rolesMap->{$host} && ! $isLocalHost);
        return 1 if(length($autoAssignedRolesMap->{$host}) && ! $isLocalHost);
    }
    return 0;
}

sub _getAddHosts {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $addHosts = $configuration->getValue('AddHosts');
    my $isAddingHosts = defined($addHosts) && !$configuration->isSkipped('AddHosts');
    my @hostsForAddition = $isAddingHosts ? $self->_filterListString($addHosts, $configuration->getAddHostsFilterRegex()) : ();
    return scalar(@hostsForAddition) > 0 ? \@hostsForAddition : undef;
}

sub _shallAddHosts {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $hostsForAddition = $self->_getAddHosts();
    return 0 if ! defined $hostsForAddition;

    my $hostsDetector = $self->_createHostsDetector($configuration);
    if(defined $hostsDetector){
        $hostsForAddition = $hostsDetector->filterExistingHosts($hostsForAddition);
        return 0 if scalar(@$hostsForAddition) == 0;
    }

    return 1;
}

sub _filterListString {
    my ($self, $listStr, $filter) = @_;
    my @list = split(/\s*,\s*/, $listStr);
    my @filteredList = grep(/$filter/, @list);
    return @filteredList;
}

sub _startSLAnalyticsMonitor {
    my $self = shift();

# Do not start the SLA Monitor for HDBLCM self-update 
    if(! $self->_isOnlyHdblcmCmpSelected()) {
        $self->SUPER::_startSLAnalyticsMonitor();
    }
}

sub _stopSLAnalyticsMonitor {
    my ($self, $executionStatusMessage) = @_;

# SLA Monitor wasn't started in case of HDBLCM self-update
    if(! $self->_isOnlyHdblcmCmpSelected()) {
        $self->SUPER::_stopSLAnalyticsMonitor($executionStatusMessage);
    }
}

sub getFinalInfoMessage{
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $hostname      = $configuration->getValue('HostName');

    my $componentManager = $self->getComponentManager();
    if ( $self->getStatus()->isInFinishedState() && $componentManager->getHANAFlavour() ne $gFlavourPlatform ) {
        my $cockpitConfigurationUtilityUrl = $configuration->{'cockpit-admin-web-app'};
        my $cockpitMonitoringConsoleUrl = $configuration->{'cockpit-web-app'};
        my $finalInfoMessage = ( defined $cockpitMonitoringConsoleUrl ? "Launch SAP HANA cockpit by opening $cockpitMonitoringConsoleUrl\n" : '' );
        $finalInfoMessage .= ( defined $cockpitConfigurationUtilityUrl ? "Launch SAP HANA cockpit manager by opening $cockpitConfigurationUtilityUrl\n" : '' );
        return $finalInfoMessage if $finalInfoMessage ne '';
    }
    return undef;
}

1;
