package LCM::Task::GenericStackTask::AddHostsTask;

use strict;
use parent qw (LCM::Task::SHASupportedTask);
use SDB::Install::Globals qw ($gLogDir);
use SDB::Install::System qw (isSidadmin);
use SDB::Install::RemoteHostctrlHosts;
use SAPDB::Install::Hostname;
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::ProcessExecutor;
use LCM::Utils::CommonUtils qw(getHdbmodifyTimeoutValues);

# Here becuse GSI has no system_user param
my $SYSTEM_USER_OPTION = 'system_user';
my $PROGRESS_MESSAGE = "Adding Additional Hosts";
my $XS_PROGRESS_MESSAGE = "Adding XS Advanced Controller Host";
my $OPERATION = 'hdblcm_add_hosts_v10';
my $PARAMETER_ID_TO_OPTION_MAP = {
    'AutoInitializeServices' => 'AIS',
    'Target' => 'SAPMNT',
    'SystemUser' => 'SU',
    'TenantUser' => 'TU',
    'ListenInterface' => 'LI',
    'OrgManagerUser' => 'OMU',
    'SkipModifySudoers' => 'SUD',
    'AcceleratorUser' => 'ASE_USER',
    'InternalNetwork' => 'INTERNAL_NETWORK',
    'InstallHostagent' => 'INSTALL_HOSTAGENT',
    'ImportContentXS2'  => 'IMPORT_XS_CONTENT',
    'UseHttp'           => 'USE_HTTP'
};

sub getId {
    return 'add_hosts';
}

sub getName {
    my $self = shift();
    return ($self->isXsControllerHostTask) ? "Add XS Advanced Controller Host" : "Add Hosts";    
}

sub getExecutionName {
    my $self = shift();    
    return ($self->isXsControllerHostTask) ? $XS_PROGRESS_MESSAGE : $PROGRESS_MESSAGE;
}

sub _getNumberOfExpectedOutputLines{
    return 150;
}

sub _executeInternal {
    my $self = shift();
    my $configuration = $self->_getConfiguration();
    my $sid = $configuration->getSID();
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    my $message = $self->getMsgLst()->addProgressMessage($self->getExecutionName() . '...');
    my $savedIsIntended = $progressHandler->setIntendationDepth(1);
    my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);

    my $returnCode = isSidadmin($sid) ? $self->_executeSHAOperation() : $self->_executeAsRoot();

    if($returnCode) {
        $self->getStatus()->_setFinishedState();
        $message->endMessage (undef, sprintf("%s finished", $self->getName()));
    } else {
        $self->getStatus()->_setErrorState();
        $message->endMessage (undef, sprintf("%s failed", $self->getName()));
    }
    $progressHandler->setIntendationDepth($savedIsIntended);
    $self->setMsgLstContext($saveContext);
}

sub _executeAsRoot {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $args = $self->_getArgs($configuration);
    my $hdbInstance = $configuration->getOwnInstance();

    if(!defined $hdbInstance){
        $self->setErrorMessage ('Unable to get HDB Instance.', $self->getErrMsgLst());
        return undef;
    }

    my $installDir = $hdbInstance->get_globalTrexInstallDir();
    my $command = File::Spec->catfile($installDir, 'bin', 'hdbmodify');
    my $stdinLines = $configuration->getXmlPasswordStream();

    if (defined $stdinLines) {
        push @$args, '--read_password_from_stdin=xml';
    }
    my $executor = new LCM::ProcessExecutor($command,$args, $stdinLines);
    $executor->setMsgLstContext($self->getMsgLstContext());
    $executor->setOutputHandler($self->getMsgLst()->getProgressHandler());
    my $exitCode = $executor->executeProgram();

    $self->_setLogLocation($self->_parseLogFileLocation($executor->getOutputLines()));

    if (!defined $exitCode || $exitCode){
        $self->getErrMsgLst()->addError("$PROGRESS_MESSAGE failed");
        return undef;
    }

    return 1;
}

sub _executeSHAOperation {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();

    my $optionMap = $self->buildSHAOptionsMap($configuration, undef, undef, 1);
    my $passwordKeys = ['Password','HostagentPassword', 'AcceleratorPassword', 'OrgManagerPassword', 'SQLSysPasswd', 'SQLTenantUserPassword'];

    my $localHost = $self->_createRemoteHostsHostctrlObject();
    my $exitCode = $localHost->executeHostctrlParallel($OPERATION, $configuration, undef, $passwordKeys, undef, undef, undef,
                                                        undef, undef, $optionMap, undef, undef, undef, 1);
    my $outputBuffer = $localHost->getOutputBuffer();
    my $executionOutput = [ split('\n', $outputBuffer->[0]) ];
    $self->_setLogLocation($self->_parseLogFileLocation($executionOutput));

    if (!defined $exitCode || ($exitCode != 0)) {
        $self->setErrorMessage( "Adding additional hosts through SAP Host Agent failed.", $localHost->getErrMsgLst());
        $exitCode = undef;
    }

    return defined($exitCode) ? 1 : undef;
}

sub _createRemoteHostsHostctrlObject {
    my ($self) = @_;
    my @localHost = (hostname());
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts(@localHost);
    $localExecutionHost->setOutputHandler($localHost[0], $self->getMsgLst()->getProgressHandler());
    $localExecutionHost->setMsgLstContext($self->getMsgLstContext());
    return $localExecutionHost;
}

sub buildSHAOptionsMap{
    my ($self) = @_;
    my $optionsMap = $self->SUPER::buildSHAOptionsMap();
    my $configuration = $self->_getConfiguration();

    my $hosts = $self->getHosts();
    $optionsMap->{ADDHOSTS} = join(',', @{$hosts});

    if($optionsMap->{INTERNAL_NETWORK} eq 'none'){
        delete($optionsMap->{INTERNAL_NETWORK});
    }

	my $ignoreOptionString = $self->_getIgnoreOptionString();
    if ($ignoreOptionString){
        my ($option, $value) = split('=', $ignoreOptionString);
        $optionsMap->{IGNORE} = $value;
    }

    my $timeoutOption = $configuration->getOptionTimeout();
    if (defined($timeoutOption)){
        my $timeoutOptionSting = $timeoutOption->getOptionStringFromArgList(getHdbmodifyTimeoutValues());
        if ($timeoutOptionSting){
            my ($option, $value) = split('=', $timeoutOptionSting);
            $optionsMap->{TIMEOUTS} = $value;
        }
    }

# SID is added explicitly, because in the resident scenario the SID parameter has value but is skipped. See bug 86105
    $optionsMap->{'SID'} = $configuration->getValue('SID');

    my $isNoStartSet = ($configuration->getValue('NoStart') =~ /$bool_true_pattern/) && !$configuration->isSkipped('NoStart');
    $optionsMap->{'NO_START'} = $isNoStartSet ? 'true' : 'false';

    my $isAutoAddingXsRoles = $configuration->getValue('AutoAddXS2Roles') && ! $configuration->isSkipped('AutoAddXS2Roles');
    $optionsMap->{'AUTOADD_XS_ROLES'} = $isAutoAddingXsRoles ? 'on' : 'off';

    return $optionsMap;
}

sub _getIgnoreOptionString {
    my ($self) = @_;
    require SDB::Install::Configuration::AnyModifyConfig;
    my $configuration = $self->_getConfiguration();
    my $ignoreOption = $configuration->getOptionIgnore();
    my $hdbmodifyIgnoreOption = new SDB::Install::Configuration::AnyModifyConfig()->getOptionIgnore();
    my $ignoreOptionString;

    if (defined $ignoreOption && defined $hdbmodifyIgnoreOption){
        my $ignoreArgs = $ignoreOption->getSetKeysValidFor($hdbmodifyIgnoreOption);
        $ignoreOptionString = $ignoreOption->getOptionStringFromArgList($ignoreArgs);
    }

    return $ignoreOptionString;
}

sub _getArgs {
    my ($self, $instconfig) = @_;
    my @args = ('-b');

    my $remoteExecution = $instconfig->getValue('RemoteExecution');
    if(defined($remoteExecution) && length($remoteExecution) > 0){
        push @args, $instconfig->getOpt('RemoteExecution') . '=' . $remoteExecution;
    }

    my $rootUser = $instconfig->getValue('RootUser');
    if($rootUser && $remoteExecution ne 'saphostagent'){
        push @args, $instconfig->getOpt('RootUser') . '=' . $rootUser;
    }

    my $hosts = $self->getHosts();
    push @args, $instconfig->getOpt('AddHosts')         . '=' . join(',', @{$hosts});
    push @args, $instconfig->getOpt('InstallHostagent') . '=' . ($instconfig->getValue('InstallHostagent') ? 'on' : 'off');
    push @args, '--skip_hostagent_password=1' if ( $instconfig->isSkipped('HostagentPassword'));
    
    my $systemUserOpt   = $instconfig->getOpt('SystemUser') || "--$SYSTEM_USER_OPTION";
    my $systemUser      = $instconfig->getValue('SystemUser');
    my $acceleratorUser = $instconfig->getValue('AcceleratorUser');
    my $orgManagerUser  = $instconfig->getValue('OrgManagerUser');
    if (!$instconfig->isSkipped('SystemUser') && $instconfig->hasValue('SystemUser')) {
        push @args, $systemUserOpt . '=' . $systemUser;
    }
    if (!$instconfig->isSkipped('AutoInitializeServices')) {
        my $autoInitUserOpt   = $instconfig->getOpt('AutoInitializeServices');
        my $autoInitValue     = $instconfig->getValue('AutoInitializeServices');
        push @args, $autoInitUserOpt . '=' . $autoInitValue;
    }
    if (!$instconfig->isSkipped('TenantUser') && $instconfig->hasValue('TenantUser')) {
        my $tenantUserOpt   = $instconfig->getOpt('TenantUser');
        my $tenantUser      = $instconfig->getValue('TenantUser');
        push @args, $tenantUserOpt . '=' . $tenantUser;
    }
    if (!$instconfig->isSkipped('AcceleratorUser')) {
        push @args, $instconfig->getOpt('AcceleratorUser') . '=' . $acceleratorUser;
    }
    if (!$instconfig->isSkipped('OrgManagerUser')) {
        push @args, $instconfig->getOpt('OrgManagerUser') . '=' . $orgManagerUser;
    }
    push @args, '--instlog_dir=' . $gLogDir;

    my $listenInterface = $instconfig->getValue('ListenInterface');
    my $internalNetwork = $instconfig->getValue('InternalNetwork');
    if (defined $listenInterface && (length($listenInterface) > 0)
        && ! $instconfig->isSkipped('ListenInterface')) {
        push @args,  $instconfig->getOpt('ListenInterface') . '=' . $listenInterface;
        if (length($internalNetwork) > 0 && $internalNetwork ne 'none' && ! $instconfig->isSkipped('InternalNetwork')) {
            push @args,  $instconfig->getOpt('InternalNetwork') . '=' . $internalNetwork;
        }
    }
    if ($instconfig->hasValue('SkipModifySudoers')) {
        my $value = $instconfig->getValue('SkipModifySudoers') ? 'on' : 'off';
        my $option = $instconfig->getOpt('SkipModifySudoers');
        push(@args, sprintf('%s=%s', $option, $value));
    }
    if ($instconfig->hasValue('ImportContentXS2')) {
        my $value = $instconfig->getValue('ImportContentXS2') ? 'on' : 'off';
        my $option = $instconfig->getOpt('ImportContentXS2');
        push(@args, sprintf('%s=%s', $option, $value));
    }
    if (!$instconfig->isSkipped('UseHttp') && $instconfig->getValue('UseHttp')) {
        push(@args, $instconfig->getOpt('UseHttp'));
    }

    my $ignoreOptionString = $self->_getIgnoreOptionString();
    if ($ignoreOptionString){
            push(@args, $ignoreOptionString);
    }

    my $timeoutOption = $instconfig->getOptionTimeout();
    if (defined($timeoutOption)){
        my $timeoutOptionSting = $timeoutOption->getOptionStringFromArgList(getHdbmodifyTimeoutValues());
        if ($timeoutOptionSting){
            push(@args, $timeoutOptionSting);
        }
    }
    push @args, '--instlog_dir=' . $gLogDir;

    my $isAutoAddingXsRoles = $instconfig->getValue('AutoAddXS2Roles') && ! $instconfig->isSkipped('AutoAddXS2Roles');
    push @args, $instconfig->getOpt('AutoAddXS2Roles') . '=' . ($isAutoAddingXsRoles ? 'on' : 'off');

    return \@args;
}

sub setHosts {
    my ($self, $hosts) = @_;
    $self->{_hosts} = $hosts;
}    

sub getHosts {
    my ($self) = @_;
    return $self->{_hosts};
}

sub setXsControllerHostTask {
    my ($self, $value) = @_;
    $self->{_isXsControllerHostTask} = $value;
}    

sub isXsControllerHostTask {
    my ($self) = @_;
    return $self->{_isXsControllerHostTask};
}

sub getParameterToOptionMapping{
    return $PARAMETER_ID_TO_OPTION_MAP;
}

sub getSlppLogFileName{
    return 'add_additional_hosts.log';
}

sub shouldPersistFinishedStep {
    return 1;
}

1;
