package LCM::Task::GenericStackTask::ComponentTask;

use strict;
use parent qw (LCM::Task::SHASupportedTask);
use LCM::Component;
use LCM::Task::GenericStackTask::Listeners::PersistUpdateTaskListener;
use LCM::Task::GenericStackTask::Listeners::PersistServerInstallTaskListener;
use LCM::Task::TaskProgressHandler::ComponentTaskProgressHandlerFactory;
use LCM::Task::CommonTask::UpdateRemoteHostsTask qw($TASK_ID_UPDATE_REMOTE_HOSTS);
use LCM::Task::CommonTask::UpdateLocalHostTask qw($TASK_ID_UPDATE_LOCAL_HOST);
use SDB::Install::Globals qw ($gKeynameLMStructure);

sub new{
	my($class,$configuration,$component,$noResume) = @_;
	my $self = shift->SUPER::new($configuration);
	$self->{_component} = $component;
    $self->{_noResume} = $noResume;
	$self->_addListeners();
	return $self;
}

sub _addListeners {
	my ($self) = @_;
	my $configuration = $self->_getConfiguration();
	my $lmStructureComponentName = $gKeynameLMStructure;
    if($self->{_noResume}){
     return;
    }
	if ($self->_isServerInstall()) {
		$self->addListener(new LCM::Task::GenericStackTask::Listeners::PersistServerInstallTaskListener());
	} elsif ($self->getId() ne $TASK_ID_UPDATE_LOCAL_HOST &&
			$self->getId() ne $TASK_ID_UPDATE_REMOTE_HOSTS &&
			$self->getId() !~ /\Q$lmStructureComponentName\E/) {
            $self->addListener(new LCM::Task::GenericStackTask::Listeners::PersistUpdateTaskListener());
	}
}

sub getId {
	my $self = shift();
	my $component = $self->getComponent();
	my $componentKeyname = $component->getComponentKeyName();
	if($component->isUpdate()){
        return 'update_'.$componentKeyname;
    } else {
        return 'install_'.$componentKeyname;
    }
}

sub getName {
    my $self = shift();
    my $component = $self->getComponent();
    my $componentName = $component->getComponentName();
    if($component->isUpdate()){
       return "Update $componentName";
    } else {
       return "Install $componentName";
    }
}

sub getStepName(){
    my $self = shift();
    my $component = $self->getComponent();
    if( defined $component && defined $component->getComponentBatchKey()){
        return $component->getComponentBatchKey();
    }
    return $self->SUPER::getStepName();
}

sub getExecutionName {
    my $self = shift();
    my $component = $self->getComponent();
    my $componentName = $component->getComponentName();
    if($component->isUpdate()){
       return "Updating $componentName";
    } else {
       return "Installing $componentName";
    }
}

sub _executeInternal {
	my $self = shift();
	my $component = $self->getComponent();
	my $configuration = $self->_getConfiguration();
    $component->setMsgLstContext($self->getMsgLstContext());
	my $taskProgressHandler = $self->getMsgLst()->getProgressHandler();
	$component->setProgressHandler($taskProgressHandler);
	LCM::ExecutionWarningsObservable->getInstance()->registerListener($self);

    if(!$self->_callComponentExecutable()){
    	$self->getStatus()->_setErrorState();
    } else {
    	$self->getStatus()->_setFinishedState();
    }

    $self->persistStep();
    LCM::ExecutionWarningsObservable->getInstance()->unregisterListener($self);
}

sub shouldPersistFinishedStep {
	my ($self) = @_;

    my $configuration = $self->_getConfiguration();
    my $component = $self->getComponent();
    if (!$configuration->isUpdate()) {
        return 1;
    }
	if (!$configuration->isOptimizedUpdate()){
		return 1;
	}
	if ($component->isInternal()) {
		return 1;
	}
	if (!$component->supportsPhases()) {
		return 1;
	}
    if ($component->supportsPhase(SYSTEM_ONLINE_PHASE) && 
        $component->getPhase() eq SYSTEM_ONLINE_PHASE) {
        return 1;
    }
    if (!$component->supportsPhase(SYSTEM_ONLINE_PHASE) && 
        $component->supportsPhase(ONLINE_PHASE) && 
        $component->getPhase() eq ONLINE_PHASE) {
        return 1;
    }
	if (!$component->supportsPhase(SYSTEM_ONLINE_PHASE) && 
	    !$component->supportsPhase(ONLINE_PHASE) &&
		$component->supportsPhase(OFFLINE_PHASE) && 
		$component->getPhase() eq OFFLINE_PHASE) {
		return 1;
	}
	return 0;
}

sub persistFinishedStep {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $component = $self->getComponent();
    $configuration->addToPersistedFinishedSteps($component->getComponentBatchKey());
    $configuration->setPersComponentVersion($component->getComponentBatchKey(), $component->getVersion());
}

sub logSkippedPersistedTask{
}

sub _getNumberOfExpectedOutputLines{
    my $self = shift();
    my $component = $self->getComponent();
    return $component->getNumberOfExpectedOutputLines();
}

sub getComponent{
	my $self = shift();
	return $self->{_component};
}

sub getComponentKeyName{
    my $self = shift();
    my $component = $self->getComponent();
    return $component->getComponentKeyName();
}

sub getLogLocation{
	my $self = shift();
    my $component = $self->getComponent();
	return $component->getLogLocation();
}

#implementing required interface by LCM::ExecutionWarningsObservable
sub onWarning{
	my ($self,$message) = @_;
	$self->_addWarningMessage($message);
}

sub isResumable{
	my($self) = @_;
	my $configuration = $self->_getConfiguration();
	return $self->getComponent()->isResumable($configuration);
}

sub getSlppLogFileName { 
	my $self = shift();
	my $component = $self->getComponent();
	return $component->getSlppLogFileName();
}

sub _callComponentExecutable{
	my $self = shift();
	my $component = $self->getComponent();
	my $configuration = $self->_getConfiguration();
	my $executionResult = 1; 
    if ( $component->isUpdate() ) {
        $executionResult = $component->updateComponent($configuration);
    } else {
        $executionResult = $component->installComponent($configuration);
    }
    return $executionResult;
}

# Override
sub isComponentTask {
	return 1;
}

sub isServerComponentTask {
	my($self) = @_;
	return $self->getComponent()->isServer();
}

sub _isServerUpdate{
    my($self) = @_;
    return $self->isServerComponentTask() && $self->getComponent()->isUpdate();
}

sub _isServerInstall{
    my($self) = @_;
    return $self->isServerComponentTask() && !$self->getComponent()->isUpdate();
}

sub initProgressHandler{
    my ($self) = @_;
    $self->getMsgLst()->setProgressHandler(LCM::Task::TaskProgressHandler::ComponentTaskProgressHandlerFactory::createComponentTaskProgressHandler($self));
}

sub getPhase{
   my($self) = @_;
   my $component = $self->getComponent();
   return $component->getPhase();
}

sub isSkipped {
    my $self = shift;
    return 1 if $self->{_skip};

    my $configuration = $self->_getConfiguration();
    return 0 if ! $configuration->can('getPersistedSteps');

    my $component = $self->getComponent();
    return 0 if ! defined $component;

    my $persistedSteps = $configuration->getPersistedSteps();
    return 0 if ! defined $persistedSteps;

    my $isUpdate = $component->isUpdate();
    if ( $isUpdate && defined $component->getManifest() ){
        my $versionObject = $component->getVersionObject();
        my $persistedVersion = $configuration->getPersComponentVersion($component->getComponentBatchKey());
        my $persistedVersionObject = new SDB::Install::Version(split('\.', $persistedVersion)) if defined $persistedVersion;
        return 0 if (defined $persistedVersionObject && $versionObject->isNewerThan($persistedVersionObject));
    }

    my $stepName = $self->getStepName();
    if ( grep( /^$stepName/, split(',', $persistedSteps) ) ){
        my $componentName = $component->getComponentName();
        my $parentTask = $self->getParentTask();
        if (defined $parentTask) {
            $parentTask->getMsgLst()->addMessage("Component '$componentName' is already installed.");
        }
        return 1;
    }

    return 0;
}

sub getParameterToOptionMapping{ return undef; }

sub shouldPersistStatus {
    return 0;
}

1;
