package LCM::Task::GenericStackTask::ComponentTaskListFactory;

use strict;
use parent 'Exporter';
use SDB::Install::Configuration qw($bool_false_pattern);
use LCM::Task::GenericStackTask::ComponentTask;
use LCM::Task::GenericStackTask::ComponentTask::ComponentListComponentTask;
use LCM::Task::GenericStackTask::ComponentTask::ComponentPhasedTask;
use LCM::Component qw(PREPARE_PHASE ONLINE_PHASE OFFLINE_PHASE SYSTEM_ONLINE_PHASE ALL);
use LCM::Utils::ComponentsDependencySort qw(sortComponentsTopologically constructComponentDependencyGraph);
use LCM::Task::StartStopHostsTask::StartSpecificHostsTask;
use LCM::Task::StartStopHostsTask::StopSpecificHostsTask;
use LCM::Task::StartStopHostsTask::StopSystemTask;
use LCM::Task::StartStopHostsTask::DisableAndStopSpecificServicesTask;
use LCM::Task::StartStopHostsTask::StartSpecificServicesTask;
use LCM::Task::StartStopHostsTask::EnableSpecificServicesTask;
use LCM::Task::GenericStackTask::DisableAutoFailoverTask;
use LCM::Task::GenericStackTask::EnableAutoFailoverTask;
use LCM::Task::ConvertToMultiDBTask::DisableXSAServicesTask;
use LCM::HostsParser;
use LCM::Utils::CommonUtils qw (removeElementFromArray);
use SDB::Install::Globals qw( $mComponentsRoles $gHostRoleWorker $gHostRoleStandby $gKeynameEngine $gKeynameXS2);
use LCM::PersistencyUtils qw ( isPendingUpdate);
use experimental qw (smartmatch);
use LCM::Task::TaskStatus qw(FINISHED);
use Scalar::Util qw(blessed);

our @EXPORT = qw(createComponentTasklist getInstance);

my $_aStoppedHosts = [];
my $instance = undef;

sub createComponentTasklist {
	my($configuration,$components) = @_;
	my $isPrepareUpdate = $configuration->isPrepareUpdate();

	if (!$configuration->isOptimizedUpdate() && !$isPrepareUpdate) {
		return _createDefaultTasklist($configuration,$components);
	}

	my $sid = $configuration->getValue('SID');
	my $phase = $configuration->getPhase();
	my $isResumeUpdateIgnored = $configuration->getIgnore('check_pending_upgrade');
	my $isPendingUpdate = $configuration->isPendingUpdate();
	my $isPhasedResumeUpdate = 	$configuration->isUpdate() && 
								$configuration->isOptimizedUpdate() && 
								$isPendingUpdate && 
								!$isResumeUpdateIgnored;
	my $isIgnoredPhasedResumeUpdate = $configuration->isOptimizedUpdate() && $isResumeUpdateIgnored;

	if (!$isPrepareUpdate && ($isPhasedResumeUpdate || $isIgnoredPhasedResumeUpdate)) {
		$phase = ALL;
	}

	if($phase eq PREPARE_PHASE){
		return _createPrepareTasklist($configuration,$components);
	} elsif($phase eq OFFLINE_PHASE){
		return _createOfflineTasklist($configuration,$components);
	} elsif($phase eq ONLINE_PHASE){
		return _createOnlineTasklist($configuration,$components);
	} elsif($phase eq SYSTEM_ONLINE_PHASE){
        return _createSystemOnlineTasklist($configuration,$components);
    }
	return _createPhasedTasklist($configuration,$components);
}

sub _createDefaultTasklist {
	my($configuration,$components) = @_;
	my $taskList = [];

	$configuration->setWillExecuteAllComponentsInAllPhases(1);

	for my $component (@{$components}) {
		if($component->getComponentName() ne 'Component List') {
			push (@{$taskList}, new LCM::Task::GenericStackTask::ComponentTask($configuration,$component));
		}
	}
	return $taskList;
}

sub _createPhasedTasklist {
	my($configuration,$components) = @_;
	my $taskList = [];
	$components = sortComponentsTopologically($configuration->getComponentManager(), $components, 0);

	return $taskList if !_recomputePhases($configuration, $components);

	my $prepareTaskList = _createPrepareTasklist($configuration, $components);
	my $offlineTaskList = _createOfflineTasklist($configuration, $components);

	my $onlineTaskList = [];
	my $systemOnlineTaskList = [];
	if(!$configuration->isServerNoStart()){
		$onlineTaskList = _createOnlineTasklist($configuration,$components);
		$systemOnlineTaskList = _createSystemOnlineTasklist($configuration,$components);
	}
	push(@{$taskList},@{$prepareTaskList});
	push(@{$taskList},@{$offlineTaskList});
	push(@{$taskList},@{$onlineTaskList});
    push(@{$taskList},@{$systemOnlineTaskList});

    if(defined $configuration->getResumeExecutionList()){
        _updateResumedTasks($configuration,$taskList);
    }
	return $taskList;
}

sub _createPrepareTasklist {
	my($configuration,$components) = @_;
	my $prepareTaskList = [];

	$configuration->setWillExecuteAllComponentsInAllPhases(0);

	for my $component (@{$components}) {
		if (! $component->canBeCalledInPhase(PREPARE_PHASE, $configuration)) {
			next;
		}
		if ($component->supportsPhase(PREPARE_PHASE)) {
			push (@{$prepareTaskList},new LCM::Task::GenericStackTask::ComponentTask::ComponentPhasedTask($configuration,$component,PREPARE_PHASE));
			next;
		}
		push (@{$prepareTaskList},new LCM::Task::GenericStackTask::ComponentTask($configuration,$component));
	}
	return $prepareTaskList;
}

sub _createOfflineTasklist {
	my($configuration,$components) = @_;
	my @offlineTaskList = ();

	$configuration->setWillExecuteAllComponentsInAllPhases(0);

	my $offlineTaskPrerequisiteTasks = _createOfflinePrerequisiteTasks($configuration, $components);
	push (@offlineTaskList, @{$offlineTaskPrerequisiteTasks});

	for my $component (@{$components}) {
		if (!$component->canBeCalledInPhase(OFFLINE_PHASE, $configuration)) {
			next;
		}
		if ($component->supportsPhase(OFFLINE_PHASE)) {
			push (@offlineTaskList, new LCM::Task::GenericStackTask::ComponentTask::ComponentPhasedTask($configuration,$component,OFFLINE_PHASE));
			next;
		}
		push (@offlineTaskList, new LCM::Task::GenericStackTask::ComponentTask($configuration,$component));
	}
	my $postComponentTasks=_createPostPhaseTaskList($configuration,OFFLINE_PHASE);
	if(scalar(@$postComponentTasks) > 0){
		push (@offlineTaskList, @$postComponentTasks);
	}
	
	return \@offlineTaskList;
}

sub _createOfflinePrerequisiteTasks {
	my ($configuration, $components) = @_;
	my @result = ();

	my $servicesToRestart = _getServicesToRestart($configuration, $components);

	if (_isSystemRestartRequired($configuration, $components)) {
		if (@$servicesToRestart) {
			push(@result, LCM::Task::StartStopHostsTask::DisableAndStopSpecificServicesTask->new($configuration, $servicesToRestart, undef));
		}
		push(@result, LCM::Task::StartStopHostsTask::StopSystemTask->new($configuration));
		return \@result;
	}

	if (@$servicesToRestart) {
		my $instancesWithServicesToRestart = _getInstancesWithServicesToRestart($configuration, $components);
		my $shouldStop = @$instancesWithServicesToRestart;
		push(@result, LCM::Task::StartStopHostsTask::DisableAndStopSpecificServicesTask->new($configuration, $servicesToRestart, $shouldStop));
	}
	my $instancesToRestart = _getInstancesToStop($configuration, $components);
	if (@$instancesToRestart) {
        push(@result, LCM::Task::GenericStackTask::DisableAutoFailoverTask->new($configuration));
        push(@result, LCM::Task::StartStopHostsTask::StopSpecificHostsTask->new($configuration, undef, $instancesToRestart));
        $configuration->{disableAutoFailoverTaskAdded} = 1;
    }
	return \@result;
}


sub _createOnlineTasklist {
	my ($configuration, $components) = @_;
	my @onlineTaskList = ();

	$configuration->setWillExecuteAllComponentsInAllPhases(1);

	my $onlineTaskPrerequisiteTasks = _createOnlinePrerequisiteTasks($configuration, $components);
	push (@onlineTaskList, @{$onlineTaskPrerequisiteTasks});

	for my $component (@{$components}) {
		if (!$component->canBeCalledInPhase(ONLINE_PHASE, $configuration)) {
			next;
		}
		if ($component->getComponentName() eq 'Component List') {
			next;
		}
		if ($component->supportsPhase(ONLINE_PHASE)) {
			push (@onlineTaskList, new LCM::Task::GenericStackTask::ComponentTask::ComponentPhasedTask($configuration,$component,ONLINE_PHASE));
			next;
		}
		push (@onlineTaskList, new LCM::Task::GenericStackTask::ComponentTask($configuration,$component))
	}
	my $postComponentTasks=_createPostPhaseTaskList($configuration,ONLINE_PHASE);
	if(scalar(@$postComponentTasks) > 0){
		push(@onlineTaskList, @$postComponentTasks);
	}
	return \@onlineTaskList;
}

sub _createOnlinePrerequisiteTasks {
	my ($configuration, $components) = @_;
	my @result = ();

	my $instancesToStartBeforeOnlinePhase = _getInstancesToStartBeforeOnlinePhase($configuration, $components);
	if (@$instancesToStartBeforeOnlinePhase) {
		my $mcm = $configuration->getComponentManager();
		my $scm = $configuration->getSystemComponentManager();
		my $isSystemReplication = LCM::Utils::CommonUtils::isSystemReplication($configuration);
		my $server = $scm->getComponentByKeyName($gKeynameEngine);
		my $isConvertToMDCStarted = defined $server ? $server->isConvertToMDCStarted() : 0;
		my $isConvertToMDC = $configuration->isConvertToMultiDbRequired() || $isConvertToMDCStarted;
		my $isXs2Selected = $mcm->isComponentSelected($gKeynameXS2);
		my $isXs2Available = $scm->isComponentAvailable($gKeynameXS2);
		if(!$isSystemReplication && $isConvertToMDC && $isXs2Available && !$isXs2Selected) {
			push(@result, new LCM::Task::ConvertToMultiDBTask::DisableXSAServicesTask($configuration));
		}

		# Start everything but non-db hosts that require instance restart
		push(@result, LCM::Task::StartStopHostsTask::StartSpecificHostsTask->new($configuration, undef, $instancesToStartBeforeOnlinePhase));
	}
	return \@result;
}

# Start everything except hosts that have a component role that requires
# restart of the instance
sub _getInstancesToStartBeforeOnlinePhase {
	my ($configuration, $components) = @_;
	my @result = ();
	my $allHosts = $configuration->getOwnInstance->get_allhosts();
	if ($configuration->isUpdate() && $configuration->isConvertToMultiDbRequired()) {
		return $allHosts;
	}
	my $instanceRestartHosts = _getInstancesToStart($configuration, $components);
	map {push(@result, $_) if (!($_ ~~ @$instanceRestartHosts))} (@$allHosts);
	return \@result;
}

sub _createSystemOnlineTasklist {
    my($configuration,$components) = @_;
    my @systemOnlineTaskList = ();

    $configuration->setWillExecuteAllComponentsInAllPhases(1);

    my $systemOnlineTaskPrerequisiteTasks = _createSystemOnlinePrerequisiteTasks($configuration, $components);
	push (@systemOnlineTaskList, @{$systemOnlineTaskPrerequisiteTasks});

    for my $component (@{$components}) {
        if (!$component->canBeCalledInPhase(SYSTEM_ONLINE_PHASE, $configuration)) {
            next;
        }
        if ($component->getComponentName() eq 'Component List') {
            next;
        }
        if ($component->supportsPhase(SYSTEM_ONLINE_PHASE)) {
            push (@systemOnlineTaskList, new LCM::Task::GenericStackTask::ComponentTask::ComponentPhasedTask($configuration,$component,SYSTEM_ONLINE_PHASE));
            next;
        }
        push (@systemOnlineTaskList, new LCM::Task::GenericStackTask::ComponentTask($configuration,$component));
    }
    my $postComponentTasks=_createPostPhaseTaskList($configuration,SYSTEM_ONLINE_PHASE);
    if(scalar(@$postComponentTasks) > 0){
        push(@systemOnlineTaskList, @$postComponentTasks);
    }
    return \@systemOnlineTaskList;
}

sub _isSystemRestartRequired {
	my ($configuration, $components) = @_;
	for my $component (@$components) {
		return 1 if ($component->isSystemStopRequired($configuration)); 
	}
	return 0;
}

sub _createSystemOnlinePrerequisiteTasks {
	my ($configuration, $components) = @_;
	my @result = ();
	my $servicesToRestart = _getServicesToRestart($configuration, $components);
	if (@$servicesToRestart) {
		push(@result, LCM::Task::StartStopHostsTask::EnableSpecificServicesTask->new($configuration, $servicesToRestart));
	}
	my $instancesToRestart = _getInstancesToStart($configuration, $components);
	if (@$instancesToRestart) {
		push(@result, LCM::Task::StartStopHostsTask::StartSpecificHostsTask->new($configuration, undef, $instancesToRestart));
	}
	my $instancesWithServicesToRestart = _getInstancesWithServicesToRestart($configuration, $components);
	if (@$instancesWithServicesToRestart) {
		push(@result, LCM::Task::StartStopHostsTask::StartSpecificServicesTask->new($configuration, $servicesToRestart));
	}
	my $isLeftDisabledFailoverFromPreviousRun = $configuration->getOwnInstance()->isAutoFailoverDisabled();
    if ( $configuration->{disableAutoFailoverTaskAdded} || $isLeftDisabledFailoverFromPreviousRun ) {
        push(@result, LCM::Task::GenericStackTask::EnableAutoFailoverTask->new($configuration));
    }
	return \@result;
}

sub _getServicesToRestart {
	my ($configuration, $components) = @_;
	my %mServicesToRestart = ();

	for my $component (@$components) {
		my $componentRoles = $component->getHostRoles();
		my $aHostsWithRoles = _getHostsWithRoles($configuration, $componentRoles);
		next if (! @$aHostsWithRoles);
		next if $component->isComponentUpdated($configuration);

		my @componentServicesToRestart = @{$component->getServicesToRestart};
		map { $mServicesToRestart{$_}++ } @componentServicesToRestart;
	}
	my @aServicesToRestart = keys %mServicesToRestart;
	return \@aServicesToRestart;
}

sub _getInstancesWithServicesToRestart {
	my ($configuration, $components) = @_;
	my %mInstancesWithServicesToRestart = ();

	for my $component (@$components) {
		my @componentServicesToRestart = @{$component->getServicesToRestart};
		if (@componentServicesToRestart) {
			my $componentRoles = $component->getHostRoles();
			my $aHostsWithRoles = _getHostsWithRoles($configuration, $componentRoles);
			map { $mInstancesWithServicesToRestart{$_}++ } @$aHostsWithRoles;
		}
	}
	my @instancesWithServicesToRestart = ();
	my $instancesToStart = _getInstancesToStart($configuration, $components);
	for my $host (keys %mInstancesWithServicesToRestart) {
		if (! ($host ~~ @$instancesToStart)) {
			push(@instancesWithServicesToRestart, $host);
		}
	}
	return \@instancesWithServicesToRestart;
}

sub _getInstancesToStart {
	my ($configuration, $components) = @_;
	my %mInstancesToRestart = ();
	my $dbHosts = _getHostsWithRoles($configuration, [$gHostRoleWorker, $gHostRoleStandby]);

	for my $component (@$components) {
		if ($component->isInstanceRestartRequired()) {
			my $componentRoles = $component->getHostRoles();
			my $aHostsWithRoles = _getHostsWithRoles($configuration, $componentRoles);
			map { $mInstancesToRestart{$_}++ if (!($_ ~~ @$dbHosts)) } @$aHostsWithRoles;
		}
	}
	my @aInstancesToRestart = keys %mInstancesToRestart;
	return \@aInstancesToRestart;
}

sub _getInstancesToStop {
	my ($configuration, $components) = @_;
	my %mInstancesToRestart = ();

	for my $component (@$components) {
		if ($component->isInstanceRestartRequired()) {
			my $componentRoles = $component->getHostRoles();
			my $aHostsWithRoles = _getHostsWithRoles($configuration, $componentRoles);
			map { $mInstancesToRestart{$_}++ } @$aHostsWithRoles;
		}
	}
	my @aInstancesToRestart = keys %mInstancesToRestart;
	return \@aInstancesToRestart;
}

sub _getHostsWithRoles {
	my ($configuration, $roles) = @_;
	my $hostDetector = new LCM::Hosts::HostsDetector (new LCM::Installer(), $configuration->getMsgLstContext(), $configuration->getOwnInstance());

	if ( $hostDetector->hasParsingErrors() ) {
		$configuration->PushError("Cannot detect hosts.");
		return [];
	}

	return $hostDetector->getHostsWithRoles($roles);
}

sub _recomputePhases {
	my ($configuration, $components) = @_;
	my $graph = constructComponentDependencyGraph($configuration->getComponentManager(), $components, 0);
	my $offlineComponentsWithDependencies = _getOfflineComponentsWithDependencies($configuration, $components, $graph);
	my %componentKeynameToComponent = map {$_->getComponentKeyName() => $_} @{$components};

	for my $componentWithDependencies (@{$offlineComponentsWithDependencies}){
		for my $dependentComponentKeyname (@{$graph->{$componentWithDependencies->getComponentKeyName()}}){
			my $dependentComponent = $componentKeynameToComponent{$dependentComponentKeyname};
			if($dependentComponent->canBeCalledInPhase(OFFLINE_PHASE, $configuration)){
				next;
			}
			if( ! $dependentComponent->canSetDependencyPhase() ){
				$configuration->getMsgLst()->addMessage("Unresolved component dependency. ".
				$componentWithDependencies->getComponentName ." requires " . $dependentComponent->getComponentName() );
				return undef;
			}
			$dependentComponent->setPhase(OFFLINE_PHASE);
		}
	}

	return 1;
}

sub _getOfflineComponentsWithDependencies {
	my ($configuration, $components, $componentDependencyGraph) = @_;
	my $offlineComponentsWithDependencies = [];

	for my $component (@{$components}) {
		if ($component->canBeCalledInPhase(OFFLINE_PHASE, $configuration) &&
			(scalar(@{$componentDependencyGraph->{$component->getComponentKeyName()}}) > 0)) {
			push @{$offlineComponentsWithDependencies}, $component;
		}
	}

	return $offlineComponentsWithDependencies;
}
sub _createPostPhaseTaskList {
    my($configuration, $phase) = @_;
    return [] if (!defined $instance->{_postPhaseTaskGenerator}->{$phase});
    my $result = undef;
    for my $taskGenerator ( @{$instance->{_postPhaseTaskGenerator}->{$phase}} ) {
        $result = $taskGenerator->createTaskList($configuration);
    }
    return defined $result ? $result : [];
}

sub addPostPhaseTaskGenerator {
    my($self, $phase, $taskGenerator) = @_;
    if (!defined $self->{_postPhaseTaskGenerator}->{$phase}) {
        $self->{_postPhaseTaskGenerator}->{$phase} = [];
    }
    push(@{$self->{_postPhaseTaskGenerator}->{$phase}}, $taskGenerator);
}

sub getInstance {
    my $class = shift;
    if (!defined $instance || !defined blessed($instance) || !$instance->isa($class)) {
        $instance = {};
        bless($instance, $class);
    }
    $instance->{_postPhaseTaskGenerator} = {};
    return $instance;
}

sub _updateResumedTasks {
    my($configuration, $taskList) = @_;
    my $resumedExecutionList = $configuration->getResumeExecutionList();
    for my $task (@$taskList) {
        my $resumeStatus = $resumedExecutionList->{$task->getId()};
        $task->setSkipped(1) if (defined $resumeStatus && $resumeStatus eq FINISHED);
    }
}
1;
