package LCM::Task::RenameTask::RenameSystemTask;

use strict;
use parent qw(LCM::Task::SHASupportedTask);

use LCM::Component;
use LCM::ShellConfigurator;
use LCM::SapHostAgentFunctions;
use File::Basename qw (basename);
use LCM::Landscape::LandscapeDescription;
use LCM::Python::SLDReconfigurationHelper;
use SDB::Install::SysVars qw($path_separator $isWin);
use SDB::Install::Globals qw($gLogDir $gProductNameSHA);

my $PROGRESS_MESSAGE = 'Renaming System';
our @EXPORT = qw ($PROGRESS_MESSAGE);

my $ext = $isWin ? '.exe' : '';

# Override
sub getId {
    return 'rename_system';
}

# Override
sub getName {
    return "Rename System";
}

# Override
sub getExecutionName {
    return $PROGRESS_MESSAGE;
}

# Override
sub _getNumberOfExpectedOutputLines {
    return 100;
}

# Override
sub _executeInternal {
    my $self = shift();
    my $configuration = $self->_getConfiguration();

    my $msg = $self->getMsgLst ()->addProgressMessage ($self->getExecutionName(). "...");
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    my $ident;
    my $progressHandler = $self->getMsgLst()->getProgressHandler();
    if(defined $progressHandler){
        $ident = $progressHandler->getIntendationDepth() if $progressHandler->can('getIntendationDepth');
        $progressHandler->setIntendationDepth( $ident + 1 ) if $progressHandler->can('setIntendationDepth');
    }

    my $exitCode = $self->_renameSystem ( $configuration );
    if(!$exitCode){
        $self->getStatus()->_setErrorState();
        $self->setErrorMessage(sprintf('%s failed', $self->getName()), $self->getErrMsgLst());      
    } else {
        $self->getStatus()->_setFinishedState();
    }

    my $endMessage = defined($exitCode) ? 'Renaming system' : 'Renaming system failed';
    $msg ->endMessage (undef, $endMessage);
    $self->setMsgLstContext($saveCntxt);
    if(defined $progressHandler){
        $progressHandler->setIntendationDepth($ident) if $progressHandler->can('setIntendationDepth');
    }
    return;
}

sub _renameSystem {
    my ( $self, $configuration ) = @_;
    my $isContinueAction   = $configuration->{isContinueOption};
    my $hdbRenameArguments = $self->_buildArguments ( $configuration );
    my $hdbRenameXmlInput  = $configuration->getXmlPasswordStream();
    if (defined $hdbRenameXmlInput) {
        push( @{$hdbRenameArguments}, '--read_password_from_stdin=xml' );
    }
    push( @{$hdbRenameArguments}, '--batch' );
    push( @{$hdbRenameArguments}, '--instlog_dir=' . $gLogDir);

    return undef if !$self->_executeHdbRename ( $configuration, $hdbRenameArguments, $hdbRenameXmlInput);
    $self->_fixINCArrayValues($configuration);
    return undef if ($configuration->isSystemIdentityChanged() && !$self->_registerInstanceService($configuration));
    return undef if (!$isContinueAction && $configuration->isSIDChanged() && !$self->_correctComponentLmStructureLocation($configuration));
    if ( $configuration->shouldRunHdbRename() ) {
        return undef if !$self->_updateSystemState($configuration);
        return undef if (!$isContinueAction && $configuration->isSIDChanged() && !$self->_reconfigureSLD($configuration));
    }
    return 1;
}

sub _registerInstanceService {
    my ( $self, $configuration ) = @_;
    $self->getMsgLst()->AddMessage('Registering Instance Service...');
    my ( $sid, $nr, $hostname ) = $configuration->getInstanceProperties();
    my $shaFunctionsHelper = LCM::SapHostAgentFunctions->new( undef, $self );
    if ($configuration->shouldSkipHostagentCalls()) {
        $self->getMsgLst()->AddMessage("Skipping registration of instance service because all $gProductNameSHA are skipped.");
        return 1;
    }
    if (! $shaFunctionsHelper->isSapHostAgentAvailable()) {
        $self->getMsgLst()->AddMessage("Skipping registration of instance service. $gProductNameSHA is not available.");
        return 1;
    }
    $shaFunctionsHelper->setMsgLstContext([$self->getMsgLst()]);
    return $shaFunctionsHelper->registerInstanceService( $sid, $nr, $hostname );
}

sub getProgressHandler {
    return $_[0]->getMsgLst()->getProgressHandler();    
}

sub _correctComponentLmStructureLocation {
    my ( $self, $configuration) = @_;

    $self->getMsgLst()->AddMessage('Correcting Component List location...');
    my $newSID = $configuration->{params}->{newSID}->{value};
    my $lmstructurePath = join( $path_separator, 'usr', 'sap', $newSID, 'lm_structure' );

    if ( -e $lmstructurePath && -l $lmstructurePath )    # if exists and is a symbolic link
    {
        $self->getMsgLst()->AddMessage('Changing location of the Component List file...');
        $self->getMsgLst()->AddMessage("Unlinking $lmstructurePath...");
        my $rc = unlink($lmstructurePath);
        return undef unless $rc;

        my $sharedSidLmStructurepath = $configuration->getSharedNewSIDPath() . $path_separator . 'lm_structure';
        $self->getMsgLst()->AddMessage("Linking $lmstructurePath to $sharedSidLmStructurepath...");
        my $rc1 = symlink( $sharedSidLmStructurepath, $lmstructurePath );
        return undef unless $rc1;
        return 1;
    } 

    $self->getMsgLst()->AddMessage('Component List file does not need correction.');
    return 1;
}

sub _updateSystemState {
    my ($self, $configuration) = @_;
    my $isContinueAction = $configuration->{isContinueOption};

    if ( !$isContinueAction ) {
        my $sid = $configuration->getValue('newSID') || $configuration->getValue('SID');
        my $progressMessage = sprintf('Updating %s...', COMPONENT_NAME_LM_STRUCTURE());
        my $failedMessage = sprintf('Updating %s failed', COMPONENT_NAME_LM_STRUCTURE());
        my $message = $self->getMsgLst()->addProgressMessage($progressMessage);
        my $landscapeDescription = new LCM::Landscape::LandscapeDescription([$message->getSubMsgLst()], $sid);

        if(!$landscapeDescription->getErrMsgLst()->isEmpty()){
            my $errorMessage = sprintf('Update of %s failed', COMPONENT_NAME_LM_STRUCTURE());
            $self->getErrMsgLst()->addError($errorMessage, $landscapeDescription->getErrMsgLst());
            $message->endMessage($failedMessage);
            return undef;
        }
        if($landscapeDescription->store()){
            $message->endMessage();
            return 1;
        }
        $message->endMessage($failedMessage);
        return undef;
    }

    if ( !$isWin && $configuration->isSIDChanged() ) {
        $self->getMsgLst()->addMessage('Updating shell configurations...');
        my $homedir = $configuration->getSidAdmHomeDir();
        my $shellConfigurator = LCM::ShellConfigurator->new($homedir);

        return undef unless $shellConfigurator->updateBashrcFile();
        return undef unless $shellConfigurator->updateBashSapenvFile();
        return undef unless $shellConfigurator->updateCshrcFile();
        return undef unless $shellConfigurator->updateCshSapenvFile();

        # TODO: check if timezone must be upadted
    }
    return 1;
}

sub _reconfigureSLD {
    my ( $self, $configuration ) = @_;

    if ($isWin) {
        $self->getMsgLst()->addMessage('SLD configuration is currently not supported under Windows.');
        return 1;
    }

    $self->getMsgLst()->addMessage('Checking SLD configuration...');
    my $ownInstanceAfterRename = $configuration->getOwnInstanceAfterRename();    # use renamed system instance
    return undef unless $ownInstanceAfterRename;

    my $sldConfigurator = LCM::Python::SLDReconfigurationHelper->new($ownInstanceAfterRename);

    my $newSID = $configuration->{params}->{newSID}->{value};
    my $isSldEnabled = $sldConfigurator->isSldEnabled( $newSID, $self->getMsgLst(), $self->getErrMsgLst() );
    return undef if not defined $isSldEnabled;

    if ($isSldEnabled) {
        my $sharedSidLmStructurepath = $configuration->getSharedNewSIDPath() . $path_separator . 'lm_structure';
        return undef
          unless $sldConfigurator->setLmStructurePath( $newSID, $sharedSidLmStructurepath, $self->getMsgLst(), $self->getErrMsgLst() );

        my $isInstanceRunning = $self->_isInstanceRunning($configuration);
        return undef if not defined $isInstanceRunning;

        if ($isInstanceRunning) {
            $self->getMsgLst()->addMessage("Configuring SLD...");

            # TODO: check what is the correct exit code
            my ( $uid, $gid ) = $sldConfigurator->getPermissions($newSID);
            my $exitCode = $sldConfigurator->reconfigureSLD( $uid, $gid );
            $self->getMsgLst()->addMessage( undef, $sldConfigurator->getMsgLst() );
            if ( !defined($exitCode) ) {
                $self->getErrMsgLst()->addMessage( undef, $sldConfigurator->getErrMsgLst() );
                return undef;
            }

        }
        else {
            $self->getMsgLst()->addMessage("There is no running Instance Service found. Skipping SLD configuration.");
        }

    }
    else {
        $self->getMsgLst()->addMessage("SLD is not enabled for SID $newSID. Skipping SLD configuration.");
    }
    return 1;
}

sub _isInstanceRunning {
    my ($self, $configuration) = @_;

    my $hdbInstance = $configuration->getOwnInstanceAfterRename();
    if (!defined $hdbInstance) {
        return undef;
    }

    my $host = $hdbInstance->get_host();
    my $instanceNumber = $hdbInstance->get_nr();
    require SDB::Install::SAPControl;
    my $sapControl = new SDB::Install::SAPControl($host,
                                                  $instanceNumber,
                                                  undef, # user
                                                  undef, # password
                                                  $configuration->isUseHttps(),
                                                  $configuration->getValue ('SSOCertificate'));

    return $sapControl->isRunning();
}

sub _buildArguments {
    my ( $self, $configuration ) = @_;

    if ($configuration->{isContinueOption}) {
        return $self->_buildScopeIndependentArguments($configuration);
    }

    my $paramIdsToSkip = ['HostMap', 'Scope', 'RootUser', 'InternalNetwork'];

    # TODO: check what should be added for Windows
    if ($isWin) {
        push(@{$paramIdsToSkip}, 'HomeDir', 'Shell', 'UID', 'GID');
    }

    my $arguments = $configuration->getCmdLineArgsFromInstconfig(new SDB::Install::Configuration::Rename(),
        $paramIdsToSkip, 'Rename', 'hdbrename');

    if ( $configuration->isAnyHostChanged () && hasValue($configuration, 'HostMap')) {
        my %hostNames = %{ $configuration->{params}->{HostMap}->{value} };
        foreach my $oldHost ( keys %hostNames ) {
            my $newHost = $hostNames{$oldHost};
            if ( $oldHost ne $newHost ) {
                push( @{$arguments}, $configuration->getOpt('HostMap')
                                . '='  . $oldHost . '=' . $newHost );
            }
        }
    }

    my $internalNetwork = $configuration->getValue('InternalNetwork');
    if(defined($internalNetwork) && !$configuration->isSkipped('InternalNetwork') && length($internalNetwork) > 0 && $internalNetwork ne 'none') {
        push(@$arguments, "--internal_network=$internalNetwork");
    }

    if (!$isWin && $configuration->isDistributedSystem(1) )
    {
        # currently not supported for Windows
        my $scopeValue = $configuration->getValue('Scope');

        if (!defined $scopeValue || ($scopeValue ne 'instance')) {
            if ( hasValue($configuration, 'RootUser') ) {
                push( @{$arguments}, $configuration->getOpt('RootUser')
                                . '='  . $configuration->{params}->{RootUser}->{value} );
            }
            $scopeValue = 'system';
        }
        push @{$arguments}, $configuration->getOpt('Scope') . '='  . $scopeValue;
    }

    return $arguments;
}

sub _buildScopeIndependentArguments {
    my ($self, $configuration) = @_;
    my $configurationParamIds = $configuration->getParamIds();
    my $scopeIndependentParamIds = ['NoStart', 'KeepUser', 'KeepUserHomeDir', 'InitUser', 'InitUserHomeDir', 'AutoStart', 'XSSpaceUserIdProd', 'XSSpaceUserIdSAP', 'XsEaDataPath'];
    my $scopeDependentParamIds = _removeElementsFromArray($configurationParamIds, $scopeIndependentParamIds);
    return $configuration->getCmdLineArgsFromInstconfig(new SDB::Install::Configuration::Rename(),
        $scopeDependentParamIds, 'Rename', 'hdbrename');
}

sub _removeElementsFromArray {
    my ($array, $elementsToRemove) = @_;
    my $modifiedArray = $array;

    foreach my $elementToRemove (@{$elementsToRemove}) {
        my $index=0;
        $index++ until @$modifiedArray[$index] eq $elementToRemove;
        splice(@$modifiedArray, $index, 1);
    }

    return $modifiedArray;
}

sub hasValue
{
    my ($configuration, $paramId) = @_;
    return 0 unless $configuration->{params}->{$paramId};
    return 0 unless $configuration->{params}->{$paramId}->{value};
    return 1;
}

sub _getHdbRenameExecutable {
    my ( $self, $configuration ) = @_;

    my $sharedSidPath = $configuration->getHanaSharedSidFromHdblcmRuntimeDir();

    my $hdbRenameExecutable =
        join( $path_separator, $sharedSidPath, 'global', 'hdb', 'install', 'bin', 'hdbrename' )
        . $ext;

    return $hdbRenameExecutable;
}

sub _executeHdbRename {
    my ( $self, $configuration, $hdbRenameArguments, $hdbRenameXmlInput ) = @_;
    my $hdbRenameExecutable = $self->_getHdbRenameExecutable ( $configuration );
    my $exer = LCM::ProcessExecutor->new ( $hdbRenameExecutable, $hdbRenameArguments,
        $hdbRenameXmlInput );
     $exer->setMsgLstContext( $self->getMsgLstContext() );
    $exer->setOutputHandler($self->getMsgLst()->getProgressHandler());
    
    my $rc = $exer->executeProgram ( );
    if(!defined $rc || $rc)
    {
        my $flavourProductName = $configuration->getFlavourProductName();
        $self->setErrorMessage ( "Error occurred while executing $flavourProductName System rename (hdbrename)", $exer->genHdbErrMsgLstFromOutput());
        $rc = undef;
       
    } else {
        $rc = 1;
    }
    return wantarray ? ($rc, $exer) : $rc;
}

sub _fixINCArrayValues {
    my($self, $configuration) = @_;
    my $targetDir = $configuration->getValue('Target');
    my $newSID = uc($configuration->getValue('newSID'));
    my $newRuntimeDir = File::Spec->catdir($targetDir, $newSID, 'hdblcm', 'instruntime');
    push(@INC, $newRuntimeDir) if (-e $newRuntimeDir);
    my $newLcmPmExtDir = File::Spec->catdir($newRuntimeDir, 'lcm_pm_ext');
    push(@INC, $newLcmPmExtDir) if (-e $newLcmPmExtDir);
}

1;
