package LCM::Task::UninstallationProcessTask;

use strict;
use parent 'LCM::Task::GenericStackProcessTask';
use SDB::Install::Globals qw($gHostRoleXS2Worker);
use LCM::Task::CommonTask::RemoveLocalRolesTask;
use LCM::Task::CommonTask::RemoveRemoteRolesTask;
use LCM::Task::CommonTask::RemoveRemoteXsControllerRolesTask;
use LCM::Task::CommonTask::RemoveRemoteHostsTask;
use LCM::Task::CommonTask::UpdateLocalHostTask;
use LCM::Task::CommonTask::UpdateRemoteHostsTask;
use LCM::Utils::RemoveHosts qw(IsRemovingRemoteHost);
use LCM::Utils::RemoveRoles qw(IsRemovingLocalRole IsRemovingRemoteRole IsRemovingRoleFromHost);
use LCM::Task::GenericStackTask::ComponentTask::UninstallTask;
use LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::SkippedComponentsManager qw ( $SCENARIO_UNINSTALL );
use LCM::Utils::ComponentsDependencySort qw(sortComponentsTopologically);
use LCM::Task qw($gFlavourProductName $gActionScope);

my $UNINSTALL_SUCCESS_END_MESSAGE = "Uninstallation of the $gFlavourProductName $gActionScope finished";
my $UNINSTALL_FAIL_END_MESSAGE    = "Uninstallation of the $gFlavourProductName $gActionScope failed";
my $UNINSTALL_ABORT_END_MESSAGE   = "Uninstallation of the $gFlavourProductName $gActionScope aborted";

my $UNINSTALL_NAME = "Uninstall $gFlavourProductName $gActionScope";
my $PROGRESS_MESSAGE_TEMPLATE = "Uninstalling $gFlavourProductName $gActionScope";

# Override
sub getId {
	my $self = shift;
	return $self->_getConfiguration()->getAction();
}

# Override
sub getName {
    return $_[0]->_formatMessage($UNINSTALL_NAME);
}

# Override
sub isUpdate {
	return 0;
}

# Override
sub getComponentManager {
	my $self = shift();
	return $self->_getConfiguration()->getSystemComponentManager();
}

#
# Only protected/private subs bellow this line
#

# Override
sub _shallAddDeployXSAComponentsTask { return 0; }

# Override
sub _executeInternal {
	my $self = shift();
	my $configuration = $self->_getConfiguration ();

	$self->_initActionMessages ();
	$self->_addProgressMsg ();
	$self->SUPER::_executeInternal ();
}

# Override
sub _createSubtasks {
	my $self = shift();
	my $configuration = $self->_getConfiguration ();
	my $componentManager = $self->getComponentManager();
	my $installedComponents = $componentManager->getAllComponents();
	my $selectedComponents = $self->_getSortedComponentList();
	my $localComponentsManager = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager($selectedComponents, $installedComponents, $SCENARIO_UNINSTALL);
	my $remoteComponentsManager = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager($selectedComponents, $installedComponents,$SCENARIO_UNINSTALL);

	if($localComponentsManager->existRegistrableComponents()){
		$self->addSubtask(new LCM::Task::CommonTask::UpdateLocalHostTask($self->_createLocalHostConfiguration($localComponentsManager)));
	}
	if($configuration->isDistributedSystem() && $remoteComponentsManager->existRegistrableComponents()){
		$self->addSubtask(new LCM::Task::CommonTask::UpdateRemoteHostsTask($self->_createRemoteHostConfiguration($remoteComponentsManager)));
	}
	$self->_addRemoveHostsAndRolesSubtasks();

	for my $selectedComponent (@{$selectedComponents}) {
# If we are uninstalling the whole system we should handle
# non-server components' failure to uninstall as a warning
		if ($self->_isServerSelected()) {
			next if $selectedComponent->getComponentBatchKey() eq 'lm_structure';

			my $isFailureCritical = ($selectedComponent->isServer()) ? 1 : 0;
			$self->addSubtask(new LCM::Task::GenericStackTask::ComponentTask::UninstallTask($configuration, $selectedComponent, $isFailureCritical));
		} else {
			$self->addSubtask(new LCM::Task::GenericStackTask::ComponentTask::UninstallTask($configuration, $selectedComponent, 1)); # 1=criticalOnFailure
		}
	}
}

sub _addRemoveHostsAndRolesSubtasks {
	my ($self) = @_;
	my $configuration = $self->_getConfiguration ();
	my $ownInstance = $configuration->getOwnInstance();
	my $xsControllerHost = $ownInstance->getXsControllerHostname();
	my $isLocalXsController = defined($xsControllerHost) && ($xsControllerHost eq $ownInstance->get_host());
	my $removeLocalXsControllerLast = $isLocalXsController && IsRemovingRoleFromHost($configuration, $xsControllerHost, $gHostRoleXS2Worker);
	my $removeRemoteXsControllerLast = $configuration->shouldRemoveXsControllerHostRolesSeparately();

	if(IsRemovingLocalRole($configuration) && !$removeLocalXsControllerLast){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveLocalRolesTask($configuration));
	}
	if(IsRemovingRemoteRole($configuration)){
		#if $removeRemoteXsControllerLast is true, a the removal of roles from the xs controller host will be skipped 
		$self->addSubtask(new LCM::Task::CommonTask::RemoveRemoteRolesTask($configuration));
	}
	if(IsRemovingRemoteHost($configuration)){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveRemoteHostsTask($configuration));
	}
	if(IsRemovingLocalRole($configuration) && $removeLocalXsControllerLast){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveLocalRolesTask($configuration));
	}
	if($removeRemoteXsControllerLast){
		$self->addSubtask(new LCM::Task::CommonTask::RemoveRemoteXsControllerRolesTask($configuration));
	}
}

# Override
sub _shallAssignAdditionalLocalRoles { return 0; }

# Override
sub _shallAssignAdditionalRemoteRoles { return 0; }

# Override
sub _initActionMessages {
	my ( $self ) = @_;
	my $status = $self->getStatus();
	$status->_setActionDoneWithWarningsMessage( "" );
	$status->_setActionDoneMessage($self->_formatMessage($UNINSTALL_SUCCESS_END_MESSAGE));
	$status->_setActionFailedMessage($self->_formatMessage($UNINSTALL_FAIL_END_MESSAGE));
	$status->_setActionAbortedMessage($self->_formatMessage($UNINSTALL_ABORT_END_MESSAGE));
}

sub _formatMessage {
    my ( $self, $message ) = @_;
    my $isServerUninstallation = $self->_isServerSelected();
    my $actionscope = $self->_isServerSelected() ? 'System' : 'Components';
    $message =~ s/$gActionScope/$actionscope/g;
    return $self->SUPER::_formatMessage($message);
}

sub _isInstallingServer {
    return 0;
}

# Override
sub _addProgressMsg {
	$_[0]->getMsgLst()->addProgressMessage($_[0]->_formatMessage($PROGRESS_MESSAGE_TEMPLATE));
}

sub _isServerSelected {
	my ($self) = @_;
	my $componentManager = $self->getComponentManager();
	return $componentManager->isHDBServerComponentSelected();
}

sub _getSortedComponentList {
	my ($self) = @_;
	my $componentManager = $self->getComponentManager();
	my $selectedComponents = $componentManager->getSelectedComponents();
	my $isServerSelected = $componentManager->isHDBServerComponentSelected();

	return $selectedComponents if($isServerSelected);
	return sortComponentsTopologically($componentManager, $selectedComponents, 1);
}

sub _createLocalHostConfiguration {
	my($self, $componentsManager) = @_;
	my $configuration = $self->_getConfiguration ();
	$componentsManager->detectComponents();
	my $skippedComponents = $componentsManager->getSkippedComponents();
	my $paramHash = {
		ComponentsRegistration => 'unregister',
		SkippedComponents => $skippedComponents,
		GenerateCertificates => 0,
		DeployShaConfigurations => 0,
		UpdateComponentListLocation => 0,
	};
	return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createLocalConfiguration($configuration, $paramHash);
}

sub _createRemoteHostConfiguration {
	my($self, $componentsManager) = @_;
	my $configuration = $self->_getConfiguration();
	$componentsManager->detectComponents();
	my $skippedComponents = $componentsManager->getSkippedComponents();
	my $remoteHostsObjects = defined($configuration->getRemoteHosts()) ? [ $configuration->getRemoteHosts() ] : [];

	my $paramHash = {
        ComponentsRegistration => 'unregister',
        SkippedComponents => $skippedComponents,
        GenerateCertificates => 0,
        DeployShaConfigurations => 0,
        UpdateComponentListLocation => 0,
    };
	return LCM::Configuration::Hosts::UpdateHosts::UpdateHostConfigurationFactory->createRemoteConfiguration($configuration, $remoteHostsObjects, $paramHash);
}

1;
