package LCM::Task::Unregister::UnregisterSubtask;

use strict;
use File::Spec;
use parent 'LCM::Task';
use LCM::ProcessExecutor;
use SDB::Install::SysVars qw($isWin);
use SDB::Install::Globals qw($gLogDir);
use SDB::Install::Configuration qw($bool_true_pattern);
use LCM::Task qw($gFlavourProductName $gActionScope);

my $UNREGISTER_NAME = "Unregister the $gFlavourProductName $gActionScope";
my $EXECUTION_NAME = "Unregistering the $gFlavourProductName $gActionScope";
my $ACTION_DONE_MESSAGE_TEMPLATE = "$gFlavourProductName $gActionScope unregistered";
my $ACTION_FAILED_MESSAGE_TEMPLATE = "Unregistration of $gFlavourProductName $gActionScope failed.";
my $ACTION_ABORTED_MESSAGE_TEMPLATE = "Unregistration of $gFlavourProductName $gActionScope aborted.";
my $ACTION_DONE_WITH_WARNINGS_MESSAGE_TEMPLATE = 'Unregistration of $gFlavourProductName $gActionScope finished with warnings';
my $PROGRESS_MESSAGE_TEMPLATE = "$EXECUTION_NAME...";
my $UNABLE_TO_GET_HDB = 'Unable to get HDB Instance';
my $PASSWORD_KEYS = ['Password', 'RootPassword', 'HostagentPassword'];

sub new {
	my ($class, $configuration) = @_;
	my $self = $class->SUPER::new($configuration);
	my $scope = ucfirst($configuration->getScope());
	my $status = $self->getStatus();

	$status->_setActionDoneMessage($self->_formatMessage($ACTION_DONE_MESSAGE_TEMPLATE));
	$status->_setActionDoneWithWarningsMessage($self->_formatMessage($ACTION_DONE_WITH_WARNINGS_MESSAGE_TEMPLATE));
	$status->_setActionFailedMessage($self->_formatMessage($ACTION_FAILED_MESSAGE_TEMPLATE));
	$status->_setActionAbortedMessage($self->_formatMessage($ACTION_ABORTED_MESSAGE_TEMPLATE));

	return $self;
}

# Override
sub getId {
	my ($self) = @_;
	my $configuration = $self->_getConfiguration();
	my $scope = lc($configuration->getScope());

	return "unregister_${scope}_internal";
}

# Override
sub getName {
	return $_[0]->_formatMessage($UNREGISTER_NAME);
}

# Override
sub getExecutionName {
    return $_[0]->_formatMessage($EXECUTION_NAME);
}

# Override
sub _executeInternal {
	my $self = shift();
	my $configuration = $self->_getConfiguration();
	my $scope = ucfirst($configuration->getScope());
	my $progressMessage = $self->_formatMessage($PROGRESS_MESSAGE_TEMPLATE);
	my $message = $self->getMsgLst()->addProgressMessage($progressMessage);
	my $saveContext = $self->setMsgLstContext([$message->getSubMsgLst()]);
	my $progressHandler = $self->getMsgLst()->getProgressHandler();
	my $currentDepth = $progressHandler->getIntendationDepth();

	$progressHandler->setIntendationDepth($currentDepth + 1);

	if (!defined $self->_unregister($configuration)) {
		$self->getStatus()->_setErrorState();
	} else {
		$self->getStatus()->_setFinishedState();
	}

	$progressHandler->setIntendationDepth($currentDepth);
	$message->endMessage(undef, $self->getStatus()->getMessage());
	$self->setMsgLstContext($saveContext);
}

# Windows is currently not supported
sub _unregister {
	my ($self, $configuration) = @_;

	if($isWin){
		$self->setErrorMessage('Windows currently not supported.');
		return undef;
	}
	my $hdbInstance = $configuration->getOwnInstance();
	if(!defined $hdbInstance){
		$self->setErrorMessage($UNABLE_TO_GET_HDB);
		return undef;
	}

	my $executor = $self->_createProcessExecutor($configuration, $hdbInstance);
	my $returnCode = $executor->executeProgram();

	$self->_setLogLocation($self->_parseLogFileLocation($executor->getOutputLines()));

	if (!defined $returnCode || $returnCode){
		my $message = $self->setErrorMessage ("Execution of hdbreg failed");
		return undef;
	}
	return 1;
}

sub _createProcessExecutor {
	my ($self, $configuration, $hdbInstance) = @_;
	my $command = File::Spec->catfile($hdbInstance->get_globalTrexInstallDir(), 'bin', 'hdbreg');
	my $arguments = $self->_createHdbToolArgs($configuration);
	my $input = $configuration->getXmlPasswordStream($PASSWORD_KEYS);

	push(@{$arguments}, '--read_password_from_stdin=xml') if (defined $input);
	push(@{$arguments}, '--skip_hostagent_calls=true') if ($configuration->shouldSkipHostagentCalls());

	my $executor = new LCM::ProcessExecutor($command,$arguments, $input);
	$executor->setMsgLstContext($self->getMsgLstContext());
	$executor->setOutputHandler($self->getMsgLst()->getProgressHandler());

	return $executor;
}

sub _createHdbToolArgs {
	my ($self, $configuration) = @_;
	my $scope = lc($configuration->getScope());
	my @args = ('-b', "--unregister=$scope", "--instlog_dir=$gLogDir");

	for my $parameterId (@{$configuration->getParamIds()}){
		next if($configuration->isSkipped($parameterId));
		next if(!$configuration->hasValue($parameterId));
		next if($configuration->getType($parameterId) =~ /passwd/);

		my $cmdOption = $configuration->getOpt($parameterId);
		my $value = $configuration->getValue($parameterId);
		if($configuration->getType($parameterId) =~ /bool/){
			$value = ($value =~ $bool_true_pattern) ? 'on' : 'off';
		}
		push(@args, sprintf('%s=%s', $cmdOption, $value));
	}

	my $timeoutsArgument = $self->_getTimeoutsOptionString($configuration);
	push @args, $timeoutsArgument if (length($timeoutsArgument) > 0);

	return \@args;
}

sub _getTimeoutsOptionString {
	my ($self, $configuration) = @_;
	my $optTimeout = $configuration->getOptionTimeout();
	if (defined $optTimeout){
		my $timeoutArgs = $optTimeout->getSetKeysValidFor ($optTimeout);
		my $timeoutStr  = $optTimeout->getOptionStringFromArgList ($timeoutArgs);
		return $timeoutStr if ($timeoutStr);
	}
	return '';
}

sub _getNumberOfExpectedOutputLines {
	return 50;
}

1;