package LCM::Utils::CommonUtils;

use strict;
use parent 'Exporter';
use SDB::Install::SAPSystem;
use SDB::Install::Globals qw($gDirNameCockpitStack);
use LCM::Manifests::GenericManifest;
use experimental qw(smartmatch);

our @EXPORT_OK = qw(
	isSingleDB
	getSidadmName
	getSystemSidadmUid
	getSystemSidadmGid
	isSystemReplication
	findAutoConfiguration
	getLocalHostSidadmUid
    getLocalHostSidadmGid
	getSpecialOptionString
	removeElementFromArray
	getHdbmodifyTimeoutValues
	getXSInstallerTimeoutValues
	getHdblcmUploadDirectory
	getCockpitProductVersion
);

sub getHdblcmUploadDirectory {
    require File::Spec;

    my ($sid) = @_;
    return File::Spec->catfile('/usr', 'sap', $sid, 'hdblcm_uploads', 'SAPHostAgentUploads');
}

sub findAutoConfiguration {
    my ($action) = @_;
    $action = $action || 'install';

    require LCM::Installer;
    require File::Spec;
    require File::Basename;

    my $installer = new LCM::Installer();
    my $runtimeDir = $installer->getRuntimeDir();
    return undef if ! -e $runtimeDir;
    my $autoConfigFile = File::Spec->catfile(File::Basename::dirname($runtimeDir), 'configurations', "auto_$action.cfg");

    return (-f $autoConfigFile) ? $autoConfigFile : undef;
}

sub getSpecialOptionString {
    my ($sourceOption, $targetOption) = @_;
    my $optionArguments = defined($sourceOption) && defined($targetOption) ? $sourceOption->getSetKeysValidFor($targetOption) : undef;
    my $optionString = defined($optionArguments) ? $sourceOption->getOptionStringFromArgList($optionArguments) : undef;

    return $optionString;
}

sub removeElementFromArray {
	my ($array, $removeElement) = @_;

	if (not ($removeElement ~~ @$array)) {
		return $array;
	}

	my $index=0;
	$index++ until @$array[$index] eq $removeElement;
	splice(@$array, $index, 1);

	return $array;
}

sub getHdbmodifyTimeoutValues {
	return ['start_instance', 'start_service']; # Should be updated if hdbmodify changes its timeout support
}

sub getXSInstallerTimeoutValues {
	return ['start_instance', 'start_service', 'stop_instance', 'stop_service']; # Should be updated if hdbinst of XSA changes its timeout support
}

sub getSidadmName {
	my ($configuration) = @_;
	return undef if ( ! defined $configuration->getValue('SID'));
	return lc($configuration->getValue('SID')) . "adm";
}

sub getSystemSidadmUid {
	my ($configuration) = @_;
	my $sapSystem = $configuration->getSAPSystem();
    my $userConfigurationFile = (defined $sapSystem) ? $sapSystem->getUserConfig() : undef;
	if (!defined $userConfigurationFile){
		$configuration->setErrorMessage(undef, $sapSystem->getErrMsgLst()) if defined ($sapSystem);
		return undef;
	}
	return $userConfigurationFile->{sidadm_id};
}

sub getSystemSidadmGid {
	my ($configuration) = @_;
	my $sapSystem = $configuration->getSAPSystem();
	my $userConfigurationFile = $sapSystem->getUserConfig();
	if (!defined $userConfigurationFile){
		$configuration->setErrorMessage( undef, $sapSystem->getErrMsgLst() );
		return undef;
	}
	return $userConfigurationFile->{sapsys_groupid};
}

sub getLocalHostSidadmUid {
	my ($configuration) = @_;
	my $sidadmName = getSidadmName($configuration);
	return undef if ( ! defined $sidadmName);
	my $localHostSidadmUid = getpwnam($sidadmName);
	return $localHostSidadmUid;
}

sub getLocalHostSidadmGid {
    my ($configuration) = @_;
    my $sidadmName = getSidadmName($configuration);
    return undef if ( ! defined $sidadmName);
    my (undef, undef, undef, $localHostSidadmGid) = getpwnam($sidadmName);
    return $localHostSidadmGid;
}

sub isSingleDB {
    require LCM::Installer;
    my $installer  = new LCM::Installer();
    my $sid        = $installer->getSid();
    my $allSystems = CollectSAPSystems( undef, 1 );
    my $sapSys     = (defined $sid) ? $allSystems->{$sid} : undef;
    my $instance   = (defined $sapSys && $sapSys->hasNewDB())
                     ? $sapSys->getNewDBInstances()->[0] : undef;
    my $dbMode     = (defined $instance) ? $instance->getDbMode() : undef;
    my $isSingleDb = ( defined $dbMode && $dbMode eq 'singledb' );

    return $isSingleDb;
}

sub isSystemReplication {
    my ($configuration) = @_;
    my $sapSys     = $configuration->getSAPSystem();
    my $instance   = defined($sapSys) && $sapSys->hasNewDB() ? $sapSys->getNewDBInstances()->[0] : undef;

    my $msgLst     = defined($configuration) ? $configuration->getMsgLst : undef;
    my $errMsgLst  = defined($configuration) ? $configuration->getErrMsgLst : undef;
    my $mode       = defined($instance) ? $instance->getReplicationMode($msgLst, $errMsgLst) : 'none';

    return undef if (!defined $mode); # Error calling 'hdbnsutil'
    return ($mode ne 'none');

}

sub isSAPEnvironment {
    require Net::Ping;

    my $slaHostAvailabilityChecker;
    my $result = 0; # false by default
    eval{
         $slaHostAvailabilityChecker = Net::Ping->new();
    };
    if ($@){
        # write message in trace file
        require SDB::Install::MsgLst;
        new SDB::Install::MsgLst ()->addError ("Cannot ping SLANALYTICSFTP: $@");
        return $result;
    }
# Since its last version (2.55), Net::Ping::ping started croaking whenever the host cannot be resolved
# The eval block prevents hdblcm from exiting with unhandled exception in such cases
    eval {
        $result = $slaHostAvailabilityChecker->ping('SLANALYTICSFTP.wdf.sap.corp', 2) ? 1 : 0;
    };
    $slaHostAvailabilityChecker->close();
    return $result;
};

sub getCockpitProductVersion {
	my ($sapSys) = @_;
	my $manifestDir = File::Spec->catdir($sapSys->get_globalSidDir(), $gDirNameCockpitStack);
	my $manifestPath = File::Spec->catfile($manifestDir, 'manifest');
	my $manifest = new LCM::Manifests::GenericManifest($manifestPath);
	my $cockpitVersion = $manifest->getVersion();
	return (defined $cockpitVersion)? $cockpitVersion : $sapSys->GetVersion();
}

1;