package LCM::Utils::ComponentActionStringGenerator::InstallUpdate;

use strict;
use SDB::Install::Version;
use parent 'LCM::Utils::ComponentActionStringGenerator';

sub getComponentKeynameToActionMap {
	my ($self, $configuration, $componentManager, $systemComponentManager) = @_;
	my $ignorePendingAction = $configuration->getIgnore('check_pending_upgrade');
	my %componentKeynameToAction = ();

	return ([], \%componentKeynameToAction) if (!defined($componentManager));

	for my $component (@{$componentManager->getAllComponents()}) {
		my $keyname = $component->getComponentKeyName();
		my $installedComponent = defined($systemComponentManager) ? $systemComponentManager->getComponentByKeyName($keyname) : undef;

		if ($self->_hasPendingAction($configuration, $component)) {
			my $cmpPendingAction = $self->_getComponentKeynameToPendingAction($installedComponent, $component, $configuration);
			if (defined $cmpPendingAction) {
				$componentKeynameToAction{$keyname} = $cmpPendingAction;
			}
			next;
		}
		if (! defined($installedComponent)) {
			$componentKeynameToAction{$keyname} = $self->_getComponentKeynameToActionInstall($component);
			next;
		}
		if($self->_canUpgradeComponent($installedComponent, $component, $configuration)) {
			$componentKeynameToAction{$keyname} = $self->_getComponentKeynameToActionUpdate($installedComponent, $component);
		}
	}

	if ($configuration->can('getDeprecatedComponentsForUninstall')) {
		for my $component (@{$configuration->getDeprecatedComponentsForUninstall()}) {
			my $keyname = $component->getComponentKeyName();
			$componentKeynameToAction{$keyname} = $self->_getComponentKeynameToActionUninstall($component);
		}
	}

	my $orderedKeynames = $self->getOrderedKeynames(\%componentKeynameToAction, $componentManager, $systemComponentManager);
	return ( $orderedKeynames, \%componentKeynameToAction );
}

sub _hasPendingAction {
	my ($self, $configuration, $component) = @_;

	return 0 if($configuration->getIgnore('check_pending_upgrade'));
	return ($component->hasPersistenceFile($configuration) && defined $component->getPersistenceXMLObject($configuration)) ? 1 : 0;
}

sub _getComponentKeynameToPendingAction {
	my ($self, $installedComponent, $targetComponent, $configuration) = @_;
	my $targetComponentName = $targetComponent->getComponentName();
	my $pendingXmlObject = $targetComponent->getPersistenceXMLObject($configuration);
	my $sourceVersionString = $pendingXmlObject->getSourceVersionString();
	my $targetVersionString = $pendingXmlObject->getTargetVersionString();
	my $targetVersionObject = new SDB::Install::Version(split('\.', $targetVersionString));
	my $availableVersionObject = new SDB::Install::Version(split('\.', $targetComponent->getVersion()));
	my $isUpdate = ($pendingXmlObject->getActionString() =~ /update/);
	my $canResume = $availableVersionObject->isEqual($targetVersionObject);

	if($canResume){
		if (!$isUpdate) {
			return sprintf("Resume installation of $targetComponentName version %s", $targetVersionString);
		}
		if (!$configuration->isPrepareUpdate()) {
			return sprintf("Resume update of $targetComponentName from version %s to %s", $sourceVersionString, $targetVersionString);
		}
	}

	if(!$isUpdate) {
		return sprintf("Install $targetComponentName version %s", $targetComponent->getVersion());
	}

	if ($self->_canUpgradeComponent($installedComponent, $targetComponent, $configuration)) {
		return $self->_getComponentKeynameToActionUpdate($installedComponent, $targetComponent);
	}

	return undef;
}

sub _getComponentKeynameToActionUpdate {
	my ($self, $installedComponent, $component) = @_;
	my $name = $component->getComponentName();
	my $installedVersion = $installedComponent->getVersion();
	my $availableVersion = $component->getVersion();
	return "Update $name from version $installedVersion to version $availableVersion";
}

sub _getComponentKeynameToActionInstall {
	my ($self, $component) = @_;
	my $name = $component->getComponentName();
	my $availableVersion = $component->getVersion();
	return "Install $name version $availableVersion";
}

sub _getComponentKeynameToActionUninstall {
	my ($self, $component) = @_;
	my $name = $component->getComponentName();
	my $availableVersion = $component->getVersion();
	return "Uninstall $name version $availableVersion";
}

sub _canUpgradeComponent {
	my ($self, $installedComponent, $targetComponent, $configuration) = @_;
	my $shouldIgnoreVersionChecks = $configuration->getIgnore('check_version');

	if (!defined $installedComponent->canUpgradeComponent()){
		my $sourceManifest = $installedComponent->getManifest();
		my $targetManifest = $targetComponent->getManifest();
		my $msgLst = new SDB::Install::MsgLst();
		my $canUpgradeComponent = $configuration->canUpgradeComponent($sourceManifest, $targetManifest, $targetComponent->getComponentName(), $msgLst);

        if (!$canUpgradeComponent) {
            if ($shouldIgnoreVersionChecks) {
                $configuration->ResetError();
                $configuration->getMsgLst()->addMessage("Ignoring error due to command line switch '--ignore'");
                $canUpgradeComponent = 1;
            } elsif (!$targetComponent->isInternal()) {
                my $restrictedComponentMessage = $msgLst->getMsgLstString();
                $configuration->addRestrictedComponentsMessage($targetComponent->getComponentName(), $$restrictedComponentMessage);
            }
        }

		$installedComponent->setCanUpgradeComponent($canUpgradeComponent);
	}
	return $installedComponent->canUpgradeComponent();
}

1;