package LCM::Utils::SlManifestXmlParser;

use strict;
use SAPDB::Install::Jar;
use SDB::Common::Utils qw(createXMLParser);
use SDB::Install::SysVars qw($currentPlatformName);
use LCM::Utils::XsApplicationDependency::InclusiveDependency;
use LCM::Utils::XsApplicationDependency::ExclusiveDependency;

my $SL_MANIFEST_PATTERN = 'META-INF.SL_MANIFEST\.XML$';
my $CAPTION_X_PATH = '/software-component-version/caption';
my $COMPONENT_KEY_X_PATH = '/software-component-version/software-component-version-key/name';
my $SCV_VERSION_X_PATH = '/software-component-version/software-component-version-key/version';
my $SP_LEVEL_X_PATH = '/software-component-version/sp/sp-key/sp-level';
my $PATCH_LEVEL_X_PATH = '/software-component-version/sp/patch-level';
my $RUNTIME_TYPE_X_PATH = '/software-component-version/runtime-type';
my $PPMS_ID_X_PATH = '/software-component-version/software-component-version-key/PPMS-ID';
my $DEPENDENCIES_X_PATH = '/software-component-version/dependencies';

sub new {
	require XML::LibXML;
	return bless({}, shift());
}

sub getParsedData {
	my ($self, $archivePath) = @_;
	my $slManifestXmlString = $self->_readSlManifestXmlAsString($archivePath);

	if(!defined($slManifestXmlString) || length($slManifestXmlString) == 0){
		return undef;
	}

	return $self->_extractDataFromXml($slManifestXmlString);
}

sub _getPpmsId {
	my ($self, $xPathContext) = @_;
	my $nodeList = $xPathContext->findnodes($PPMS_ID_X_PATH);
	if($nodeList->size() > 0) {
		return $nodeList->get_node(1)->textContent();
	}
	return $self->_getComponentKey($xPathContext);
}

sub _getRuntimeType {
	my ($self, $xPathContext) = @_;
	my $nodeList = $xPathContext->findnodes($RUNTIME_TYPE_X_PATH);
	return $nodeList->get_node(1)->textContent();
}

sub _getComponentKey {
	my ($self, $xPathContext) = @_;
	my $nodeList = $xPathContext->findnodes($COMPONENT_KEY_X_PATH);
	return lc($nodeList->get_node(1)->textContent());
}

sub _getScvVersion {
	my ($self, $xPathContext) = @_;
	return $xPathContext->findnodes($SCV_VERSION_X_PATH)->get_node(1)->textContent();
}

sub _getSpPatchLevel {
	my ($self, $xPathContext) = @_;
	return $xPathContext->findnodes($SP_LEVEL_X_PATH)->get_node(1)->textContent();
}

sub _getPatchLevel {
	my ($self, $xPathContext) = @_;
	return $xPathContext->findnodes($PATCH_LEVEL_X_PATH)->get_node(1)->textContent();
}

sub _getCaption {
	my ($self, $xPathContext) = @_;
	my $nodeList = $xPathContext->findnodes($CAPTION_X_PATH);
	if($nodeList->size() > 0) {
		my $captionString = $nodeList->get_node(1)->textContent();
		return $captionString if(length($captionString) > 0);
	}
	return $self->_getComponentKey($xPathContext);
}

sub _extractDataFromXml {
	my ($self, $xmlString) = @_;
	my $parser = createXMLParser();
	my $tree = $parser->parse_string($xmlString);
	my $xPathContext = XML::LibXML::XPathContext->new($tree);

	$xPathContext->registerNs('tns' => '');

	return {
		'compversion-id' => $self->_getPpmsId($xPathContext),
		'keycaption' => $self->_getCaption($xPathContext),
		'component-key' => $self->_getComponentKey($xPathContext),
		'keyname' => $self->_getComponentKey($xPathContext),
		'runtime-type' => $self->_getRuntimeType($xPathContext),
		'platform' => $currentPlatformName,
		'release' => $self->_getScvVersion($xPathContext),
		'rev-number' => $self->_getSpPatchLevel($xPathContext),
		'rev-patchlevel' => $self->_getPatchLevel($xPathContext),
		'dependencies-array' => $self->_getDependenciesArray($xPathContext),
	};
}

sub _readSlManifestXmlAsString {
	my ($self, $archivePath) = @_;
	my $archive = SAPDB::Install::Jar->new();

	$archive->OpenArchive($archivePath);

	for (my $i = 0; $i < $archive->{'num_of_files'}; $i++) {
		my $file = $archive->Next();

		if ($file->{'filename'} =~ /$SL_MANIFEST_PATTERN/) {
			$archive->Open();
			my $result = '';
			while(1) {
				my $line = $archive->ReadLine();
				last if(!defined($line));
				$result .= $line;
			}
			$archive->Close();
			$archive->CloseArchive();
			return $result;
		}
	}
	$archive->CloseArchive();
	return undef;
}

sub _getDependenciesArray {
	my ($self, $xPathContext) = @_;
	my $nodeList = $xPathContext->findnodes($DEPENDENCIES_X_PATH);

	return [] if($nodeList->size() == 0);

	my $node = $nodeList->get_node(1)->nonBlankChildNodes()->get_node(1);
	my $nodeName = $node->nodeName();

	if($nodeName =~ /requires/){
		return [ $self->_getRequiresTagDependencies($xPathContext, $node) ];
	}
	if ($nodeName =~ /not/) {
		return [ $self->_getNotTagDependencies($xPathContext, $node) ];
	}
	if ($nodeName =~ /and/) {
		return $self->_getAndTagDependencies($xPathContext, $node);
	}
	if ($nodeName =~ /or/){
		return $self->_getOrTagDependencies($xPathContext, $node);
	}
	return [];
}

sub _getNotTagDependencies {
	my ($self, $xPathContext, $node) = @_;
	my $technicalName = $xPathContext->findvalue('software-component-version-key/name', $node);
	my $caption = $xPathContext->findvalue('caption', $node);
	my $scvVersion = $xPathContext->findvalue('software-component-version-key/version', $node);
	my $spLevel = $xPathContext->findvalue('sp/sp-key/sp-level', $node);
	my $patchLevel = $xPathContext->findvalue('sp/patch-level', $node);

	return [ new LCM::Utils::XsApplicationDependency::ExclusiveDependency($caption, lc($technicalName), $scvVersion, $spLevel, $patchLevel) ];
}

sub _getAndTagDependencies {
	my ($self, $xPathContext, $node) = @_;
	my @childNodes = $node->nonBlankChildNodes();
	my $result = [[]];

	for my $childNode (@childNodes) {
		my $nodeName = $childNode->nodeName();

		if($nodeName =~ /requires/) {
			for my $existingDependencies (@{$result}) {
				push(@{$existingDependencies}, @{$self->_getRequiresTagDependencies($xPathContext, $childNode)});
			}
		} elsif ($nodeName =~ /not/){
			for my $existingDependencies (@{$result}) {
				push(@{$existingDependencies}, @{$self->_getNotTagDependencies($xPathContext, $childNode)});
			}
		} else {
			my $dependencies = ($nodeName =~ /and/) ? $self->_getAndTagDependencies($xPathContext, $childNode) : $self->_getOrTagDependencies($xPathContext, $childNode);
			my $updated = [];
			for my $dependecy (@{$dependencies}){
				for my $existingDependencies (@{$result}) {
					my $newDependency = [ @{$dependecy}, @{$existingDependencies} ];
					push(@{$updated}, $newDependency);
				}
			}
			$result = $updated;
		}
	}
	return $result;
}

sub _getOrTagDependencies {
	my ($self, $xPathContext, $node) = @_;
	my @childNodes = $node->nonBlankChildNodes();
	my $result = [];

	for my $childNode (@childNodes) {
		my $nodeName = $childNode->nodeName();

		if($nodeName =~ /requires/) {
			push(@{$result}, $self->_getRequiresTagDependencies($xPathContext, $childNode));
		} elsif ($nodeName =~ /not/){
			push(@{$result}, $self->_getNotTagDependencies($xPathContext, $childNode));
		} elsif ($nodeName =~ /and/) {
			push(@{$result}, @{$self->_getAndTagDependencies($xPathContext, $childNode)});
		} elsif ($nodeName =~ /or/) {
			push(@{$result}, @{$self->_getOrTagDependencies($xPathContext, $childNode)});
		}
	}
	return $result;
}

sub _getRequiresTagDependencies {
	my ($self, $xPathContext, $node) = @_;
	my $technicalName = $xPathContext->findvalue('software-component-version-key/name', $node);
	my $caption = $xPathContext->findvalue('caption', $node);
	my $scvVersion = $xPathContext->findvalue('software-component-version-key/version', $node);
	my $spLevel = $xPathContext->findvalue('sp/sp-key/sp-level', $node);
	my $patchLevel = $xPathContext->findvalue('sp/patch-level', $node);

	return [ new LCM::Utils::XsApplicationDependency::InclusiveDependency($caption, lc($technicalName), $scvVersion, $spLevel, $patchLevel) ];
}

1;
