package SDB::Install::App::AnyApp;

use base SDB::Install::App;

use Getopt::Long;

use SDB::Install::Log;
use SDB::Install::Globals qw($gLogDir $gLogTime $gProductName $gSapsysGroupName);
use SDB::Install::System  qw(getFilesTimestampPID isAdmin isPASE makedir);
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Tools   qw(getFilenameTimeStamp iso8601DateTime);

use strict;


#-------------------------------------------------------------------------------
# Constructor
sub new {

    my $self = shift->SUPER::new();

    return $self;
}

sub setCommonLogDir {

	my ($self, $envVarName, $filenameTemplate) = @_;

	if (!defined $envVarName || !defined $filenameTemplate) {
		return undef;
	}

	if (!exists $ENV{$envVarName}) {
		# common log dir not set in environment
		return 1;
	}

	my $dir = $ENV{$envVarName};
	if (-d $dir) {
		unless (opendir (DH, $dir)){
			$self->setErrorMessage ("Cannot open existing log directory \"".$dir."\": $!");
			return undef;
		}
		my @files = readdir (DH);
		closedir (DH);
		if (@files > 2) {
			$self->setErrorMessage ("Existing log directory \"".$dir."\" is not empty.");
			return undef;
		}
	} elsif (-e $dir) {
		$self->setErrorMessage ("\"".$dir."\" exists but is not a directory.");
		return undef;
	}

	# create common log dir
	if (!defined makedir($dir)) {
		$self->setErrorMessage ("Cannot create log directory \"".$dir."\".");
		return undef;
	}

	$ENV{SDBINSTLOGFILE} = $dir . $path_separator . $filenameTemplate;

	return 1;

}

#-------------------------------------------------------------------------------
# Adds the log file to the trace directory without displaying the log file name.
#
# The procedure 'addLogFileLocation' replaces the pattern '_LOGDATETIME'
# with the current data/time.
#
# Parameters: SDB::Install::SAPSystem $sapSys
#             string                  $programName  # e.g. hdbrename

sub addLogToTraceDir {

    my ($self,
        $sapSys,         # SDB::Install::SAPSystem
        $programName,
        $checkOnly,
        $wantedHostname, # if undef, local hostname of TrexInstance is used
        $wantedUid,      # if undef, uid from $sapSys is used
        $wantedGid       # if undef, gid from $sapSys is used
       ) = @_;

    my $instance = $sapSys->getNewDBInstances ()->[0];

    if (!defined $instance) {
        return 0;
    }

    my $traceDir = $instance->get_hostNameTraceDir($wantedHostname);
    my $actionID = $self->{options}->{action_id};
    my $extension= ($checkOnly) ? '.check.log' : '.log';
    my $traceFilename;

    my $oldLogFiles = getFilesTimestampPID($traceDir, $programName, $extension);

    foreach my $currOldLog (@$oldLogFiles) {
        unlink $traceDir . $path_separator . $currOldLog;
    }

    if (defined $actionID) {
        my $pattern = '^' . $programName;
        $traceFilename =
            ($self->{options}->{is_action_master} && ($actionID =~/^$programName/))
            ? $actionID
            : $programName . '_' . $actionID;
    }
    else {
        $traceFilename = $programName . '_LOGDATETIME';
    }

    $traceFilename .= $extension;

    my $slaveCopy = undef;

    if ($self->{options}->{is_action_master}) {
        my $hosts = $instance->get_hosts();
        if (defined $hosts && @$hosts) {
            my $logGID = $self->{options}->{instlog_gid};
            if (!$isWin && !defined $logGID) {
                $logGID = getgrnam($gSapsysGroupName);
            }
            $slaveCopy = {'actionID'    => $actionID,
                          'instanceDir' => $instance->get_instanceDir(),
                          'destinDir'   => $gLogDir,
                          'hostNames'   => $hosts,
                          'uid'         => $self->{options}->{instlog_uid},
                          'gid'         => $logGID,
                         };
        }
    }

    my $uid = undef;
    my $gid = undef;

    if (!$isWin) {
        $uid = (defined $wantedUid) ? $wantedUid : $sapSys->getUID();
        $gid = (defined $wantedGid) ? $wantedGid : $sapSys->getGID();
    }

    $self->{log}->addLogFileLocation(LOG_FORMAT_PLAIN,
                                     $traceDir,
                                     3,
                                     $traceFilename,
                                     LOG_DONT_INDICATE_LOCATION_ON_STDOUT,
                                     $uid,
                                     $gid,
                                     LOG_DESTINATION_TRY_ONLY,
                                     $slaveCopy);

     return 1;
}


#-------------------------------------------------------------------------------
# Adds message containing the start date.
#
# Parameters: $msglst      SDB::Install::MsgLst
#             $programInfo string  (e.g. hdbrename)

sub addStartDateMsg {

    my ($self, $msglst, $programInfo) = @_;

    my $strTime = substr (iso8601DateTime(), 0, 10);

    # message: 20:36:32.713 - INFO: Start Date: 2013-01-17  -  hdbrename
    $msglst->addMessage("Start Date: $strTime  -  $programInfo");
    if (defined $self->{options}->{action_id}) {
        $msglst->addMessage('Log File ID: ' . $self->{options}->{action_id});
    }
}

#-------------------------------------------------------------------------------
# Returns undef and writes an error message if working without administrator
#                                                                     privilege.
# Parameter: string $programName   # e.g. hdbrename
#
# Returns undef in case of an error


sub checkIsAdmin {

    my ($self, $programName) = @_;

    if (!isAdmin ()){
        $self->setErrorMessage ( $isWin ? "Please start $programName as local administrator!" :
                                ( isPASE() ? "Please restart $programName as user QSECOFR!" :
                                             "Please restart $programName as user root!"));
        return undef;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Initializes this application
# Without parameters
# Returns int retCode

sub InitApp {

    my ($self) = @_;

    $self->SUPER::InitApp();
    return 1;
}


#-------------------------------------------------------------------------------
# Returns the name of this program

sub getProgramName {
    return $SAPDB::Install::Config{ProgramName};
}


#-------------------------------------------------------------------------------
# Returns the action of this program.

sub getAction {
    my $action = $_[0]->{action};
    return (defined $action) ? $action : 'Execute';
}


#-------------------------------------------------------------------------------
# Returns the completed action of this program.

sub getActionDone {
    my $actionDone = $_[0]->{actionDone};
    return (defined $actionDone) ? $actionDone : 'executed';
}


#-------------------------------------------------------------------------------
# Returns the progressive action of this program.

sub getActionProgressive {
    my $actionProgressive = $_[0]->{actionProgressive};
    return (defined $actionProgressive) ? $actionProgressive : 'Executing';
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing help/info options of the program.

sub GetHelpSwitches {
    my ($self, $indent)= @_;
    return [@{$self->SUPER::GetHelpSwitches($indent)}, '--list_systems'];
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing help/info options
# with additional help options

sub GetModifiedHelpSwitches {

    my ($self, $additionalHelpOptions, $leftIndent) = @_;

    my $switches = $self->SUPER::GetHelpSwitches($leftIndent);

    foreach my $i (0 .. (scalar @$switches -1)) {

        if ($switches->[$i] =~ /help/) {

            my $currLineLen = 0;
            my $isFirst     = 1;
            my $line        = ' [';
            my $indent      = $leftIndent . ( ' ' x length($switches->[$i]) );
            my $lineLimit   = 100 - length($indent);

            foreach my $currOpt (@$additionalHelpOptions) {

                if ($isFirst) {
                    $line       .= $currOpt;
                    $currLineLen = length($line);
                    $isFirst     = 0;
                    next;
                }

                my $len = length($currOpt);

                if ($currLineLen + $len > $lineLimit) {
                    $line       .= "\n$indent  ";
                    $currLineLen = 0;
                }
                $line        .= " | $currOpt";
                $currLineLen += $len;
            }
            $switches->[$i] .= $line . ']';
            last;
        }
    }

    return [@$switches, '--list_systems'];
}


#-------------------------------------------------------------------------------
# Returns a derived class of SDB::Install::Configuration::AnyConfig

sub getInstconfig {
    return $_[0]->{instconfig};
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of password tags.

sub GetPasswordUsage {
    if (!defined $_[0]->getInstconfig()){
        return [];
    }
    return $_[0]->getInstconfig()->GetPasswordUsageFromParams();
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing the program options
# without help/info options.

sub GetSwitches{

    my $instconfig = $_[0]->getInstconfig();

    return (defined $instconfig) ? $instconfig->GetSwitchesFromParams() : [];
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays
# containg the description of program options without common batch/help options.

sub GetUsage{
    my $usage = [];
    if (defined $_[0]->getInstconfig()){
        $usage = $_[0]->getInstconfig()->GetUsageFromParams();
    }
    return [@$usage,
            ['--list_systems', '-L', "Shows installed $gProductName systems"]];
}


#-------------------------------------------------------------------------------
# Displays the program result and returns an evaluated return code.

sub handleReturnCodes() {

    my ($self, $err, $rc) = @_;

    my $msglst  = $self->getMsgLst ();
    my $retcode = $rc;

    if ($err){
        my $signalInfoText;
        if ($err =~ /^__SIGINT__/){
            $msglst->addMessage ('User canceled '
                                 . $self->getProgramName() . ' with Ctrl + c');
            $retcode = 2;
        }
        elsif ($err =~ /^__SIGPIPE__/){
            $self->setErrorMessage ('Broken pipe',
                                   $self->{stackBacktraceMsglst});
            $signalInfoText = $self->getSignalInfoText ();
            undef $retcode;
        }
        elsif ($err =~ /^__SIG(\w+)__/){
            $self->setErrorMessage ("Caught signal $1",
                                   $self->{stackBacktraceMsglst});
            $signalInfoText = $self->getSignalInfoText ();
            undef $retcode;
        }
        else{                        
            $self->setErrorMessage ('unhandled exception: '. $err,
                                   $self->{stackBacktraceMsglst});
            undef $retcode;
        }
        if ($signalInfoText){
            $self->appendErrorMessage ($signalInfoText);
        }
    }
    $self->printMessageByReturnCode($retcode);

    return $retcode;
}

sub printMessageByReturnCode{
	my ($self, $retcode) = @_;
	my $msglst  = $self->getMsgLst ();
	
	if (defined $retcode) {
		if ($retcode == 2){
            $msglst->addProgressMessage($self->getAbortedProgressMessage());
        }
        elsif ( defined $self->getActionDone() ){
            $msglst->addProgressMessage($self->getDoneProgressMessage());
        }

    }
    else{
        $self->ShowErrorMsg($self->getFailedProgressMessage(), $self->getErrMsgLst());
    }
}

sub getAbortedProgressMessage {
	return $_[0]->getAction() . ' ' . $_[0]->getActionScope() . ' aborted!';
}

sub getFailedProgressMessage {
	return $_[0]->getActionProgressive() . ' ' . $_[0]->getActionScope() . ' failed!';
}

sub getDoneProgressMessage {
	return $_[0]->getActionScope() . ' ' . $_[0]->getActionDone() . '.';
}

sub getActionScope {
    my ($self) = @_;
    my $scopeName  = $self->getScope();
    my $systemName = $gProductName;
    $systemName   .= (defined $scopeName) ? ' ' . $scopeName : ' instance';
    return $systemName;
}

sub getScope {
    my ($self) = @_;
    my $instconfig = $self->getInstconfig();
    return (defined $instconfig) ? $instconfig->getValue('Scope') : undef;
}

#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs {

    my ($self, $args, $pass_through) = @_;
    if (defined $self->{cmdLineArgsAfterInit}){
        return $self->{cmdLineArgsAfterInit};
    }

    if (!defined $pass_through){
        $pass_through = 0;
    }

    my $instconfig = $self->getInstconfig();
    if (defined $instconfig ){
        $instconfig->SetProgressHandler ($self->getMsgLst()->getProgressHandler ());
        if (!$instconfig->InitDefaults ()){
            $self->setErrorMessage (undef, $instconfig->getErrMsgLst ());
            return undef;
        }
    }

    my $rc = $self->SUPER::InitCmdLineArgs ($args,
                            defined $self->getInstconfig () || $pass_through);
    if (!defined $rc){
        return undef;
    }

    if (!defined $self->{options}->{action_id}) {
        $self->{options}->{is_action_master} = 1;
        $self->{options}->{action_id} = $self->getProgramName()
                                        . '_' . getFilenameTimeStamp($gLogTime)
                                        . '_' . $$;  # process id
    }

    local *ARGV = $rc;
    my $optctrl              =  {};
    $optctrl->{list_systems} = \$self->{options}->{list_systems};
    $optctrl->{L}            = \$self->{options}->{list_systems};

    if (!defined $self->getOptions ($optctrl,
                            defined $self->getInstconfig () || $pass_through)){
        $self->{return} = -1;
        return undef;
    }
    $self->{cmdLineArgsAfterInit} = \@ARGV;
    if ($self->{options}->{list_systems}) {
        $self->PrintText ($self->getListSystems());
        $self->{return} = 0;
        return [];
    }
    if (defined $instconfig){
       if (!defined $self->ParseConfigCmdLineArgs ($rc, $pass_through)){
            return undef;
        }
    }
    if (defined $instconfig && defined $self->{configfile}
                                       && !defined $instconfig->{config_file}) {
        $instconfig->readConfigfile($self->{configfile});
    }
    return \@ARGV;
}

sub ParseConfigCmdLineArgs{
    my ($self, $args, $pass_through) = @_;

    if (!defined $pass_through){
        $pass_through = 0;
    }
    local @ARGV = @$args;
    my $instconfig = $self->getInstconfig();


    my $cfgDumpConfigXmlRc = $self->handleOptionDumpConfigXML();
    if (!defined $cfgDumpConfigXmlRc){
        $self->{return} = 1;
        return undef;
    }

    if ($cfgDumpConfigXmlRc){
        $self->{return} = 0;
        return [];
    }


    my $instopt_ctrl = $instconfig->GetOptionCtrl();
	
    if (! defined $self->getOptions  ($instopt_ctrl, $pass_through)){
        $self->{return} = -1;
        return undef;
    }
    my $errlst = new SDB::Install::MsgLst ();
    my $saveCntxt;
    foreach my $optWithArgs (
                        $instconfig->getOptionIgnore (),
                        $instconfig->getOptionTimeout ()){
        if (!defined $optWithArgs){
            next;
        }
        $saveCntxt = $optWithArgs->setMsgLstContext ([$self->getMsgLst (), $errlst]);
        my $optName = $optWithArgs->getOptionName ();
        if (!defined $optWithArgs->parseArgs ($self->{options}->{$optName})){
                $self->{return} = -1;
        }
        $optWithArgs->setMsgLstContext ($saveCntxt);
    }
    if (defined $self->{return} && $self->{return} == -1){
        $self->WrongUsage ($errlst);
        return undef;
    }

    return \@ARGV;
}

# to be overridden by subclasses:
sub shouldWarnIfCalledStandalone{
    return 0;
}

sub warnIfShouldWarnIfCalledStandalone {
    my ($self) = @_;
    my $instconfig = $self->getInstconfig();
    if($instconfig->shouldWarnIfCalledStandalone() && $self->shouldWarnIfCalledStandalone()) {
        my $exeName = $instconfig->getExeName();
        my $exePhrase = (defined $exeName) ? " of $exeName" : "";
        my $hdblcm = $instconfig->getResidentHdblcmPath();
        if(!defined $hdblcm) {
            $hdblcm = "hdblcm";
        }
        my $standaloneWarning = "Warning: Direct usage$exePhrase is not supported. Use $hdblcm instead.";
        $self->getMsgLst()->addProgressMessage("\n" . ('#' x length ($standaloneWarning)) ."\n".
                                                      $standaloneWarning . "\n" .
                                                      ('#' x length ($standaloneWarning)) . "\n");
    }
}

1;
