#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: base class of installation console applications


package SDB::Install::App::Console;

use SDB::Install::App::AnyApp;
use SDB::Install::ProgressHandler;
use SDB::Install::SysVars qw ($isWin);
use SDB::Install::Tools;
use SDB::Common::BuiltIn;
use strict;

our @ISA = qw (SDB::Install::App::AnyApp SDB::Install::ProgressHandler);

sub new {
	my $self = shift->SUPER::new ();

	 #
     # switch STDIO autoflush on
     #
     $| = 1;

	$self->{fp_out} =
        sub {
            if (defined $_[0]){
                print $_[0] . "\n";
            }
       };
	$self->getMsgLst ()->setProgressHandler ($self);
	$self->SetErrorOutHandler (sub {print STDERR $_[0] . "\n"});
	return $self;
}

sub GetProgressHandler{
    return $_[0];
}

sub InitApp{
	my $self = shift;
	$self->SUPER::InitApp (@_);
	if ($SAPDB::Install::Config{SilentMode}){
		$self->{batch_mode} = 1;	
	}
}

sub SetErrorOutHandler{
	my ($self,$handler) = @_;
	if (defined $handler){
		$self->{fp_error_out} = $handler;
	}	
	else{
		delete $self->{fp_error_out};
	}	
}

#-------------------------------------------------------------------------------
# Prints the specified text to standard output.
#
# Parameter string $text

sub PrintText {
    my ($self, $text) = @_;
    print $text;
}

sub printToOutChannel{
	my ($self,$msg) = @_;
	&{$self->{fp_out}} ($msg);
}

sub printToErrorChannel{
	my ($self,$msg) = @_;
	&{$self->{fp_error_out}} ($msg);
}

sub SetProgress;
*SetProgress = \&printToOutChannel;


sub WrongUsage{
    my ($self, $errlst) = @_;
    my $instconfig = $self->{instconfig};
	$self->ShowErrorMsg ('Wrong usage!', $errlst);
    $self->printToOutChannel ('');	
	$self->PrintUsage();
}


sub ShowErrorMsg{
    my ($self,$text,$list, $justShow) = @_;

    if ($justShow){
        my $msglst;
        my $msg;
        if (defined $text){
            $msglst = new SDB::Install::MsgLst ();
            $msg = $msglst->addError ($text, $list);
        }
        else{
            $msglst = $list;
        }
        if (!defined $msglst){
            return undef;
        }
        &{$self->{fp_error_out}} (${$msglst->getMsgLstString ()});
        return $msg;
    }
	if (defined $self->{fp_error_out}){
        my $msg;
        if (!defined $text){
            $self->getErrMsgLst()->setMsgLst($list);
        }
        else{
		    $msg = $self->setErrorMessage ($text,$list,3);
        }
		&{$self->{fp_error_out}} (${$self->getErrMsgLst ()->getMsgLstString ()});
		return $msg;	
	}
	else {
		return shift->SUPER::ShowErrorMsg (@_);
	}
}

sub ConfigureInteractive {
	my ($self, $config) = @_;
	my $params = $config->{params};
	my $flag = 1;
	my $saveContext = $config->setMsgLstContext ([$self->getMsgLst()]);
	foreach my $currID (@{$config->getParamIds ()}){
		if($flag) {
			print "\n";
			$flag = 0;
		}
		my $paramID = $currID;

		my $param   = $params->{$paramID};
		if (defined $param->{retry_param}) {
			if (defined $config->getValue($param->{retry_param})) {
				next;
			}
			$paramID = $param->{retry_param};
			$param   = $params->{$paramID};
		}
		next if ($param->{DEPRECATED});
		if (defined $param->{value}){
			if ($param->{type} =~ /map/){
				if ($param->{'all_set'}){
					next;
				}
			}
			else{
				next;
			}
		}
		if ($param->{skip}) {
			$self->getMsgLst()->addMessage($config->getParamName($paramID) . ' is skipped');
			next;
		}
		if (defined $param->{set_interactive} && !$param->{set_interactive}){
			if (defined $param->{batchValue}){
				if (!$config->setBatchValue ($paramID)){
					$self->setErrorMessage ("Checking command line parameter '--$param->{opt}' failed.", $config->getErrMsgLst ());
					$config->setMsgLstContext ($saveContext);
					return undef;
				}
			} elsif (defined($param->{persFileValue})) {
				my $hasBeenSet = $config->setPersistedValue($paramID, $param, $param->{persFileValue});
				if(! $hasBeenSet){
					$self->setErrorMessage ("Checking persisted value for parameter '--$param->{opt}' failed.", $config->getErrMsgLst ());
					$config->setMsgLstContext ($saveContext);
					return undef;
				}
				next;
			} elsif ($param->{init_with_default}){
				if ($param->{type} =~ /map/) {
					if (!$config->setDefaultMapValues($paramID)) {
					$config->setMsgLstContext ($saveContext);
					return undef
					}
				}
				else {
					my $default = $config->getDefault ($paramID);
					if (defined $default){
						$config->addLogValueMsg($paramID,
						$default,
						'is set with default value =',
						$self->getMsgLst());

						if (!$config->setValue ($paramID, $default)){
							$self->setErrorMessage (undef, $config->getErrMsgLst ());
							$config->setMsgLstContext ($saveContext);
							return undef;
						}
					}
				}
			}
			if (!defined $param->{value}) {
				$self->getMsgLst()->addMessage($config->getParamName($paramID) . ' is not set');
			}			
			next;
		}
		if ($param->{type} =~ /map/) {		    
            if (defined($param->{persFileValue})) {
                my $hasBeenSet = $config->setPersistedValue($paramID, $param, $param->{persFileValue});
                if(! $hasBeenSet){
                    $self->setErrorMessage ("Checking persisted value for parameter '--$param->{opt}' failed.");
                    return undef;
                }
                next;
            }		    		    
			if (!$self->readCheckMapValues($config, $paramID)) {
				$config->setMsgLstContext ($saveContext);
				return undef;
			}
			next;
		}

		if (!$self->getParam ($config, $paramID)) {
			$config->setMsgLstContext ($saveContext);
			return undef;
		}
	}
	$config->setMsgLstContext ($saveContext);
	return 1;
}

#-------------------------------------------------------------------------------
# Reads a value and assigns it to the parameter entry 'value'.
#
# Parameters SDB::Install::Configuration $config
#                          reference to e.g. SDB::Install::Configuration::HdbReg
#
#            string $paramID  identifies the parameter
#
# Returns int retCode 

sub getParam{
    my ($self, $config, $paramID) = @_;
    
    my ($logvalue, $value, $default, $UIDefault, $confirm_password, $param, $str, $retryCount);
    my ($mvalue, $i, $confirmed);
    my @arrayValue;
    my $defaultAssigned = 0;

    my $shouldUseDefault = undef;
    
    $param     = $config->{params}->{$paramID};
    $default   = $config->getDefault($paramID);
    $UIDefault = $config->getDefaultUIString ($paramID);
    $str       = $param->{interactive_str} // ($param->{str} // $paramID);
    $retryCount= $param->{retry_count};
    while (1){
		if ($param->{no_retry}) {
			return undef;
		}
		if (defined $param->{batchValue}) {
		    my $rc = $config->setBatchValue ($paramID);
		    delete $param->{batchValue};
            if ($rc) {
                return 1;
            }
            if (!$param->{no_retry}){
                $self->ShowErrorMsg (undef , $config->getErrMsgLst ());
            }
            else{
                $self->setErrorMessage (undef , $config->getErrMsgLst ());
            }
            next; # retry interactive
        }
		if (defined($param->{persFileValue})) {
			my $hasBeenSet = $config->setPersistedValue($paramID, $param, $param->{persFileValue});
			if(! $hasBeenSet){
				$self->setErrorMessage ("Checking persisted value for parameter '--$param->{opt}' failed.", $config->getErrMsgLst());
				return undef;
			}
			return 1;
		}
        if ($confirm_password){
            print "Confirm $str: ";
        }
        else {
			if ($param->{f_init_interactive_values}) {
				&{$param->{f_init_interactive_values}}();
			}
            my $prompt = '';
            if (defined $param->{valid_values}) {
            	if (defined $param->{f_get_console_description}){
                    $prompt = &{$param->{f_get_console_description}}();
                }
                else{
                    my @defaultValues;
                    my $defaultFound = 0;
                    if ($param->{interactive_index_selection}){
                        if ($param->{type} eq 'csv'){ # comma-separated values
                            @defaultValues = split(/,/, $default);
                            $UIDefault="";
                            $defaultFound = 1;
                        }
                        elsif ($param->{type} eq 'array' && ref ($default)){
                            @defaultValues = @$default;
                            $UIDefault="";
                            $defaultFound = 1;
                        }
                    }

                    my $inputValues = (exists $param->{visible_alias_values})
                                      ? $param->{visible_alias_values}
                                      : $param->{valid_values};

                    if (!$defaultFound && @$inputValues) {
                        foreach my $curr (@$inputValues) {
                            if ($curr eq $default) {
                                $defaultFound = 1;
                                last;
                            }
                        }
                        if (!$defaultFound && defined $default) {
                            $default   = $inputValues->[0];
                            $UIDefault = $default;
                        }
                    }

                    my $header = (defined $param->{console_text})
                                 ? "\n$param->{console_text}\n" : "\n";
                    $prompt = $header;
                    my @rows;
                    my @headerRow;
                    if ($param->{interactive_index_selection}) {
                        push @headerRow, 'Index';
                    }

                    push @headerRow, $param->{str};

                    if (defined $param->{ui_values}) {
                        my $colHeader = $param->{ui_column_header};
                        $colHeader    = 'Description' if (!defined $colHeader);
                        push @headerRow, $colHeader;
                    }

                    push @rows, \@headerRow;

                    foreach my $j (0 .. scalar (@$inputValues) - 1){
                    
                        my @currRow          = ();
                        my $currValidValue   = $inputValues->[$j];
                        my $interactiveValue = $currValidValue;

						if (grep {$currValidValue eq $_} @{$param->{hidden_valid_values}}){
							next;
						}

                        if ($param->{interactive_index_selection}) {
                            $interactiveValue = $j+1;
                            push @currRow, $interactiveValue;
                        }

                        push @currRow, $currValidValue;

                        if (defined $param->{ui_values}) {
                            push @currRow, $param->{ui_values}->[$j];
                        }

                        push @rows, \@currRow;
                            
                        if ($param->{interactive_index_selection}) {
                            if ($param->{type} eq 'csv' || # comma-separated values
                                $param->{type} eq 'array'){

                                if ( grep( /^$currValidValue$/, @defaultValues ) ) {
                                    $UIDefault .= "," unless !$UIDefault;
                                    $UIDefault .= $interactiveValue;
                                }
                            }
                            elsif (defined $default &&
                                  (($default eq $currValidValue) ||
                                   ($default eq $param->{valid_values}->[$j]))) {
                                $UIDefault = $interactiveValue;
                            }
                        }
                    }
                    my $table = printTableToArrayOfLines (\@rows, ' | ', 1);
                    $prompt .= '  ' . join ("\n  ", @$table) . "\n\n";
                }
                $UIDefault = 1 if ( (! defined $UIDefault || $UIDefault eq "") && scalar @{$param->{valid_values}} == 1);
            }

            my $action;
            if (defined $param->{interactive_str}) {
                $action = "Enter $param->{interactive_str}";
            }
            elsif ($param->{interactive_index_selection}) {
                $action = "Select $str / Enter Index";
            }
            elsif ($param->{console_omit_word_Enter}) {
                $action = $str;
            }
            else {
                $action = "Enter $str";
            }

            $action .= (defined $default   &&
                        defined $UIDefault && (length($UIDefault) > 0))
                           ? " [$UIDefault]: " : ': ';
            my $hasUIDefault = (defined $UIDefault && $UIDefault ne '') ? 1 : 0;
            $shouldUseDefault = ($config->getUseDefault($paramID) && (defined $default || $hasUIDefault)) ? 1 : 0;
            if (!defined $param->{custom_input} && !$shouldUseDefault) {
                print($prompt);
                print($param->{recommendation}."\n") if defined $param->{recommendation};
                print($action);
            }
            if ($shouldUseDefault){
                $param->{no_retry} = 1;
            }
        }
        $confirmed = 0;
        my $actualUserInput;    # The user may have entered something but on failed validation value will be undef
# Custom input parameters should handle the
# {use_default} option in their own implementation
        if (defined $param->{custom_input}) {
            $value =  $param->{custom_input}->($self, $config);
        }
        elsif ($shouldUseDefault) {
            $value = undef;
        }
        else {
            ($value, $actualUserInput) = $self->readInput ($param->{type} eq 'passwd' || $param->{type} eq 'initial_passwd', \$confirm_password, \$confirmed);
        }
        if (!defined $value && $self->isStdinError()){
            return undef;
        }
        if ($confirmed){
            last;
        }
        if($param->{interactive_index_selection}) {
        	$value = $self->convertFromIndexToValue($param, $value);
        	if (!defined $value) {
        		next;
        	}
        }
      
        if (defined $value && ($value !~ /\S/)){
            $value = undef;
        }

        if (!defined $value && (($param->{type} !~ /array/) || !@arrayValue)) {

            if (defined $default){
                $value = $default;
                $defaultAssigned = 1;
            }
            elsif ($param->{mandatory}){
                if ( ($param->{type} eq 'passwd' || $param->{type} eq 'initial_passwd') && ($actualUserInput =~ /\S/) ) { 
                    next;
                }
                $self->ShowErrorMsg ("Value is empty", undef, 1);
                next;
            }
            else{
                last;
            }
        }
        $config->resetError();

        if (($param->{type} =~ /array/) && defined $value && !$defaultAssigned){

            push @arrayValue, $value;
            if (askConfirmation
                        ("Additional input for parameter '$param->{str}'?")) {
                next;
            }

            print("\n");
            $value = \@arrayValue;
        }

        my $rc = $config->setValue ($paramID, $value);
        if (!$rc){
            @arrayValue      = ();
            $defaultAssigned = 0;

            if (!$param->{no_retry}){
				$self->ShowErrorMsg (undef , $config->getErrMsgLst (),1);
            }
            else{
				$self->setErrorMessage (undef , $config->getErrMsgLst ());
            }
            if (!defined $rc){
                next;
            }
            if (defined $retryCount && !($param->{type} eq 'initial_passwd')) {
                $retryCount--;
                if ($retryCount > 0) {
                    next;
                } else {
                    return 0;
                }
            } else {
                next;
            }
        }
        if (defined $value && $param->{type} eq 'initial_passwd'){
            $confirm_password = $value;
            next;
        }
        last;
    }
    
    my $paramName = $config->getParamName($paramID);

    if (defined $value) {
        my $info = ($defaultAssigned)
                   ? 'is set by user (confirmed the default value), value ='
                   : 'is set by user, value =';

        $config->addLogValueMsg($paramID, $value, $info, $self->getMsgLst());
        if ($param->{type} eq 'number'){
            $value = int ($value);
        }
    }
    else{
        $self->getMsgLst()->addMessage ($config->getParamName($paramID)
                                        . ' is not set (skipped by user)');
    }
    return 1;
}

sub convertFromIndexToValue{
	my ($self, $param, $csvIndices) = @_;
	if (length($csvIndices) == 0) {
		return '';
	}
	my @indices = undef;
	
	if(!$self->_isIndexesFormValid($csvIndices)) {
		$self->ShowErrorMsg ("The provided input '" . $csvIndices . "' is not valid.", undef, 1);
		return undef;
	}
	
	if ("csv" eq $param->{type}) { # comma-separated values
		@indices = split(/\s*,\s*/, $csvIndices);	
	} 
	else{
		@indices = ($csvIndices);
	}
	
	if (length($csvIndices) >0 and scalar @indices == 0) {
		$self->ShowErrorMsg ("The provided input '" . $csvIndices . "' is not valid.", undef, 1);
		return undef;
	}
	
	if(!$self->_areIndexesUnique(\@indices)) {
		$self->ShowErrorMsg ("The provided input '" . $csvIndices . "' contains duplicate indexes.", undef, 1);
		return undef;
	}
	
	my $csvValues = "";
	for my $index( @indices) {
		if (($index !~ /^[1-9][0-9]*$/) or $index < 1 or (scalar @{$param->{valid_values}} < $index)) {			
			$self->ShowErrorMsg ("The provided index '" . $index . "' is not valid.", undef, 1);
			return undef;
		}
		my $value = $param->{valid_values}->[$index-1];

		if($param->{hidden_valid_values} && grep $_ eq $value, @{$param->{hidden_valid_values}}){
			$self->ShowErrorMsg ("The provided index '" . $index . "' is not valid.", undef, 1);
			return undef;
		}
		$csvValues .= $value. ",";	
	}
	
	if (length($csvValues) > 0) {
		$csvValues = substr($csvValues, 0, length($csvValues)-1);
	}
		
	return $csvValues;
}

sub _handleStdinError{
    my ($self) = @_;
    $self->setErrorMessage ("Cannot read from stdin handle. Handle might be already closed.");
    $self->{_stdin_error} = 1;
}

sub isStdinError{
    return $_[0]->{_stdin_error};
}


sub _isIndexesFormValid {
	my($self, $csvIndices) = @_;
	
	return 1 if ($csvIndices eq "");
	
	if($csvIndices !~ m/^\d+(,\d+)*$/){
		return 0;
	}
	return 1;
}

sub _areIndexesUnique {
	my ($self, $indexes_reference) = @_;
	my @indexes = sort @{$indexes_reference};
	my $last_index = @indexes-1;
	if ($last_index == 0) {
		return 1;
	}
	for(0 .. ($last_index-1)) {
		if ($indexes[$_] == $indexes[$_+1]) {
			return 0;
		} 
	}
	
	return 1;
}

sub printWarnings{
    my ($self) =@_;
    my $warnings    = $self->{instconfig}->getWarningList();
    if (@$warnings) {
        print "\n";
    }
    foreach my $currWng (@$warnings) {
        print "Note: $currWng\n";
    }
}


#-------------------------------------------------------------------------------
# Returns a string that is read from standard input.

sub readInput{
    my ($self,$password, $rconfirm_password, $rconfirmed) = @_;
    my $value;
    if ($password){
        require Term::ReadKey;
        Term::ReadKey::ReadMode('noecho');
        eval{
            $value = Term::ReadKey::ReadLine();
        };
        print "\n";
       Term::ReadKey::ReadMode('restore');
       if ($@){
            die ($@);
       }
       if (!defined $value){
            $self->_handleStdinError ();
            return undef;
       }
       if ($isWin){
            local $/ = "\r\n";
            chomp ($value);
       }
       chomp ($value);
       if (defined $rconfirm_password && defined $$rconfirm_password){
            if ($$rconfirm_password eq $value){
                $$rconfirmed = 1;
                return wantarray ? ($value, $value) : $value;
            }
            else{
                $self->ShowErrorMsg ("Password confirmation failed", undef, 1);
                $$rconfirm_password = undef;  
                return wantarray ? (undef, $value) : undef;
            }
       }
    }
    else{
        $value = <STDIN>;
        if (!defined $value){
            $self->_handleStdinError ();
            return undef;
        }
        if ($isWin){
            local $/ = "\r\n";
            chomp ($value);
        }
        chomp ($value);
    }
    return wantarray ? ($value, $value) : $value;
}


#-------------------------------------------------------------------------------
# Reads hash map values and checks the entire hash map (checkEntries<paramID>).
# If reading fails, undef is returned.
# If check entries fails, reading is repeated with new default values.
#
# Parameters SDB::Install::Configuration  $config   1)
#            string                       $paramID  identifies the parameter
#
# Returns int retCode
#
# 1) reference to e.g. SDB::Install::Configuration::HdbReg that provides
#                         the parameter hash ( $config->{params}->{$paramID} )

sub readCheckMapValues {

    my ($self, $config, $paramID) = @_;

    my $param   = $config->{params}->{$paramID};

    while (1) {

        if (!$self->readMapValues($config, $paramID)) {
            return undef; # skip retry in case of read errors
        }

        $config->resetError();
        if ($config->checkAllMapValues($paramID)) {
            last;
        }

        # check host map entries failed --> retry with updated default values

        $self->ShowErrorMsg (undef, $config->getErrMsgLst (), 1);

        foreach my $currOrigin (@{$param->{origin_values}}) {

            my $lcCurrOrigin = lc($currOrigin);

            if (defined $param->{batchValue} &&
                defined $param->{batchValue}->{$lcCurrOrigin}) {

                $param->{default_map}->{$lcCurrOrigin} =
                                          $param->{batchValue}->{$lcCurrOrigin};
                delete $param->{batchValue}->{$lcCurrOrigin};
            }

            if (defined $param->{value} &&
                defined $param->{value}->{$lcCurrOrigin}) {

                $param->{default_map}->{$lcCurrOrigin} =
                                               $param->{value}->{$lcCurrOrigin};
                delete $param->{value}->{$lcCurrOrigin};
            }
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Reads values in order to create a hash map that is assigned to the parameter
# entry 'value'.
# According to the param entry 'origin_values', the created hash map consists
# of already existing values and new input values.
# Reading of map entries is done in the order of sorted origin values.
# If the  hash 'own_origin_value' is defined in the map parameter,
# the value of this entry is handled first (e.g. to enter the local host before
# entering remote hosts).
#
# Parameters SDB::Install::Configuration  $config   1)
#            string                       $paramID  identifies the parameter
#        
# Returns int retCode
#
# 1) reference to e.g. SDB::Install::Configuration::HdbReg that provides
#                         the parameter hash ( $config->{params}->{$paramID} )

sub readMapValues {
    
    my ($self, $config, $paramID) = @_;
    
    my $subroutineShowMap = $config->can("show$paramID"); # e.g. showHostMap

    if (defined $subroutineShowMap) {
        # calls e.g. showHostMap if subroutine exists
        &$subroutineShowMap($config);
    }

    my $param = $config->{params}->{$paramID};
    if (exists $param->{custom_input}) {
        my $returnCode = $param->{custom_input}->($self, $config);
        $self->setErrorMessage(${$config->getErrMsgLst()->getMsgLstString()}) if (!$returnCode);
        return $returnCode;
    }
    my $sortedOriginValues = $param->{origin_values};  # already sorted
    my $ownOriginValue     = $param->{own_origin_value};

    if (defined $ownOriginValue
                     && (lc($sortedOriginValues->[0]) ne lc($ownOriginValue))) {

        # move own host to front of list

        my @auxSorted = ($ownOriginValue);

        foreach my $currOrigin (@{$param->{origin_values}}) {
            if (lc($currOrigin) ne lc($ownOriginValue)) {
                push @auxSorted, $currOrigin;
            }
        }
        $sortedOriginValues = \@auxSorted;
    }

    if (scalar @$sortedOriginValues < 1) {
        $self->ShowErrorMsg ("Source $param->{str} not found");
        return undef;
    }

    foreach my $currOrigin (@$sortedOriginValues) {

        my $lcCurrOrigin = lc($currOrigin);

        if (defined $param->{value}->{$lcCurrOrigin}){
            next; # skip reading input for current $valKey
        }

        my $default   = $param->{default_map}->{$lcCurrOrigin};
        my $str       = sprintf ($param->{str_templ}, $currOrigin);
        my $paramName = $config->getParamName($paramID, $currOrigin);
        my $inputVal  = undef;

        if (defined $param->{batchValue}) {
            $inputVal = $param->{batchValue}->{$lcCurrOrigin};
        }

        while (1) {
            my $okInfoPrefix = (defined $inputVal)
                               ? "$paramName is given via command line option"
                               : "$paramName is set by user";

            if (defined $default && $default ne '' && $config->getUseDefault($paramID)) {
                $inputVal = $default;
                $okInfoPrefix .= " (confirmed the default value)";
            }
            if (!defined $inputVal) {
                my $prompt = (!defined $param->{console_omit_word_Enter} ? 'Enter ' : '')
                    . $str . (defined $default ? " [$default]: " : ': ');

                print($prompt);
                $inputVal = $self->readInput();
                if (!defined $inputVal && $self->isStdinError()){
                    return undef;
                }
                if ($inputVal !~ /\S/) {
                    # hit return without any input
                    $inputVal = (defined $default) ? $default : undef;

                    if (!defined $inputVal && $param->{mandatory}) {
                        $self->ShowErrorMsg ("Value is empty");
                        next; # try again reading input for current $valKey
                    }
                    $okInfoPrefix .= " (confirmed the default value)";
                }
            }
            
            if (!$config->setMapValueItem($paramID, $currOrigin, $inputVal)) {
                
                $self->ShowErrorMsg (undef, $config->getErrMsgLst(), 1);
                undef $inputVal;
                next; # try again reading input for current $valKey
            }

            $self->getMsgLst()->addMessage ($okInfoPrefix . ", value = '$inputVal'");
            last;
        }
    }
    return 1;
}

sub waitForKeyPress{
    require Term::ReadKey;
    SDB::Common::BuiltIn->get_instance()->print("\npress any key...");
    Term::ReadKey::ReadMode('raw');
    Term::ReadKey::ReadKey(0);
}

sub shouldWarnIfCalledStandalone{
    return 0;
}

1;
