# CollectHostInfo
#
# Parameters:
# ------------
#
# --dataPath=<data_path>                   # checks the existence of this data path (xml tag 'dataPath')
#
# --logPath=<log_path>                     # checks the existence of this log path (xml tag 'logPath')
#
# --collect_only=<xml_tag>[,<xml_tag>]     # selects these xml tags only (e.g. 'SAPSystems,users')
#                                          # <xml_tag>: main tags except 'currentUser', 'datapath', 'logPath'
#
# --skip_collect=all|<xml_tag>[,<xml_tag>] # does not select these xml tags
#                                          # (cannot be applied together with '--collect_only')
#                                          # <xml_tag>: main tags except 'currentUser', 'datapath', 'logPath'
#
# --sapadm=<sapadm_username>               # if xml tag 'users' is skipped the properties
#                                          # of this sapadm are collected
#
# --sidadm=<sidadm_username>               # if xml tag 'users' is skipped the properties
#                                          # of this sidadm are collected
#
# --userid=<uid>                           # if xml tag 'users' is skipped the properties
#                                          # of this sidadm user ID are collected
#
# --groupid=<gid>                          # if xml tag 'groups' is skipped the properties
#                                          # of this sapsys group ID are collected
#
# --usernames=<user1>[,<user2>,...]        # if xml tag 'users' is skipped the properties
#                                          # of all users in the list are collected
#
# --groupnames=<group1>[,<group2>,...]     # if xml tag 'groups' is skipped the properties
#                                          # of all groups in the list are collected

#
#
# Formatted Sample Output:
# ------------------------
#
# <HostDescription>
#
#     <currentUser isAdmin="1" />
#
#     <sapsysid>79</sapsysid>
#
#     # if sapsys exists, collection of other groups (except --groupid) is suppressed
#     <groups>
#         <group id="17">audio</group>
#         <group id="79">sapsys</group>
#     </groups>
#
#     <users>
#         <user id="1" gid="1">bin</user>
#         <user id="2" gid="2">daemon</user>
#         <user id="0" gid="0">root</user>
#         <user id="1002" gid="79">db7adm</user>
#         <user id="1003" gid="79">db1adm</user>
#     </users>
#
#    <SAPSystems>
#         <System sid="DB1">
#             <Landscape id="51ddf0d6-83bd-188a-e100-00007f000002"></Landscape>
#             <Instances>
#                 <Instance type="HDB">01</Instance>
#             </Instances>
#         </System>
#         <System sid="DB7">
#             <Landscape id="51d52dce-8eb9-0d3f-e100-00007f000002"></Landscape>
#             <Instances>
#                 <Instance type="HDB">07</Instance>
#             </Instances>
#         </System>
#     </SAPSystems>
#
#     <networkAddresses>
#         <ip>100.100.100.100/24</ip>
#         <ip>fe80::21a:6bff:fe51:fb16/64</ip>
#         <ip>127.0.0.1/8</ip>
#         <ip>127.0.0.2/8</ip>
#         <ip>::1/128</ip>
#     </networkAddresses>
#
#     <saphostagentVersion>7.20.162</saphostagentVersion>
#	  <certificateHostname>myhanahost</certificateHostname>
#	  <dataPath isExists="1" />
#	  <logPath isExists="1" />
# </HostDescription>


package SDB::Install::App::Console::CollectHostInfo;

use SDB::Install::Globals qw ($gSapsysGroupName);
use SDB::Install::Configuration::NewDB;
use SDB::Install::SAPSystem;
use SDB::Install::Saphostagent qw(getSHAVersion);
use SDB::Install::SysInfo;
use SDB::Common::Utils qw(toBase64);
use LCM::RegenerateCertificates;
use SDB::Install::System qw (isAdmin);
use Getopt::Long;

use strict;

sub _maskContent ($$){
    my ($refData, $refIsBase64Encoded) = @_;
    if ($$refData =~ /[^[:print:]]/){
        $$refIsBase64Encoded = 1;
        $$refData = toBase64($$refData);
    }
    else{
        $$refIsBase64Encoded = 0;
        $$refData =~ s/&/&amp;/sg;
        $$refData =~ s/</&lt;/sg;
        $$refData =~ s/>/&gt;/sg;
        $$refData =~ s/"/&quot;/sg;
    }
}


sub main{
    local @ARGV = @_;

    my $certificateHostnameTag = 'certificateHostname';
    my $dataPathTag            = 'dataPath';           # param --dataPath only
    my $groupsTag              = 'groups';
    my $logPathTag             = 'logPath';            # param --logPath only
    my $networkAddressesTag    = 'networkAddresses';
    my $saphostagentVersionTag = 'saphostagentVersion';
    my $sapsysidTag            = 'sapsysid';
    my $sidcryptidTag          = 'sidcryptid';
    my $SAPSystemsTag          = 'SAPSystems';
    my $usersTag               = 'users';
    my $systemCheckErrorsTag   = 'systemCheckErrors';

    my $dataPath    = '';
    my $logPath     = '';
    my $collectOnly = '';
    my $skipCollect = '';
    my $sapadm      = '';
    my $sidadm      = '';
    my $sidcrypt    = '';
    my $sidcryptGroupid = '';
    my $userid      = '';
    my $groupid     = '';
    my $usernames   = '';
    my $groupnames  = '';
    my $compilerVersion = '';

    GetOptions ('dataPath=s'     => \$dataPath,
                'logPath=s'      => \$logPath,
                'collect_only=s' => \$collectOnly,
                'skip_collect=s' => \$skipCollect,
                'sapadm=s'       => \$sapadm,
                'sidadm=s'       => \$sidadm,
                'sidcrypt=s'     => \$sidcrypt,
                'usernames=s'    => \$usernames,
                'groupnames=s'    => \$groupnames,
                'userid=s'       => \$userid,
                'groupid=s'      => \$groupid,
                'sidcrypt_groupid' => \$sidcryptGroupid,
                'compiler_version=s' => \$compilerVersion,
                );

    my %collect = ($certificateHostnameTag => 1,
                   $groupsTag              => 1,
                   $networkAddressesTag    => 1,
                   $saphostagentVersionTag => 1,
                   $sapsysidTag            => 1,
                   $sidcryptidTag          => 1,
                   $SAPSystemsTag          => 1,
                   $usersTag               => 1,
                   $systemCheckErrorsTag   => 1,
                  );

    if ($collectOnly ne '') {
        %collect = ();
        foreach my $item (split(',', $collectOnly)) {
            $collect{$item} = 1;
        }
    }
    elsif ($skipCollect ne '') {
        foreach my $item (split(',', $skipCollect)) {
            if ($item eq 'all') {
                %collect = ();
                last;
            }
            else {
                delete $collect{$item};
            }
        }
    }

    print '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    print "<HostDescription>\n";

    print sprintf ("<currentUser isAdmin=\"%d\" />\n", isAdmin () ? 1 : 0);

    my $nameOfGroupid = undef;
    my $sapsysid      = undef;
    my $nameOfSidcryptGroupid = undef;
    my $sidcryptid = undef;

    if ($collect{$sapsysidTag}) {
        #sapsys group
        $sapsysid = getgrnam ($gSapsysGroupName);
        if (defined $sapsysid){
            print sprintf ("<$sapsysidTag>%d</$sapsysidTag>\n", $sapsysid);
            if ($groupid eq $sapsysid) {
                $nameOfGroupid = $gSapsysGroupName;
            }
        }
    }

    if (!defined $nameOfGroupid && ($groupid ne '')) {
        $nameOfGroupid = getgrgid($groupid);
    }

    if ($collect{$sidcryptidTag}) {
        $sidcryptid = getgrnam ($sidcrypt);
        if (defined $sidcryptid){
            printf ("<$sidcryptidTag>%d</$sidcryptidTag>\n", $sidcryptid);
            if ($sidcryptGroupid eq $sidcryptid) {
                $nameOfSidcryptGroupid = $sidcrypt;
            }
        }
    }

    if (!defined $nameOfSidcryptGroupid && ($sidcryptGroupid ne '')) {
        $nameOfSidcryptGroupid = getgrgid($sidcryptGroupid);
    }
    my $collectGroups = defined $sapsysid || defined $sidcryptid || $collect{$groupsTag} ||
                        $groupnames || (defined $nameOfGroupid && $groupid);

    my $groupFormat       = "<group id=\"%d\">%s</group>\n";
    my $groupFormatBase64 = "<group id=\"%d\"\" encoding=\"base64\">%s</group>\n";
    my $isBase64Encoded;
    print "<$groupsTag>\n" if $collectGroups;
    if (defined $sapsysid && defined $sidcryptid) {
        printf ($groupFormat, $sapsysid, $gSapsysGroupName);
        if (defined $nameOfGroupid && ($groupid ne $sapsysid)) {
            printf ($groupFormat, $groupid, $nameOfGroupid);
        }
        printf ($groupFormat, $sidcryptid, $sidcrypt);
        if (defined $nameOfSidcryptGroupid && ($sidcryptGroupid ne $sidcryptid)) {
            printf ($groupFormat, $sidcryptGroupid, $nameOfSidcryptGroupid);
        }
    }
    elsif ($collect{$groupsTag} && !$groupnames) {
        my ($name, $gid);
        setgrent();
        while (($name, undef, $gid) = getgrent()) {
            _maskContent (\$name,\$isBase64Encoded);
            print sprintf ($isBase64Encoded ? $groupFormatBase64 : $groupFormat, $gid, $name) if ($gid != 0);
        }
        endgrent();
    }
    elsif (defined $nameOfGroupid && ($groupid ne '')) {
        print sprintf ($groupFormat, $groupid, $nameOfGroupid);
    }

    if ($groupnames){
        my ($name, $gid);
        foreach my $groupname (split (',', $groupnames)){
            # collect each user in the list
            ($name, undef, $gid) = getgrnam ($groupname);
            if (defined $gid) {
                _maskContent (\$name,\$isBase64Encoded);
                print sprintf ($isBase64Encoded ? $groupFormatBase64 : $groupFormat, $gid, $name) if ($gid != 0);
            }
        }
    }

    print "</$groupsTag>\n" if $collectGroups;

    my $userFormat       = "<user id=\"%d\" gid=\"%d\">%s</user>\n";
    my $userFormatBase64 = "<user id=\"%d\" gid=\"%d\" encoding=\"base64\">%s</user>\n";
    if ($collect{$usersTag}) {
        print "<$usersTag>\n";
        my ($name, $uid, $gid);
        setpwent ();
        while (($name, undef, $uid, $gid) = getpwent()) {
            _maskContent (\$name,\$isBase64Encoded);
            print sprintf ($isBase64Encoded ? $userFormatBase64 : $userFormat, $uid, $gid, $name) if ($uid != 0);
        }
        endpwent ();
        print "</$usersTag>\n";
    }
    else {

        my $sapadmInfo = undef;
        my $sidadmInfo = undef;
        my $useridInfo = undef;
        my $sidadmUID  = undef;
        my $usersInfo  = undef;

        if ($sapadm ne '') {
            # collect system administrator only
            my ($name, undef, $uid, $gid) = getpwnam($sapadm);
            if (defined $uid) {
                $sapadmInfo = sprintf ($userFormat, $uid, $gid, $name);
            }
        }

        if ($sidadm ne '') {
            # collect system administrator only
            my ($name, undef, $uid, $gid) = getpwnam($sidadm);
            if (defined $uid) {
                $sidadmUID  = $uid;
                $sidadmInfo = sprintf ($userFormat, $uid, $gid, $name);
            }
        }

       if ($usernames ne '') {
            foreach my $username (split (',', $usernames)){
                # collect each user in the list
                my ($name, undef, $uid, $gid) = getpwnam($username);
                if (defined $uid) {
                    if (!defined $usersInfo){
                        $usersInfo = sprintf ($userFormat, $uid, $gid, $name);
                    }
                    else{
                        $usersInfo .= sprintf ($userFormat, $uid, $gid, $name);
                    }
                }
            }
        }

        if (($userid ne '') && (!defined $sidadmUID || ($sidadmUID != $userid))){

            my ($name, undef, $uid, $gid) = getpwuid($userid);
            if (defined $name && ($name ne $sidadm)) {
                $useridInfo = sprintf ($userFormat, $uid, $gid, $name);
            }
        }

        if (defined $sapadmInfo || defined $sidadmInfo || defined $useridInfo ||
            defined $usersInfo) {
                print "<$usersTag>\n";
                print $sapadmInfo if (defined $sapadmInfo);
                print $sidadmInfo if (defined $sidadmInfo);
                print $useridInfo if (defined $useridInfo);
                print $usersInfo  if (defined $usersInfo);
                print "</$usersTag>\n";
        }
    }

    if ($collect{$SAPSystemsTag}) {
        print "<$SAPSystemsTag>\n";
        my $systems = CollectSAPSystems ();
        my $instances;
        my ($reservedInstanceNrs, $attribReservedInstanceNrs);

        foreach my $sid (keys (%$systems)){
            print sprintf ("<System sid=\"%s\">\n", $sid);
            my $currSystem = $systems->{$sid};

            # landscape
            if ($currSystem->hasNewDB()) {
                my $instance    = $currSystem->getNewDBInstances()->[0];
                my $landscapeID = (defined $instance) ? $instance->getLandscapeID()
                                                      : undef;
                if (defined $landscapeID) {
                    print sprintf ("<Landscape id=\"%s\"></Landscape>\n", $landscapeID);
                }
            }
            # instances
            print "<Instances>\n";
            $instances = $currSystem->get_instances ();
            if (defined $instances){
                foreach my $num (keys (%$instances)){
                    $reservedInstanceNrs = $instances->{$num}->getReservedInstanceNumbers();
                    if ($reservedInstanceNrs > 0){
                        $attribReservedInstanceNrs = " reserved_instance_numbers=\"$reservedInstanceNrs\"";
                    }
                    else{
                        $attribReservedInstanceNrs = '';
                    }
                    print sprintf ("<Instance type=\"%s\"%s>%02d</Instance>\n",
                                   $instances->{$num}->get_type (),
                                   $attribReservedInstanceNrs, $num);
                }
            }
            print "</Instances>\n";
            print "</System>\n";
        }
        print "</$SAPSystemsTag>\n";
    }

    if ($collect{$networkAddressesTag}) {
        # network interfaces
        print "<$networkAddressesTag>\n";
        my $sysinfo = new SDB::Install::SysInfo ();
        my $netInterfaces = $sysinfo->interfaceInfo ();
        foreach my $addresses (values %$netInterfaces){
            foreach my $ip (@$addresses){
                print "<ip>$ip->{addr}/$ip->{netbits}</ip>\n";
            }
        }
        print "</$networkAddressesTag>\n";
    }

    if ($collect{$saphostagentVersionTag}) {
        my $agentVersion = getSHAVersion();
        if (!defined $agentVersion) {
            $agentVersion = '';
        }
        print "<$saphostagentVersionTag>$agentVersion</$saphostagentVersionTag>\n";
    }

    if ($collect{$certificateHostnameTag}) {
        my $certificateHostname = LCM::RegenerateCertificates::getCertOwner();
        $certificateHostname =~ s/CN=//;
        print "<$certificateHostnameTag>$certificateHostname</$certificateHostnameTag>\n";
    }

    if ($collect{$systemCheckErrorsTag} && $compilerVersion) {
        my $newDbConfig   = SDB::Install::Configuration::NewDB->new();
        my $sysinfo       = SDB::Install::SysInfo->new();
        require SDB::Install::Manifest;
        my $dummyManifest = SDB::Install::Manifest->new();
        $dummyManifest->{data}->{'compiler-version'} = $compilerVersion;
        $newDbConfig->checkSystemRequirements($sysinfo, $dummyManifest);
        my $errors = ${$newDbConfig->getErrMsgLst()->getMsgLstString()};
        if ($errors) {
            print "<$systemCheckErrorsTag>$errors</$systemCheckErrorsTag>\n";
        }
    }

	# data and log path existence
	if ($dataPath ne '') {
		my $isDataPathExists = -d $dataPath;
		print sprintf ("<$dataPathTag isExists=\"%d\" />\n", $isDataPathExists ? 1 : 0 );
	}
	if ($logPath ne '') {
		my $isLogPathExists = -d $logPath;
		print sprintf ("<$logPathTag isExists=\"%d\" />\n", $isLogPathExists ? 1 : 0 );
	}

    print "</HostDescription>\n";
    return 0;
}

sub shouldWarnIfCalledStandalone{
    return 0;
}

1;
