#!/usr/bin/perl
#
# Desc: Registers or unregisters a SAP HANA instance
#       using an existing installation path (e.g. /hana/shared/<sid>).
#
# This tool checks the configuration and detects:
#   - missing users
#   - missing groups
#   - missing or incomplete sapinit files
#   - missing sapservices
#
# Like hdbrename, the tool repairs the detected configuration problems.


package SDB::Install::App::Console::HdbReg;

use base SDB::Install::App::Console;

use SDB::Install::Configuration::HdbReg;
use SDB::Install::Log;
use SDB::Install::SAPSystemUtilities;
use SDB::Install::SAPSystemRegister;

use SDB::Install::Configuration qw ($bool_true_pattern);
use SDB::Install::Globals qw ($gProductName $gProductNameInstaller);
use SDB::Install::SysVars qw ($isWin $path_separator);
use SDB::Install::Tools   qw (askConfirmation);

use strict;

our $HDBREG_PROGRAM_VERS = '0059';
our $PARAM_ID_START_ONLY = 'StartOnly';
our $PARAM_ID_STOP_ONLY  = 'StopOnly';
our $PARAM_ID_UNREGISTER = 'Unregister';

#-------------------------------------------------------------------------------
# Constructor
sub new {
    
    my $self = shift->SUPER::new ();
    
    return $self;
}


#-------------------------------------------------------------------------------
# Initializes this application
# Without parameters

sub InitApp {
    
    my ($self) = @_;

    $self->SUPER::InitApp();

    return 1;
}


#-------------------------------------------------------------------------------
# Initializes hdbreg, performs several checks and calls SAPSystemRegister.
#
# Without parameters
# Returns undef in case of an error

sub init {
    
    my ($self) = @_;

    if ($isWin){
        $self->setErrorMessage ($self->getProgramName()
                                 . ' is not yet implemented for this platform');
        return undef;
    }

    if (!$self->checkIsAdmin($self->getProgramName())) {
        return undef;
    }

    my $instconfig          = $self->getInstconfig();
    my $msglst              = $self->getMsgLst ();
    my $strProgramName      = $self->getProgramName();
    my $strConfigOrInput    = 'Configuration';
    my $strInstanceOrSystem = 'Instance';
    my $unregStartStop      = undef;
    my $isChangePasswdOnly  = 0;

    $instconfig->setMsgLstContext ([$msglst]);

    foreach my $paramID ($PARAM_ID_UNREGISTER,
                         $PARAM_ID_START_ONLY,
                         $PARAM_ID_STOP_ONLY) {

        my ($rc, $scope) = $instconfig->scanParamUnregStartStop($paramID);

        if (!$rc) {
            $self->setErrorMessage (undef, $instconfig->getErrMsgLst());
            return undef;
        }

        if (defined $scope) {

            if (defined $unregStartStop) {
                $self->setErrorMessage("Option '"
                    . $instconfig->getOpt($unregStartStop)
                    . "' is already set, cannot handle '"
                    . $instconfig->getOpt($paramID) . "'", undef);
                return undef;
            }

            $unregStartStop      = $paramID;
            $strInstanceOrSystem = $self->getScopeName($scope);
            $strConfigOrInput    = 'Input';

            if ($paramID eq $PARAM_ID_UNREGISTER) {
                $self->{action}            = 'Unregister';
                $self->{actionProgressive} = 'Unregistering';
                $self->{actionDone}        = 'unregistered';
                my $prog                   = $SAPDB::Install::Config{ProgramName};
                $strProgramName           .= '_unregister';
            }
            elsif ($paramID eq $PARAM_ID_START_ONLY) {
                $self->{action}            = 'Start';
                $self->{actionProgressive} = 'Starting';
                $self->{actionDone}        = 'started';
                $strProgramName           .= '_start';
            }
            else {
                $self->{action}            = 'Stop';
                $self->{actionProgressive} = 'Stopping';
                $self->{actionDone}        = 'stopped';
                $strProgramName           .= '_stop';
            }
        }
    }

    if (!defined $unregStartStop) {

        my $changePw = $instconfig->getBatchValue('ChangeSystemPasswdOnly');

        if (defined $changePw
            &&
            (($changePw eq '') || ($changePw =~ /$bool_true_pattern/i))) {

            $isChangePasswdOnly        = 1;
            $self->{action}            = 'Change system password';
            $self->{actionProgressive} = 'Changing system password';
            $self->{actionDone}        = 'system password changed';
            $strProgramName           .= '_change_system_password';
            $instconfig->initChangeSystemPasswdOnly();
        }
    }

    $self->warnIfShouldWarnIfCalledStandalone();
    my $strHeadline = "$gProductNameInstaller - $strProgramName  "
                                               . $self->GetInstaller->{version};

    # info: SAP HANA Database Installation Manager - hdbreg/Unregister <version>
    #       ********************************************************************
    #
    #       Register/Unregister SAP HANA Instance

    my $info = "\n\n" . $strHeadline . "\n" . ('*' x length($strHeadline))
             . "\n\n" . $self->getAction()
             . " $gProductName $strInstanceOrSystem\n";

    if (defined $unregStartStop && ($unregStartStop eq $PARAM_ID_UNREGISTER)) {
        $info .= "Data, log and installation directories will not be deleted,\n"
              .  "because they are required if you decide to register the system again later.\n";
    }
    $msglst->addProgressMessage($info);

    $self->addStartDateMsg($msglst, $self->getProgramName()
                       . ' (' . $self->getAction() . ")  $HDBREG_PROGRAM_VERS");


    my $msg = $msglst->addMessage ("Checking $strConfigOrInput");

    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);

    my $rcParamCheck = $instconfig->CheckParams ($self->{batch_mode});

    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();

    $msg->setEndTag ("$strConfigOrInput Check");

    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    };

    if($self->{batch_mode}){

        if (!$rcParamCheck){
            if (defined $rcParamCheck){
                $self->setErrorMessage ("Running in batch mode",
                                        $instconfig->getErrMsgLst ());
            }
            else{
                $self->setErrorMessage ("$strConfigOrInput error (batch mode):",
                                        $instconfig->getErrMsgLst ());
            }
            return undef;
        }
    }
    else{
        if (!$rcParamCheck && $instconfig->ErrorState ()){
            $self->ShowErrorMsg ("$strConfigOrInput error:",
                                 $instconfig->getErrMsgLst ());
            if (!defined $rcParamCheck){
                return undef;
            }
        }
        
        if (!defined $self->ConfigureInteractive ($instconfig)){
            return undef;
        }
    }

    $instconfig->setMsgLstContext ([$msglst]);

    if ($instconfig->isShowPending()) {
        $instconfig->showUnfinishedSlaves($self->getActionProgressive(),
                                          $self->getProgramName(), 1);
        $self->{return} = 0;
        return 1;
    }

    if ($instconfig->isCheckOnly()) {
        my $originAction = $self->{action};
        $self->{action}            = "Check $originAction";
        $self->{actionProgressive} = "Checking $originAction";
        $self->{actionDone}        = "checked ($originAction)";
    }

    $msg = $msglst->addProgressMessage($self->getActionProgressive()
                                       . " $gProductName $strInstanceOrSystem '"
                                       . $instconfig->getSID() . "'...");

    if (!$self->defineLog($strProgramName,
                          $strProgramName,
                          $instconfig->isCheckOnly(),
                          $instconfig->getSID())) {
        return undef;
    }

     $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);

    my $sapSys = $instconfig->getSAPSystem();

    if (defined $sapSys) {
        $sapSys->SetProgressHandler ($msglst->getProgressHandler());
        $self->addLogToTraceDir($sapSys,
                                $strProgramName,
                                $instconfig->isCheckOnly());
    }

    if (!$isWin){

        my $rcHandler;
        my $registerHandler = new SDB::Install::SAPSystemRegister();

        $instconfig->dumpConfigFile($strProgramName);

        if (defined $sapSys) {
            $sapSys->setMsgLstContext ([$msg->getSubMsgLst()]);
        }

        if (defined $unregStartStop && ($unregStartStop ne $PARAM_ID_UNREGISTER)) {

             $rcHandler = $registerHandler->startStopOnly($self,
                                                          $sapSys,
                                                          $unregStartStop,
                                                          undef,
                                                          1); # handle stop service error
        }
        elsif ($isChangePasswdOnly) {

            my $instance = $instconfig->getOwnInstance();
            if (!defined $sapSys || !defined $instance) {
                 $self->setErrorMessage("$gProductName instance does not exist");
                 $rcHandler = undef;
            }
            else {
                $rcHandler = $registerHandler->changeSystemUserPassword
                                              ($sapSys, $instance, $instconfig);
            }
        }
        else {
            $instconfig->displayParameterSummary();
            if (!$self->{batch_mode} && !$instconfig->isCheckOnly()
                       && !$self->{options}->{noprompt} && !askConfirmation()) {
                $msglst->addProgressMessage($self->getActionProgressive()
                             . " $gProductName $strInstanceOrSystem cancelled");
                $self->{return} = 0;
                return 1;
            }

            if (defined $unregStartStop) {
                # $sapSys may be undef in case of unregister incomplete instance
                $rcHandler = $registerHandler->unregisterInstance($self, $sapSys);
            }
            else {
                $rcHandler = $registerHandler->registerSystem($self, $sapSys);
            }
        }

        if (!$rcHandler) {
            if (defined $sapSys) {
                $self->appendErrorMessage (undef, $sapSys->getErrMsgLst ());
            }
            return undef;
        }

        if (!defined $unregStartStop &&
            !defined $instconfig->checkSystemRequirements($self->GetSysInfo())) {

            $self->setErrorMessage ("Checking system requirements failed!",
                                    $instconfig->getErrMsgLst());
            return undef;
        }
    }

    if (!defined $unregStartStop) {
        $self->showUnfinishedRegisterSlaves($msg);
    }
    elsif ($unregStartStop eq $PARAM_ID_UNREGISTER) {
        $self->showAdditionalUnregisterInfo($msg, $msglst);
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs {
    
    my ($self, $args) = @_;

    $self->{instconfig} =
            new SDB::Install::Configuration::HdbReg ($self->{options});

    $self->{action}            = 'Register';
    $self->{actionProgressive} = 'Registering';
    $self->{actionDone}        = 'registered';

    if (!$self->additionalInit()) {
        return undef;
    }

    return $self->SUPER::InitCmdLineArgs ($args);
}


#-------------------------------------------------------------------------------
# Additional initialization - may be overridden

sub additionalInit {
    return 1;
}


#-------------------------------------------------------------------------------
# Returns the name of the instance or system depening on the scope.

sub getActionScope {
    my ($self) = @_;
    return $gProductName . ' ' . $self->getScopeName($self->getScope());
}


#-------------------------------------------------------------------------------
# Converts the scope enumeration into a string

sub getScopeName {
    my ($self, $scope) = @_;

    if (!defined $scope) {
        return 'instance';
    }

    if ($scope eq 'all_instances') {
        return 'instances';
    }

    return ($scope eq 'system_only') ? 'system (without sapstartsrv)'
                                     : $scope;
}


#-------------------------------------------------------------------------------
# Returns true if the instance should be started after this instance
# is registered.

sub isStartInstance {
    return 1;
}


#-------------------------------------------------------------------------------
# Returns true if the SAP HANA system should be stopped before
# registering this instance.

sub isStopSystem {
    return 1;
}


#-------------------------------------------------------------------------------
# Displays additional information of unregister

sub showAdditionalUnregisterInfo {

    my ($self, $msg, $msglst) = @_;

    my $instconfig = $self->getInstconfig();
    my $sapSys     = $instconfig->getSAPSystem();
    my $instance   = $instconfig->getOwnInstance();

    if (defined $instance) {

        my $instPath = $sapSys->get_globalSidDir();
        my $dataPath = $instance->getDataPath(1);
        my $logPath  = $instance->getLogPath(1);
        my $progPath = $instance->get_globalTrexInstallProgamDir();
        my $info     = "\nThese directories still exist:\n"
                     . "   Installation: '$instPath'\n"
                     . "   Data........: '$dataPath'\n"
                     . "   Log.........: '$logPath'\n\n";

        $info .= "The instance can be registered with the following command:\n"
                . $progPath . $path_separator . "hdbreg\n"
                if (!$instconfig->isCheckOnly());

        $msglst->addProgressMessage($info);
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Shows the host names of outstanding slaves.

sub showUnfinishedRegisterSlaves {

    my ($self, $msg) = @_;

    my $instconfig = $self->getInstconfig();
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
    $instconfig->showUnfinishedSlaves($self->getActionProgressive(),
                                      $self->getProgramName());
    return 1;
}


#===============================================================================
#   M A I N

sub main{
    my $app = new __PACKAGE__;
    return $app->mainFunction(\@_);
}


#-------------------------------------------------------------------------------
# This function can also be called from other classes:
#     require SDB::Install::App::Console::HdbReg;
#     my $register = new SDB::Install::App::Console::HdbReg();
#     $register->mainFunction(\@ARGV);

sub mainFunction {
    my ($app, $args) = @_;

    my $rc;
    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval{

        if (!defined $app->InitCmdLineArgs ($args)){
           return undef;
        }

        if (defined $app->{return}){
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            return $app->{return};
        }

        $rc = $app->init ();

    };

    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }

    $rc = $app->handleReturnCodes($@, $rc);

    $app->CleanUp ();

    undef $app;

    return defined $rc ? 0 : 1;
}

sub shouldWarnIfCalledStandalone{
    return 1;
}


1;
