#!/usr/bin/perl
#

package SDB::Install::App::Console::HdbupdConf;

use base SDB::Install::App::Console;
use SDB::Install::Installation qw (SDB_O_CREAT);
use SDB::Install::SAPSystem;
use SDB::Install::Globals qw ($gProductNameInstaller $gShortProductNameEngine
                              $gProductNameEngine gPrepareLogPath $gLogDir);
use SDB::Install::SysVars qw ($isWin $path_separator);
use Getopt::Long;
use SDB::Install::System qw (isEmptyDir getSAPDrive);
use SDB::Install::Tools qw (printTableToArrayOfLines);
use SAPDB::Install::Hostname;
use Cwd qw (getcwd);
use SDB::Install::Configuration::HdbupdConf;
use SDB::Install::Log;


use strict;

#----------------------------------------------------------------------------------

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}

#----------------------------------------------------------------------------------

sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}

#----------------------------------------------------------------------------------

sub init{
    my ($self) = @_;
    my $retval = 1;
    my $msglst = $self->getMsgLst ();
    my $installerVersion = $self->GetInstaller->GetVersion();
    my $caption = "$gProductNameInstaller - HdbupdConf " .$installerVersion; 
    $msglst->addProgressMessage("\n\n".$caption . "\n" . ('*' x length ($caption)) . "\n\n");
    my $instconfig = $self->{'instconfig'};
    $retval = $instconfig->CheckParams($self->{batch_mode});
    if (!$retval && $instconfig->ErrorState()) {
        $self->setErrorMessage ('Configuration error:', $instconfig->getErrMsgLst ());
    }
    else {
        my $drive;
        my $hdbupdConfCtrl;
        my $runtimeDir = $self->GetInstaller->GetRuntimeDir();
        # extract sid directory (e.g. /usr/sap/<sid> or /hana/shared/<sid>) from prefix of the runtime dir,
        # e.g. '/hana/shared/AAA/global/hdb/install', or '/usr/sap/AAA/global/hdb/install',
        # or '/hana/shared/AAA/SYS/global/hdb/install' or '/usr/sap/AAA/SYS/global/hdb/install'
        #
        # greed is good, as we know; but in this case we need the greed coming from the right and going to the left:
        my $reverseRuntimeDir = scalar reverse $runtimeDir;
        # e.g. llatsni\/bdh\/labolg\/(SYS\/)?(\w{3}\/.*)
        my $reverseUsrSapSidPattern = join (quotemeta ($path_separator), ('llatsni', 'bdh', 'labolg', '(SYS'.quotemeta ($path_separator).')?(\w{3}'.quotemeta ($path_separator).'.*)'));
        my (undef, $reverseUsrSapSid) = ($reverseRuntimeDir =~ /$reverseUsrSapSidPattern/i);
        my $usrSapSid = scalar reverse $reverseUsrSapSid;
        # e.g. .*\/(\w{3})
        my $sidPattern = join (quotemeta ($path_separator), ('.*', '(\w{3})'));
        my ($sid) = $usrSapSid =~ /$sidPattern/;
        if ($isWin){
            $drive = getSAPDrive ($self->getMsgLst ());
        }
        if($usrSapSid) {
            my $sapSys;
            if (-d $usrSapSid){
                $sapSys = new SDB::Install::SAPSystem();
                if ($isWin){
                    $sapSys->initSAPSystem($usrSapSid, SDB_O_CREAT, undef, $drive);
                }
                else{
                    $sapSys->initSAPSystem($usrSapSid);
                }
            }
            # this is a 'TrexInstance' object:
            $hdbupdConfCtrl = $sapSys->getNewDBInstances()->[0];
            if ($isWin){
                $hdbupdConfCtrl->set_localUsrSapSidDir($drive.$path_separator.'usr'.$path_separator.'sap'.$path_separator.$sid);
            }
        }
        else {
            $retval = undef;
        }
        my $errlst = new SDB::Install::MsgLst();
        my $oldPwd = getcwd();
        # we never write logfiles to this directory (local, under /var/tmp);
        # 'gPrepareLogPath' just creates it and chdirs to it
        # in case we have to dump core, or similar:
        if(!gPrepareLogPath($sid, 'hdbupdConf', $errlst, $self->{options}->{instlog_dir})) {
            $self->setErrorMessage('Could not get log directory', $errlst);
            return undef;
        }

        if (!defined $hdbupdConfCtrl) {
            $self->setErrorMessage("$gProductNameEngine '$sid' not found\n",
                                                                       $errlst);
            $retval = undef;
        }
        else {
            $hdbupdConfCtrl->handleRestartEvent($self);
        }

        # when we are done, we can remove it if it is empty:
        if(isEmptyDir($gLogDir)) {
            chdir($oldPwd);
            my $rc = rmdir($gLogDir);
            if(!$rc) {
                $self->setErrorMessage("Could not remove directory \"$gLogDir\".");
                return undef;
            }
        }
    }
    return $retval;
}

#----------------------------------------------------------------------------------

sub InitCmdLineArgs{
    my (
        $self,
        $args
    ) = @_;
    $self->{'options'} = {};
    $self->{'instconfig'} = new SDB::Install::Configuration::HdbupdConf(
        $self->{'options'},
        $self->{'configfile'}
    );
    $self->{'instconfig'}->InitDefaults();
    my $rc = $self->SUPER::InitCmdLineArgs($args);
    if(!defined $rc) {
        return undef;
    }
    local @ARGV = @$rc;
    my $optctrl = $self->{'instconfig'}->GetOptionCtrl();
    if(!defined $self->getOptions ($optctrl, 0)) {
        $self->{'return'} = -1;
        return undef;
    }
    return \@ARGV;
}

#----------------------------------------------------------------------------------
# Returns a reference to an array containing program options
# without help/info options.

sub GetSwitches{

    return $_[0]->{'instconfig'}->GetSwitchesFromParams();
}

#----------------------------------------------------------------------------------
# Returns a reference to an array of arrays containg the description
# of program options without common batch/help options.

sub GetUsage{

    return $_[0]->{instconfig}->GetUsageFromParams();
}

#----------------------------------------------------------------------------------

sub main{
    my $app = new __PACKAGE__;
    my $rc;

    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval { 
        $app->InitCmdLineArgs(\@_);
        if(defined $app->{'return'}) {
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            return $app->{'return'};
        }
        $rc = $app->init();
    };
    if(defined $app->{'return'}) {
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{'return'};
    }
    my $msglst = $app->getMsgLst ();
    if($@) {
        my $signalInfoText;
        if($@ =~ /^__SIGINT__/) {
            $msglst->addMessage('User canceled hdbupdConf with Ctrl + c.');
            $rc = 2;
        }
        elsif ($@ =~ /^__SIGPIPE__/) {
            $app->setErrorMessage ('Broken pipe.', $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        elsif ($@ =~ /^__SIG(\w+)__/) {
            $app->setErrorMessage ("Caught signal $1 .", $app->{stackBacktraceMsglst});
            $signalInfoText = $app->getSignalInfoText ();
            undef $rc;
        }
        else {
            $app->setErrorMessage ('unhandled exception: '. $@, $app->{stackBacktraceMsglst});
            $rc = undef;
        }
        if ($signalInfoText){
            $app->appendErrorMessage ($signalInfoText);
        }
    }
    if(defined $rc) {
        if($rc == 2) {
            print "\n";
            $msglst->addProgressMessage('hdbupdConf aborted.');
        }
        else {
            $msglst->addProgressMessage('hdbupdConf done.');
        }
    }
    else {
        $app->ShowErrorMsg('hdbupdConf failed.',$app->getErrMsgLst ());
    }
    $app->CleanUp();   
    undef $app;
    return defined $rc ? 0 : 1; 
}

#----------------------------------------------------------------------------------

sub shouldWarnIfCalledStandalone{
    return 0;
}

1;



