#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
# Desc: rename an existing NewDB (change SID, InstanceNr, host names)
#   required by storabe based system copy 

package SDB::Install::App::Console::Rename;

use base SDB::Install::App::Console;
use SDB::Install::SAPSystem;
use SDB::Install::SAPSystemUtilities;
use SDB::Install::SAPSystemRename;
use SDB::Install::Globals qw ($gProductNameInstaller
                              $gProductNameSystem
                              $gShortProductNameEngine);
use SDB::Install::SysVars qw ($isWin $path_separator);
use Getopt::Long;
use SDB::Install::Tools qw (askConfirmation printTableToArrayOfLines);
use SAPDB::Install::Hostname;
use SDB::Install::Configuration::Rename;
use SDB::Install::Log;

use strict;

our $programVers = '0060';

sub new {
    my $self = shift->SUPER::new ();
    return $self;
}


#-------------------------------------------------------------------------------
# Returns the title of this program

sub getProgramTitle {
    return "$gShortProductNameEngine Rename";
}


sub InitApp{
    my ($self) = @_;
    $self->SUPER::InitApp();
    return 1;
}

#-------------------------------------------------------------------------------
# Initializes hdbrename, performs several checks and calls SAPSystemRename.
#
# Without parameters
# Returns undef in case of an error

sub init{
    my ($self) = @_;
    my $rc = 1;

    if ($isWin){
        $self->setErrorMessage ('System rename is not yet implemented for this platform');
        return undef;
    }

    if (!$self->checkIsAdmin($self->getProgramName())){
        return undef;
    }

    my $msglst     = $self->getMsgLst ();
    my $instconfig = $self->getInstconfig();
    $instconfig->setMsgLstContext ([$msglst]);
    if (!$self->isHelp() &&
        !defined $instconfig->checkSystemRequirements ($self->GetSysInfo())){
        $self->setErrorMessage ("Checking system requirements failed!", $instconfig->getErrMsgLst ());
        return undef;
    }

    $self->warnIfShouldWarnIfCalledStandalone();
    my $caption = $gProductNameInstaller . ' - ' . $self->getProgramTitle()
                                        . '  ' . $self->GetInstaller->{version};
    
    $msglst->addProgressMessage ("\n\n".$caption . "\n" . ('*' x length ($caption)) . "\n");

    $self->addStartDateMsg($msglst, $self->getProgramName() . "  $programVers");

    my $msg = $msglst->addMessage ("Checking configuration");
    $msg->setEndTag ('Configuration Check');
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
    $rc = $instconfig->CheckParams ($self->{batch_mode});

    my $cfgDumpRc = $self->handleOptionDumpConfigFileTemplate();
    if (!defined $cfgDumpRc){
        $self->{return} = 1;
        return undef;
    }
    if ($cfgDumpRc){
        $self->{return} = 0;
        return 1;
    }

    if($self->{batch_mode}){
        if (!$rc){
            if (defined $rc){
                $self->setErrorMessage ("Running in batch mode", $instconfig->getErrMsgLst ());
            }
            else{
                $self->setErrorMessage ('Configuration error (batch mode):', $instconfig->getErrMsgLst ());
            }
            return undef;
        }
    }
    else{
        if (!$rc && $instconfig->ErrorState ()){
            $self->setErrorMessage ('Configuration error:',$instconfig->getErrMsgLst ());
            if (!defined $rc){
                return undef;
            }
        }
        if (!defined $self->ConfigureInteractive ($instconfig)){
            return undef;
        }   
    }
    $instconfig->setMsgLstContext ([$msglst]);
    if ($instconfig->isShowPending()) {
        $instconfig->showUnfinishedSlaves($self->getActionProgressive(),
                                          $self->getProgramName(), 1);
        $self->{return} = 0;
        return 1;
    }
    $instconfig->setMsgLstContext ([$msg->getSubMsgLst ()]);
    if ($instconfig->isCheckOnly()) {
        my $originAction = $self->{action};
        $self->{action}            = "Check $originAction";
        $self->{actionProgressive} = "Checking $originAction";
        $self->{actionDone}        = "checked ($originAction)";
    }

    my $sapSys = $instconfig->getSAPSystem();

    $sapSys->SetProgressHandler ($msglst->getProgressHandler());
    $instconfig->resetSAPSystemHashes();

    if (!$self->defineLog($self->getProgramName(),
                          $self->getProgramName(),
                          $instconfig->isCheckOnly(),
                          $instconfig->getValue('newSID'))) {
        return undef;
    }

    $instconfig->dumpConfigFile($self->getProgramName());
    $instconfig->displayParameterSummary();
    if (!$self->{batch_mode} && !$instconfig->isCheckOnly()
                       && !$self->{options}->{noprompt} && !askConfirmation()) {
        $msglst->addProgressMessage($self->getActionProgressive()
                                    . " $gProductNameSystem cancelled");
        $self->{return} = 0;
        return 1;
    }

    $msg = $msglst->addProgressMessage
                           ($self->getActionProgressive() . ' instance...');

    my $renameHandler = new SDB::Install::SAPSystemRename();

    $sapSys->setMsgLstContext ([$msg->getSubMsgLst ()]);
    
    if ($instconfig->{isSlave}){

        if (!$instconfig->isCheckOnly() &&
            !defined $renameHandler->renameSlave ($sapSys,
                                                  $instconfig,
                                                  $self->GetSysInfo())){
            $self->appendErrorMessage($self->getActionProgressive().' slave failed!',
                                    $sapSys->getErrMsgLst ());
            return undef;
        }
        
    }
    else{
        my $rc;
        eval {
            $rc = $renameHandler->renameMaster ($self,$sapSys);
            if (!defined $rc){
                $self->setErrorMessage ($self->getActionProgressive().' master failed!',
                                        $sapSys->getErrMsgLst ());
                if($renameHandler->isDisabledAutoFailOver()){
                    $renameHandler->enableAutoFailover($instconfig,$self->getMsgLst ());
                }
            }
        };
        if($@){
            if($renameHandler->isDisabledAutoFailOver()){
                    $renameHandler->enableAutoFailover($instconfig,$self->getMsgLst ());
            }
        }
        return undef if (!defined $rc);
    }

    $self->addLogToTraceDir($sapSys,
                            $self->getProgramName(),
                            $instconfig->isCheckOnly(),
                           );  # now trace directory is renamed
    $self->showAdditionalInfo();

    return 1;
}


#-------------------------------------------------------------------------------
# Initializes the arguments of the command line
#
# Parameters string-array arguments
#
# Returns a reference to a string array containing the arguments
#         or undef in case of an error

sub InitCmdLineArgs{
    my ($self,$args) = @_;

    $self->{instconfig} =
            new SDB::Install::Configuration::Rename ($self->{options});

    $self->{action}            = 'Rename';
    $self->{actionProgressive} = 'Renaming';
    $self->{actionDone}        = 'renamed';

    if (!$self->additionalInit()) {
        return undef;
    }

    return $self->SUPER::InitCmdLineArgs ($args);
}


#-------------------------------------------------------------------------------
# Additional initialization - may be overridden

sub additionalInit {
    return 1;
}


#-------------------------------------------------------------------------------
# Displays additional information
#
# Parameter: $sapSys SBD::Install::SAPSystem

sub showAdditionalInfo {

    my ($self) = @_;

    my $instconfig = $self->getInstconfig();

    if (!$instconfig->isCheckOnly()) {
        $instconfig->showUnfinishedSlaves($self->getActionProgressive(),
                                          $self->getProgramName());
    }

    return 1;
}


sub main{
    my $app = new __PACKAGE__;
    return $app->mainFunction(\@_);
}


#-------------------------------------------------------------------------------
# This function can also be called from other classes:
#     require SDB::Install::App::Console::Rename;
#     my $rename = new SDB::Install::App::Console::Rename();
#     $rename->mainFunction(\@ARGV);

sub mainFunction {
    my ($app, $args) = @_;

    my $rc;
    $app->{stackBacktraceMsglst} = new SDB::Install::MsgLst ();
    eval{

        $app->InitCmdLineArgs ($args);

        if (defined $app->{return}){
            LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
            return $app->{return};
        }

        $rc = $app->init ();
    };

    if (defined $app->{return}){
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace();
        return $app->{return};
    }

    $rc = $app->handleReturnCodes($@, $rc);

    $app->CleanUp ();   
    
    undef $app;
    
    return defined $rc ? 0 : 1; 
}

sub shouldWarnIfCalledStandalone{
    return 1;
}

1;



