#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: tar / tar.gz archive


package SAPDB::Install::Untgz;

bootstrap SAPDB::Install::Untgz;

sub AUTOLOAD {
  die "cannot load symbol ".$AUTOLOAD."\n"
    unless ($AUTOLOAD =~ /bootstrap$/);
  
  require XSLoader;
  XSLoader::load (__PACKAGE__);
}

1;

package SDB::Install::Archive;

use SDB::Install::BaseLegacy;

our @ISA = qw (SDB::Install::BaseLegacy);

#--------------------------------------------------------------------------------------------------

sub new{
	my $self = shift->SUPER::new ();
	($self->{archive}) = @_;
	$self->open();
	return $self;
}

#--------------------------------------------------------------------------------------------------

sub open{
	my ($self,$archive) = @_;
	
	if (defined $archive){
		$self->{archive} = $archive;
	}
	
	unless (defined $self->{archive}){
		$self->AddError ('no archive file defined');
		return undef;
	}
	
	if (defined $self->{arch_handle}){
		$self->close ();
	}
	
	my $untgz = SAPDB::Install::Untgz::new ();
	
	unless (defined $untgz->Open ($self->{archive})){
		$self->AddError ('cannot open archive "'.$self->{archive}.'": ' .
				 join (': ',$untgz->GetErr()));
		return undef;
	}
	$self->{arch_handle} = $untgz;
	return 1;
}

#--------------------------------------------------------------------------------------------------

sub read_meta{
	my ($self) = @_;
	my $data = $self->get_file_content ();
	my $result = {};

	unless (defined $data){
		$self->AddError ('cannot extract meta data from archive "'.$self->{archive}.'"');
		return undef;
	}
	foreach my $line (split ("\n",$data)){
		next if $line !~ /\S/;
		next if $line =~ /^\s*#/;
		my ($key,$value) = ($line =~ /(\S+)\s+=\s+"(.*)"/);
		if (defined $key){
			$result->{$key} = $value;
		}
	}
	return $result;
}

#--------------------------------------------------------------------------------------------------

sub read_filelist{
	my ($self) = @_;

	my $filelist_value =  $self->get_file_content ();

	my $files = {};

    my $MODE_BIT_LEN = 12; # no of bits for file permissions
    my $MODE_SHIFT  = 32 - $MODE_BIT_LEN; # no of shifts needed to shift file permissions to 'right' end.
	foreach my $line (split "\n", $filelist_value){
		next if $line !~ /\S/ or $line =~ /^\s*#/;
		my ($file,$stat) = ($line =~ /\"(.+)\"\s+(\S+)/);
		my %file_info;
		my ($size_hex,$mtime_hex,$mode_flags_hex);
		($file_info{checksum},
		 $size_hex,
		 $mtime_hex,
		 $mode_flags_hex
		) = split (',',$stat);
		$file_info{size} = hex ($size_hex);
		$file_info{mtime} = hex ($mtime_hex);
		$file_info{mode} = hex ($mode_flags_hex) >> $MODE_SHIFT;
		my $flags = hex ($mode_flags_hex);
		my $i = 1;
		foreach my $flag (reverse ('forceExtract','substituteMacros','setRootAsOwner', 'specialFile', 'isSymlink')){
			if ($flags & (1 << ($MODE_SHIFT - $i++))) {
				$file_info{$flag} = 1;
			}
		}
		$files->{$file} = \%file_info;
	}
	return $files;
}


#--------------------------------------------------------------------------------------------------

sub next{
	my ($self) = @_;
	my $file = {};
	
	( $file->{mode},
	  $file->{size},
	  $file->{time},
	  $file->{name}) = $self->{arch_handle}->Next ();
	
	unless (defined $file->{name}){
		my @err = $self->{arch_handle}->GetErr ();
		if (@err){
			# error
			$self->AddError ('error reading archive: ' .
					 join (': ',@err));
			return undef;
		}
		# eof reached		
		return {};
	}
	$self->{current_file} = $file;
	return $file; 
}


#--------------------------------------------------------------------------------------------------

sub get_file_content{
	my ($self) = @_;
	my $result;
	unless (defined $self->{arch_handle}->ExtractScalar ($result)){
		my @err = $self->{arch_handle}->GetErr ();
		$self->AddError ('cannot extract file content' . @err ? (': ' .
				  join (': ',@err)) : '');
		return undef;
	}
	return $result;
}	

#--------------------------------------------------------------------------------------------------

#--------------------------------------------------------------------------------------------------

sub close ($){
	undef $_[0]->{arch_handle};
}

#--------------------------------------------------------------------------------------------------

sub rewind ($){
	$_[0]->{arch_handle}->Rewind ();
}

#--------------------------------------------------------------------------------------------------

sub set_owner{ 
	shift->{arch_handle}->SetOwner (@_);
}

#--------------------------------------------------------------------------------------------------

sub GetArchiveHandle ($){
	$_[0]->{arch_handle};
}

#--------------------------------------------------------------------------------------------------

sub Extract{
	my ($self,$destdir) = @_;

	my $archive_handle = $self->{arch_handle};

	if (defined $destdir){
		unless (-d $destdir){
			$self->AddError ('cannot set destination directory: ' . $!);
			return undef;
		} 
		$archive_handle->SetDestDir ($destdir);
	}
	my $file;
	my $rc;
	my $msg;
	
	while (1){
		$file = $self->next ();
		# error
		return undef unless (defined $file);

# 		eof reached
		last if ( ( scalar keys %$file ) == 0);

		if ($file->{mode} & 040000){
			# extract a directory
			$rc = $archive_handle->ExtractFile ();
			unless (defined $rc and $rc == 0){
				$self->AddError (join (': ',$archive_handle->GetErr));
				return undef;
			}
			next;
		}
				
		$msg = $self->AddProgressMessage ('extracting '.$file->{name});
		$rc = $archive_handle->ExtractFile ();
		unless (defined $rc and $rc == 0){
			$self->AddError (join (': ',$archive_handle->GetErr));
			return undef;
		}
		$msg->{submsg} = [$self->GenMsg (undef, sprintf ("file info: mode = 0%o, size = %d bytes", $file->{mode} & 07777,$file->{size}))];
	}
	return 1;
}

#--------------------------------------------------------------------------------------------------

sub SetDestDir ($$){
	$_[0]->{arch_handle}->SetDestDir($_[1]);
}

#--------------------------------------------------------------------------------------------------

sub ExtractFile ($){
	$_[0]->{arch_handle}->ExtractFile();
}

#--------------------------------------------------------------------------------------------------

sub main{
	local @ARGV = @_;
	require Getopt::Long;
	import Getopt::Long;
	
	my $usage = 'usage: -f <tar|tar.gz archive> [-C <destination directory>]';
	
	my ($opt_f, $opt_C);
	
	if (!GetOptions ('f=s' => \$opt_f,
				'C=s' => \$opt_C) || not defined $opt_f){
		print "$usage\n";
		return 1;		
	}

	my $arch = new SDB::Install::Archive($opt_f);
	
	$arch->SetProgressHandler (sub {print $_[0] . "\n"});

	
	if ($arch->ErrorState){
		print STDERR $arch->GetErrorString();
		return 1;
	}
		
	unless (defined $arch->Extract ($opt_C)){
		print STDERR $arch->GetErrorString();
		return 1;
	}
	
	print "done\n";
	return 1;
}



1;

