package SDB::Install::Configuration;

use strict;

use SDB::Install::PersistencyMngr;
use SDB::Install::Tools qw (readini);
use SDB::Install::System qw (getSysProgPath getHomeDir isAdmin);
use SDB::Install::SysVars;
use SDB::Install::IniFile;
use SDB::Common::Utils qw(createSysAdminUserName);
use SDB::Install::DebugUtilities qw (dumpThings);
use SDB::Install::Configuration::OptionIgnore;
use SDB::Install::Configuration::OptionTimeout;
use SAPDB::Install::Hostname;
use SDB::Install::Globals qw ($gLogDir
                              $gProductName
                              $gProductNameSystem
                              $gSapsysGroupName
                              $gUseDefaultValue);
use experimental qw (smartmatch);
use SDB::Install::ConfigurationChangeEvent;
use File::Spec;

our $sysProgPath = $isWin || $isApple ? getSysProgPath () : '/usr';
our $userProgPath = getHomeDir ();
our $bool_false_pattern = '^off$|^false$|^0$|^-$|^no$|^n$';
our $bool_true_pattern =  '^on$|^true$|^1$|^\+$|^x$|^yes$|^y$';

our @EXPORT = qw ($bool_false_pattern $bool_true_pattern);

our @ISA = qw (SDB::Install::PersistencyMngr Exporter);

my $PREFIX_OPT       = '--';
my $PREFIX_SHORT_OPT = '-';
my $READ_PW_SYNTAX   = '--read_password_from_stdin=xml';
my $READ_PW_HELP     = 'Reads passwords via XML input stream in batch mode';

sub getPersistentFile {return undef}

sub new{
	my $self = shift->SUPER::new ();
	
	($self->{options}, $self->{config_file}) = @_;

	if (defined $self->{config_file}){
		$self->readConfigfile();
    }
        
     #   
     #   types validation regex
     #
     $self->{type_pattern} = {
        'string' => '.+',   # at least one character
        'path' => '^[\\\/\w_\-\+\.\(\)]+$',     # at least one character
        'number' => '\d+' , # at least one number
     };
    $self->{warnings} = [];
    return $self;
}

sub readConfigfile {
    my ($self, $configfilename) = @_;

    if (defined $configfilename) {
        $self->{config_file} = $configfilename;
    }

    if (!defined $self->{config_file}) {
        return 1;
    }
    my $errMsgLst = new SDB::Install::MsgLst ();
    my $cfg = readini ($self->{config_file}, $errMsgLst, 1);

    if (!defined $cfg){
        $self->setErrorMessage ("Cannot read config file '$self->{config_file}'", $errMsgLst);
        return undef;
    }
    $self->{cfgfile_data} = $cfg;
    return 1;
}


sub enumExistingInstallations{
	return {};
}


sub isInstallationSelected{
	return 0;
}


sub InitDefaults{
    return 1;
}

sub getProductName{
	return "Unknown product";
}

sub getShortProductName{
	return "Unknown";
}

sub getCustomLogSuffix {
    return undef;
}

sub getWarningList{
	return $_[0]->{warnings};
}

sub Reset{
	my ($self) = @_;
	foreach my $key (@{$self->pers_keys}){
		if ($key eq '_kitversion'){
			next;
		}
		if (exists $self->{$key}){
			delete $self->{$key};
		}
	}
	my $params = $self->{params};
	foreach my $id (keys (%$params)){
		$params->{$id}->{value} = undef;
	}
}

sub checkSystemRequirements{
	return 1;
}

sub getIgnoreValues{
    return undef;
}

sub getHiddenIgnoreValues{
    return undef;
}

sub getTimeoutValues{
    return undef;
}

sub getOptionIgnore{
    my ($self) = @_;
    if (!exists $self->{optionIgnore}){
        if (!defined $self->getIgnoreValues() && !defined $self->getHiddenIgnoreValues()){
            $self->{optionIgnore} = undef;
            return undef;
        }
        $self->{optionIgnore} = new SDB::Install::Configuration::OptionIgnore (
                $self->getIgnoreValues(),
                $self->getHiddenIgnoreValues());
    }
    return $self->{optionIgnore};
}

sub setOptionIgnore{
    my ($self, $value) = @_;
    $self->{optionIgnore} = $value;
}

sub getOptionTimeout{
    my ($self) = @_;
    if (!exists $self->{optionTimeout}){
        if (!defined $self->getTimeoutValues()){
            $self->{optionTimeout} = undef;
            return undef;
        }
        $self->{optionTimeout} = new SDB::Install::Configuration::OptionTimeout (
                $self->getTimeoutValues());
    }
    return $self->{optionTimeout};
}


#
# required disk space beside the extracted installation packages
#

sub getFootprints{
    my ($self) = @_;
    my ($param, $path);
    my @result;
    foreach my $id (keys %{$self->{params}}){
        $param = $self->{params}->{$id};
        if ($param->{skip}){
            next;
        }
        if ($param->{type} ne 'path'){
            next;
        }
        $path = $self->getValue ($id);
        if ($path && $param->{required_disk_space}){
            if ($param->{diskspace_check_subdir}){
                $path .= $path_separator . $param->{diskspace_check_subdir};
            }
            push @result, [$path, $param->{required_disk_space},
                defined $param->{footprint_str} ? $param->{footprint_str} : $param->{str}];
        }
    }
    return \@result;
}

sub getIgnore{
    my ($self, $value) = @_;
    if (defined $self->{optionIgnore}){
        return $self->{optionIgnore}->getArg ($value);
    }
    return undef;
}

sub getTimeout{
    my ($self, $value) = @_;
   if (defined $self->{optionTimeout}){
        return $self->{optionTimeout}->getArg ($value);
    }
    return undef;
}

sub GetPathByName{
	my ($self, $name, $installations) = @_;
	
	if (!defined $installations){
		$installations = $self->{installations};
		return '' if !defined $installations;
	}
	
	foreach my $inst (keys (%$installations)){
		if (exists $installations->{$inst}->{name} && $installations->{$inst}->{name} eq $name){
			return $inst;
		}
	}
	return '';
}



sub getDefaultUIString{
	my ($self,$id,$substitute,$no_default) = @_;
	if (exists $self->{params}->{$id} && defined $self->getDefault($id)){
		if ($self->{params}->{$id}->{type} =~ /bool/){
			my $defaultValue = $self->getDefault($id);
			if ($defaultValue =~ /$bool_true_pattern/){
				return 'y';
			} elsif ($defaultValue =~ /$bool_false_pattern/){
				return 'n';
			}
		}
		return $self->getDefault ($id,$substitute,$no_default);
	}
	return undef;
}

sub getUIStringFromValue{
	my ($self,$id,$value) = @_;

	my $param = $self->{params}->{$id};

	if (defined $param->{ui_values}) {

		foreach my $i (0..scalar(@{$param->{valid_values}})-1) {

			if (($value eq $param->{valid_values}->[$i])
		        ||
		        (exists $param->{visible_alias_values}
		         && ($value eq $param->{visible_alias_values}->[$i]))) {

				return $param->{ui_values}->[$i]
			}
		}
	}

	return $value;
}

sub getValueFromUIString{
	my ($self,$id,$uistring) = @_;

	my $param = $self->{params}->{$id};

	if (defined $param->{ui_values}) {

		foreach my $i (0..scalar(@{$param->{ui_values}})-1) {

			if ($uistring eq $param->{ui_values}->[$i]) {

				return (exists $param->{visible_alias_values})
				       ? $param->{visible_alias_values}->[$i]
				       : $param->{valid_values}->[$i];
			}
		}
		return undef;
	}
	return $uistring;
}

sub getParamIdByOpt{
    my ($self, $opt) = @_;
    foreach my $paramId (keys (%{$self->{params}})){
        if ($self->{params}->{$paramId}->{opt} eq $opt){
            return $paramId;
        }
    }
    return undef;
}




sub substituteParamsByReplaceHash{
    my ($self, $value, $substitute, $replaceHash) = @_;
    if (!defined $value) {
        return undef;
    }
    my $refValue = ref($value);
    if (($refValue !~ /ARRAY/) && ($refValue !~ /HASH/) && ($value =~ /\$/)) {
        my @param_ids = ($value =~ /\$([\w_]+)/g);
        push @param_ids, ($value =~ /\$\{(.+?)\}/g);
        my $pattern;
        my $replacement;
        foreach my $param_id (@param_ids){
            $replacement = undef;
            if ($param_id){
                $pattern = '\$\{'.$param_id.'\}|\$'.$param_id;
                if (defined $substitute){
                    $value =~ s/$pattern/$substitute/g;
                }
                elsif (exists $replaceHash->{$param_id}){
                    $replacement = $replaceHash->{$param_id};
                    $value =~ s/$pattern/$replacement/g;
                }
            }
        }
    }
    return $value
}


sub substituteValues{
    my ($self, $value, $substitute, $no_default) = @_;
    my %replaceHash;
    my ($param, $replacement);
    foreach my $paramId (@{$self->getParamIds}){
        $param = $self->{params}->{$paramId};
        if (defined $param->{value}){
            $replacement = $param->{value};
        }
        elsif (defined $param->{batchValue}){
            $replacement = $param->{batchValue};
        }
        elsif (!$no_default && defined $param->{default}){
            $replacement = $param->{default};
        }
        else{
            next;
        }
        $replaceHash{$paramId} = $replacement;
        if (defined $param->{opt}){
            $replaceHash{$param->{opt}} = $replacement;
        }
    }

    if (defined $self->{helper_params}){
        foreach my $paramId (keys %{$self->{helper_params}}){
           $replaceHash{$paramId} = $self->{helper_params}->{$paramId};
        }
    }
    return $self->substituteParamsByReplaceHash ($value, $substitute, \%replaceHash);
}


sub getDefault{
	my ($self, $id, $substitute, $no_default) = @_;

	return undef if(!exists($self->{params}->{$id}));

	my $internalDefault = $self->{params}->{$id}->{default};
	my $configFileDefault = ($self->getType($id) !~ /passwd/) ? $self->getDefaultValueFromConfigfile($id) : undef;
	my $defaultValue = defined($configFileDefault) ? $configFileDefault : $internalDefault;

	return $self->substituteValues($defaultValue, $substitute, $no_default);
}

sub setDefault{
    my ($self,$id,$value,$initWithDefaultFlag) = @_;
    if (exists $self->{params}->{$id}){
        $self->{params}->{$id}->{default} = $value;
        if (defined $initWithDefaultFlag){
            $self->{params}->{$id}->{init_with_default} = $initWithDefaultFlag;
        }
    }
    return 1;
}



#
# handle --ignore and --timeout from config file
#
sub _handleOptionFromCfgFile{
    my ($self, $option, $section) = @_;
    if (!defined $option){
        # no acceptable values defined
        return 1;
    }
    my $optionName = $option->getOptionName();
    if (defined $self->{options}->{$optionName}){
        #already set via command line option
        return 1;
    }

    # default section is 'General'
    $section //= 'General';

    my $cfg_data = $self->{cfgfile_data};
    if (!defined $cfg_data || !defined $cfg_data->{$section} || !defined $cfg_data->{$section}->{$optionName}){
        # option not in config file
        return 1;
    }
    $option->setMsgLstContext($self->setMsgLstContext());
    return $option->parseArgs(join(',', @{$cfg_data->{$section}->{$optionName}}));
}

#
# check the current configuration
#

sub CheckParams{
	my ($self, $batchmode) = @_;

	my $rc = 1;
	my $params = $self->{params};

	$self->ResetError();

	if (!defined $self->_handleOptionFromCfgFile($self->getOptionIgnore())){
		return undef;
	}

	if (!defined $self->_handleOptionFromCfgFile($self->getOptionTimeout())){
		return undef;
	}

	my $result;
	my $batch_error = 0;
	my $fatal_error = 0;
	foreach my $currID (@{$self->getParamIds ()}){
		my $id    = $currID;
		my $param = $params->{$id};
        if (defined $param->{retry_param}) {
            my $retryID = $param->{retry_param};
            if (defined $params->{$retryID}->{batchValue} ||
                defined $params->{$retryID}->{value}) {
                next;
            }
            $id    = $retryID;
            $param = $params->{$id};
        }
        if ($param->{DEPRECATED}) {
            next;
        } 
        if ($param->{skip} && $batchmode) {
            $self->AddMessage ($self->getParamName($id) . ' is skipped');
            next;
        }
        if (defined $param->{value}){
            # parameter is already set

            if ($param->{type} =~ /map/) {

                foreach my $val (keys %{$param->{value}}) {
                    $self->AddMessage($self->getParamName($id, $val)
                        ." is set to value = '$param->{value}->{$val}'");
                }
            }
            else {
                $self->addLogValueMsg($id, $param->{value},'is set to value =');
            }
			next;
        }
		my $value;

        if ($batchmode && defined $self->{options}->{read_password_from_stdin} &&
            ($param->{type} eq 'passwd' || $param->{type} eq 'initial_passwd')){

            my $passwd;
            if ($self->{options}->{read_password_from_stdin} eq 'xml'){
                $passwd = $self->getPasswordFromXML($id, $param);
            }
            else{
                print "Start reading from input channel...\n";
                $passwd = <STDIN>;
                chomp ($passwd);
                print "... Done.\n";
            }
            if (defined $passwd){
                $result = $self->setValue ($id, $passwd);
                if ($result){
                    $self->addLogValueMsg($id, $passwd,
                                          'is given via input stream, value =');
                }
                $rc   &&= $result;
                next;
            }
            else {
                $self->AddMessage ($self->getParamName($id) .
                                   ' is not part of input stream');
            }
        }

		my $optionValue;
        if (defined $param->{opt}){
            $optionValue = $self->{options}->{$param->{opt}};
        }

		if (defined $optionValue){
            # parameter is specified within the command line

			if ($param->{type} =~ /bool/){
				$value = $self->assignBoolValue($id, $optionValue);
				if (!defined $value) {
					$batch_error = 1;
					$rc = 0;
					next;
				}
			}
			elsif ($param->{optional_password} && (length($optionValue) == 0)) {
                $param->{set_interactive} = 1;
            }
            else {
			    if (($param->{type} =~ /string/)
			        && (length($optionValue) == 0)
                    && defined $self->getDefault($id)
                    && !$param->{init_with_default}) {
                    # assign default value if string option is set without any value
                    $optionValue = $self->getDefault($id);
                }
				
				$value = $optionValue;
			}
		}
        my $isReadInputForbidden = (exists $param->{set_interactive}
                                       && !$param->{set_interactive}) ? 1 : 0;
		if (defined $value){
            # parameter is specified within the command line

			if ($param->{can_be_switched_off}){
				if (!$value || $value eq '0' || ($value =~ /$bool_false_pattern/i)){
					$self->AddMessage($self->getParamName($id)
						              . ' is disabled via command line option');
					$param->{value} = undef;
					$param->{skip} = 1;
					next;
				}
			}

			if ($param->{type} =~ /map/) {
				foreach my $val (keys %$value){

                    if (!$batchmode) {
                        $param->{batchValue}->{$val} = $value->{$val};
                    }
                    elsif (!$self->setMapValueItem($id, $val, $value->{$val})) {
                        $fatal_error = 1;
                        next;
			        }

			        $self->AddMessage($self->getParamName($id, $val)
			            ." is given via command line option, value = '$value->{$val}'");
    			}
    			if( !$fatal_error && $batchmode && !$self->checkAllMapValues($id) ){
    			    $fatal_error = 1;
    			}
				$rc &&= ($fatal_error ? 0 : 1);
    			$value = undef;
				next;
			}
			else {
				$self->addLogValueMsg($id, $value,
				                  'is given via command line option, value =');
			    if (!$batchmode) {
			    	$param->{batchValue} = $value;
				    next;
				}

				$result = $self->setValue($id,$value);
                if (!defined $result || $param->{no_retry}){
                    $fatal_error = 1;
                }
				$rc &&= $result;
				if (!$result && $isReadInputForbidden){
					$fatal_error = 1;
				}
				next;
			}
		}

		$value = $self->getValueFromConfigfile($id);
        my $valueSourceStr = "is given via configuration file";
        if (!defined $value) {
            $value = $self->getValueFromEnv($id);
            $valueSourceStr = "is given via environment variable";
        }

		if (defined $value){
		    # parameter is contained in the configuration file or environment

			if ($param->{type} =~ /map/){
				my ($old,$new);
				foreach my $pair (@$value) {
					($old,$new) = split('=', $pair);
					if ($old && $new){
                        $self->AddMessage ($self->getParamName($id, $old)
                            ." $valueSourceStr, value = '$new'");
                        if (!$batchmode) {
                            $param->{batchValue}->{$old} = $new;
                        }
                        elsif (!$self->setMapValueItem($id, $old, $new)) {
                            $fatal_error = 1;
                        }
                    }
                    else {
                        $self->AddMessage ($self->getParamName($id)
                            .": value pair missing in '$pair'");
                    }
				}
				$rc &&= ($fatal_error ? 0 : 1);
				next if ($fatal_error);
				$value = undef;
			}
			else {
				if ($param->{type} =~ /bool/) {
					$value = $self->assignBoolValue($id, $value);
					if (!defined $value) {
						$batch_error = 1;
						$rc = 0;
						next;
					}
				}
				elsif ($param->{type} !~ /array/) {
					$value = $self->substituteValues ($value, undef, 1);
				}
				$self->addLogValueMsg($id, $value, "$valueSourceStr, value =");
				if (!$batchmode) {
					$param->{batchValue} = $value;
					next;
				}

                $result = $self->setValue($id,$value);
	            if (!defined $result || $param->{no_retry}){
	                $fatal_error = 1;
	            }
                $rc &&= $result;
                if (!$result && $isReadInputForbidden){
                    $fatal_error = 1;
                }
                next;
            }
		}
		if (defined($param->{persFileValue}) && $batchmode){
			if(!$self->setPersistedValue($id, $param, $param->{persFileValue})){
				$fatal_error = 1;
				$rc &&= ($fatal_error ? 0 : 1);
			}
			next;
		}

		if ($param->{type} =~ /map/) {
			if ($batchmode) {
				if ($param->{init_with_default} ||
				    $param->{init_batch_with_default}) {

					if (!$self->setDefaultMapValues($id)) { # calls checkAllMapValues
						$fatal_error = 1;
						next;
					}
				}
				elsif (!$self->checkAllMapValues($id)) {
					$fatal_error = 1;
					next;
				}
				$rc &&= ($fatal_error ? 0 : 1);
				next if (defined $param->{value});
			}
		}
		else {
			my $configFileDefault = $self->getDefaultValueFromConfigfile($id);
			my $shallInitWithDefault = $param->{init_with_default} || $param->{init_batch_with_default} || defined($configFileDefault);

			if ($batchmode && $shallInitWithDefault) {
				my $default = $self->getDefault ($id);
				$self->addLogValueMsg($id, $default, 'is set with default value =');
				$result = $self->setValue ($id, $default);
				$rc   &&= $result;
				return $rc if(!$rc);
				next;
			}
		}
		if (!$batchmode && $isReadInputForbidden && $param->{init_with_default}){
			my $default = $self->getDefault ($id, undef, 1);
			if (defined $default){
			    # default value is set by SDB::Install::App::Console::ConfigureInteractive
				next;
			}
		}

		if($param->{mandatory}){
			if ($batchmode){
				$self->PushError
				     ($self->getParamName($id, undef, 'Mandatory parameter')
				     . ' is missing or invalid');
				$batch_error = 1;
                $rc = 0;
                last;
			}
		}
		elsif ($batchmode){
			$self->AddMessage ($self->getParamName($id) . ' is not set');
		}
	}
	
	if ($batch_error){
		$rc = 0;
	}
	elsif (!$rc){
		$rc = ($fatal_error xor $batchmode) ? undef : 0;
	}
	$self->{check_state} = $rc;
	return $rc;
}

# 0 == Failed, 1 == Success
sub setPersistedValue {
	my ($self, $id, $param, $persistedValue) = @_;

	if ($param->{type} =~ /map/){
		if (ref($persistedValue) ne 'HASH') {
			my ($key, $value) = split('=', $persistedValue);
			if (defined $key && defined $value) {
				$persistedValue = {};
				$persistedValue->{$key} = $value;
			}
		}
		foreach my $key (keys(%{$persistedValue})) {
			my $value = $persistedValue->{$key};
			$self->AddMessage ($self->getParamName($id, $key) ." was found in a persistence file, value = '$value'");
			if ($param->{shouldUseSetterOnDeserialization}){
				return undef if (!$self->setMapValueItem($id, $key, $value));
			} else {
				$param->{value}->{$key} = $value;
				return undef if (!$self->notifyParameterListeners($id, $value, $key));
			}
		}
		return 1;
	}
	if($param->{type} =~ /bool/){
		$persistedValue = $self->assignBoolValue($id, $persistedValue);
	}

	$self->addLogValueMsg($id, $persistedValue, 'was found in a persistence file, value = ');
	if($param->{shouldUseSetterOnDeserialization}){
		return $self->setValue($id,$persistedValue) ? 1 : 0;
	}
	$param->{value} = $persistedValue;
	return $self->notifyParameterListeners($id, $persistedValue);
}


#-------------------------------------------------------------------------------
# Converts the given value into a boolean value.
#
# Returns undef in case of an error.

sub assignBoolValue {
    my ($self, $id, $value) = @_;

    if (!defined $value || ($value =~ /$bool_false_pattern/i)) {
    	return 0;
    }
    if (($value =~ /$bool_true_pattern/i) || ($value eq '')) {
        return 1;
    }
    $self->PushError
            ($self->getParamName($id) . " cannot decode boolean value '$value'");
    return undef;
}

#-------------------------------------------------------------------------------
# Adds a message containing the specified value

sub addLogValueMsg {
	my ($self,
	    $id,
	    $value,
	    $info,  # e.g. 'is given via command line option, value ='
	    $msglst # may be omitted
	   ) = @_;

    $msglst = $self->getMsgLst() if (!defined $msglst);

    my $prefix = $self->getParamName($id) . " $info '";
    if (defined $value && ($self->{params}->{$id}->{type} =~ /array/)) {
        foreach my $currVal (@$value) {
            $msglst->addMessage($prefix . $currVal . "'");
        }
    }
    else {
        $msglst->addMessage($prefix . $self->getLogValue($id, $value) . "'");
    }
}

#-------------------------------------------------------------------------------
# Reads the password of the spcified parameter from the xml file.
#
# Parameter: string $id
#            hash   $param
#
# Returns the password (string) or undef if not found.

sub getPasswordFromXML {

    my ($self, $id, $param , $notVerbose) = @_;

    my $password = undef;

    if (!defined $self->{stdin_passwd_parser}){
        if(!$notVerbose){
            print "Start reading from input channel...\n";
        }
        require SDB::Install::Configuration::XMLPasswdParser;
        $self->{stdin_passwd_parser} = new SDB::Install::Configuration::XMLPasswdParser(\*STDIN);

        if (!defined $self->{stdin_passwd_parser}->parse()){
            $self->PushError ("Cannot read password", $self->{stdin_passwd_parser});
            return undef;
        }
        if(!$notVerbose){
            print "... Done.\n";
        }
    }

    my $alias      = $param->{alias_opts};
    my $xmlTagList = [$param->{opt},
                      $param->{short_opt},
                      (defined $alias) ? @{$alias} : undef,
                      $id,
                     ];

    foreach my $tag (@$xmlTagList) {

        if (defined $tag) {
            $password = $self->{stdin_passwd_parser}->getPassword($tag, $param->{opt});
            if (defined $password) {
                last;
            }
        }
    }

    if (!defined $password && $param->{mandatory}) {
        $self->PushError (undef ,$self->{stdin_passwd_parser});
    }

    return $password;
}

sub getPhase{
    return undef;
}

sub setValue{
    my ($self,$id,$value, $msglst) = @_;
    my $param = $self->{params}->{$id};

    if ($param->{type} =~ /bool/) {
        $value = $self->assignBoolValue($id, $value);
        if (!defined $value) {
            return 0;
        }
    }

    # trim leading whitespace in user input of path parameters
    if ($param->{type} eq 'path') {
        $value =~ s/^\s+//;
    }

    if (exists $param->{visible_alias_values}) {

        my $arrayValues = ($param->{type} =~ /array/) ? $value : [$value];
        my $arrayValIdx = 0;

        foreach my $paramVal (@$arrayValues) {
            my $i = 0;
            foreach my $currAlias (@{$param->{visible_alias_values}}) {
                if ($currAlias eq $paramVal) {
                    if ($param->{type} =~ /array/) {
                        $value->[$arrayValIdx] = $param->{valid_values}->[$i];
                    }
                    else {
                        $value = $param->{valid_values}->[$i];
                    }
                    last;
                }
                $i++;
            }
            $arrayValIdx++;
        }
    }

    # calls e.g. setSID, setUID, etc. if subroutine exists
    my $sub = $self->can ("set$id"); 
    if ($sub){
        my $result = &$sub ($self, $value);
        if($result){
            my $setValue = $self->getValue ($id);
            $result = $self->notifyParameterListeners($id, $setValue);
            if (!$result && defined $setValue){
                $param->{value} = undef;
            }
        }
        return $result;
    }

    my $rc = $self->checkValue($id,$value,$msglst);
    if (!$rc) {
        return $rc;
    }

    $rc = $self->notifyParameterListeners($id, $value);
    if (!$rc) {
        return $rc;
    }
    $param->{value} = $value;
    return 1;
}

sub checkValue{
    my ($self,$id,$value,$msglst_arg) = @_;
    my $param = $self->{params}->{$id};
    my $msglst = defined $msglst_arg ? $msglst_arg : $self;

    if (defined $param->{valid_values} && defined $value) {
		my $publicValidValues;
		if (defined $param->{hidden_valid_values}){
			for my $validValue (@{$param->{valid_values}}){
				if (not ($validValue ~~ @{$param->{hidden_valid_values}})){
					push @$publicValidValues, $validValue;
				}
			}
		}

        if ($param->{type} eq 'csv') { # comma-separated values
        	my @csvTokens = split(/\s*,\s*/,$value);
        	my $found = 0;
        	for my $csvToken (@csvTokens) {
                if (not ($csvToken ~~ @{$param->{valid_values}})) {
                    my $msg = "Value \"$csvToken\" is invalid.";
                    my $paramValidValues = $param->{valid_values};
                    my $appendMsg;

                    if ( (defined $publicValidValues) && (scalar (@$publicValidValues) > 0) ) {
                        $appendMsg = " Please use one of: " . join(', ', @$publicValidValues);
                    } elsif ( (defined $paramValidValues) && (scalar (@$paramValidValues) > 0) ) {
                        $appendMsg = " Please use one of: " . join(', ', @$paramValidValues);
                    }

                    if (defined $appendMsg) {
                        $msg .= $appendMsg;
                    }

                    $msglst->AddError ($msg);

                    return 0;
        		}
        	}

        } else {
	        my $found = 0;

            my $arrayValues = ($param->{type} =~ /array/) ? $value : [$value];

            foreach my $paramVal (@$arrayValues) {

                foreach my $valid_value(@{$param->{valid_values}}) {
                    if ($valid_value eq $paramVal) {
	                   $found = 1;
	                   last;
                    }
                }

                if (!$found){

                    my $refValidValues = (defined $publicValidValues)
                                         ? $publicValidValues
                                         : (exists $param->{visible_alias_values})
                                             ? $param->{visible_alias_values}
                                             : $param->{valid_values};

                    my $msg = "Value '$paramVal' is invalid.";
                    my $appendMsg;

                    if ( (defined $refValidValues) && (scalar (@$refValidValues) > 0) ) {
                        $appendMsg = " Please use one of: " . join(', ', @$refValidValues);
                    }

                    if (defined $appendMsg) {
                        $msg .= $appendMsg;
                    }

                    $msglst->AddError ($msg);

                    return 0;
                }
	        }
        }
    }
    else{

        if ($param->{type} eq 'number'){
            if (defined $value && ($value =~ /\D/)) {
                $msglst->AddError ("Value \"$value\" is no positive decimal number");
                return 0;
            }
        }
        elsif ($param->{type} =~ /bool/){
            if ($value !~  /$bool_false_pattern|$bool_true_pattern/i){
                $msglst->AddError ("Cannot decode boolean value '$value'");
                return 0;
            }
        }
        elsif ($param->{type} eq 'existing_path'){
            if (-e $value){
                $msglst->AddError ("Path \"$value\" doesn't exist");
                return 0;
            }
        }
        elsif ($param->{type} eq 'path'){
            if ($^O =~ /mswin/i){
                my $test_value = $value;
                $test_value =~ s/^[a-z]://i;
                if ($test_value =~ /["\*:\<\>\?\|]/){
                    $msglst->AddError ("Value \"$value\" contains invalid characters");
                    return 0;
                }
            }
        }
    }

    # calls e.g. checkSID, checkUID, etc. if subroutine exists
    my $sub = $self->can ("check$id");

    if ($sub){
        return &$sub ($self, $value, $msglst_arg);
    }
    return 1;
}


#
# import config file 
# from another configuration to avoid 
# reading it more than once
#


sub ImportConfigFile{
	my ($self,$cfg) = @_;
	$self->{config_file} = $cfg->{config_file};
	$self->{cfgfile_data} = $cfg->{cfgfile_data};
}



#
# generate a opt_ctrl hash as input 
# for Getopt::Long::GetOptions () method
#


sub GetOptionCtrl{
	my ($self) = @_;
	my %opt_ctrl;
	my $options = defined $self->{options} ? $self->{options} : {};
	$self->{options} = $options;
    my $optName;
    foreach my $optWithArgs ($self->getOptionIgnore, $self->getOptionTimeout){
        if (!defined $optWithArgs) {
            next;
        }
        $optName = $optWithArgs->getOptionName ();
        $options->{$optName} = undef;
        $opt_ctrl{$optWithArgs->getGetOptionSpec()} = \$options->{$optName};
    }

	# Builds option specification argument for Getopt::Long::GetOptions
    foreach my $id (keys (%{$self->{params}})) {
        my $param = $self->{params}->{$id};

        # read primary option name 
        my $opt = $param->{opt};       
        if (!defined $opt){
            next;
        }
        
        # add short option and other aliases if needed
        if (defined $param->{short_opt}) {
            $opt .= '|';
            $opt .= $param->{short_opt};
        }
        
        if (defined $param->{alias_opts}) {
            foreach my $alias (@{$param->{alias_opts}}) {
                $opt .= '|';
                $opt .= $alias;
            }
        }
        
        # decides if option is mandatory and the type of its argument
        if ($param->{type} =~ /string/) {
            $opt .= (!$param->{init_with_default}
                     && defined $self->getDefault($id)) ? ':s'  # optional
                                                   : '=s'; # mandatory
        }
        elsif ($param->{type} =~ /passwd/) {
            $opt .= ($param->{optional_password})  ? ':s'  # optional
                                                   : '=s'; # mandatory
        }
        elsif ($param->{type} =~ /path/) {
            $opt .= '=s';
        }
        elsif ($param->{type} =~ /number/) {
            $opt .= '=i';
        }
        elsif ($param->{type} =~ /bool/) {
            $opt .= ':s';
        }
        elsif ($param->{type} =~ /map/) {
            $opt .= '=s%';
        }
        elsif ($param->{type} =~ /array/) {
            $opt .= '=s@';
        }
        elsif ($param->{type} =~ /csv/) { # comma-separated values
            $opt .= '=s';
        }

        #  Option format: 
        #  opt[|short_opt][|aliases...] ( = | : ) (s | i) [ @ | % ] 
        $opt_ctrl{$opt} = \$options->{$param->{opt}};        
    }
	return \%opt_ctrl;
}


#-------------------------------------------------------------------------------
# Checks the specified parameter (except map type) and returns a reference
# to a hash containing the keys:
#    'isBool'     - matches 'bool'    (returns true if parameter is specified or has the value 'on')
#    'isMap'      - matches 'map'     ($self->getValue('<id>') returns a hash map)
#    'isPassword' - matches 'passwd'  (output of the value is suppressed)
#    'isPath'     - matches 'path'    (part if the path may be changed according to sapmnt)
#    'arg'        - no match          (normal parameter with value)
#
# Parameters: string  $paramID   # used to access '$self->{params}->{paramID}'
#             boolean $isSwitch

sub GetArgumentsFromParam {

    my ($self, $paramID, $isSwitch) = @_;

    my %result;
    my $param = $self->{params}->{$paramID};

    if (defined $param->{type}) {

        if ($param->{type} =~ /bool/) {
            $result{isBool} = 1;
        }
        elsif ($param->{type} =~ /map/) {
            $result{isMap} = 1;
        }
        elsif ($param->{type} =~ /passwd/) {
            $result{isPassword} = 1;
        }
        elsif ($param->{type} =~ /path/) {
            $result{isPath} = 1;
        }
        elsif ($param->{type} =~ /array/) {
            $result{isArray} = 1;
        }
    }

    if (!$result{isPassword} && (!$result{isMap} || !$isSwitch)) {

        if ($isSwitch && defined $param->{opt_arg_switch}) {

            $result{arg} = $param->{opt_arg_switch};
        }
        elsif (defined $param->{opt_arg}) {

            $result{arg} = $param->{opt_arg};
        }
        elsif ($result{isBool}) {

            $result{arg} = 'off';
        }
        elsif ($param->{valid_values}
                              && !defined $param->{f_get_console_description}) {
            my @outputValues;
            my $validValues = (exists $param->{visible_alias_values})
                              ? $param->{visible_alias_values}
                              : $param->{valid_values};

            foreach my $val (sort @$validValues) {
                if (!exists $param->{skip_help_valid_values}->{$val}) {
                    push @outputValues, $val;
                }
            }
            $result{arg} = join ('|', @outputValues);
        }

        if (!defined $result{arg}) {

            # returns converted paramID as argument
            # e.g. converts 'BasePathLogVolumes' to '<base_path_log_volumes>'

            my $argument =  $paramID;
            $argument    =~ s/([a-z])([A-Z][a-z])/$1_$2/g;
            $result{arg} =  '<' . lc ($argument). '>';
        }
    }

    return \%result;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays containg the description
# of password tags that are created from the hash 'params'.

sub GetPasswordUsageFromParams {
    my ($self) = @_;

    my @result;
    my $passwordFound = 0;

    foreach my $id (keys (%{$self->{params}})) { # unsorted

        my $param = $self->{params}->{$id};
        my $tag   = $param->{opt};

        if (!defined $tag || (defined $param->{hidden} && $param->{hidden})) {
            next;
        }

        my $argType = $self->GetArgumentsFromParam($id);

        if (!$argType->{isPassword}) {
            next;
        }

        my $description = $self->getParamDesc($param);
        my $constraint  = $param->{constraint};
        $description   .= ' [' . lc($constraint) . ']' if (defined $constraint);

        push @result, [$tag, $description];
    }

    return \@result;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of arrays containg the description
# of program options that are created from the hash 'params'.

sub GetUsageFromParams {
	my ($self) = @_;

	my @result;
	my $passwordFound = 0;

    foreach my $id (keys (%{$self->{params}})) { # unsorted

		my $param = $self->{params}->{$id};
		my $opt   = $param->{opt};

		if (!defined $opt || (defined $param->{hidden} && $param->{hidden})) {
			next;
		}

        my $argType = $self->GetArgumentsFromParam($id);

        if ($argType->{isPassword}) {
            $passwordFound = 1;
            if (!$param->{optional_password}) {
                next;
            }
        }

        my $shortOpt      = $param->{short_opt};
        my $description   = $self->getParamDesc($param);
        my $isOptionalVal = 0;
        my $interactive   = ($param->{set_interactive} ||
                             $param->{may_be_interactive}) ? ' [interactive'
                                                           : undef;

        my $defaultPrefix = (defined $interactive)
                            ? "$interactive, default"
                            : ' [default';

        if ($argType->{isBool}) {

            if ($self->getDefault($id)) {
                $description .= $defaultPrefix . ': ' . $PREFIX_OPT . $opt .']';
            }
            elsif (defined $interactive) {
                 $description .= $interactive . ']';
            }
        }
        elsif (defined $self->getDefault($id) && !$argType->{isMap}) {

            my $defaultVal = $self->getDefault($id);

            if ($argType->{isPath}) {

                # converts e.g. '/usr/sap/$SID/home' to '/usr/sap/<SID>/home'

                my $doSubstitute = 1;

                while ($doSubstitute) {

                    my ($varName) = ($defaultVal =~ /\$([a-zA-Z0-9]+)/);

                    if (defined $varName) {

                        # replaces '$Target' by '<installation_path>'
                        my $outVarName = ($varName eq 'Target')
                                         ? 'installation_path'
                                         : $varName;

                        $defaultVal =~ s/\$$varName/\<$outVarName\>/;
                    }
                    else {
                        $doSubstitute = 0;
                    }
                }
            }
            elsif (!$param->{init_with_default} &&
                   !$param->{help_with_default}) {
                $isOptionalVal = 1;
            }

            my $defaultMsg;
            if ($isOptionalVal) {
                $defaultMsg = (ref($defaultVal) =~ /ARRAY/)
                              ? " value: '" . join("', '", @$defaultVal) . "'"
                              : " value: '$defaultVal'";
            }
            else {
                my $optStr  = $PREFIX_OPT . $opt . '=';
                $defaultMsg = (ref($defaultVal) =~ /ARRAY/)
                              ? ': ' . $optStr . join(" $optStr", @$defaultVal)
                              : ': ' . $optStr . $defaultVal;
            }
            $description .= $defaultPrefix . $defaultMsg . ']';
        }
        elsif (defined $interactive) {
            $description .= $interactive . ']';
        }

        if (defined $param->{additional_desc}) {
            $description .= "\n" . $param->{additional_desc};
        }

        my ($normSyntax, $shortSyntax);

        if ($argType->{isBool}) {

            $normSyntax  = $PREFIX_OPT . $opt;
            $shortSyntax = (defined $shortOpt) ? $PREFIX_SHORT_OPT.$shortOpt
                                               : undef;

            if ($self->getDefault($id)) {
                $normSyntax  .= '[=off]';
                $shortSyntax .= ' [off]' if (defined $shortOpt);
            }
        }
        elsif ($param->{optional_password}) {
            $normSyntax  = $PREFIX_OPT . $opt;
            $shortSyntax = undef;
            $description = "Enable interactive input of $param->{str}";
        }
        elsif ($isOptionalVal) {

            $normSyntax  = $PREFIX_OPT . $opt . '[=' . $argType->{arg} . ']';

            $shortSyntax = (defined $shortOpt)
                ? $PREFIX_SHORT_OPT . $shortOpt . ' [' . $argType->{arg} . ']'
                : undef;
        }
        else {

            $normSyntax  = $PREFIX_OPT . $opt . '=' . $argType->{arg};

            $shortSyntax = (defined $shortOpt)
                           ? $PREFIX_SHORT_OPT . $shortOpt .' '. $argType->{arg}
                           : undef;
        }

        if (defined $param->{constraint}) {
            push @result, [$normSyntax, $shortSyntax, $description, $param->{constraint}];
        }
        else {
            push @result, [$normSyntax, $shortSyntax, $description];
        }
	}

	if ($passwordFound) {
        push @result, [$READ_PW_SYNTAX, undef, $READ_PW_HELP];
	}

    foreach my $optWithArgs ($self->getOptionIgnore, $self->getOptionTimeout){
        if (defined $optWithArgs && !$optWithArgs->isHidden ()) {
            push @result, [ $optWithArgs->getSyntax (), undef,
                 $optWithArgs->getDescription ()];
        }
    }

	return \@result;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing program options
# without help/info options. The options are created from the hash 'params'.
#
# Returns an empty array if parameters are not defined.
#
# The following hash keys of '$self->{params}' are used by this function:
#
#    Hash Key                  | Description
#    --------------------------+------------------------------------------
#    f_get_console_description | ??
#    hidden                    | do not show the parameter
#    opt                       | long  parameter name (e.g. '--keep_user')
#    short_opt                 | short parameter name (e.g. '-k')
#    type                      | boolean, hash map, path, string, etc.
#    opt_arg_switch            | parameter argument (e.g. '<host%s>=<host%s`>')
#    opt_arg                   | argument (e.g. '<path>') if opt_arg_switch is undefined
#    valid_values              | array containing the valid values

sub GetSwitchesFromParams {

    my ($self) = @_;

    my $passwordFound = 0;
    my @switches;

    foreach my $id (keys %{$self->{params}}) {

        my $param = $self->{params}->{$id};

        if (!defined $param->{opt}
            || (defined $param->{hidden} && $param->{hidden})) {

            next;
        }

        my $argType = $self->GetArgumentsFromParam($id, 1);

        if ($argType->{isPassword}) {
            $passwordFound = 1;
            if ($param->{optional_password}){
                push(@switches, $PREFIX_OPT . $param->{opt});
            }
            next;
        }

        my $currSwitch = $PREFIX_OPT . $param->{opt};

        if ($argType->{isBool}) {

            $currSwitch .= '=off' if ($self->getDefault($id));
        }
        else {

            my $isOptionalVal = (!$param->{init_with_default} &&
                                 !$param->{help_with_default}
                                 && defined $self->getDefault($id)
                                 && !$argType->{isMap}
                                 && !$argType->{isPath});

            $currSwitch .= ($isOptionalVal) ? '[=' : '=';

            if ($argType->{isMap}) {

                # e.g. output: " [-H <old1>=<new1>[ -H <old2>=<new2>]...]"

                my $arg0    = sprintf ($param->{opt_arg_switch}, 1, 1);
                my $arg1    = sprintf ($param->{opt_arg_switch}, 2, 2);
                $currSwitch .= $arg0 . '[ ' . $currSwitch.$arg1 . ']...';
            }
            elsif ($argType->{isArray}) {

                # e.g. output: " [--addhost=<host1>[ --addhost=<host2>]...]"

                my $argTemplate = $param->{opt_arg_switch};
                if (defined $argTemplate) {
                    my $arg0     = sprintf ($param->{opt_arg_switch}, 1);
                    my $arg1     = sprintf ($param->{opt_arg_switch}, 2);
                    $currSwitch .= $arg0 . '[ ' . $currSwitch . $arg1 . ']...';
                }
                else {
                    $currSwitch .= $argType->{arg}
                                . '[ ' . $currSwitch . $argType->{arg} . ']...';
                }
            }
            else {
                $currSwitch .= $argType->{arg};
                $currSwitch .= ']' if ($isOptionalVal);
            }
        }
        push(@switches, $currSwitch);
    }

    foreach my $optWithArgs ($self->getOptionIgnore, $self->getOptionTimeout){
        if (defined $optWithArgs && !$optWithArgs->isHidden ()) {
            push (@switches, $optWithArgs->getSyntax ());
        }
    }

    if ($passwordFound) {
        push(@switches, $READ_PW_SYNTAX);
    }

	return \@switches;
}

sub isHidden{
	if (!defined $_[0]->{params}->{$_[1]}){
		return undef;
	}
	return $_[0]->{params}->{$_[1]}->{hidden} ? 1 : 0;
}

sub isMandatory{
	if (!defined $_[0]->{params}->{$_[1]}){
		return undef;
	}
	return $_[0]->{params}->{$_[1]}->{mandatory} ? 1 : 0;
}

sub isSkipped{
	if (!defined $_[0]->{params}->{$_[1]}){
		return undef;
	}
	return $_[0]->{params}->{$_[1]}->{skip} ? 1 : 0;
}

#-------------------------------------------------------------------------------
# If the specified parameter exists, skip is set (default: 1)

sub setSkip {

    my ($self, $paramID, $skip, $warning) = @_;

    if (exists $self->{params}->{$paramID} ) {
    	my $oldValue = $self->{params}->{$paramID}->{skip};
        my $newValue = (defined $skip) ? $skip : 1;

        if (!defined $oldValue || ($oldValue != $newValue)) {
            $self->{params}->{$paramID}->{skip} = $newValue;
            $self->{params}->{$paramID}->{skip_warning} = $warning if $newValue && defined $warning;
            if (defined $self->{_changeEventHandlers}) {
                my $cfgChangeEvent = SDB::Install::ConfigurationChangeEvent->new
                                    ($self,$paramID,'skip',$oldValue,$newValue);
                $self->notifyConfigurationChangeEventHandlers($cfgChangeEvent)
            }
        }
    }
}

sub setUseDefault {
    my ($self, $paramId, $value) = @_;
    return if (!exists $self->{params}->{$paramId});

    my $newValue = $value || 1;
    $self->{params}->{$paramId}->{use_default} = $newValue;
}

sub getUseDefault {
    my ($self, $paramId) = @_;
    return undef if (!exists $self->{params}->{$paramId});
    return $self->{params}->{$paramId}->{use_default};
}

sub setType {
    my ($self, $paramID, $type) = @_;

    if (exists $self->{params}->{$paramID}) {
        $self->{params}->{$paramID}->{type} = $type;
    }
}

sub getType {
	my ($self, $paramID) = @_;

	if (exists $self->{params}->{$paramID}) {
		return $self->{params}->{$paramID}->{type};
	}

	return undef;
}

sub setNoRetry {
    my ($self, $paramID, $flag) = @_;

    if (exists $self->{params}->{$paramID}) {
        $self->{params}->{$paramID}->{no_retry} = $flag;
    }
}

#-------------------------------------------------------------------------------
# If the specified parameter exists, mandatory is set (default: 1)

sub setMandatory {
    my ($self, $paramID, $skip) = @_;

    if (exists $self->{params}->{$paramID}) {
        $self->{params}->{$paramID}->{mandatory} = defined($skip) ? $skip : 1;
    }
}

sub setHidden {
    my ($self,$param_id, $isHidden) = @_;
    if (exists $self->{params}->{$param_id}){
        $self->{params}->{$param_id}->{hidden} = defined $isHidden ? $isHidden : 1;
    }
}

sub setInteractive {
    my ($self,$param_id, $isInteractive) = @_;
    if (exists $self->{params}->{$param_id}){
        $self->{params}->{$param_id}->{set_interactive} = defined $isInteractive ? $isInteractive : 1;
    }
}

sub hasValue ($){
    my ($self,$param_id) = @_;
   
    if (!exists $self->{params}->{$param_id}){
        return undef;
    }
    
    defined $self->{params}->{$param_id}->{value} ? 1 : 0;
}

sub getValue{
    my ($self,$param_id) = @_;
   
    if (!exists $self->{params}->{$param_id}){
        return undef;
    }
    return $self->{params}->{$param_id}->{value};
}

sub getOpt {
    my ($self, $param_id) = @_;

    if (!exists $self->{params}->{$param_id}){
        return undef;
    }
    my $opt = $self->{params}->{$param_id}->{opt};
    return (defined $opt) ? '--' . $opt : undef;
}

sub getCommandLineValue {
    my ($self, $paramId) = @_;
    my $opt = $self->getOpt($paramId);
    return undef if (!defined $opt);
    $opt = substr($opt, 2);
    return $self->{options}->{$opt};
}

sub getString {
    my ($self, $param_id) = @_;

    if (!exists $self->{params}->{$param_id}){
        return undef;
    }
    return $self->{params}->{$param_id}->{str};
}

sub getHostctrlOpt {
    my ($self, $param_id) = @_;

    if (!exists $self->{params}->{$param_id}){
        return undef;
    }
    return $self->{params}->{$param_id}->{hostctrl_opt};
}

sub getValueOrBatchValue {
    my ($self,$param_id) = @_;

    if (!exists $self->{params}->{$param_id}){
        return undef;
    }
    if (exists $self->{params}->{$param_id}->{batchValue}){
        return $self->{params}->{$param_id}->{batchValue};
    }
    return $self->{params}->{$param_id}->{value};
}

sub getValidValues {
    my ( $self,$param_id ) = @_;

    if ( ! exists $self->{params}->{$param_id} ) {
    	return undef;
    }
    
    if ( ! exists $self->{params}->{$param_id}->{valid_values} ) {
        return undef;
    }
    
    return $self->{params}->{$param_id}->{valid_values};	
}

sub isValuePreset{
    my ($self,$param_id) = @_;
    if (!exists $self->{params}->{$param_id}){
        return undef;
    }
    return defined $self->{params}->{$param_id}->{value} ||
        defined $self->{params}->{$param_id}->{batchValue};
}

sub getBatchValue{
    my ($self,$param_id) = @_;
    my $param = $self->{params}->{$param_id};

    if (!defined $param){
        return undef;
    }

    my $value = $param->{batchValue};

    if (!defined $value) {
        $value = $param->{value};
    }

    if (!defined $value && defined $param->{opt}) {
        $value = $self->{options}->{$param->{opt}};
    }

    if (!defined $value) {
        $value = $self->getValueFromConfigfile($param_id);
    }

    if (!defined $value) {
        $value = $self->getValueFromEnv($param_id);
    }

    if (!defined $value) {
        $value = $param->{persFileValue};
    }

    return $value;
}


sub getValueFromEnv {
    my ($self, $paramId) = @_;
    my $paramNameForShellEnv = $self->getParamNameForShellEnv($paramId);
    return $ENV{$paramNameForShellEnv};
}

sub getParamNameForShellEnv {
    my ($self, $paramId) = @_;
    my $ucParamOpt = uc($self->getOpt($paramId));
    $ucParamOpt = substr($ucParamOpt, 2);
    return "HDB_INSTALLER_$ucParamOpt";
}

sub getValueFromConfigfile {
	my ($self, $paramId, $returnActualValue) = @_;

	my $configFileData = $self->{cfgfile_data};
	my $param = $self->{params}->{$paramId};
	my $alternativeSections = defined($param->{alternative_sections}) ? $param->{alternative_sections} : [];

	return undef if (!defined($configFileData));

	for my $relevantSection ($param->{section}, @{$alternativeSections}){
		next if (! exists($configFileData->{$relevantSection}));

		my $dataSection = $configFileData->{$relevantSection};
		my $firstArrValue = undef;
		my $value = $self->_getValueFromDataSection($paramId, $param, $dataSection);
		$firstArrValue = $value->[0] if (ref($value) eq 'ARRAY');

		if ($value eq $gUseDefaultValue || $firstArrValue eq $gUseDefaultValue) {
			return $gUseDefaultValue if($returnActualValue);

			$self->setUseDefault($paramId);
			$value = undef;
		}
		return $value if(defined($value));
	}
	return undef;
}

sub getDefaultValueFromConfigfile {
	my ($self, $paramId) = @_;
	my $param = $self->{params}->{$paramId};
	my $configFileData = $self->{cfgfile_data} || {};
	my $alternativeSections = $param->{alternative_sections} || [];


	for my $relevantSection ($param->{section}, @{$alternativeSections}){
		next if (! exists($configFileData->{$relevantSection}));

		my $dataSection = $configFileData->{$relevantSection};
		my $value = $self->_getDefaultValueFromDataSection($paramId, $param, $dataSection);

		return $value if(defined($value));
	}
	return undef;
}

sub _getDefaultValueFromDataSection {
	my ($self, $paramId, $param, $configfileDataSection) = @_;
	my @aliasOptions = exists($param->{alias_opts}) ? @{$param->{alias_opts}} : ();
	my @options = ((defined($param->{opt}) ? ($param->{opt}) : ()), (defined($param->{short_opt}) ? ($param->{short_opt}) : ()));
	my @searchIds = map { (sprintf('%s::DEFAULT', $_), sprintf('%s::default', $_)) } ($paramId, @options, @aliasOptions);

	for my $id (@searchIds){
		next if(!defined($configfileDataSection->{$id}));
		my $value = $configfileDataSection->{$id};
		return ($value->[0] ne '') ? $value->[0] : undef;
	}
	return undef;
}

sub _getValueFromDataSection {
	my ($self, $paramId, $param, $configfileDataSection) = @_;
	my $value = $configfileDataSection->{$paramId};

	if (!defined $value){
		$value = $configfileDataSection->{$param->{opt}}
	}

	if (!defined $value && exists $param->{short_opt}) {
		$value = $configfileDataSection->{$param->{short_opt}};
	}

	if (!defined $value && exists $param->{alias_opts}) {
		foreach my $aliasOpt (@{$param->{alias_opts}}) {
			$value = $configfileDataSection->{$aliasOpt};
			last if (defined $value);
		}
	}

	if (defined $value  && ($param->{type} !~ /array|map/) ){
		$value = $value->[0] ne '' ? $value->[0] : undef;
	}
	return $value;
}

sub getLogValue{
	my ($self,$param_id, $value) = @_;
	if (!exists $self->{params}->{$param_id}){
		return undef;
	}
	my $param = $self->{params}->{$param_id};

    if ($param->{type} eq 'passwd' || $param->{type} eq 'initial_passwd'){
        return '***';
    }

    my $logValue = $value;

	if (!defined $logValue) {
		if($param->{type} =~ /map/ and defined $param->{value}){
			my $mapParamValue = $param->{value};
			my $mapKeysCount = keys (%$mapParamValue);
			my $i=0;
			$logValue="";
    		while(my ($mapKey, $mapValue) = each (%$mapParamValue)){
                $logValue .= (($mapKey eq $mapValue) && !$param->{'summary_full_print'})
                          ? $mapKey
                          : (defined $mapValue) ? "$mapKey -> $mapValue"
                                                : "$mapKey -> <not defined>";
    			$i++;
				if($i < $mapKeysCount){
					$logValue .= ", ";
				}
    		}
		}
		else{
			$logValue = $param->{value};
		}   	
	}

	if (defined $param->{log_value}){
		return &{$param->{log_value}} ($logValue);
	}

	if (!defined $logValue){
		return '<not defined>';
	}
	
	if ($param->{type} =~ /bool/){
		return $logValue ? 'Yes' : 'No';
	}

	if (($param->{type} =~ /array/) && (ref($logValue) =~ /ARRAY/)) {

		if (exists $param->{visible_alias_values}) {
			foreach my $i (0..scalar(@$logValue)-1) {
				my $k = 0;
				foreach my $currValid (@{$param->{valid_values}}) {
					if ($logValue->[$i] eq $currValid) {
						$logValue->[$i] = $param->{visible_alias_values}->[$k];
						last;
					}
					$k++;
				}
			}
		}
		return join(',', @$logValue);
	}
	elsif (exists $param->{visible_alias_values}) {
		my $k = 0;
		foreach my $currValid (@{$param->{valid_values}}) {
			if ($logValue eq $currValid) {
				$logValue = $param->{visible_alias_values}->[$k];
				last;
			}
			$k++;
		}
	}

	return $logValue;
}


#-------------------------------------------------------------------------------
# Returns a string containing a summary item for the specified parameter ID.
# If the parameter should be skipped, undef is returned.
#
# Returned string: '<parameterName>: <value>'

sub getSummaryItem {

	my ($self, $param_id, $isUIStringWanted, $getResultArray) = @_;

    my $param = $self->{params}->{$param_id};
    
    if (!defined $param
        || $param->{skip}
        || !defined $param->{value}
        || ($param->{type} eq 'passwd')
        || ($param->{type} eq 'initial_passwd')
        || (defined $param->{hidden} && $param->{hidden})
        || (($param->{type} =~ /bool/)   &&
             $param->{init_with_default} &&
             ($param->{value} eq $self->getDefault($param_id))) ) {

        return undef;
    }

    my $paramName = (defined $param->{str}) ? $param->{str} : $param->{desc};

    if ($getResultArray && ($param->{type} =~ /array/)) {
        my @result;
        foreach my $currVal (@{$param->{value}}) {
            push @result, "$paramName: $currVal";
        }
        return \@result;
    }

    my $value = $self->getLogValue($param_id);
    if($param->{pluralStr}) {
        my $pluralName = $self->_getParamPluralName($param);
        $paramName = $pluralName if $pluralName;
    }
    if ($isUIStringWanted) {
        $value = $self->getUIStringFromValue ($param_id, $value);
    }
    elsif (defined $param->{summaryTitle}) {
        $value = $param->{summaryTitle};
    }

    $value = $param->{value} if (!defined $value);
    my $result = "$paramName: $value";
    return ($getResultArray) ? [$result] : $result;
}

sub _getParamPluralName {
    my ($self, $param) = @_;
    return undef if ($param->{type} ne 'mapping_list');
    my @values = keys %{$param->{value}};
    my $valuesCount = scalar(@values);
    return ($valuesCount > 1) ? $param->{pluralStr} : undef;
}

sub pers_keys{
	return ['_kitversion', 'step', 'installed_units'];
}

sub setStep{
    my ($self, $step, $isFirstStep) = @_;
    my $hash = {};

    $self->{step} = $step;

    for my $key (@{$self->pers_keys()}){
        if (defined($self->{$key})) {
            $hash->{$key} = $self->{$key};
        }
    }

    for my $id (@{$self->getParamIds()}){
        next if($self->isSkipped($id));
        next if(!$self->hasValue($id));
        next if(!$self->getType($id) =~ /passwd/);
        next if(!defined($self->getPersStep($id)));

        my $value = $self->getValue($id);

        if ($self->getType($id) !~ /map/) {
            $hash->{$id} = $self->getValue($id);
            next;
        }
        for my $key (keys(%{$value})) {
            $hash->{$id.'='.$key} = $value->{$key};
        }
    }

    return $self->pers_store(undef, $hash);
}

sub getNextPhaseFromCurrent {
	my ( $self, $currentPhase ) = @_;
	return undef if ( ! defined $currentPhase || $currentPhase eq '' );

	my $nextPhase;
	if ( $currentPhase eq 'prepare' ) {
		$nextPhase = 'offline';
	} elsif ( $currentPhase eq 'offline' ) {
		$nextPhase = 'online';
	}
	return $nextPhase;
}

sub addInstalledDUToPersistenceFile {
	my ( $self, $installedDU ) = @_;
	my %hash = ('installed_units' => $installedDU);
	$self->{'installed_units'} = $installedDU;
	return $self->addNewPersistenceInfo(\%hash);
}

sub addPhaseToPersistenceFile {
	my ( $self, $phase ) = @_;
	my %hash = ('next_phase' => $phase);
	$self->{'next_phase'} = $phase;
	return $self->addNewPersistenceInfo(\%hash);
}

sub addNewPersistenceInfo {
	my ($self, $hash) = @_;
	my $persistenceData = $self->pers_load();

	for my $key ( keys %$hash ) {
		next if ( exists $persistenceData->{$key} && $persistenceData->{$key} eq $hash->{$key} );
		$persistenceData->{$key} = $hash->{$key};
	}
	return $self->pers_store(undef, $persistenceData);
}

sub getStep{
	defined $_[0]->{step} ? $_[0]->{step} : 0;
}

#
# get list of sorted paramerter ids
#

sub getParamIds{
    my ($self) = @_;
    if (!defined $self->{paramIds}){
        my $params = $self->{params};
        $self->{paramIds} =
            [sort {$params->{$a}->{order} <=> $params->{$b}->{order}} keys (%$params)];
    }
    return $self->{paramIds};
}

sub getParamOriginValues {
    my ($self, $paramId) = @_;

	return defined $self->{params}->{$paramId} ?
			$self->{params}->{$paramId}->{origin_values} :
			undef;
}

sub clearCachedParamIds{
    my ($self) = @_;
    undef($self->{paramIds});
}

#
# required for gui installer
# initialize all parameters until $paramId with batchValue
# or default
#

sub setBatchValueOfRequiredParams{
    my ($self, $paramId) = @_;

    my $param;
    my $defaultValue;

    if (!defined $self->{params}->{$paramId}){
        return undef;
    }

    my $params = $self->{params};
    my $last = 0;

    foreach my $currID (@{$self->getParamIds ()}){
        my $id = $currID;
        if ($last){
            last;
        }
        if ($id eq $paramId){
            # all parameter until $parmId were intialized
            $last = 1;
        }

        my $retryID = $self->{params}->{$id}->{retry_param};
        if (defined $retryID) {
            if (defined $params->{$retryID}->{batchValue} ||
                defined $params->{$retryID}->{value}) {
                next;
            }
            $id = $retryID;
        }

        $param = $self->{params}->{$id};
        if ($param->{skip}){
            # skipped
            next;
        }

        if (defined $param->{value}){
            #already set
            next;
        }

        my $valueAlreadySet = $self->getValue($id);
        my $valueToSet      = $self->getBatchValue($id);
        if (!$valueAlreadySet && defined $valueToSet) {
            if ($param->{type} =~ /map/) {
                if (ref($valueToSet) eq 'HASH') {
                    foreach my $key (keys(%$valueToSet)){
                        if (!$self->setMapValueItem($id, $key,$valueToSet->{$key}) && defined $param->{set_interactive} && !$param->{set_interactive}) {
                            return undef;
                        }
                    }
                }
                next;
            }
            if (!$self->setValue($id, $valueToSet) && defined $param->{set_interactive} && !$param->{set_interactive}) {
            	return undef;
            }
            next;
        }

		if (!defined $param->{set_interactive} || $param->{set_interactive}){
			next;
		}

        if ($param->{init_with_default}){
			if ($param->{type} =~ /map/) {
				$self->setDefaultMapValues($id);
				next;
			} else {
				$defaultValue = $self->getDefault ($id);
				if (defined $defaultValue
				    && (($param->{type} =~ /array/) ||
				        ($defaultValue !~ / \$/))) {
					$self->setValue ($id, $defaultValue);
					next;
				}
			}
        }
    }
    $self->ResetError();
    return 1;
}


sub validatePersistency{
	my ($self, $pers, $paramIdsToSkip) = @_;

	if (!defined $pers){
		$pers = $self->pers_load();
	}
	
	if (!defined $pers){
		return undef;
	}

	foreach my $id (@{$self->getParamIds ()}){
        if (grep {$id eq $_} (@{$paramIdsToSkip})) {
            next;
        }

        if (!exists $pers->{$id}) {
            next;
        }

        my $param = $self->{params}->{$id};

        if (defined $param->{persStep}
                             && (int $param->{persStep} <= int $pers->{step})) {
            if ($param->{type} =~ /map/) {
                $self->_setMapParamValue($pers, $id);
                $param->{'all_set'} = 1;
                next;
            }
            $param->{persFileValue} = $pers->{$id};
		}
		else {
            if ($param->{type} =~ /map/) {
                $self->_setMapParamDefaultValue($pers, $id);
                next;
            }
		    $self->setDefault($id, $pers->{$id});
		}
	}

    foreach my $key (@{$self->pers_keys}){
        if ($key eq '_kitversion'){
                next;
        }
        if (exists $pers->{$key}){
            $self->{$key} = $pers->{$key};
        }
    }


    $self->{params}->{SID}->{value}         = $self->{current_sid};
    $self->{params}->{SID}->{persFileValue} = $self->{current_sid};


	return 1;
}


sub _setMapParamValue {
	my ($self, $pers, $id) = @_;
	my $param = $self->{params}->{$id};

	if (ref($pers->{$id}) eq 'HASH') {
		my $persistenceHash = $pers->{$id};
		my @originValues = keys (%{$persistenceHash});
		$param->{origin_values} = [@originValues];
		foreach my $key (keys (%{$persistenceHash})) {
			$param->{persFileValue}->{$key} = $persistenceHash->{$key};
		}
	} elsif (!ref($pers->{$id})) {
		if ($pers->{$id} =~ /([^=]*)=(.*)/) {
			if (!defined $1 || !defined $2) {
				return;
			}
			$param->{origin_values} = [$1];
			$param->{persFileValue}->{$1} = $2;
		}
	}
}

sub _setMapParamDefaultValue {
	my ($self, $pers, $id) = @_;

	my $param = $self->{params}->{$id};

	if (ref($pers->{$id}) eq 'HASH') {
		foreach my $key (keys (%{$pers->{$id}})) {
			$param->{default}->{$key} = $pers->{$id}->{$key};
		}
	} elsif (!ref($pers->{$id})) {
		if ($pers->{$id} =~ /([^=]*)=(.*)/) {
			if (!defined $1 || !defined $2) {
				return;
			}
			$param->{default}->{$1} = $2;
		}
	}
}


#>------------------------------ types validation -----------------------------<#

#validateType
#
#@return
#   1 = valid
#   0 = not valid

sub validateType{
    my ($self,$newValue,$type) = @_;
    my $pattern = $self->{type_pattern}->{$type};
    if (!$newValue =~ m/$pattern/){
          $self->PushError ("Invalid value for type \"$type\"");
          return 0;     
    }
    1;
}

sub getHWCheckScript{
    return undef;
}

sub checkSystemRequirementsAfterConfig{
    return 1;
}

sub checkMinMem{
    return 1;
}

sub dumpConfigFileTemplate {
    my ($self, $filename) = @_;

    if (!open (FD,">$filename")){
        $self->AddError ("Cannot create file '$filename': $!");
        return undef;
    }
    close (FD);

    my %paramIdOptionMap;
    for my $paramId (@{$self->getParamIds()}){
        my $paramOpt = $self->{params}->{$paramId}->{opt};
        if (defined $paramOpt){
            $paramIdOptionMap{$paramId} = '${' . $paramOpt . '}';
        }
    }

    my $ini = new SDB::Install::IniFile ($filename);
    my @passwordIDs;

    for my $id (@{$self->getParamIds ()}){
        my $param = $self->{params}->{$id};
        my ($section, $value) = ($param->{section}, undef);
        my $opt = $param->{opt} ? $param->{opt} : $id;

        next if (!$section || $param->{excludeFromConfigDump} || $param->{hidden});

        if (! $param->{leaveEmptyInConfigDump}){
            $value = defined $param->{value} ? $param->{value} :
                                               (defined $param->{batchValue} ? $param->{batchValue} :
                                                                               $self->getDefault($id));
            $value = $self->substituteParamsByReplaceHash($value, undef, \%paramIdOptionMap);
        }
        my $comment = $self->_prepareIniFileComment($param, $value, $id,\%paramIdOptionMap);
        $value = $self->_getConfigfileValue($value, $id);
        $ini->setValue($section,$opt, $value, $comment);
        if ($param->{type} =~ /passwd/) {
            push @passwordIDs, $id;
        }
    }
    if (!defined $ini->write ()){
        $self->AddError ("Cannot write config file '$filename'", $ini);
        return undef;
    }

    if (@passwordIDs) {
        if (!$self->dumpXmlPasswordFile($filename, \@passwordIDs, 1)) {
            return undef;
        }
    }
    return 1;
}

sub dumpConfigFile {
    my ($self, $name, $extension) = @_;
    $extension   = '.cfg' if (!defined $extension);
    my $filename = join($path_separator, $gLogDir, $name) . $extension;
    my @passwordIDs;

    if (!open (FD,">$filename")){
        $self->AddMessage("Cannot create file '$filename': $!");
        return undef;
    }
    close (FD);

    my $ini = new SDB::Install::IniFile ($filename);
    my $scenario = $self->can('getAction') ? $self->getAction() : undef;

    if($scenario){
        $ini->setValue('Action', 'action', $scenario, 'Execution scenario');
    }

    for my $id (@{$self->getParamIds()}){
        my $param = $self->{params}->{$id};
        my ($section, $comment, $value) = ($param->{section}, $param->{str}, $param->{value});
        my $opt = $param->{opt} ? $param->{opt} : $id;

        next if (!$section || $param->{excludeFromConfigDump});
        next if (!defined($value));
        if ($param->{type} =~ /passwd/) {
            push @passwordIDs, $id if (!$param->{skip});
            next;
        }

        $value = $self->_getConfigfileValue($value, $id);
        $ini->setValue($section,$opt, $value, $comment);
    }

    if (!defined $ini->write ()){
        $self->AddMessage("Cannot write config file '$filename'");
        return undef;
    }

    if (!$isWin && ($> == 0)) {   # $> (effective UID) == 0 (root user id)

        chmod(0640, $filename); # rw-r-----
        my $gid = getgrnam ($gSapsysGroupName);
        if (defined $gid) {
            chown(-1, $gid, $filename);
        }
        # ignore chown errors
    }

    if (@passwordIDs) {
        if (!$self->dumpXmlPasswordFile($filename, \@passwordIDs)) {
            return undef;
        }
    }

    return (1, $filename);
}


sub dumpXmlPasswordFile {

    my ($self, $configFilename, $passwordIDs, $isTemplate) = @_;

    my $xmlStream = $self->getXmlPasswordStream($passwordIDs, 1, $isTemplate);

    if (defined $xmlStream) {
        my $xmlFilename = $configFilename . '.xml';

        if (!open (FD, ">$xmlFilename")) {
            $self->AddMessage("Cannot create file '$xmlFilename': $!");
            return undef;
        }
        if (! print FD @$xmlStream){
            $self->AddMessage ("Cannot write file '$xmlFilename': $!");
            close (FD);
            return undef;
        }
        close (FD);

        if (!$isTemplate && !$isWin && ($> == 0)) { # $> (effective UID) == 0 (root user id)

            chmod(0640, $xmlFilename); # rw-r-----
            my $gid = getgrnam ($gSapsysGroupName);
            if (defined $gid) {
                chown(-1, $gid, $xmlFilename);
            }
            # ignore chown errors
        }
    }
    return 1;
}

sub configuration2Xml {
    my ($self) = @_;
    my $class = "$self";
    $class =~ s/=.*//;
    my $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    my $installer_version = getProductVersion ();
    my $component_version = $self->getProductVersion ();
    $xml .= "<install_configuration class=\"$class\" installer_version=\"$installer_version\" component_version=\"$component_version\">\n";
    $xml .= "  <params>\n";
    my ($param,$key, $value);
    foreach my $id (@{$self->getParamIds ()}){
        $param = $self->{params}->{$id};
        $xml .= "    <param id=\"$id\">\n";
        foreach $key (keys (%$param)){
            $value = $param->{$key};
            if (!defined $value || ref ($value)){
                $value = '';
            }
            else{
                $value =~ s/&/&amp;/sg;
                $value =~ s/</&lt;/sg;
                $value =~ s/>/&gt;/sg;
                $value =~ s/"/&quot;/sg;
            }
            $xml .= "      <$key>$value</$key>\n";
        }
        $xml .= "    </param>\n";
    }
    $xml .= "  </params>\n";
    my $timeouts = $self->getTimeoutValues();
    if (defined $timeouts && @$timeouts){
        $xml .= "  <timeouts>\n";
        foreach my $timeout (@$timeouts){
            $xml .= "    <timeout>$timeout</timeout>\n";
        }
        $xml .= "  </timeouts>\n";
    }
    my $ignoreValues = $self->getIgnoreValues ();
    my $hiddenIgnoreValues = $self->getHiddenIgnoreValues ();
    if (defined $ignoreValues && @$ignoreValues ||
        defined $hiddenIgnoreValues && @$hiddenIgnoreValues){

        $xml .= "  <ignore_values>\n";
        if (defined $ignoreValues && @$ignoreValues){
            foreach my $ignoreValue (@$ignoreValues){
                $xml .= "    <ignore_value>$ignoreValue</ignore_value>\n";
            }
        }
        if (defined $hiddenIgnoreValues && @$hiddenIgnoreValues){
            foreach my $hiddenIgnoreValue (@$hiddenIgnoreValues){
                $xml .= "    <ignore_value hidden=\"1\">$hiddenIgnoreValue</ignore_value>\n";
            }
        }
        $xml .= "  </ignore_values>\n";
    }
    $xml .= "</install_configuration>\n";
    return \$xml;
}


sub _prepareIniFileComment{
	my ($self, $param, $value, $id, $replaceParamidOptHash) = @_;
	my $comment = $param->{str};
	my $commentText;
    my $default = $param->{type} eq 'boolean' ?
            $self->getDefaultUIString($id) :
            $self->getDefault($id, undef, 1);
    if (defined $default ){
        if(defined $value and not ref($value) =~ /ARRAY/ and  not ref($value) =~ /HASH/){
            $default = $self->substituteParamsByReplaceHash
                                      ($default, undef, $replaceParamidOptHash);
            $commentText .= "Default: $default";
        }
    }
    if(defined $param->{valid_values}){
        my $refValues = (exists $param->{visible_alias_values})
                        ? $param->{visible_alias_values}
                        : $param->{valid_values};
        $commentText .= "; " if defined $commentText;
        $commentText .= 'Valid values: ' . join(' | ', @$refValues);
    }
    $comment .= " ( $commentText )" if defined $commentText;
    return $comment;
}

sub _getConfigfileValue{
	my ($self, $value, $id) = @_;
	if (defined $value and $self->{params}->{$id}->{type} =~ /bool/){
        if ($value =~ /$bool_false_pattern/i){
            $value = "n";
        } elsif ($value =~ /$bool_true_pattern/i){
            $value = "y";
        }
    }
    return $value;	
}

sub get_InstallationPath{
	$_[0]->getValue ('PATH');
}

sub set_InstallationPath{
	return $_[0]->setValue ('PATH', $_[1]);
}

sub get_InstallationPathDefault{
	$_[0]->getDefault ('PATH');
}

sub set_InstallationPathDefault{
	return $_[0]->setDefault('PATH', $_[1]);
}


#-------------------------------------------------------------------------------
# Assigns a hash map to the param entry 'value' of the specified map parameter.
# According to the param entry 'origin_values', the assigned hash map consists
# of already existing values and added default values.
#
# Parameters string $paramID  the hash key of the parameter
#
# Returns int retCode 

sub setDefaultMapValues {

    my ($self, $paramID) = @_;
    my $param = $self->{params}->{$paramID};

    foreach my $valKey (@{$param->{origin_values}}) {

        my $lcValKey = lc($valKey);

        if (defined $param->{value} && $param->{value}->{$lcValKey}){
            next;
        }

        if (defined $param->{default_map}) {

            my $default = $param->{default_map}->{$lcValKey};

            if (defined $default) {

                if (!$self->setMapValueItem($paramID, $valKey, $default)) {
                    return undef;
                }

                $self->AddMessage($self->getParamName($paramID, $valKey)
                        . " is set with default value = '$default'");
            }
        }
    }
    
    return $self->checkAllMapValues($paramID);
}


#-------------------------------------------------------------------------------
# Adds or replaces a hash item of the parameter entry 'value'
# in case of parameter type 'map'.
#
# Assignment of new value:
#   params->{$paramID}->{value}->{$valKey} = $newVal
#
# Parameters string $paramID  selects the parameter
#            string $valKey   specifies the item of the hash map under 'value'
#            string $newVal   new value of the specified hash map item
#
# Returns int retCode 

sub setMapValueItem {
    
    my ($self, $paramID, $valKey, $newVal) = @_;

    my $param            = $self->{params}->{$paramID};
    my $subroutineSetAny = $self->can("set$paramID"); # e.g. setHostMap
    
    if (defined $subroutineSetAny) {
            
        # calls e.g. setHostMap if subroutine exists
        if (!&$subroutineSetAny($self, $valKey, $newVal)) {
            return undef;
        }
        if( !$self->notifyParameterListeners($paramID, $newVal, $valKey)){
        	return undef;
        }
    }
    else {
        
        my $subroutineCheckAny = $self->can ("check$paramID"); # e.g. checkHostMap
            
        if (defined $subroutineCheckAny) {
                
            # calls e.g. checkHostMap if subroutine exists
            if (!&$subroutineCheckAny($self, $valKey, $newVal)) {
                return undef;
            }
        }
        if( !$self->notifyParameterListeners($paramID, $newVal, $valKey)){
        	return undef;
        }
        $param->{value}->{lc($valKey)} = $newVal;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# This method has to be called when all hash entries are knwon.
# Calls 'checkEntries<paramID>' if such a subroutine exists.

sub checkAllMapValues {

    my ($self, $paramID) = @_;

    my $subroutineCheckEntries = $self->can("checkEntries$paramID");
                                                      # e.g. checkEntriesHostMap

    if (defined $subroutineCheckEntries) {
        return &$subroutineCheckEntries($self);
    }
    return 1;
}


sub isUpdate {
	return 0;
}

our $CDATA_START = '<![CDATA[';
our $CDATA_END = ']]>';
our $CDATA_END_REGEX = quotemeta ($CDATA_END);
our $CDATA_TEMPL = $CDATA_START.'%s'.$CDATA_END;
our $CDATA_JOIN = ']]' . $CDATA_END . $CDATA_START . '>';

sub _getCDATA ($){
    my ($value) = @_;
    if ($value !~ /$CDATA_END_REGEX/){
        return sprintf ($CDATA_TEMPL, $value);
    }
    my $cdata = $CDATA_START;
    $cdata .= join ($CDATA_JOIN, split($CDATA_END_REGEX, $value, -1));
    $cdata .= $CDATA_END;
    return $cdata;
}


sub getXmlPasswordStream{
	my ($self,
	    $wantedPasswordIDs, # if specified, these passwords are used instead of all existing passwords
	    $createFile,
	    $isTemplate,
	   ) = @_;

	my @paramIDs = (defined $wantedPasswordIDs) ? @$wantedPasswordIDs
	                                            : keys (%{$self->{params}});
	my $buf = '';
	my $indent = ($createFile) ? "\n    " : '';
	foreach my $id (@paramIDs){
		my $param = $self->{params}->{$id};
		if (!defined $param) {
			next;
		}
		if (!defined $param->{type} ||
		    ($param->{type} ne 'passwd' && $param->{type} ne 'initial_passwd')){
			next;
		}
		if (!$isTemplate && (!defined $param->{value} || $param->{skip})) {
			next;
		}
		my $tag = $param->{opt};
		my $value = ($createFile) ? '***' : $param->{value};
		$buf .= $indent . '<' . $tag . '>' . _getCDATA ($value) . '</' . $tag . '>';
	}
	if ($buf ne '') {
		my $result = ($createFile)
		             ? '<?xml version="1.0" encoding="UTF-8"?>' . "\n"
		               . '<!-- Replace the 3 asterisks with the password -->'
		               . "\n<Passwords>$buf\n</Passwords>\n"
		             : "<Passwords>$buf</Passwords>";
		return [$result];
	}
	return undef;
}


#-------------------------------------------------------------------------------
# Returns the description of a parameter.
#
# Parameter:  $param  - reference to the parameter

sub getParamDesc {

    my ($self, $param) = @_;

    if (defined $param->{desc}) {
        return $param->{desc};
    }

    return $self->convertDescIntoLowerCase($param->{str});
}


#-------------------------------------------------------------------------------
# Returns the given description converted into a lower case sentence.

sub convertDescIntoLowerCase {

    my ($self, $desc) = @_;

    my $description = uc(substr($desc,0,1))
                    . lc(substr($desc,1));

    $description =~ s/^Sap hana system /SAP HANA System /;
    $description =~ s/ sap hana system / SAP HANA System /;
    $description =~ s/ sap hana system$/ SAP HANA System/;
    $description =~ s/ sap hana xs / SAP HANA XS /;
    $description =~ s/ sap hana xs$/ SAP HANA XS/;
    $description =~ s/ sap ase / SAP ASE /;
    $description =~ s/ sap ase$/ SAP ASE/;
    $description =~ s/^Sap /SAP /;
    $description =~ s/ sap / SAP /;
    $description =~ s/ sap$/ SAP/;
    $description =~ s/ hana / HANA /;
    $description =~ s/ hana$/ HANA/;
    $description =~ s/^Id /ID /;
    $description =~ s/ id / ID /;
    $description =~ s/ id$/ ID/;
    $description =~ s/ user \(system\)/ user \(SYSTEM\)/;

    return $description;
}

#-------------------------------------------------------------------------------
# Returns a string used to identify the parameter within a message.
#
# Output Examples:
#     Parameter 'versionID'
#     Parameter '--nostart'
#     Parameter '--sapmnt' (Target)
#
# Parameters:  $paramID  - string
#             [$entryKey - string] # specifies a specific hash map entry

sub getParamName {

    my ($self, $paramID, $entryKey, $prefix) = @_;

    my $option    = $self->{params}->{$paramID}->{opt};
    my $paramName = (defined $prefix) ? $prefix . ' ' : 'Parameter ';
    my $id        = (defined $entryKey) ? "$paramID:$entryKey" : $paramID;

    if (!defined $option) {
        $paramName .= "'$id'";
    }
    else {
        $paramName .= "'$option' ($id)";
    }
    return $paramName;
}

sub setBatchValue{
    my ($self, $paramId) = @_;
    my $param = $self->{params}->{$paramId};
    my $batchValue = $param->{batchValue};

    if (!defined $batchValue){
        return 0;
    }

    if ($batchValue =~ /\$/){
        $batchValue = $self->substituteValues ($batchValue);
    }
    
    if($param->{type} =~ /map/){
        while(my ($key, $value) = each %$batchValue){
            if(!$self->setMapValueItem($paramId, $key, $value)){
                return undef;
            }
        }
        return $self->checkAllMapValues($paramId);
    }

    return $self->setValue ($paramId, $batchValue);
}

sub isShowPending {
    return 0;
}

sub isAdminRequired{
    return 1;
}

sub getDefaultSysProgPath{
    return $sysProgPath;
}

sub getDefaultUserProgPath{
    return $userProgPath;
}

sub getDefaultProgPath{
    if ($_[0]->isUserMode ()){
		return $_[0]->getDefaultUserProgPath();
    }
	return $_[0]->getDefaultSysProgPath();
}
sub isUserMode{
	return !$_[0]->isAdminRequired () && !isAdmin ();
}

sub addParameterListener {
	my ($self, $paramId, $listener, ) = @_;

	my $param = $self->{params}->{$paramId};

	if(!defined $param) {
		return undef;
	}
	if (defined $param->{listeners}) {
		push (@{$param->{listeners}}, $listener);
	} else {
		$param->{listeners} = [$listener];
	}
}

sub notifyParameterListeners {
	my ($self, $paramId, $value, $valKey) = @_;

	my $parameter = $self->{params}->{$paramId};
	if (!defined $parameter->{listeners}) {
		return 1;
	}

	my @listeners = @{$parameter->{listeners}};
	my $rc = 1;
	for(@listeners) {
		if((ref $_ eq 'CODE')) {
			$rc = $_->($value, $self);
		} else {
			if($self->{params}->{$paramId}->{type} =~ /map/){
				$rc = $_->onMapValueChange($valKey,$value, $self);
			} else {
				$rc = $_->onValueChange($value, $self);
				
			}
		}
		last if (!$rc);
	}
	return $rc;
}

sub resetParam {
	my ($self, $paramId) = @_;

	my $param = $self->{params}->{$paramId};
	if ($param) {
		$param->{value} = undef;
		$param->{batchValue} = undef;
		$self->{options}->{$param->{opt}} = undef;
	}

	my $sub = $self->can ("reset$paramId");
	if ($sub) {
		return &$sub ($self);
	}
}

sub addParameterWarning
{
    my ($self, $paramId, $warning) = @_;
    my $param = $self->{params}->{$paramId};
    if (not exists $param->{warnings}) {
        $param->{warnings} = [];
    }
    push (@{$param->{warnings}}, $warning);
    return $param->{warnings};
}

sub clearParameterWarnings {
    my ($self, $paramId) = @_;
	if (exists($self->{params}->{$paramId})){
        $self->{params}->{$paramId}->{warnings} = [];
    }
}

sub getParameterWarnings {
    my ($self, $paramId) = @_;
    my $warnings = $self->{params}->{$paramId}->{warnings};
    return (!defined $warnings || !@$warnings) ? [] : $warnings;
}

#-------------------------------------------------------------------------------
# No default implementation available

sub getSystemHosts {
    return undef;
}


#-------------------------------------------------------------------------------
# Returns a local host name provided by operating system.
# There may exist alias names.

sub getLocalHanaHost {
    my $self = shift;
    return lc( hostname() );
}


#-------------------------------------------------------------------------------
# Returns name of the System Administrator User for the specified SID.

sub getSysAdminUserName {
    my ($self, $sid) = @_;

    return defined($sid) ? createSysAdminUserName($sid) : undef;
}


sub addConfigurationChangeEventHandler {
    my($self,$handler) = @_;
    if(!defined $self->{_changeEventHandlers}){
        require SDB::Install::ConfigurationChangeEvent;
        $self->{_changeEventHandlers} = [$handler];
    }
    else {
        push(@{$self->{_changeEventHandlers}},$handler);
    }
}

sub notifyConfigurationChangeEventHandlers {
    my($self,$event) = @_;
    foreach my $handler(@{$self->{_changeEventHandlers}}){
        $handler->configurationChanged($event);
    }
}

sub setUpdateParams{
	return 1;
}

sub getProductVersion {
    my $installer = new SDB::Install::Installer();
    return $installer->GetVersion();
}

sub getAbsPath{
    return File::Spec->rel2abs ($_[1], $SAPDB::Install::Config{CallerDir});
}

#-------------------------------------------------------------------------------
# Returns 0 if parameter '--use_http' is specified or in case of Windows.

sub isUseHttps {
    my ($self) = @_;
    return ($isWin || $self->getValue('UseHttp') || $self->getValue('SAPControlUseHttp')) ? 0 : 1;
}

sub setPersStep {
	my ($self, $paramId, $persStep) = @_;
	if ( ! defined $paramId || ! defined $self->{params}->{$paramId} ) {
		return;
	}

	$self->{params}->{$paramId}->{persStep} = $persStep;
}

sub getPersStep {
    my ($self,$paramId) = @_;
    return undef if(!exists($self->{params}->{$paramId}));
    return $self->{params}->{$paramId}->{persStep};
}

sub setPendingInstallation {
	$_[0]->{pendingInstallation} = $_[1];
}

sub isPendingInstallation {
	return $_[0]->{pendingInstallation};
}

sub getPendingConfiguration { return undef; }

sub getParamXsEaDataPath {
    my ($self, $order, $section, $constraint) = @_;
    return {
        'skip' => 1,
        'order' => $order,
        'type' => 'path',
        'value' => undef,
        'mandatory' => 0,
        'section' => $section,
        'set_interactive' => 1,
        'init_with_default' => 1,
        'opt' => 'xs_app_working_path',
        'opt_arg' => '<path>',
        'alias_opts' => ['xs_ea_data_path'],
        'constraint' => $constraint,
        'str' => 'XS Advanced App Working Path',
        'desc' => 'XS Advanced App Working Path',
    };
}


sub getParamOrgManagerUser {
    my ($self, $order, $section, $constraint) = @_;
    return {
        'order' => $order,
        'opt' => 'org_manager_user',
        'opt_arg' => '<name>',
        'type' => 'string',
        'section' => $section,
        'value' => undef,
        'default' => 'XSA_ADMIN',
        'str' => 'XS Advanced Admin User',
        'desc' => 'XS Advanced Admin User',
        'skip' => 1,
        'mandatory' => 1,
        'set_interactive' => 1,
        'init_with_default' => 1,
        'constraint' => $constraint,
        'hostctrl_opt' => 'ORG_MANAGER_USER',
    };
}

sub getParamOrgManagerPassword {
    my ($self, $order, $section, $constraint,$userParamId) = @_;
    return {
        'order' => $order,
        'opt' => 'org_manager_password',
        'type' => 'initial_passwd',
        'section' => $section,
        'value' => undef,
        'str' => 'XS Advanced Admin User Password',
        'desc' => 'XS Advanced Admin User Password',
        'skip' => 1,
        'mandatory' => 1,
        'set_interactive' => 1,
        'constraint' => $constraint,
        'user_paramid' => $userParamId
    };
}

sub getParamXsDomainName {
    my ($self, $order, $section, $constraint) = @_;
    return {
        'order' => $order,
        'opt' => 'xs_domain_name',
        'type' => 'string',
        'section' => $section,
        'value' => undef,
        'default' => undef,
        'str' => 'XS Advanced Domain Name (see SAP Note 2245631)',
        'desc' => 'XS Advanced Domain Name (see SAP Note 2245631)',
        'skip' => 1,
        'mandatory' => 0,
        'init_with_default' => 1,
        'set_interactive' => 1,
        'constraint' => $constraint,
    };
}

sub addWarning {
    my ($self, $warning) = @_;
    push (@{$self->{warnings}}, $warning);
    $self->getMsgLst()->addWarning($warning);
}

sub getUserParameterId{
    my ($self,$id) =@_;
    return undef if ($self->{params}->{$id}->{type} !~ /passwd/ );
    my $user_paramId = $self->{params}->{$id}->{'user_paramid'};
    if(defined $self->{params}->{$user_paramId}){
        return $user_paramId;
    }
    return undef;
}

sub isInteractive {
    my ($self,$param_id,) = @_;
    return undef if (!exists $self->{params}->{$param_id});
    return exists $self->{params}->{$param_id}->{set_interactive} ? $self->{params}->{$param_id}->{set_interactive} : 1;
}
1;
