package SDB::Install::Configuration::AddHost;

use strict;
use SDB::Install::Configuration::NewServerConfig;

use SAPDB::Install::Hostname qw(hostname);
use SDB::Install::Globals    qw($gDriveWin
                                $gHostRoleAcceleratorWorker
                                $gHostRoleStandby
                                $gHostRoleWorker
                                $gHostRoleXS2Standby
                                $gHostRoleXS2Worker
                                $gProductName
                                $gSapmntDirUx
                                $gShortProductNameAccelerator
                                $gShortProductNameTenant
                                $gShortProductNameXS2
                                $gXSParametersAddHostOrRoleConstraint
                                $gDirNameAccelerator
                                $gDirNameEs
                                $gDirNameRDSync
                                $gDirNameStreaming);
use SDB::Install::Installation qw (SDB_O_CREAT);
use SDB::Install::SAPSystem    qw (@ADDHOST_STEP_NAMES
                                   $STEP_ADDHOST_CREATE_DIRECTORIES);
use SDB::Install::System     qw(R_OK W_OK X_OK);
use SDB::Install::SysVars    qw($isWin $path_separator $isPowerPC);
use SDB::Install::Tools      qw(trim);
use SDB::Install::Configuration::AnyMultiHostConfig qw ($hostRoleDefaultValue
                                                        $validHostRoles
                                                        getSortedHostRoleNames);
use SDB::Install::Configuration::AutoInitFamilyServicesParamHandler;
use SDB::Install::Configuration::TenantUserListener;
use SDB::Install::LayeredConfig qw(CFG_LAYER_DEFAULT);
require SDB::Install::Configuration::NewDB;

our @ISA = qw (SDB::Install::Configuration::NewServerConfig);


#-------------------------------------------------------------------------------
# Initializes {_hwCheckScript} if hard ware check is required
#
# Parameters: SDB::Install::SAPInstance::TrexInstance $instance
#             SDB::Install::SAPSystem                 $sapSys
#
# Returns int retCode = 1

sub initHardwareCheck {
    
    my ($self, $instance, $sapSys) = @_;

    if (!$isWin  && !$isPowerPC && ($sapSys->getManifest ()->isCustomerReleaseBranch () ||
        exists $ENV{'HDB_FORCE_HANA_HARDWARE_CHECK'})){

        my $script = $sapSys->getHanaHwCheckScript();
        if (-f $script){
            $self->{_basepath_logvolumes}  = $instance->getLogPath (1);
            $self->{_basepath_datavolumes} = $instance->getDataPath (1);
            $self->{_hwCheckScript}        = $script;
        }
        else{
            $self->AddWarning ("Hardware check script '$script' not found");
        }
        $self->{instance} = $instance;
        $self->{userCfg} = $sapSys->getUserConfig ();
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Initializes and checks the volume paths
#
# Parameters: SDB::Install::SAPInstance::TrexInstance $instance
#             SDB::Install::SAPSystem                 $sapSys
#
# Returns int retCode

sub initVolumePaths {
    
    my ($self, $instance, $sapSys) = @_;

    if ($instance->usesStorageApi()){
        return 1;
    }

    my $datapath = $instance->getDataPath ();
    my $logpath  = $instance->getLogPath ();
    my $rc       = 1;
    my $usercfg  = $sapSys->getUserConfig();
    
    foreach my $path ($datapath, $logpath){
        next if !$path;
        next if $path =~ /^\$/;
        $self->AddMessage ("Checking volume path '$path'");
        
        if (!-d $path){
            $self->PushError ("Volume path '$path' not found: $!");
            $rc = 0;
            next;
        }
        
        if (!$self->check_path_access
                                 ($path,R_OK|W_OK|X_OK, $usercfg->{sidadm_id},
                                  $usercfg->{sapsys_groupid})){
            $rc = 0;
        }
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Constructor
sub new{
	my $self = shift->SUPER::new (@_);

	my $section = 'AddHost';
	my $order   = 0;

	my $acceleratorOnly = "Valid with $gHostRoleAcceleratorWorker/standby only";
    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();
	$self->{params} = {
        'UseHttp'          => $self->getParamUseHttp         ($order++, $section),
        'CheckOnly'        => $self->getParamCheckOnly       ($order++, $section),
        'InstallHostagent' => $self->getParamInstallHostagent($order++, $section),
        'HostName'         => $self->getParamHostName        ($order++, $section),
        'SID'              => $self->getParamSID             ($order++, $section),
        'AddHostCalledInCompatiblityMode' => $self->getParamAddHostCalledInCompatiblityMode($order++, $section),
        'AutoInitializeServices' => $self->getParamAutoInitializeServices($order++, $section, undef, $tenantCredentialsConstraint),
        ($isWin
            ? ('Drive'     => $self->getParamDrive           ($order++, $section), )
            : ('Target'    => $self->getParamTarget          ($order++, $section), )
         ),
        'Password'               => $self->getParamPassword              ($order++, $section),
        'HostRoles'              => $self->getParamHostRoles             ($order++, $section),
        'HostRole'               => $self->getParamHostRole              ($order++, $section),
        'HostFailoverGroup'      => $self->getParamHostFailoverGroup     ($order++, $section),
        'StoragePartitionNumber' => $self->getParamStoragePartitionNumber($order++, $section),
        'WorkerGroup'            => $self->getParamWorkerGroup           ($order++, $section,1 , $STEP_ADDHOST_CREATE_DIRECTORIES),
        'NoStart'                => $self->getParamNoStart               ($order++, $section,
                                    'Do not start the instance after installation', 'Does not start the instance after installation', 0),
        'MachineUtilization'     => $self->getParamMachineUtilization    ($order++, $section),
    ($isWin ? (
        'HostagentUserName'      => $self->getParamHostagentUserName     ($order++, $section))
            : ()),
        'HostagentPassword'      => $self->getParamHostagentPassword     ($order++, $section),
        'LandscapeID'            => $self->getParamLandscapeID           ($order++, $section),
        'SystemUser'             => $self->getParamSystemUser            ($order++, $section, 1, $acceleratorOnly),
        'SQLSysPasswd'           => $self->getParamSQLSysPasswd          ($order++, $section, 'passwd', 1, undef, $acceleratorOnly),
        'AcceleratorUser'        => $self->getParamAcceleratorUser       ($order++, $section, $acceleratorOnly),
        'AcceleratorPassword'    => $self->getParamAcceleratorPassword   ($order++, $section, $acceleratorOnly),
        'OrgManagerUser'         => $self->getParamOrgManagerUser        ($order++, $section, $gXSParametersAddHostOrRoleConstraint),
        'OrgManagerPassword'     => $self->getParamOrgManagerPassword    ($order++, $section, $gXSParametersAddHostOrRoleConstraint),
        'AutoAddXS2Roles'        => $self->getParamAutoAddXS2Roles       ($order++, $section, undef, 0),
        'ImportContentXS2'       => $self->getParamImportContentXS2      ($order++, $section),
         ($isWin
                ? ()
                : ('SkipModifySudoers' => $self->getParamSkipModifySudoers($order++, $section)),
         ),
        'TenantUser'            => $self->getParamTenantUser    ($order++, $section,1, $tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++,$section,1, $tenantCredentialsConstraint),
	};

	$self->{params}->{NoStart}->{additional_desc} =
		"(Cannot be applied together with host role of $gShortProductNameAccelerator)";

	# Target and SID are set by SBD::Install::Installer::getInstallation
	$self->{params}->{SID}->{hidden}    = 1;
	$self->{params}->{Target}->{hidden} = 1;

	$self->{params}->{HostName}->{persStep} = undef;
	$self->{params}->{Target}->{persStep}   = undef;

	foreach my $id (qw (MachineUtilization SystemUser AcceleratorUser)){
		$self->{params}->{$id}->{persStep} = $STEP_ADDHOST_CREATE_DIRECTORIES;
	}
    $self->addParameterListener('HostRoles',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('HostRoles'));
    $self->addParameterListener('HostRole',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('HostRole'));
    $self->addParameterListener('TenantUser',SDB::Install::Configuration::TenantUserListener->new());

    $self->setSummaryOrder(['Target',
                            'SID',
                            'HostName',
                            'HostRole',
                            'HostRoles',
                            'HostFailoverGroup',
                            'WorkerGroup',
                            'StoragePartitionNumber',
                            'SystemUser',
                            'AcceleratorUser',
                            'CheckOnly',
                            'NoStart',
                            'TenantUser']);
	return $self;
}

sub checkWorkerGroup {
    my ($self, $workerGroupCSV) = @_;
    my @workerGroups = split(',', $workerGroupCSV);

    for my $group (@workerGroups){
        return undef if(!$self->SUPER::checkWorkerGroup($group));
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Checks the specified landcsape ID with the landscape ID contained in the
# installation path. In case of a remote execution, the master adds its
# landscape ID to the parameters of hdbaddhost. If e.g. '/hana/shared' is linked
# to another system, this can be detected.
#
# Returns int retCode

sub checkLandscapeID {

    my ($self, $wantedID) = @_;

    if (!defined $wantedID) {
        return 1; # skip checking ID
    }

    my $instance     = $self->{instance};
    my $sid          = $self->getValue('SID');
    my $instanceName = (defined $sid) ? "instance '$sid'" : 'instance';

    if (!defined $instance) {
         my $sapSys = $self->{sapSys};
         $instance  = $sapSys->getNewDBInstances()->[0] if (defined $sapSys);
    }

    if (!defined $instance || !$instance->isNewDB()) {
        $self->PushError ("Cannot access $gProductName $instanceName");
        return 0;
    }

    my $landscapeID = $instance->getLandscapeID();

    if (!defined $landscapeID) {
        $self->PushError ("Landscape ID of $gProductName $instanceName not found");
        return 0;
    }

    if ($landscapeID ne $wantedID) {
        $self->PushError ("Installation path is not linked to $gProductName"
                . " master $instanceName: landscape ID '$landscapeID' found"
                . " ('$wantedID' expected)");
        return 0;

    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks if the specified SID and the system administrator user
#
# Parameter: string $wantedSID
#
# Returns int retCode 

sub checkSID{
	my ($self, $wantedSID) = @_;
	
	my $properties = $self->{params}->{SID};
	my $sapSys     = $self->getSAPSystem();
	my $instance   = $sapSys->getNewDBInstances()->[0] if (defined $sapSys);
	if (!defined $sapSys || !defined $instance) {
		$self->PushError ("$gProductName instance '$wantedSID' does not exist");
		return 0;
	}

	$self->{pers_filename_addhost} =
			$self->getAddHostPersFile($instance, $self->getValue('HostName'));

	if ($self->pers_exists()) {
		my $persData = $self->pers_load();
		if (defined $persData && defined $persData->{InstallDate}) {
			$self->PushError ("$gProductName instance '$wantedSID' already"
			    . " installed on host '" . $self->getValue('HostName')
			    . "' at $persData->{InstallDate}");
			return 0;
		}
		if (defined $persData->{step}) {
			$self->{step} = $persData->{step};
			foreach my $persID (keys %$persData) {
				my $currParam = $self->{params}->{$persID};
				if (defined $currParam) {
					$currParam->{batchValue}             = $persData->{$persID};
					$self->{options}->{$currParam->{opt}}= $persData->{$persID};
				}
			}
		}
	}
	elsif (!$self->checkAdditionalSID($wantedSID, $properties->{str})) {
		return 0;
	}
	if (!$self->verifyPMemConfig()){
		return 0;
	}
	my $user = new SDB::Install::NewDBUser ($wantedSID);
	my $userName = $user->getSidAdmName ();
    if (!$user->exists()) {
        $self->{params}->{Password}->{type} = 'initial_passwd';
    }
    elsif ($self->getStep() == 0) { # suppress message if restarting hdbaddhost
        $self->AddProgressMessage ("The defined user '$userName' already exists on the system. Neither the password, nor any other attribute of the user will be changed.");
        $self->AddProgressMessage ('Verify that the user is correct.');
    }
    
	$self->changePasswordStr($userName);
	$self->enableHostagentPasswordIfRequired();

	return 1;
}



sub verifyPMemConfig{
    my ($self) = @_;
    my $hanaInstance = $self->getOwnInstance();
    my $errorMessage = 'Cannot verify pmem config';
    if (!defined $hanaInstance){
        $self->appendErrorMessage ("$errorMessage: HANA instance is not defined");
        return undef;
    }
    my $msglst = $self->getMsgLst();
    $hanaInstance->setMsgLstContext([$msglst]);
    my $global_ini = $hanaInstance->getGlobalIni();
    if (!defined $global_ini){
        $self->appendErrorMessage ($errorMessage, $hanaInstance->getErrMsgLst());
        return undef;
    }
    $global_ini->setMsgLstContext([$msglst]);
    my $iniFileLayer;
    my $basePathPMemVols = $global_ini->getValue('persistence', 'basepath_persistent_memory_volumes', \$iniFileLayer);
    if (!$basePathPMemVols || $iniFileLayer == CFG_LAYER_DEFAULT){
        $msglst->addMessage ('No pmem volumes configured.');
        return 1;
    }
    my $rc = 1;
    if (!$self->_checkNVMMounts($errorMessage)){
        return 0;
    }
    foreach my $volDir (split (';', $basePathPMemVols)){
        if (!$self->_checkDaxFileSystem($volDir, $errorMessage)){
            $rc = 0;
        }
    }
    return $rc;
}



#-------------------------------------------------------------------------------
# Checks the path consisting of the given target directory and SID
#
# Parameter: string $targetDirectory
#
# Returns int retCode

sub checkTarget{
	my ($self, $targetDirectory) = @_;
	my $sid = $self->{params}->{SID}->{value};
	delete $self->{internalLocalIp};
	my $path = $targetDirectory . $path_separator . $sid;
	if (!-d $path){
		$self->PushError ("Directory '$path' doesn't exist");
		return 0;
	}

	my $sapSys;
    my $globalSapSid = "$targetDirectory$path_separator$sid";
    $self->{_globalSapSid} = $globalSapSid;
    $self->{params}->{Target}->{no_retry} = 1;
    if (-d $globalSapSid){
        $sapSys = $self->getNewSAPSystem($globalSapSid);

        if (!$sapSys->hasNewDB){
            $self->AddError ("System installed in '$path' has no HDB instance");
            return 0;
        }

		my $instance = $sapSys->getNewDBInstances()->[0];
        my $listenInterface = $instance->getTrexNetListenInterface ();
        my $internalNetworkPrefix = $instance->getInternalNetworkPrefix ();

        my $errMsg1 = "Parameter 'listeninterface' is set to '.local'.";
        my $errMsg1Append1 = "Please reconfigure your master node:";
        my $errMsg1Append2 = "Perform 'hdbnsutil -reconfig --hostnameResolution=global' as sidadm on master node.";
        my $errMsg2 = "Listen interface is set to '.internal', but internal_network is unknown.";

        if (! $self->validateListenInterface($listenInterface, $internalNetworkPrefix, $errMsg1, $errMsg1Append1, $errMsg1Append2, $errMsg2)) {
        	return 0;
        }

    }
    $self->{sapSys} = $sapSys;
	# checking r-x of patent directories
	return $self->check_path_access($targetDirectory, 0);
}


#-------------------------------------------------------------------------------
# Checks that the partition number is greater 1
#
# Parameter: int $value
#
# Returns int retCode

sub checkStoragePartitionNumber{
    my ($self, $value) = @_;
    my $properties = $self->{params}->{StoragePartitionNumber};
    
    if(!defined $value or $value eq '') {
    	$self->PushError ("$properties->{str} is empty");
        return 0;
    }
    
    if( $value !~ m/^([0-9]+)$/) {
    	$self->PushError ("$properties->{str} has to be a number");
        return 0;
    }
    if ($value < 2){
        $self->PushError ("$properties->{str} has to be greater than one");
        return 0;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the specified sidadm password
#
# Parameter: string $sidadmPassword
#
# Returns int retCode

sub checkPassword{
    my ($self, $sidadmPassword) = @_;

    my $properties = $self->{params}->{Password};

    if (!$sidadmPassword){
        $self->PushError ("Empty $properties->{str} not allowed");
        return 0;
    }
    my $success = 1;
    my $sapSys   = $self->{sapSys};
    if (-d $self->{_globalSapSid} && defined $sapSys) {
        my $instance = $sapSys->getNewDBInstances()->[0];
        if (defined $instance){
            my $masters = $instance->getMasters();
            my $sidadm = $sapSys->getUser()->getSidAdmName();
            my ($errlst, $rc);
            foreach my $master (@$masters){
                $errlst = new SDB::Install::MsgLst ();
                $self->AddMessage ("Checking password via sapstartsrv of master host ($master)");
                $rc = $instance->checkSapStartSrv ($sidadm,
                                                   $sidadmPassword,
                                                   $master,
                                                   $errlst,
                                                   $self->isUseHttps());
                if ($rc){
                    if ($rc == 1){
                        $self->PushError ("Cannot verify password: sapstartsrv of host '$master' is not running", $errlst);
                        $success = 0;
                        next;
                    }
                    if ($rc != 6){
                        $self->PushError ("Cannot verify password:", $errlst);
                        return 0;
                    }
                    $self->PushError ('Wrong password', $errlst);
                    return 0;
                }
                return 1;
            }
        }
    }
    if (!$success){
        $properties->{no_retry} = 1;
    }
    return $success;
}


#-------------------------------------------------------------------------------
# Returns a string containing the filename of the hardware check script 

sub getHWCheckScript{
    return $_[0]->{_hwCheckScript};
}


#-------------------------------------------------------------------------------
# Returns a string array specifying hardware and platform checks 

sub getHiddenIgnoreValues{
	return [qw (check_hardware check_platform check_hostname)];
}

sub getIgnoreValues{
    return [qw (check_timediff check_pmem)];
}

sub getTimeoutValues{
    return [qw (start_instance start_service)];
}

#
# performing hardware check with data and log cmdln options for space check
#

sub checkSystemRequirementsAfterConfig{
    my ($self, $sysinfo, $manifest) = @_;
    if ($isWin){
        delete $self->{instance};
        return 1;
    }

    if (!defined $self->checkSystemRequirements($sysinfo, $manifest)){
        return undef;
    }

    if (!defined $self->getHWCheckScript()){
        #
        # no check script
        #
        delete $self->{instance};
        return 1;
    }

    if ($self->{_isStandby}) {
        #
        # standy has no own volumes
        #
        delete $self->{instance};
        delete $self->{_basepath_datavolumes};
        delete $self->{_basepath_logvolumes};
        return 1;
    }

    my $partition = $self->getValue ('StoragePartitionNumber');

    my $mounter;
    my $useStorageApi = $self->{instance}->usesStorageApi ();
    if (defined $partition){
        my ($uid,$gid);
        if ($useStorageApi){
            my $msg = $self->AddMessage ("Mounting storage devices...");
            $mounter = $self->{instance}->mountStorageDevices ($partition);
            if (!defined $mounter){
                $self->AddError ("Cannot mount storage devices",$self->{instance});
                delete $self->{instance};
                return undef
            }
            $self->AddSubMsgLst($msg, $mounter);
            $uid = $self->{userCfg}->{sidadm_id};
            if (defined $uid){
                $uid = int ($uid);
            }
            $gid = $self->{userCfg}->{sapsys_groupid};
            if (defined $gid){
                $gid = int ($gid);
            }
        }
        my $path;
        if (defined $self->{_basepath_datavolumes}){
            $path = $self->{_basepath_datavolumes} . $path_separator . sprintf ('mnt%05d', int ($partition));
            if (-d $path){
                $self->{_basepath_datavolumes} = $path;
                if (defined $uid && defined $gid){
                    if (chown ($uid,$gid, $path)){
                        $self->getMsgLst()->addMessage ("Owner of $path changed (uid = $uid, gid = $gid)");
                    }
                    else{
                        $self->getMsgLst()->addWarning ("Cannot change owner of $path: $!");
                    }
                }
            }
        }
        if (defined $self->{_basepath_logvolumes}){
            $path = $self->{_basepath_logvolumes} . $path_separator . sprintf ('mnt%05d', int ($partition));
            if (-d $path){
                $self->{_basepath_logvolumes} = $path;
                if (defined $uid && defined $gid){
                    if (chown ($uid,$gid, $path)){
                        $self->getMsgLst()->addMessage ("Owner of $path changed (uid = $uid, gid = $gid)");
                    }
                    else{
                        $self->getMsgLst()->addWarning ("Cannot change owner of $path: $!");
                    }
                }
            }
        }
    }
    elsif ($useStorageApi){
        # skip disk space check,
        # if storage connector is used and storage partition is unknown
        delete $self->{instance};
        delete $self->{_basepath_datavolumes};
        delete $self->{_basepath_logvolumes};
        return 1;
    }
    my $rc = $self->runHWCheckScript ();
    if (defined $mounter){
        $mounter->resetMsgLstContext ();
        my $msg = $self->AddMessage ("Umounting storage devices...");
        if (!defined $mounter->umount ()){
            $self->AddError ("Cannot umount storage devices", $mounter);
            $rc = undef;
        }
        $self->AddSubMsgLst($msg, $mounter);
    }
    delete $self->{instance};
    return $rc;
}


#-------------------------------------------------------------------------------
# Returns a new instance of SDB::Install::SAPSystem
#
# Parameter: String $pasthUsrSapSID   database path (e.g. usr/sap/DB1)

sub getNewSAPSystem {

    my ($self, $pathUsrSapSID) = @_;

    $self->resetSAPSystemHashes();

    my $sapSys = new SDB::Install::SAPSystem();

    if ($isWin){
        $sapSys->initSAPSystem($pathUsrSapSID, SDB_O_CREAT,
                                              undef, $self->getValue ('Drive'));
    }
    else{
        $sapSys->initSAPSystem($pathUsrSapSID);
    }

    $sapSys->SetProgressHandler($self->GetProgressHandler());

    return $sapSys;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--group'.

sub getParamHostFailoverGroup {
    my ($self, $order, $section) = @_;

    return {
        'order'     => $order++,
        'opt'       => 'group',
        'opt_arg'   => '<name>',
        'type'      => 'string',
        'section'   => $section,
        'value'     => undef,
        'default'   => undef,
        'str'       => 'Host Failover Group',
        'desc'      => 'Specifies the failover group of this host',
        'init_with_default' => 0,
        'set_interactive'   => 1,
        'persStep'          => $STEP_ADDHOST_CREATE_DIRECTORIES,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--role'.

sub getParamHostRole {
    my ($self, $order, $section) = @_;

    my $sortedRoles = getSortedHostRoleNames();
    my @sortedRoleUIValues;
    foreach my $role (@$sortedRoles) {
        push @sortedRoleUIValues, $validHostRoles->{$role}->{desc};
    }

    return {
        'order'           => $order,
        'opt'             => 'role',
        'opt_arg'         => '<role>',
        'opt_arg_switch'  => '<role%s>',
        'type'            => 'array',
        'section'         => $section,
        'value'           => undef,
        'default'         => [$hostRoleDefaultValue],
        'str'             => 'Host Role',
        'desc'            => 'Specifies a role of this host',
        'additional_desc' => "(The parameter has to be specified for each host role)\n"
                             . $self->getHostRoleDescription(),
        'valid_values'    => $sortedRoles,
        'ui_values'       => \@sortedRoleUIValues,
        'set_interactive' => 1,
        'init_with_default'           => 1,
        'interactive_index_selection' => 1,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--host_roles'.

sub getParamHostRoles {
    my ($self, $order, $section) = @_;

    return {
        'order'           => $order,
        'opt'             => 'roles',
        'opt_arg'         => '<role1>[,<role2>]...',
        'type'            => 'string',
        'section'         => $section,
        'value'           => undef,
        'default'         => undef,
        'str'             => 'Host Roles',
        'desc'            => 'Specifies the roles of this host',
        'set_interactive' => 0,
        'hidden'          => 1,
        'persStep'        => $STEP_ADDHOST_CREATE_DIRECTORIES,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--landscape_id'.

sub getParamLandscapeID {
    my ($self, $order, $section) = @_;

    return {
        'order'           => $order,
        'opt'             => 'landscape_id',
        'opt_arg'         => '<id>',
        'type'            => 'string',
        'section'         => $section,
        'value'           => undef,
        'default'         => undef,
        'str'             => 'Landscape ID',
        'desc'            => 'Checks this landscape ID with the ID contained in the installation path',
        'set_interactive' => 0,
        'hidden'          => 1,
        'persStep'        => $STEP_ADDHOST_CREATE_DIRECTORIES,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--group'.

sub getParamStoragePartitionNumber {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order++,
        'opt'               => 'storage_partition',
        'opt_arg'           => '<number>',
        'type'              => 'number',
        'section'           => $section,
        'value'             => undef,
        'str'               => 'Storage Partition Number',
        'desc'              => 'Specifies the storage partition number of this host',
        'init_with_default' => 0,
        'set_interactive'   => 0,
        'skip'              => 1,
        'persStep'          => $STEP_ADDHOST_CREATE_DIRECTORIES,
    };
}

#-------------------------------------------------------------------------------
# Returns an array containing the host role names.

sub getHostRoles {
    return $_[0]->{_hostRoles};
}


#-------------------------------------------------------------------------------
# Checks and sets a host role
#
# Assigns valid host roles to the array $self->{_hostRoles}.
# This subroutine can be called multiple.

sub handleOneHostRole {

    my ($self, $newRole) = @_;

    trim (\$newRole);
    $newRole  = lc($newRole);
    my $rc = 1;

    if (!exists $validHostRoles->{$newRole}) {
        $self->PushError ("Unkown host role '$newRole': only '"
            . join ('\', \'', @{getSortedHostRoleNames()}) ."' are allowed");
        $rc = 0;
    }

    my $hdbInstance  = $self->getOwnInstance();

    if (!$isWin && $validHostRoles->{$newRole}->{isColumnStore}){
        my $sapSys   = $self->{sapSys};
        if (!$self->initVolumePaths ($hdbInstance, $sapSys)) {
            $rc = 0;
        }
        if (!$self->initHardwareCheck ($hdbInstance, $sapSys)) {
            $rc = 0;
        }
    }
    my $shallSkipWorkerGroup = $validHostRoles->{$newRole}->{hasWorkerGroup} ? 0 : 1;
    $self->setSkip ('WorkerGroup', $shallSkipWorkerGroup);

    if ($rc && !$self->isSpecialComponentInstalled($newRole)) {
        $self->appendErrorMessage("Role '$newRole' is invalid: required component is not installed");
        $rc = 0;
    }

    if ($rc) {

        if ($validHostRoles->{$newRole}->{hasPartition}) {
            $self->setSkip('StoragePartitionNumber', 0);
        }

        my $isStandby       = $validHostRoles->{$newRole}->{isStandby};
        $self->{_isStandby} = $isStandby;

        if (defined $self->{_hostRoles}) {

            foreach my $currRole (@{$self->{_hostRoles}}) {

                if ($newRole eq $currRole) {
                    $self->PushError ("Duplicate host role '$newRole'");
                    $rc = 0;
                    last;
                }

                my $currStandby = $validHostRoles->{$currRole}->{isStandby};

                if (( $isStandby && !$currStandby) ||
                    (!$isStandby &&  $currStandby)) {
                    $self->PushError ("Host role '$newRole' is not allowed in"
                                   . " combination with host role '$currRole'");
                    $rc = 0;
                    last;
                }
            }
            if ($rc) {
                push @{$self->{_hostRoles}}, $newRole;
                $self->{_mapHostRoles}->{$newRole} = 1;
            }
        }
        else {
            $self->{_hostRoles} = [$newRole];
            $self->{_mapHostRoles} = {$newRole => 1};
        }

        if (defined $validHostRoles->{$newRole}->{defaultGroup}) {

            $self->setSkip('HostFailoverGroup',0);

            my $paramGroup = $self->{params}->{HostFailoverGroup};
            my $targetDefaultValue = (defined $self->getDefault('HostFailoverGroup'))
                                     ? $validHostRoles->{worker  }->{defaultGroup}
                                     : $validHostRoles->{$newRole}->{defaultGroup};
            $self->setDefault('HostFailoverGroup', $targetDefaultValue);
            $paramGroup->{init_with_default} = 1;
        }
    }

    if ($newRole eq 'standby' && $hdbInstance->isIsolatedMultiDb ()){
            if (!defined $hdbInstance->getTenantDatabases()){
                $rc = 0;
            }
            my $hostname = $self->getValue('HostName');
            my $saveContext = $self->resetMsgLstContext ();
            my $tenantUserCheck = $self->checkLocalTenantDBUsersAndGroups ($hostname);
            my $error = $self->getErrMsgLst ();
            $self->setMsgLstContext ($saveContext);

            if (!defined $tenantUserCheck){
                $self->appendErrorMessage ("Requirements for host role '$newRole' are not met on host '$hostname'.");
                $self->appendErrorMessage ("A '$newRole' host has to be able to take over each $gShortProductNameTenant.", $error);
                $rc = 0;
            }
    }

    if ($rc && $validHostRoles->{$newRole}->{isAccelerator}) {

        $self->setSkip('SystemUser',          0);
        $self->setSkip('SQLSysPasswd',        0);
        $self->setSkip('AcceleratorUser',     0);
        $self->setSkip('AcceleratorPassword', 0);
        $self->setSkip('NoStart');
    }

    if($rc && $validHostRoles->{$newRole}->{isXS2}){
        $rc &= $self->_checkXSSpaceIsolationParameters();
    }

    if (!$rc) {
        delete $self->{_hostRoles};
        delete $self->{_mapHostRoles};
        delete $self->{_isStandby};
        $self->setDefault('HostFailoverGroup', undef);
        $self->{params}->{HostFailoverGroup}->{init_with_default} = 0;
        $self->setSkip('StoragePartitionNumber');
    }

    return $rc;
}

#-------------------------------------------------------------------------------
# If this step is outstanding, 1 is returned and the step written into the
# pers file.

sub isOutstandingAddHostStep {

    my ($self, $step) = @_;

    my $resumeStep = $self->getStep();

    if ($resumeStep < $step) {
        $self->setStep($step);  # ignore write error of pers file
    }
    return ($resumeStep <= $step) ? 1 : 0;
}


#-------------------------------------------------------------------------------
# Overrides the function of SDB::Install::PersistencyMngr
# Each host has its own persistent file.
#
# Returns the filename if addhost was started at this host before, otherwise undef

sub pers_filename{
    return $_[0]->{pers_filename_addhost};
}

#-------------------------------------------------------------------------------
# Returns the name of the name of the current step or undef if not defined.

sub pers_getstepname {
    my ($self, $step) = @_;
    return $ADDHOST_STEP_NAMES[defined $step ? $step : $self->{step}];
}


#-------------------------------------------------------------------------------
# Checks and sets the multiple option '--host_role=<role>'
# Assigns valid host roles to the array $self->{_hostRoles}.

sub setHostRole {

    my ($self, $roleArray) = @_;

    foreach my $currRole (@$roleArray) {
        my $rc = $self->handleOneHostRole($currRole);
        if (!$rc) {
            return $rc;
        }
    }
    $self->{params}->{HostRole}->{value} = $roleArray;
    $self->{params}->{HostRoles}->{value}= join (',', @$roleArray);
    $self->unskipLoadInitialContentParamsIfNecessary(join(',', @$roleArray));
    return 1;
}


#-------------------------------------------------------------------------------
# Checks and sets the option '--host_roles=<role1>[<role2>]...'
# Assigns valid host roles to the array $self->{_hostRoles}.

sub setHostRoles {

    my ($self, $roleList) = @_;

    foreach my $currRole (split (',', $roleList)) {
        my $rc = $self->handleOneHostRole($currRole);
        if (!$rc) {
            return $rc;
        }
    }
    $self->{params}->{HostRoles}->{value} = $roleList;
    $self->setSkip('HostRole');
    $self->unskipLoadInitialContentParamsIfNecessary($roleList);
    return 1;
}


#-------------------------------------------------------------------------------
# Adds an xs_worker or xs_standby if a worker or standby is added and
# the parameter '--autoadd_xs2_roles' is set.

sub tryAddHostAutoAddXS2 {
    my ($self) = @_;

    my $rc = 1;

    if ($self->getValue('AutoAddXS2Roles')) {

        my $instance = $self->getOwnInstance();

        if (defined $instance && $instance->existsXS2Dir()) {

            if (!$self->{_mapHostRoles}->{$gHostRoleXS2Worker} &&
                 $self->{_mapHostRoles}->{$gHostRoleWorker}) {

                 $rc = $self->handleOneHostRole($gHostRoleXS2Worker);
                 $self->unskipLoadInitialContentParamsIfNecessary($gHostRoleXS2Worker) if ($rc);
            }
            elsif (!$self->{_mapHostRoles}->{$gHostRoleXS2Standby} &&
                    $self->{_mapHostRoles}->{$gHostRoleStandby}) {

                 $rc = $self->handleOneHostRole($gHostRoleXS2Standby);
            }
        }
    }
    return $rc;
}

#-------------------------------------------------------------------------------
# Loads the initial content of XS2 if a xs_worker is added
# and xscontroller is running on this host.

sub tryAddHostLoadXS2InitialContent {

    my ($self) = @_;

    if ($self->{_mapHostRoles}->{$gHostRoleXS2Worker}
        &&  $self->getValue('ImportContentXS2')
        && !$self->getValue('NoStart')) {

        my $instance = $self->getOwnInstance(1); # 1 - $no_cache

        if (defined $instance && $instance->hasXS2Controller()) {

            if (!$self->loadInitialContentXS2()) {
                return undef;
            }
        }
    }
    return 1;
}


#===============================================================================
# import check functions from SDB::Install::Configuration::NewDB

sub checkSystemRequirements;
*checkSystemRequirements = \&SDB::Install::Configuration::NewDB::checkSystemRequirements;

sub runHWCheckScript;
*runHWCheckScript = \&SDB::Install::Configuration::NewDB::runHWCheckScript;

sub isAutoInitializeServicesApplicableParameterCallback{
    my($self,$paramId,$valKey,$roleValue) = @_;
    $self->{_requiresTenantUserCredentials} = 0;
    my $hdbInstance = $self->getOwnInstance();
    return 0 if (!defined $hdbInstance);
    if($paramId eq 'HostRoles'){
        foreach my $currRole (split (',', $roleValue)) {
            return 1 if $self->_doesRoleRequireAutoInitialize($currRole);
        }
    }
    if($paramId eq 'HostRole'){
        foreach my $currRole (@$roleValue) {
            return 1 if $self->_doesRoleRequireAutoInitialize($currRole);
        }
    }
    return 0;
}

sub _doesRoleRequireAutoInitialize {
    my($self, $role) = @_;
    my $isDbRole = $validHostRoles->{$role}->{isDbRole};
    my $isXs2Role = $validHostRoles->{$role}->{isXS2};
    my $isStandbyRole = $validHostRoles->{$role}->{isStandby};
    my $isExtendedStorage = $validHostRoles->{$role}->{isExtendedStorage};
    if ($isExtendedStorage || $isDbRole || $isXs2Role || $isStandbyRole){
        return 0;
    }
    my $componentDir;
    my $componentDirName;
    my $hdbInstance = $self->getOwnInstance();
    if ($validHostRoles->{$role}->{isStreaming}) {
        $componentDir = $hdbInstance->getStreamingLcmDir();
        $componentDirName = $gDirNameStreaming;
    }
    elsif ($validHostRoles->{$role}->{isExtendedStorage}) {
        $componentDir = $hdbInstance->getExtendedStorageLcmDir();
        $componentDirName = $gDirNameEs;
    }
    elsif ($validHostRoles->{$role}->{isAccelerator} && !$validHostRoles->{$role}->{isStandby}) {
        $componentDir = $hdbInstance->getAcceleratorLcmDir();
        $componentDirName = $gDirNameAccelerator;
    }
    elsif ($validHostRoles->{$role}->{isRDSync}) {
        $componentDir = $hdbInstance->getRemoteDataSyncLcmDir();
        $componentDirName = $gDirNameRDSync;
    }
    if(defined $componentDir && -d $componentDir){
        my $manifest = SDB::Install::Manifest->new($hdbInstance->getInstalledComponentManifestPath($componentDirName));
        if($manifest->isAutoInitializeServiceSupported()){
            $self->getMsgLst()->addMessage ("Component directory $componentDirName found, which requires auto initializing of services.");
            if(!$validHostRoles->{$role}->{isExtendedStorage}){
                $self->{_requiresTenantUserCredentials} = 1; 
            }
            return 1;
        }
    }
    return 0;
}

sub isSystemInCompatibilityMode{
    my $self = shift;
    return $self->getValue('AddHostCalledInCompatiblityMode');
}

sub checkRequiresTenantUserCredentials{
    my ($self) = @_;
    return $self->{_requiresTenantUserCredentials};
}

sub shouldWarnIfCalledStandalone{
    my ($self) = @_;
    return $self->_calledStandaloneCriterion();
}

sub getExeName{
    return "hdbaddhost";
}

sub getResidentHdblcmPath{
    my ($self) = @_;
    return $self->_constructResidentHdblcmPath();
}

1;
