package SDB::Install::Configuration::AnyModifyConfig;

use strict;

use SDB::Install::Configuration::AnyMultiHostConfig qw (@listenInterfaceValidValues
                                                        @listenInterfaceUIValues);
use SDB::Install::Configuration::AnyRegisterRename;
use SDB::Install::RemoteHosts;
use SDB::Install::Globals qw ($gProductNameSystem
                              $gShortProductNameXS2);
use SDB::Install::SysVars qw ($path_separator);
use Cwd qw(abs_path);

our @ISA = qw (SDB::Install::Configuration::AnyRegisterRename);


#-------------------------------------------------------------------------------
# Constructor
sub new {
    my $self = shift->SUPER::new (@_);

    return $self;
}


#-------------------------------------------------------------------------------
# Checks the option 'addhosts'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Returns int retCode

sub checkAddHosts {
    my ($self, $value, $msglst_arg, $notHandleListenInterfaceParam) = @_;
    my $sapmntPath = $self->getValue('Target');

    if (defined $sapmntPath) {
        my $sid = $self->getSID();
        my $absoluteSapmntPath = abs_path($sapmntPath);
        my $usrSapSidPattern = quotemeta( join($path_separator, '', 'usr', 'sap', $sid, '') ); # e.g. /usr/sap/<SID>/

        if ($absoluteSapmntPath =~ "^$usrSapSidPattern?\$") { # Treat the trailing '/' as optional
            $self->PushError("Installation path '$absoluteSapmntPath' is not allowed for a multiple host system");
            return 0;
        }
    }

    # checkSID has tried to create the class RemoteHosts before
    # in order to check additional hosts by the following subroutine

    if (!$self->handleAddOrRemoveHosts($value, 'AddHosts')) {
        return 0;
    }

    if (!defined $self->getAdditionalRemoteHosts()) {
        return 1;
    }

	if (!$notHandleListenInterfaceParam){
        $self->restrictListenInterfaceMultiHost();
	}
    $self->setSkip('Password', 0);

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the option 'internal_network'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Returns int retCode

sub checkInternalNetwork {
    my ($self, $value) = @_;

    if (!defined $value) {
        return 1;
    }
    $self->setSkip('Password', 0);
    return $self->SUPER::checkInternalNetwork($value);
}

#-------------------------------------------------------------------------------
# Checks the option 'removehosts' and enables the options 'keep_user'
# and 'keep_user_home_dir'.
#
# This subroutine calls setRemovingRemoteHosts.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Returns int retCode

sub checkRemoveHosts {

    my ($self, $value, $msglst_arg, $doNotFailOnLocalHost) = @_;

    # checkSID has tried to create the class RemoteHosts before
    if (!defined $self->getRemoteHosts) {
        $self->PushError ('Remote hosts do not exist');
        return 0;
    }

    if (!$self->handleAddOrRemoveHosts($value, 'RemoveHosts', $doNotFailOnLocalHost)) {
        return 0;
    }

    my $removeHosts = $self->getRemovingRemoteHosts();

    if (!defined $removeHosts) {
        return 1;
    }

    if (!$self->isUseSAPHostagent()) {
        if (!$removeHosts->copyConnections($self->getRemoteHosts(), $self)) {
            return 0;
        }
    }

    $self->setSkip('KeepUser',        0);
    $self->setSkip('KeepUserHomeDir', 0);

    if ($self->getNumberOfDefinedHosts() == 1) {
        # allow listen interface 'local'
        my $interfaceParam              = $self->{params}->{ListenInterface};
        $interfaceParam->{valid_values} = \@listenInterfaceValidValues;
        $interfaceParam->{ui_values}    = \@listenInterfaceUIValues;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the root user name in case of a  distributed system.
# Furthermore this method tries to use or create an authorization key
# to access remote hosts.
# An authoization key can only be created if all hosts provide the same
# root user name.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkRootUser {

	my ($self, $remoteRootUser) = @_;
	return $self->tryRemoteKeyAuthorization($remoteRootUser);
}


#-------------------------------------------------------------------------------
# Checks the specified SID
#
# Parameter: String $sid
#
# Returns int retCode

sub checkSID {

    my ($self, $sid, $msglst_arg) = @_;

    if (!$self->isValidSID($sid)) {
    	return 0;
    }
    if (!$self->initRemoteHostsForInstance()) {
    	return 0;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the path consisting of the given target directory and SID
#
# Parameter: string $targetDirectory
#
# Returns int retCode

sub checkTarget {
	my ($self, $path) = @_;

    if (!-d $path){
        $self->PushError ("Directory '$path' does not exist");
        return 0;
    }
	return 1;
}


#-------------------------------------------------------------------------------
# Returns the new listen interface depending on the specified parameter,
# the existing listen interface and additional hosts.

sub detectNewListenInterfaceType {

    my ($self, $paramValue, $existingType) = @_;

    my $wantedType = (defined $paramValue) ? $paramValue : $existingType;

    if (($wantedType eq 'local') && ($existingType eq 'local')
        && defined $self->getAdditionalRemoteHosts()) {

        $wantedType = 'global';
    }

    if (($wantedType eq 'local') && ($existingType ne 'local')
        &&
        (defined $self->getRemoteHosts() ||
         defined $self->getAdditionalRemoteHosts())) {

        $wantedType = $existingType;
    }

    return $wantedType;
}


#-------------------------------------------------------------------------------
# Returns a reference to a string array specifying hardware and platform checks

sub getHiddenIgnoreValues {
    return [qw (check_hardware check_platform check_hostname)];
}


#-------------------------------------------------------------------------------
# Returns the number of all hosts
# (including additional hosts, without hosts to be removed)

sub getNumberOfDefinedHosts {
    my ($self) = @_;

    my $cntHosts    = 1;
    my $remoteHosts = $self->getRemoteHosts();

    if (defined $remoteHosts) {
        my $add    = $self->getAdditionalRemoteHosts();
        my $remove = $self->getRemovingRemoteHosts();
        my $cntAdd = (defined $add)    ? $add->getNumberOfHosts()    : 0;
        my $cntDel = (defined $remove) ? $remove->getNumberOfHosts() : 0;
        $cntHosts  = 1 + $remoteHosts->getNumberOfHosts() + $cntAdd - $cntDel;
    }

    return $cntHosts;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--add_local_roles'.

sub getParamAddLocalRoles {
    my ($self, $order, $section) = @_;

    return {
        'order'           => $order,
        'opt'             => 'add_local_roles',
        'hostctrl_opt'    => 'ADD_LOCAL_ROLES',
        'opt_arg'         => '<role1>[,<role2>]...',
        'type'            => 'string',
        'section'         => $section,
        'value'           => undef,
        'default'         => undef,
        'str'             => 'Additional Local Host Roles',
        'desc'            => 'Specifies additional roles for this host only',
        'additional_desc' => $self->getHostRoleDescription(1),
        'set_interactive' => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--add_roles.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamAddRoles{
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'add_roles',
        'opt_arg'           => '<host1>[,<host2>]...',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Additional Host Roles',
        'desc'              => "Specifies additional host roles for local and remote hosts",
        'additional_desc'   => "Syntax of a host entry:\n"
                             . "<host_name>:role=<role1>[:role=<role2>]...\n"
                             . $self->getHostRoleDescription(1),
        'init_with_default' => 0,
        'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
sub getParamLoadInitialContentXS2 {
    my ($self, $order, $section) = @_;

    my $str = "Loads the initial content of $gShortProductNameXS2";

    return {
        'order'             => $order,
        'opt'               => 'load_initial_xs_content',
        'hostctrl_opt'      => 'LOAD_INITIAL_XS_CONTENT',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => $str,
        'desc'              => $str . ' if xs worker is already installed',
        'init_with_default' => 0,
        'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'removehosts'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamRemoveHosts {
    my ($self, $order, $section) = @_;

    my $optArg = '<hostname1>[,<hostname2>]...';

    return {
        'order' => $order,
        'opt'               => 'removehosts',
        'opt_arg'           => $optArg,
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Remove Hosts',
        'interactive_str'   => 'comma-separated list of hosts to remove',
        'desc'              => "Specifies remote hosts to remove from $gProductNameSystem",
        'additional_desc'   => 'Hosts are removed before new hosts are added.',
        'init_with_default' => 0,
        'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--remove_roles.

sub getParamRemoveRoles{
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'remove_roles',
        'hostctrl_opt'      => 'REMOVE_ROLES',
        'opt_arg'           => '<host1>[,<host2>]...',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Remove Roles',
        'desc'              => "Removes roles from local and remote hosts",
        'additional_desc'   => "Syntax of a host entry:\n"
                             . "<host_name>:role=<role1>[:role=<role2>]...\n"
                             . $self->getHostRoleDescription(1,0,1),
        'init_with_default' => 0,
        'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns a reference to a string array containing valid timeout arguments

sub getTimeoutValues {
    return [qw (start_instance start_service)];
}

sub isValidSID{
	my($self,$sid) = @_;

	my $instance = $self->getOwnInstance();

	if (!defined $instance) {
		$self->PushError("$gProductNameSystem '$sid' does not exist");
		return 0;
	}

	my $sapSys = $self->getSAPSystem();
	if (defined $sapSys) {
		$self->changePasswordStr($sapSys->getUser()->getSidAdmName());
	}

	$self->showSystemProperties();

	return 1;
}

sub initRemoteHostsForInstance{
	my ($self) = @_;

	my $instance = $self->getOwnInstance();

	if ($instance->exists_remote_host()) {

		$self->restrictListenInterfaceMultiHost();
		my $scope = $self->getValue('ScopeModify');
		if (!$self->{isSlave} && (!defined $scope || ($scope eq 'system'))) {
			if (!$self->initRemoteHosts()) {
				return undef;
			}
			$self->enableMultipleHostParameters(1); # 1 - $useSidadmForHostctrl
		}
	}

	return 1;
}

1;
