package SDB::Install::Configuration::AnyRegisterRename;

use strict;

use SDB::Install::Configuration qw ($bool_true_pattern);
use SDB::Install::Configuration::AnyMultiHostConfig;
use SAPDB::Install::Hostname qw(hostname);
use SDB::Install::Globals    qw($gProductName
                                $gProductNameSystem
                                $gSapsysGroupName);
use SDB::Install::System     qw($hostname_regex nslookup isAdmin);
use SDB::Install::SysVars    qw($isWin $path_separator);
use SDB::Install::User;
use SDB::Common::Utils qw(createXSSpaceSAPUserName createXSSpaceProdUserName createSysAdminUserName createUsername checkOsUserID);

our @ISA = qw (SDB::Install::Configuration::AnyMultiHostConfig);


# Hash keys of this class: 'localHostName'    (string)
#                          'isSlave'          (int 1 / undef)
#                          'ownInstance'      (SDB::Install::SAPInstance::TrexInstance)
#                          'params'           (hash)
#                          'sapSys'           (SDB::Install::SAPSystem)


#-------------------------------------------------------------------------------
# Constructor
sub new {
    my $self = shift->SUPER::new (@_);

    return $self;
}

#-------------------------------------------------------------------------------
# Adds the content of HostMap to the specified message list.

sub addMsgHostMap {

    my ($self, $msglst) = @_;

    my $lines = $self->getHostMapParamInfo();

    if (defined $lines) {
        foreach my $currLine (@$lines) {
            $msglst->AddMessage($currLine);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the remote SID, instance number, sidadm properties
# and updates $self->{existingSidAdm}.
#
# COLLECTED HOST INFO:
#                           instNr         host        SID  instType  landscapeID
# $self->{remoteInstances}->{'01'}->[0]->['lu123456', 'DB1', 'HDB',  '51ddf0d6-83bd...']
#                                   [1]->['lu222222', 'DB1', 'HDB',  '51ddf0d6-83bd...']
#                           {'06'}->[0]->['lu222222', 'DB6', 'HDB',  '51d3f1ef-84b5...']
#
#                      userID         host        user     gid
# $self->{remoteUids}->{1003}->[0]->['lu222222', 'db6adm', 79]
#                      {1010}->[0]->['lu123456', 'db1adm', 79]
#                              [1]->['lu222222', 'db1adm', 79]

sub checkCollectedHostInfo {
    my ($self, $ownSID, $ownInstanceNumber) = @_;

    my $IDX_HOST          = 0;
    my $IDX_SID           = 1;
    my $IDX_LANDSCAPE_ID  = 3;

    my $retcode        = 1;
    my $ownUserName    = $self->getSysAdminUserName($ownSID);
    my $instance       = $self->getOwnInstance();
    my $ownLandscapeID = (defined $instance) ? $instance->getLandscapeID()
                                             : undef;

    foreach my $currInstanceNumber (keys %{$self->{remoteInstances}}) {

        my $instHostList = $self->{remoteInstances}->{$currInstanceNumber};

        if (!defined $instHostList) {
            next;
        }

        foreach my $currHostInfo (@$instHostList) {

            if (defined $ownLandscapeID
                && ($ownLandscapeID eq $currHostInfo->[$IDX_LANDSCAPE_ID])) {
                next; # instance is part of own SAP HANA system
            }

            if ($ownSID eq $currHostInfo->[$IDX_SID]) {
                $self->PushError("SAP system ID '$ownSID' is already in use on"
                                 . " host '$currHostInfo->[$IDX_HOST]'");
                $retcode = 0;
            }

            if (defined $ownInstanceNumber
                && ($ownInstanceNumber eq $currInstanceNumber)) {
                $self->PushError("Instance number '$ownInstanceNumber' is"
                    . " already in use on host '$currHostInfo->[$IDX_HOST]'");
                $retcode = 0;
            }
        }
    }

    my $remoteUids = $self->{remoteUids};

    if (!$retcode || !defined $remoteUids) {
        return $retcode;
    }

    foreach my $uid (keys %$remoteUids) {

        foreach my $userProperties (@{$remoteUids->{$uid}}) {

            if ($ownUserName eq $userProperties->[1]) {

                if (defined $self->{existingSidAdm}) {
                    $retcode =
                        $self->compareExistingSidAdm ($uid, $userProperties);
                }
                else {
                    $self->{existingSidAdm} = {'uid'  => $uid,
                                               'prop' => $userProperties};

                    $retcode = $self->compareExistingSidAdmWithCfg
                                                  ($self->getValue('InitUser'));
                }
            }
        }
    }

    if (!defined $self->checkMultiDBIsolationPrerequisitesRemote()){
        $retcode = 0;
    }

    return $retcode;
}


#-------------------------------------------------------------------------------
# Checks one host map entry and returns 1
# if the new host is unique in the host map and the new host is a valid hostname.
#
# Parameters: string $originHost
#             string $newHost
#
# Returns int retCode

sub checkHostMap {

    my ($self, $originHost, $newHost, $dontFailOnAlreadyDefinedHost) = @_;

    my $param        = $self->{params}->{HostMap};
    my $lcOriginHost = lc($originHost);
    my $lcNewHost    = lc($newHost);

    if (!defined $param->{default_map}->{$lcOriginHost}) {
        $self->PushError ("Host name '$originHost' is not part of the system");
        return 0;
    }

    if ($lcOriginHost ne $lcNewHost) {
        foreach my $currNewHost (values %{$param->{value}}) {
            if (lc($currNewHost) eq $lcNewHost) {              
                if($dontFailOnAlreadyDefinedHost) {
                    return 1;   
                }
                $self->PushError("Host name '$newHost' is already defined");
                return 0;
            }
        }
    }
    
    my $instance = $self->getOwnInstance();

    if (($lcNewHost eq lc(hostname())) || $instance->isHostLocal($lcNewHost)) {
        $self->{localHostName} = $lcNewHost;
    }

    if (!$self->verifyNewHostName ($newHost, 1)){
        if ($self->getIgnore ('check_hostname')){
            $self->getMsgLst()->addMessage ("Ignoring error due to command line switch '--ignore=check_hostname'");
            $self->resetError ();
        }
        else{
            return 0;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks all entries of the host map and returns 1
# if all new values are unique and only one local host name exists.
#
# The local host name is assigned to the hash 'localHostName'.

sub checkEntriesHostMap {

    my ($self) = @_;

    my $instance      = $self->getOwnInstance();
    my %newToOldHostMap = ();
    my $oldToNewHostMap = $self->getValueOrBatchValue('HostMap');
    my $oldLocal      = undef;

    my $standardLocalHost = (defined $self->{localHostName})
                            ? $self->{localHostName}
                            : $instance->get_host();

    $standardLocalHost = lc(hostname()) if (!defined $standardLocalHost);

    my ($oldHost, $newHost);
    my ($lastKey, $lastValue);
    while (($oldHost, $newHost) = each %{$oldToNewHostMap}) {

        my $lcNewHost = lc($newHost);

        if (exists $newToOldHostMap{$lcNewHost}) {

            $self->PushError("Duplicate host name '$newHost' for source hosts '".
                 "$newToOldHostMap{$lcNewHost}' and '$oldHost'");
            return 0;
        }
        else {
            $newToOldHostMap{$lcNewHost} = $oldHost;
        }

        if (($lcNewHost eq $standardLocalHost)
                                        || $instance->isHostLocal($lcNewHost)) {

            if (defined $oldLocal) {

                $self->PushError
                    ("Local host name '$newHost' for source host '$oldHost' is "
                    ."already defined as alias '"
                    .$self->{localHostName} ."' for source host '$oldLocal'");

                return 0;
            }
            else {
                $oldLocal              = $oldHost;
                $self->{localHostName} = $lcNewHost;
            }
        }
        elsif (!$self->isHostAccessible($newHost)) {
            return 0;
        }
        $lastKey = $oldHost;
        $lastValue = $newHost;
    }
    
    if (!defined $self->{localHostName}) {
    	my $value = $self->{params}->{HostMap}->{value}->{$standardLocalHost};
        if (defined $value and $value ne "" ) {
        	$newHost = $value; 	
        }
        $self->{params}->{HostMap}->{value}->{$standardLocalHost} = undef;
        $self->PushError("Local host name '$lastValue' is not specified");
        return 0;
    }

    if (!$instance->hasLocalHost()) {
        $instance->setLocalHost($oldLocal);
    }

    delete $instance->{_mdcDatabases};

    if ($instance->isIsolatedMultiDb ()){
        if (!defined $self->checkMultiDBIsolationPrerequisitesLocal ()){
            return 0;
        }
    }

    if (! $self->{handleInitParamsCalled} && ! $self->handleInitParams()) {
        return 0;
    }

    $self->setDefault('XsDomainName', $self->getDefaultXsDomainName()) if ! $self->isSkipped('XsDomainName');

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the option '--nostart'

sub checkNoStart {

    my ($self, $value) = @_;

    if ($value =~ /$bool_true_pattern/i){
        if ($self->getValue('ChangeSystemPasswd')) {
            $self->PushError ("Parameter '" . $self->getOpt('NoStart')
                    . "' cannot be applied together with '"
                    . $self->getOpt('ChangeSystemPasswd') . "'");
            return 0;
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the layout of the target password against the password policy

sub checkTrgSQLSysPasswd {

    my ($self, $newPassword) = @_;

    return 1 if ($newPassword eq 'manager');
    my $msglst = new SDB::Install::MsgLst ();
    my $rc = $self->complySqlPasswordPolicy ($newPassword, 1, $msglst, 'TrgSQLSysPasswd');
    if (!$rc){
        my $paramName = $self->{params}->{TrgSQLSysPasswd}->{str};
        $self->PushError ("$paramName is invalid", $msglst);
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Compares the given properties with those of the existing sidadm.
# Returns 1 if all properties match or if a sidadm does not exist.

sub compareExistingSidAdm {

    my ($self, $otherUID, $otherProp, $isOtherCfg, $skipErrMsg) = @_;

    my $own = $self->{existingSidAdm};

    if (!defined $own) {
        return 1;
    }

    my $IDX_HOST = 0;
    my $IDX_USER = 1;
    my $IDX_GID  = 2;

    my $ownInfo =
        "of '$own->{prop}->[$IDX_USER]' on host '$own->{prop}->[$IDX_HOST]'";

    my $otherInfo = ($isOtherCfg)
                    ? 'of the SAP HANA configuration'
                    : "of the administrator on host '$otherProp->[$IDX_HOST]'";

    if ($own->{uid} != $otherUID) {
        if (!$skipErrMsg) {
            $self->PushError("The user ID '$own->{uid}' $ownInfo does not match"
                            . " the user ID '$otherUID' $otherInfo.");
        }
        return 0;
    }

    if ($own->{prop}->[$IDX_GID] ne $otherProp->[$IDX_GID]) {
        if (!$skipErrMsg) {
            $self->PushError("The group ID '$own->{prop}->[$IDX_GID]' $ownInfo"
                            . " does not match the group ID"
                            . " '$otherProp->[$IDX_GID]' $otherInfo.");
        }
        return 0;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# An error is thrown if the uid or gid of an existing sidadm does not match
# the ownership of the files.

sub compareExistingSidAdmWithCfg {

    my ($self, $ignoreMismatches) = @_;

    my $existingSidAdm = $self->{existingSidAdm};

    if (!defined $existingSidAdm) {
        return 1;
    }

    my $userCfg = $self->{sapSys}->getUserConfig();

    if (!defined $userCfg){
        $self->AddError ('Cannot get user configuration', $self->{sapSys});
        return undef;
    }

    my $cfgUsername      = $self->{sapSys}->getUser()->getSidAdmName();
    my $existingUsername = $existingSidAdm->{prop}->[1];

    if ($existingUsername ne $cfgUsername) {
        return 1; # in case of rename, users are different
    }

    my $hostname = defined ($self->{localHostName}) ? $self->{localHostName}
                                                    : hostname();
    my $userCfgProp = [$hostname,
                       $existingUsername,
                       $userCfg->{sapsys_groupid}];

    my $rc = $self->compareExistingSidAdm ($userCfg->{sidadm_id},
                                           $userCfgProp,
                                           1,                  # $isOtherCfg
                                           $ignoreMismatches); # $skipErrMsg

    if ($rc) {
        $self->{params}->{HomeDir}->{value} = $userCfg->{sidadm_home};
        $self->{params}->{Shell  }->{value} = $userCfg->{sidadm_shell};
        $self->setSkip('HomeDir');
        $self->setSkip('Shell');
    }

    if (!$rc && $ignoreMismatches) {
        $rc = 1;
        $existingSidAdm->{changeUserCfg} = 1;
        $self->setDefault('Shell', $userCfg->{sidadm_shell});
    }

    if ($rc) {
        $self->{params}->{UID}->{value} = $existingSidAdm->{uid};
        $self->{params}->{GID}->{value} = $existingSidAdm->{prop}->[2];
        $self->setSkip('UID');
        $self->setSkip('GID');
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array of lines containing the host map parameters.
# Returns undef in case of single host.

sub getHostMapParamInfo {

    my ($self) = @_;

    my $param = $self->{params}->{HostMap};

    if (scalar @{$param->{origin_values}} <= 1) {
        return undef;
    }

    my @info = ('Host Names:');

    foreach my $originHost (@{$param->{origin_values}}) { # already sorted

        my $lcOriginHost = lc($originHost);
        my $newHost      = undef;
        my $newHostType  = '';

        if (defined $param->{value}) {

            $newHost = $param->{value}->{$lcOriginHost};
        }

        if (!defined $newHost && defined $param->{batchValue}) {

            $newHost = $param->{batchValue}->{$lcOriginHost};
        }

        if (!defined $newHost && defined $param->{default_map}) {

            $newHost     = $param->{default_map}->{$lcOriginHost};
            $newHostType = 'default ';
        }

        if (defined $newHost && (lc($newHost) eq $lcOriginHost)) {
            $newHost = undef;
        }

        if (defined $newHost) {
            push (@info, "   $originHost (source host name), "
                         . $newHostType . "target host name: $newHost");
        }
        else {
            push (@info, "   $originHost");
        }
    }
    return \@info;
}


#-------------------------------------------------------------------------------
sub getParamChangeSystemPasswd {

    my ($self, $order, $section, $interactive, $changeOnly, $constraint) = @_;

    my $opt  = 'change_system_user_password';
    my $desc = 'Changes the password of the database user';

    if ($changeOnly) {
        $opt  .= '_only';
        $desc .= ' without registering the system';
    }

    return {
        'order'     => $order,
        'opt'       => $opt,
        'type'      => 'boolean',
        'section'   => $section,
        'value'     => undef,
        'default'   => 0,
        'str'       => 'Change the password of the Database user',
        'str_templ' => 'Change the password of the Database user (%s)',
        'desc'      => $desc,
        'set_interactive'         => $interactive,
        'init_with_default'       => 1,
        'console_omit_word_Enter' => 1,
        'constraint'              => $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'hostmap'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'register only')

sub getParamHostMap {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'hostmap',
            'short_opt'         => 'H',
            'opt_arg_switch'    => '<old_host_name%s>=<new_host_name%s>',
            'opt_arg'           => '<old>=<new>',
            'type'              => 'mapping_list',
            'section'           => $section,
            'origin_values'     => undef,  # array    (all source host names)
            'value'             => undef,  # hash map (key  : source  host name,
                                           #           value: new     host name)
            'default_map'       => undef,  # hash map (key  : source  host name,
                                           #           value: default host name)
            'default'           => undef,
            'str'               => 'Host Name',
            'pluralStr'         => 'Host Names',
            'str_templ'         => 'Target Host Name for Source Host \'%s\'',
            'desc'              => 'Renames one host',
            'additional_desc'   => "The parameter has to be specified for each renamed host:\n"
                                 . '-H <old_host_name1>=<new_host_name1> -H <old_host_name2>=<new_host_name2> ...',
            'set_interactive'   => 1,
            'init_with_default' => 1,
            'constraint'        => $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'InitUser'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'unregister only')

sub getParamInitUser {

    my ($self, $order, $section, $constraint, $userAttr, $aliasOptions) = @_;

    my $attr = (defined $userAttr) ? "$userAttr " : '';
    my $str  = "Use Existing $attr".'System Administrator';
    my $desc = 'Uses the properties of an existing ' . lc($attr)
             . 'System Administrator';

    return {'order'             => $order,
            'opt'               => 'init_user',
            (defined $aliasOptions ?
              ('alias_opts'     => $aliasOptions) :
              ()
            ),
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'str'               => $str,
            'str_templ'         => $str . ' (%s)',
            'desc'              => $desc,
            'default'           => 0,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'InitUserHomeDir'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'unregister only')

sub getParamInitUserHomeDir {

    my ($self, $order, $section, $constraint, $userAttr, $aliasOptions) = @_;

    my $attr = (defined $userAttr) ? "$userAttr " : '';
    my $str  = "Init Existing Home Directory of $attr".'System Administrator';
    my $desc = 'Initializes an existing home directory of the ' . lc($attr)
             . 'System Administrator';

    return {'order'             => $order,
            'opt'               => 'init_user_home_dir',
            (defined $aliasOptions ?
              ('alias_opts'     => $aliasOptions) :
               ()
            ),
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'str'               => $str,
            'str_templ'         => $str . ' (%s)',
            'desc'              => $desc,
            'default'           => 0,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'revert_to_snapshot'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'register only')

sub getParamRevertToSnapshot {

    my ($self, $order, $section, $constraint) = @_;
    my %paramRevertToSnapshot = (
            'order'   => $order,
            'opt'     => 'revert_to_snapshot',
            'type'    => 'boolean',
            'section' => $section,
            'value'   => undef,
            'str'     => "Resets $gProductNameSystem to previously created snapshot",
            'set_interactive' => 0,
            'constraint'        => $constraint,
            );

    return \%paramRevertToSnapshot;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'sr_cleanup'.
#
# Parameter: int    $order
#            string $section
#            string $constraint (e.g. 'register only')

sub getParamTrgSysReplicationCleanUp {

    my ($self, $order, $section, $constraint) = @_;
    my %paramTrgSysReplicationCleanUp = (
            'order'   => $order,
            'opt'     => 'sr_cleanup',
            'type'    => 'boolean',
            'section' => $section,
            'value'   => undef,
            'default' => 0,
            'str'     => "Clean up system replication configuration on the target system",
            'set_interactive' => 0,
            'constraint'      => $constraint,
            );

    return \%paramTrgSysReplicationCleanUp;
}

#-------------------------------------------------------------------------------
sub getParamSrcSQLSysPasswd {

    my ($self, $order, $section, $constraint) = @_;

    return {
        'order'     => $order,
        'opt'       => 'source_system_user_password',
        'type'      => 'passwd',
        'opt_arg'   => '<database_password>',
        'section'   => $section,
        'value'     => undef,
        'default'   => undef,
        'str'       => 'Source Database User Password',
        'str_templ' => 'Source Database User (%s) Password',
        'init_with_default' => 0,
        'log_value' => sub {'***';},
        'skip'      => 1,
        'mandatory' => 1,
        'constraint'=> $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'StartOnly'.

sub getParamStartOnly {
    my ($self, $order, $section) = @_;

    return {'order'             => $order,
            'opt'               => 'start_only',
            'hostctrl_opt'      => 'START_ONLY',
            'type'              => 'string',
            'opt_arg_switch'    => 'all_instances|system|system_only',
            'opt_arg'           => '<scope>',
            'valid_values'      => ['instance', 'all_instances', 'system', 'system_only'],
            'section'           => $section,
            'value'             => undef,
            'default'           => 'instance',
            'str'               => "Start $gProductName",
            'desc'              => "Starts services (sapstartsrv) and $gProductName instances without registering the system",
            'additional_desc'   => "Scope: 'instance'      - starts the local service and the local $gProductName instance\n"
                                 . "       'all_instances' - starts all services and all $gProductName instances in parallel\n"
                                 . "       'system'        - starts all services and the $gProductName system\n"
                                 . "       'system_only'   - starts the $gProductName system\n",
            'init_with_default' => 0,
            'set_interactive'   => 0,
            'skip'              => 1,
            'leaveEmptyInConfigDump' => 1,
    }
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'StopOnly'.

sub getParamStopOnly {
    my ($self, $order, $section) = @_;

    return {'order'             => $order,
            'opt'               => 'stop_only',
            'hostctrl_opt'      => 'STOP_ONLY',
            'type'              => 'string',
            'opt_arg_switch'    => 'system|system_only',
            'opt_arg'           => '<scope>',
            'valid_values'      => ['instance', 'system', 'system_only'],
            'section'           => $section,
            'value'             => undef,
            'default'           => 'instance',
            'str'               => "Stop $gProductName",
            'desc'              => "Stops $gProductName instances and services (sapstartsrv) without registering the system",
            'additional_desc'   => "Scope: 'instance'    - stops the local $gProductName instance and the local service\n"
                                 . "       'system'      - stops the $gProductName system and all services\n"
                                 . "       'system_only' - stops the $gProductName system\n",
            'init_with_default' => 0,
            'set_interactive'   => 0,
            'skip'              => 1,
            'leaveEmptyInConfigDump' => 1,
    }
}


#-------------------------------------------------------------------------------
sub getParamTrgSQLSysPasswd {

    my ($self, $order, $section, $constraint) = @_;

    return {
        'order'     => $order,
        'opt'       => 'target_system_user_password',
        'type'      => 'initial_passwd',
        'opt_arg'   => '<database_password>',
        'section'   => $section,
        'value'     => undef,
        'default'   => undef,
        'str'       => 'Target Database User Password',
        'str_templ' => 'Target Database User (%s) Password',
        'init_with_default' => 0,
        'log_value' => sub {'***';},
        'skip'      => 1,
        'mandatory' => 0,
        'constraint'=> $constraint,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter 'Unregister',
# 'StartOnly' or 'StopOnly'.

sub getParamUnregister {

    my ($self, $order, $section) = @_;

    return {'order'             => $order,
            'opt'               => 'unregister',
            'type'              => 'string',
            'opt_arg_switch'    => 'system',
            'opt_arg'           => 'instance|=system',
            'section'           => $section,
            'value'             => undef,
            'default'           => 'instance',
            'str'               => "Unregisters $gProductName",
            'desc'              => "Unregisters $gProductName instance (local host only) or system (all hosts)",
            'init_with_default' => 0,
            'set_interactive'   => 0,
            'skip'              => 1,
            'leaveEmptyInConfigDump' => 1,
    };
}

sub requiresChangeOfInternalNetwork {
	return 0;
}

#-------------------------------------------------------------------------------
# Returns true, if a new host name is contained in the host map.

sub isAnyHostChanged {
    my ($self) = @_;

    my $paramHostMap = $self->{params}->{HostMap};

    if (defined $paramHostMap) {

        foreach my $sourceHost (keys %{$paramHostMap->{value}}) {

            my $targetHost = $paramHostMap->{value}->{$sourceHost};
            if (defined $targetHost && (lc($sourceHost) ne lc($targetHost))) {
                return 1;
            }
        }
    }

    return 0;
}


#-------------------------------------------------------------------------------
# Shows the content of HostMap

sub showHostMap {

    my $lines = $_[0]->getHostMapParamInfo();

    if (defined $lines) {
        print "\n";
        foreach my $currLine (@$lines) {
            print "$currLine\n";
        }
        print "\n";
    }
    return 1;
}


#-------------------------------------------------------------------------------
# This method is called from handleInitParams and initializes succeding
# parameters (hash 'params') on slave hosts.
#
# Parameter: hash $pers_data  *)
#
#    *) contains the content of the persistent file created for slaves by
#       the master (filename: '.../HDB<nr>/<host>/<persistentFileName>')
#
# Returns int retCode

sub handleSlaveInitParams {

    my ($self, $pers_data) = @_;
    $self->{isSlave} = 1;
    $self->showSystemProperties('Target');
    if (defined $self->{params}->{ListenInterface}){
        $self->{params}->{ListenInterface}->{value}      = undef;
        $self->{params}->{ListenInterface}->{batchValue} = undef;
        $self->setSkip('ListenInterface');
    }
    if (defined $self->{params}->{InternalNetwork}){
        $self->{params}->{InternalNetwork}->{value}      = undef;
        $self->{params}->{InternalNetwork}->{batchValue} = undef;
        $self->setSkip('InternalNetwork');
    }
    if (defined $self->{params}->{SystemUsage}){
        $self->{params}->{SystemUsage}->{value}      = undef;
        $self->{params}->{SystemUsage}->{batchValue} = undef;
        $self->setSkip('SystemUsage');
    }
    my $instance = $self->getOwnInstance(1);
    my $listenInterface = $instance->getTrexNetListenInterface ();
    if (defined $listenInterface){
        if ($listenInterface eq '.local' && !$self->isSlaveUpgrade ()){
            $self->setErrorMessage ("Slave operation not supported due to 'listeninterface=.local'");
            return undef;
        }
        if ($listenInterface eq '.internal' || $listenInterface eq '.global'){
            my $internalNetwork = $instance->getInternalNetworkPrefix ();
            if (!defined $internalNetwork && $listenInterface eq '.internal'){
                $self->setErrorMessage ("Slave operation not supported due to 'listeninterface=.internal' and unknown subnet");
                return undef;
            }
            if (defined $internalNetwork && !$self->checkInternalNetwork ($internalNetwork)){
                return undef;
            }
        }
    }

    if ($self->isCheckOnly()) {
        $self->setSkip('Password');
        $self->setSkip('UID');
        $self->setSkip('GID');
        $self->setSkip('HomeDir');
        $self->setSkip('Shell');
    }

    return 1;
}

sub isScopeInstance {
    my ($self) = @_;
    my $scope = $self->getBatchValue('Scope');
    return (defined $scope && $scope eq 'instance') ? 1 : 0;
}

#-------------------------------------------------------------------------------
# required for slave operations based on SDB::Install::App::Console::DBUpgradeHost
# in this case isSlaveUpgrade () returns true
#

sub isSlaveUpgrade{
    return (defined $_[0]->{_is_slave_upgrade} && $_[0]->{_is_slave_upgrade} ? 1 : 0);
}

sub isLocalHostContext{
    if ($_[0]->isScopeInstance() ||
        !$_[0]->isDistributedSystem() ||
        $_[0]->{isSlave}){

        return 1;
    }
    return 0;
}

#-------------------------------------------------------------------------------
# setIsSlaveUpgrade ()
# Parameter:
#    boolean $enable     optional, default is true
#

sub setIsSlaveUpgrade{
    my ($self, $enable) = @_;
    $self->{_is_slave_upgrade} = (defined $enable) ? $enable : 1;

    if ($self->isSlaveUpgrade()) {
        $self->{isSlave} = 1;
    }
}


#-------------------------------------------------------------------------------
# This method is called from handleInitParams and initializes succeding
# parameters (hash 'params') on a host where hdbreg/hdbrename is executed
# at first (this host can de differ from the master host of the database)
#
# Returns int retCode

sub handleMasterInitParams {

    my ($self) = @_;

    delete $self->{isSlave};
    $self->showSystemProperties();

    #
    # check whether the files are owned by original sidadm or nobody (nfs4 support)
    #
    # file <hana_shared>/<SID>/global/hdb/install/support/cfg is used to
    # determine the current uid (via stat() ) and to get the
    # original sidadm uid (via file conent)
    #

    my $sapSys   = $self->{sapSys};
    if (!$self->verifyUserConfigFile($sapSys)){
        return 0;
    }

    my $instance = $self->getOwnInstance();
    if ($instance->exists_remote_host()) {
        $self->restrictListenInterfaceMultiHost();
    }

    my $listenInterface = $self->getValue('ListenInterface');
    my $internalNetwork = undef;
    $internalNetwork = $instance->getInternalNetworkPrefix()
            if (!defined $listenInterface || ($listenInterface ne 'local'));
    if (defined $internalNetwork) {
        my $prefix = $self->getIPPrefix($internalNetwork);
        my $ipList = $self->getLocalIPs();
        if (defined $prefix && defined $ipList) {
            my $internalIp = $prefix->findSubnetDeviceIps($ipList)->[0];
            if (defined $internalIp){
                $self->{internalLocalIp}       = $internalIp->short();
                $self->{internalNetworkPrefix} = $prefix->print();
            }
        }
    }

    
    $self->setDefault('ListenInterface', $self->getListenInterfaceType());

    if (defined $self->{params}->{'SystemUsage'}){
        my $systemUsage = $instance->getSystemUsage();
        if (defined $systemUsage) {
            $self->setDefault('SystemUsage', $systemUsage);
        }
    }

    if (exists $self->{params}->{'AutoStart'}) {
        $self->setSkip('AutoStart', 0);
        $self->setDefault('AutoStart', ($instance->isAutoStartEnabledLocal() ? 1 : 0));
    }

    my $userConfig = $sapSys->getUserConfig() || {};
    my $shallSkipXSSpaceUserSAP = $instance->existsXS2Dir() && exists($userConfig->{xs_space_user_sap}) ? 0 : 1;
    my $shallSkipXSSpaceUserProd = $instance->existsXS2Dir() && exists($userConfig->{xs_space_user_prod}) ? 0 : 1;
    my $shallSkipXSEADataPath = $instance->existsXS2Dir() && defined($instance->getXSEADataPath()) ? 0 : 1;
    my $shallSkipXsDomainName = $instance->existsXS2Dir() ? 0 : 1;
    $self->setSkip('XSSpaceUserIdSAP', $shallSkipXSSpaceUserSAP);
    $self->setSkip('XSSpaceUserIdProd', $shallSkipXSSpaceUserProd);
    $self->setSkip('XsEaDataPath', $shallSkipXSEADataPath);
    $self->setSkip('XsDomainName', $shallSkipXsDomainName);
    return 1;
}

sub getDefaultXsDomainName {
    my ($self) = @_;

    my $instance = $self->getOwnInstance();
    my $oldValue = $instance->getXsDomainName();
    return undef if (! defined $oldValue);
    my $hostMap = $self->getValue ('HostMap');
    foreach my $oldHostname (keys %$hostMap) {
        my $newHostname = $hostMap->{$oldHostname};
        if (index($oldValue, $oldHostname) == 0) {
            if ($instance->isHostLocal($newHostname)) {
                my $fqdnHostName = undef;
                eval {
                    require Net::Domain;
                    $fqdnHostName = Net::Domain::domainname();
                };
                return (index($fqdnHostName, sprintf('%s.', $newHostname)) == 0) ? $fqdnHostName : $newHostname;
            } else {
                return $newHostname;
            }
        }
    }

    return $oldValue;
}

sub checkXsDomainName {
    my ($self, $value) = @_;

    if (!$value){
        return 1;
    }

    if ($value !~ /$hostname_regex/){
        $self->PushError ("\'$value' is no valid host name.");
        return 0;
    }

    if (!$self->isHostAccessible($value)) {
        $self->PushError ("Host name $value is not accessible.");
        return 0;
    }

    my $instance = $self->getOwnInstance();
    my $routingMode = $instance->getXsRoutingMode();
    if ($routingMode eq 'hostnames') {
        my $subdomain = "subdomaintest.".$value;
        if (!$self->isHostAccessible($subdomain)) {
            $self->PushError ("XSA Domain Name needs to allow wildcard subdomains. Please configure your DNS accordingly (see SAP Note 2245631).");
            return 0;
        }
    }

    if (!$self->isXsDomainNameSameHostAsBefore($value)) {
        $self->PushError ("XSA Domain Name host should be the same host as the one before rename.");
        return 0;
    }

    return 1;
}

sub isXsDomainNameSameHostAsBefore {
    my ($self, $value) = @_;

    my $instance = $self->getOwnInstance();
    my $oldValue = $instance->getXsDomainName();
    return 1 if (! defined $oldValue);
    my $hosts = $instance->get_allhosts();
    my $hostMap = $self->getValue ('HostMap');
    my $address = $self->_getIpAddress($oldValue);
    my $newAddress = $self->_getIpAddress($value);

    foreach my $currHost (@{$hosts}) {
        if ($self->_isSameIpAddress($currHost, $address)) {
            my $newHostname = $hostMap->{$currHost};
            return $newHostname ? $self->_isSameIpAddress($newHostname, $newAddress) : 1;
        }
    }

    return 1;
}

sub _isSameIpAddress {
    my ($self, $host, $address) = @_;

    my $hostAddress = $self->_getIpAddress($host);
    return ($address eq $hostAddress);
}

sub _getIpAddress {
    my ($self, $hostname) = @_;
    my $errlst   = new SDB::Install::MsgLst();
    my $nslookup = nslookup($hostname, $errlst);

    if (!defined $nslookup || !defined $nslookup->{address}) {
        return undef;
    }

    return $nslookup->{address};
}

sub verifyUserConfigFile {
    my ($self, $sapSys) = @_;
    my $userCfg = $sapSys->getUserConfig(); # read the sidadm configuration

    if (!defined($userCfg) || !defined($userCfg->{sidadm_id})){
        $self->setErrorMessage("Cannot read sidadm user configuration", $sapSys->getErrMsgLst());
        return 0;
    }

    my $storedUID = $userCfg->{sidadm_id};
    my $userConfigFile = $sapSys->getUserConfigFile();
    my (undef, undef, undef, undef, $ownerUID) = stat ($userConfigFile);

    if (!defined($ownerUID)){
        $self->setErrorMessage ("Cannot stat '$userConfigFile': $!");
        return 0;
    }
    if ($ownerUID != $storedUID && $ownerUID != getpwnam ('nobody')){
        $self->setErrorMessage ("Source files are not owned by the original sidadm user (uid = $storedUID)");
        return 0;
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Initializes the host map entries and displays the local hostname.
# If the local hostname matches an origin name, this name is added
# under 'own_origin_value' to the host map parameter.
#
# Parameter:  SDB::Install::SAPInstance::TrexInstance  $instance

sub handleInitHostMap {

    my ($self, $instance) = @_;

    my $paramHostMap               = $self->{params}->{HostMap};
    my @sortedOriginHosts          = sort @{$instance->get_allhosts()};
    $paramHostMap->{origin_values} = \@sortedOriginHosts;

    delete $paramHostMap->{own_origin_value};
    foreach my $currSrcHost (@sortedOriginHosts) {

        $paramHostMap->{default_map}->{lc($currSrcHost)} = $currSrcHost;

        if ( ! defined $paramHostMap->{own_origin_value} && $instance->isHostLocal($currSrcHost)) {
            $paramHostMap->{own_origin_value} = $currSrcHost;
            $self->{localHostName}            = $currSrcHost;
            $self->AddProgressMessage( "Local Host Name: $paramHostMap->{own_origin_value}" );
        }
    }

    if (!defined $paramHostMap->{own_origin_value}) {

        my $hostWorkingOn = lc(hostname());
        my $srcHost       = $instance->get_host();

        if (defined $srcHost) {
            if ($srcHost eq $hostWorkingOn) {
                $srcHost = undef;
            }
            elsif ($instance->isHostLocal($srcHost)) {
                $hostWorkingOn = $srcHost;
                $srcHost       = undef;
            }
        }

        $self->AddProgressMessage("Own Host Name: $hostWorkingOn");
        if (defined $srcHost) {
             $self->AddProgressMessage("Source Host Name: $srcHost");
        }

        if (!$self->{isSlave}) {

            # assign hostWorkingOn to first master host
            # if master host is an origin host

            my $srcMasterList = $instance->getMasters();
            my $srcMasterHost = defined $srcMasterList ? lc($srcMasterList->[0])
                                                       : undef;

            if (defined $srcMasterHost &&
                defined $paramHostMap->{default_map}->{$srcMasterHost}) {
                $paramHostMap->{default_map}->{$srcMasterHost} = $hostWorkingOn;
                $paramHostMap->{own_origin_value}              = $srcMasterHost;
            }
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# This method initializes the succeding parameters (hash 'params').
# The method is called from 'checkSID' when a valid SID and a valid installation
# path is detected and the hash 'sapSys' exists.
#
# Returns int retCode

sub handleInitParams {

    my ($self) = @_;

    if ($self->{handleInitParamsCalled}) {
        return 1;
    }

    $self->{handleInitParamsCalled} = 1;

    my $sapSys   = $self->{sapSys};
    my $sid      = $sapSys->get_sid();
    my $instance = $self->getOwnInstance();

    my $rc = 1;

    if ($instance->hasLocalHost()){
        #
        # found local host => checking for a slave operation
        #
        my $hostDir = $instance->get_hostNameDir();

        if (-d $hostDir) {
            $rc = $self->detectSlave($hostDir);
        }
    }
    else {
        $self->PushError
            ("Local host name of $gProductName instance '$sid' is not defined");
        $rc = 0;
    }

    if ($rc && !$self->{isSlave}) {
        $rc = $self->handleMasterInitParams();
    }

    if ($rc) {
        $self->tryEnableScopeParams();
        $self->enableHostagentPasswordIfRequired();
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Sets the hash 'isSlave' and exeuctes 'handleSlaveInitParams' in case of slave.
# Subroutine has to be overridden by HdbReg and Rename.

sub detectSlave {
	return 1;
}


#-------------------------------------------------------------------------------
# Initializes the user specific parameters of a slave.
# The IDs for sidadm user and sapsys group have to match those of the master.
# If the user or group does not exist, these IDs have to be free at the slave.
#
# The method tryHandleExistingGroupAndUser has to be called before.
#
# Returns int retCode

sub handleSlaveInitUserParams {

    my ($self)= @_;

    my $sapSys = $self->{sapSys};
    my $user   = $sapSys->getUser();
    my $cfg    = $sapSys->getUserConfig();

    if (!defined $cfg){
        $self->AddError ('Cannot get user configuration', $sapSys);
        return undef;
    }

    if (!defined $self->{localSapsysGID}) {

        if (!$self->checkGID ($cfg->{sapsys_groupid})) {
            return 0;
        }

        $self->{params}->{GID}->{value} = $cfg->{sapsys_groupid};
        $self->setSkip('GID');
    }

    if (!defined $self->{existingSidAdm}) {

        if (!$self->checkUID ($cfg->{sidadm_id})) {
            return 0;
        }

        $self->{params}->{UID}->{value}     = $cfg->{sidadm_id};
        $self->{params}->{HomeDir}->{value} = $cfg->{sidadm_home};
        $self->{params}->{Shell}->{value}   = $cfg->{sidadm_shell};

        $self->setSkip('UID');
        $self->setSkip('HomeDir');
        $self->setSkip('Shell');
    }

    $self->{_sidAdmUser}                = $user->getSidAdm();
    if (defined $self->{existingSidAdm}){
        $self->{params}->{Password}->{type} = 'passwd';
    }
    else{
        $self->{params}->{Password}->{type} = 'initial_passwd'
    }
    $self->changePasswordStr($user->getSidAdmName());

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the password of the system administrator

sub checkPassword{

    my ($self, $password) = @_;

    my $param = $self->{params}->{Password};

    if (($param->{type} eq 'passwd')
                              && (!$self->{isSlave} || !$self->isCheckOnly())) {

        if (!defined $self->{_sidAdmUser}) { # For rename scenarios the newSID must be used (see Bug: 100924)
            my $sid = $self->hasValue('newSID') ? $self->getValue('newSID') : $self->getSID();
            my $sidAdmUsername = createSysAdminUserName($sid);
            my $sidAdmUser = new SDB::Install::User($sidAdmUsername);

            if (defined $sidAdmUser && $sidAdmUser->exists()) {
                $self->{_sidAdmUser} = $sidAdmUser;
            }
        }

        if (!defined $self->{_sidAdmUser}) {
            $self->PushError ('System administrator user does not exist');
            return 0;
        }

        if (!$self->verifySidAdmPassword($self->{_sidAdmUser},
                                         $password,
                                         $param->{str})) {
            return 0;
        }
    }
    elsif (!defined $password || length ($password) < 8){
        $self->PushError ("$param->{str} must contain at least 8 characters");
        return 0;
    }

    return 1;
}

sub checkUID {
    my ($self, $value, $msglst, $doNotFailOnExistingRemoteUID) = @_;

    return undef if (!$self->_checkAlreadyUsedIds('UID', $value));
    return $self->SUPER::checkUID($value, $msglst, $doNotFailOnExistingRemoteUID);
}

#-------------------------------------------------------------------------------
# Checks the UID for the OS user for SAP space

sub checkXSSpaceUserIdSAP {
	my ($self, $value) = @_;
	my $targetSID = $self->getValue('newSID') || $self->getSID();
	my $userName = createXSSpaceSAPUserName($targetSID);

	return undef if(!checkOsUserID($self, 'XSSpaceUserIdSAP', $userName, $value, 0)); # Cannot be in group 'sapsys'

    return $self->_checkAlreadyUsedIds('XSSpaceUserIdSAP', $value);
}

#-------------------------------------------------------------------------------
# Checks the UID for the OS user for customer space

sub checkXSSpaceUserIdProd {
	my ($self, $value) = @_;
	my $targetSID = $self->getValue('newSID') || $self->getSID();
	my $userName = createXSSpaceProdUserName($targetSID);

	return undef if(!checkOsUserID($self, 'XSSpaceUserIdProd', $userName, $value, 0)); # Cannot be in group 'sapsys'

    return $self->_checkAlreadyUsedIds('XSSpaceUserIdProd', $value);
}

sub _checkAlreadyUsedIds {
    my ($self, $paramId, $value) = @_;

    my $reservedIds = $self->_getReservedUIDs($paramId);
    my $otherId = $reservedIds->{$value};
    if(defined $otherId){
        my $targetSID = $self->getValue('newSID') || $self->getSID();
        my $sid = (@$otherId[1] eq 'new') ? $targetSID : $self->{sapSys}->get_sid();
        my $username = createUsername(@$otherId[0], $sid);
        $self->getErrMsgLst()->AddError("User ID '$value' is already taken for user $username");
        return undef;
    }
    return 1;
}

sub _getUIDtoUserConfigMap {
    return { 'UID'               => 'sidadm_id',
             'XSSpaceUserIdSAP'  => 'xs_space_user_sap_id',
             'XSSpaceUserIdProd' => 'xs_space_user_prod_id',
           };
}

sub _getReservedUIDs {
    my ($self, $parameterId) = @_;

    my %result = ();

    my $uidToUserConfigMap = $self->_getUIDtoUserConfigMap();
    my $userCfg = $self->{sapSys}->getUserConfig() || {};
    for my $id (keys %{$uidToUserConfigMap}) {
        next if($id eq $parameterId);
        my $takenOldUID = $userCfg->{$uidToUserConfigMap->{$id}};
        if(defined($takenOldUID) && (length($takenOldUID) != 0)) {
            $result{$takenOldUID}=[$id, 'old'];
        }
        my $takenUID = $self->getValue($id) || $self->getDefault($id);
        if(defined($takenUID) && (length($takenUID) != 0)) {
            $result{$takenUID}=[$id, 'new'];
        }
    }

    return \%result;
}

sub handleDefaultUserId {
    my ($self, $parameterId) = @_;

    my $oldUID = $self->_getOldUserId($parameterId);
    my @reservedIDs = keys %{$self->_getReservedUIDs($parameterId)};
    my $targetID = $self->getFreeOsUserID($oldUID, \@reservedIDs);
    $self->setDefault($parameterId, $targetID);
}

sub getIDsNotApplicableForSidadm {
    my ($self) = @_;
    my $reservedIds = $self->SUPER::getIDsNotApplicableForSidadm();
    push (@$reservedIds, keys %{$self->_getReservedUIDs('UID')});
    return $reservedIds;
}

sub _getOldUserId {
    my ($self, $parameterId) = @_;

    my $uidToUserConfigMap = $self->_getUIDtoUserConfigMap();
    my $userCfg = $self->{sapSys}->getUserConfig() || {};
    my $oldUID = $userCfg->{$uidToUserConfigMap->{$parameterId}};
    return $oldUID;
}

#-------------------------------------------------------------------------------
# Checks the host map with the given $origin and $new strings
# and assigns $new converted into lower-case to $origin specific
# HostMap parameter
#
# Parameters: string $originHost
#             string $newHost
#
# Returns int retCode

sub setHostMap {
    my ($self, $originHost, $newHost, $dontFailOnAlreadyDefinedHost) = @_;

    if (!$self->checkHostMap ($originHost, $newHost, $dontFailOnAlreadyDefinedHost)) {
        return 0;
    }

    $self->{params}->{HostMap}->{value}->{lc($originHost)} = lc($newHost);

    return 1;
}


#-------------------------------------------------------------------------------
# Shows the host names of slaves if hdbreg/hdbrename has not been finished for
# that slave.
#
# Parameter: $action      string  # 'Registering' or 'Renaming'
#            $programName string  # 'hdbreg'   or 'hdbrename'
#            $showDoneMsg boolean # displays a message if all hosts are finished
#
# Returns int retCode

sub showUnfinishedSlaves {

    my ($self, $actionProgressive, $programName, $showDoneMsg) = @_;

    if ($self->isCheckOnly()) {
        return 1;
    }

    my $outstandingHosts = $self->getOutstandingHosts();

    if (defined $outstandingHosts) {

        $self->AddProgressMessage
            ($actionProgressive . ' of this system is not yet finished.');

        $self->AddProgressMessage('   To complete this action, run '
                                  . $programName . ' on ' . $outstandingHosts);
    }
    elsif ($showDoneMsg) {
        $self->AddProgressMessage("Pending hosts not found.");
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Returns a string containing outstanding slave hosts
# or undef if all slaves are finished.

sub getOutstandingHosts {

    my ($self, $arrayWanted) = @_;

    my $instance         = $self->getOwnInstance();
    my $hosts            = (defined $instance) ? $instance->get_allhosts() : undef;
    my $outstandingHosts = undef;
    my $hostOrHosts      = 'host ';
    my $persFilename     = $self->getPersistentFile();
    my @arrayOutstandingHosts;

    if (defined $hosts && defined $persFilename) {

        foreach my $currHost (sort(@{$hosts})) {

            my $currHostDir  = $instance->get_hostNameDir($currHost);
            my $currPersFile = join ($path_separator,
                                     $currHostDir,
                                     $persFilename);

            if (-f $currPersFile) {

                if (defined $outstandingHosts) {
                    $hostOrHosts      = 'hosts ';
                    $outstandingHosts = "$outstandingHosts, '$currHost'";
                }
                else {
                    $outstandingHosts = "'$currHost'";
                }
                push @arrayOutstandingHosts, $currHost;
            }
        }
    }

    return undef if (!defined $outstandingHosts);

    return ($arrayWanted) ? \@arrayOutstandingHosts
                          : $hostOrHosts . $outstandingHosts;
}

#-------------------------------------------------------------------------------
#
# returns true if LCM slave operations must not start its own host
#

sub noHostStart{
    my ($self, $instance) = @_;

    if ($self->{isNoHostStart}) {
        return 1;
    }

    $instance    = $self->getOwnInstance() if (!defined $instance);
    my $allhosts = $instance->get_allhosts();

    if (!defined $allhosts || scalar @$allhosts < 2){
        # single host system
        return 0;
    }

    if (defined $self->getValue('InternalNetwork') # reconfigure internal net
        || $instance->isUsingInternalInterface()
        || $self->existsNonColumnStoreHostRole($instance)) {

        $self->{isNoHostStart} = 1;
        return 1;
    }

    return 0;
}


sub newHostIsWorkerOrStandby{
    my ($self, $newhost, $new2oldHostMap, $hostRolesInfo) = @_;
    my $oldhost = $new2oldHostMap->{$newhost};
    my $hostRoles;
    if (defined $oldhost){
        $hostRoles = $hostRolesInfo->{$oldhost};
    }
    if (!defined $hostRoles){
        # try new name as fallback
        $hostRoles = $hostRolesInfo->{$newhost};
    }
    if (!defined $hostRoles){
        $self->appendErrorMessage ("Cannot get host roles of host '$newhost' (former '$oldhost')");
        return undef;
    }
    return ((grep {/^worker$|^standby$/} split(',', $hostRoles)) ? 1 : 0);
}


sub checkMultiDBIsolationPrerequisitesLocal{
    my ($self) = @_;
    my $instance = $self->getOwnInstance ();

    if (!defined $instance->getTenantDatabases ()){
        return undef;
    }

    if (!defined $instance->{_mdcDatabases}){
        return 1;
    }

    my ($database, $container, $username, $groupname, $uid, $gid, $user,$group);

    my $old2newHostMap = $self->getValue ('HostMap');
    my %new2oldHostMap;
    if (defined $old2newHostMap){
        %new2oldHostMap = reverse %$old2newHostMap;
    }
    my $hostRolesInfo = $instance->getHostRolesInfo ();
    my $newhost = $self->{localHostName};
    my $isHanaHost = $self->newHostIsWorkerOrStandby ($newhost, \%new2oldHostMap, $hostRolesInfo);

    if (!$isHanaHost){
        # local host is no hana host (worker or standby)
        return 1;
    }
    return $self->checkLocalTenantDBUsersAndGroups ($newhost);
}

our $mcdErrorMessageTemplate;
import SDB::Install::Configuration::AnyConfig qw ($mcdErrorMessageTemplate);

sub checkMultiDBIsolationPrerequisitesRemote{
    my ($self) = @_;
    my $rc = 1;
    my $hdbInstance = $self->getOwnInstance();
    my $tenantDatabases = $hdbInstance->getTenantDatabases();

    if (!defined $tenantDatabases || !@$tenantDatabases){
        return $rc;
    }
    if (!defined $self->{remoteUsersAndGroups}){
        return $rc;
    }

    my ($database, $container, $username, $groupname, $uid, $gid);
    my $usersHash;
    my $groupsHash;

    my $old2newHostMap = $self->getValue ('HostMap');
    my %new2oldHostMap;
    if (defined $old2newHostMap){
        %new2oldHostMap = reverse %$old2newHostMap;
    }
    my $instance = $self->getOwnInstance ();
    my $hostRolesInfo = $instance->getHostRolesInfo ();

    foreach my $entry (@$tenantDatabases){
        ($database, $container, $username, $groupname, $uid, $gid) = @$entry;
        foreach my $host (sort keys (%{$self->{remoteUsersAndGroups}})){
            if (!$self->newHostIsWorkerOrStandby ($host, \%new2oldHostMap, $hostRolesInfo)){
                next;
            }
            $usersHash = $self->{remoteUsersAndGroups}->{$host}->{users};
            if (!exists $usersHash->{$username}){
                $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "required operating system user '$username' doesn't exist on host '$host'"));
                $rc = undef;
            }
            elsif ($usersHash->{$username}->[0] != $uid){
                $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "operating system user '$username' has wrong uid '" .
                    $usersHash->{$username}->[0]."' on host '$host', expected '$uid'"));
                $rc = undef;
            }
            $groupsHash = $self->{remoteUsersAndGroups}->{$host}->{groups};
            if (!exists $groupsHash->{$groupname}){
                $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "required operating system group '$groupname' doesn't exist on host '$host'"));
                $rc = undef;
            }
            elsif ($groupsHash->{$groupname}->[0] != $gid){
                $self->appendErrorMessage (sprintf ($mcdErrorMessageTemplate, $database, "operating system group '$groupname' has wrong gid '" .
                    $groupsHash->{$groupname}->[0]."' on host '$host', expected '$gid'"));
                $rc = undef;
            }
        }
    }
    return $rc;
}

#-------------------------------------------------------------------------------
# Sets the name of the unregister persistent file in order to remove it later.

sub setUnregisterPersFile {
    my ($self, $filename) = @_;
    $self->{_pers_fn_unregister} = $filename;
}


#-------------------------------------------------------------------------------
# If the wanted sidadm already exist, $self->{existingSidAdm} is created.
# This method is called before any host information is collected.

sub tryHandleExistingGroupAndUser {

    my ($self, $sidadmName, $remoteUserExists, $isRenameSID) = @_;

    if ($isWin) {
        return 1;
    }

    my $sapSys   = $self->{sapSys};
    my $usercfg  = $sapSys->getUserConfig();

    if (!defined $usercfg){
        $self->AddError ('Cannot get user configuration', $sapSys);
        return undef;
    }

    $self->changePasswordStr($sidadmName);

    my $params    = $self->{params};
    my $sapsysGID = getgrnam($gSapsysGroupName);
    my $instance  = $self->getOwnInstance();
    my $hostname  = defined ($self->{localHostName}) ? $self->{localHostName}
                                                     : hostname();

    my $ignoreCfgMismatches = 0;

    if (!$self->{isSlave} &&
        ($isRenameSID || (!$remoteUserExists && $self->getValue('InitUser')))) {
        $ignoreCfgMismatches = 1;
    }

    if (defined $sapsysGID) {

        if (!$ignoreCfgMismatches && ($sapsysGID != $usercfg->{sapsys_groupid})) {
            $self->PushError ("Group '$gSapsysGroupName' has wrong id '$sapsysGID'"
                              . " (expected: '$usercfg->{sapsys_groupid}')!");
            return 0;
        }

        my $message = "$params->{GID}->{str}: $sapsysGID";
        if ($params->{GID}->{silent}) {
            $self->AddMessage($message);
        }
        else {
            $self->AddProgressMessage($message);
        }

        $self->setSkip('GID');
        $params->{GID}->{value}          = $sapsysGID;
        $self->{sapsysids}->{$sapsysGID} = [$hostname];
        $self->{localSapsysGID}          = [$sapsysGID, $hostname];
    }
    elsif ($self->{isSlave} || ($remoteUserExists && !$isRenameSID)) {
        my $gid  = $usercfg->{sapsys_groupid};
        my $name = getgrgid($gid);
        if (defined $name) {
            $self->PushError("Group id '$gid' is already in use by group '$name'");
            return 0;
        }
        $params->{GID}->{value} = $gid;
        $self->setSkip('GID');
    }
    elsif (!defined $self->getBatchValue('GID')
           && !$self->getIgnore('scan_password_database')) {

        if ($instance->exists_remote_host()) {
            # CollectOtherHostInfos calls getFreeGroupID with usercfg_sapsys_groupid
            $self->{usercfg_sapsys_groupid} = $usercfg->{sapsys_groupid};
        }
        else {
            # single-host system
            $self->setDefault('GID', $self->getFreeGroupID($usercfg->{sapsys_groupid}));
        }
    }

    my ($name, undef, $uid, $gid, undef, undef, undef, $home, $shell) =
                                                          getpwnam($sidadmName);

    if (defined $uid) {
        my $groupName = getgrgid($gid);

        if (!defined $groupName) {
            $self->PushError("The group id $gid of user '$name' is unknown");
            return 0;
        }

        if ($groupName ne $gSapsysGroupName) {
            $self->PushError("Existing user '$name' belongs to group '$groupName'"
                             . " (id $gid) instead of group '$gSapsysGroupName'");
            return 0;
        }

        $self->{existingSidAdm}->{uid}  = $uid;
        $self->{existingSidAdm}->{prop} = [$hostname, $sidadmName, $gid];

        if (!$self->compareExistingSidAdmWithCfg($ignoreCfgMismatches)) {
            delete $self->{existingSidAdm}->{$uid};
            return 0;
        }

        $params->{UID}->{value}     = $uid;
        $params->{HomeDir}->{value} = $home;
        $params->{Shell}->{value}   = $shell;
        $params->{Password}->{type} = 'passwd';

        $self->setSkip('UID');
        $self->setSkip('HomeDir');
        $self->setSkip('Shell');
        if ($self->isLocalHostContext()){
            $self->setSkip('Password');
        }
    }
    elsif ($self->{isSlave} || ($remoteUserExists && !$isRenameSID)) {
        $uid  = $usercfg->{sidadm_id};
        $name = getpwuid($uid);
        if (defined $name) {
            $self->PushError("User id '$uid' is already in use by user '$name'");
            return 0;
        }
        $params->{Password}->{type} = 'initial_passwd';
        $params->{UID}->{value}     = $uid;
        $params->{HomeDir}->{value} = $usercfg->{sidadm_home};
        $params->{Shell  }->{value} = $usercfg->{sidadm_shell};
        $self->setSkip('UID');
        $self->setSkip('HomeDir');
        $self->setSkip('Shell');
    }
    else {
        $params->{Password}->{type} = 'initial_passwd';
        $self->setDefault('Shell', $usercfg->{sidadm_shell});
        # home dir has the default '/usr/sap/$newSID/home'
        $params->{UID}->{str}       = sprintf($params->{UID}->{str_templ},    $sidadmName);
        $params->{HomeDir}->{str}   = sprintf($params->{HomeDir}->{str_templ},$sidadmName);
        $params->{Shell}->{str}     = sprintf($params->{Shell}->{str_templ},  $sidadmName);

        if (!defined $self->getBatchValue('UID')
            && !$self->getIgnore('scan_password_database')) {

            if ($instance->exists_remote_host()) {
                # CollectOtherHostInfos calls getFreeUserID with usercfg_sidadm_id
                $self->{usercfg_sidadm_id} = $usercfg->{sidadm_id};
            }
            else {
                # single-host system
                $self->setDefault('UID', $self->getFreeUserID($usercfg->{sidadm_id}, $self->getIDsNotApplicableForSidadm()));
            }
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Deletes the unregister persistent file, if it exists.

sub tryRemoveUnregisterPersFile {
	my ($self) = @_;

	if (defined $self->{_pers_fn_unregister}) {
        $self->pers_remove($self->{_pers_fn_unregister});
    }
    return 1;  # ignore error
}


#-------------------------------------------------------------------------------
# Enables or disables autostart if necessary.

sub trySetAutoStartLocal {
    my ($self, $instance, $msglst) = @_;

    my $isEnabled = ($instance->isAutoStartEnabledLocal()) ? 1 : 0;
    my $param     = $self->{params}->{AutoStart};
    my $wanted    = (defined $param && $param->{value}) ? 1 : 0;

    if ($isEnabled != $wanted) {
        if (!$instance->enableAutoStartLocal($wanted, $msglst)) {
            return undef;
        }
    }
    return 1;
}

1;
