package SDB::Install::Configuration::AnyRename;

use strict;

use SDB::Install::Configuration::AnyRegisterRename;
use SDB::Install::IniFile;
use SDB::Install::NewDBUser;
use SDB::Install::SAPProfiles;
use SDB::Install::Configuration qw ($bool_true_pattern $bool_false_pattern);
use SDB::Install::Globals       qw ($gHostRoleAcceleratorStandby
                                    $gHostRoleAcceleratorWorker
                                    $gHostRoleEsStandby
                                    $gHostRoleEsWorker
                                    $gHostRoleRDSync
                                    $gProductNameSystem
                                    $gShortProductNameRDSync);
use SDB::Install::SysVars       qw ($path_separator);

use File::Basename qw(dirname);
use SDB::Install::System qw(isSameFile);
use SDB::Install::Tools qw (isMountPoint);

require SDB::Install::Configuration::NewDB;
require SDB::Install::Configuration::NewServerConfig;

our @ISA = qw (SDB::Install::Configuration::AnyRegisterRename);

our $hdbregPersFilename = 'hdbreg';
our $peristentFileName  = 'hdbrename';
sub getPersistentFile {return $peristentFileName}

#-------------------------------------------------------------------------------
# Constructor
sub new {

    my $self = shift->SUPER::new (@_);

    return $self;
}


#-------------------------------------------------------------------------------
# Checks the access of the specified directory

sub checkExistingDir {

    my ($self, $path) = @_;

    if (!-e $path){
        $self->PushError ("Directory '$path' does not exist");
        return 0;
    }
    if (!-d $path ){
        $self->PushError ("'$path' is no directory");
        return 0;
    }

    # checking r-x of patent directories
    return $self->SDB::Install::Configuration::NewServerConfig::check_path_access ($path,0);
}


#-------------------------------------------------------------------------------
# Checks if the specified instance number matches the own instance or
# is not already in use by any other instance.
#
# Returns int retCode

sub checkInstanceNumber {

    my ($self,
        $number # instance number has to contain digits only
       ) = @_;

    my $param = $self->{params}->{InstanceNumber};
    if ($number =~ /\D/){
        $self->PushError ("$param->{str} '$number' is not a decimal number");
        return 0;
    }

    if ($self->getOwnInstance()->get_nr() == $number) {
        # instance number remains unchanged
        return 1;
    }

    if (!$self->SUPER::checkInstanceNumber($number)) {
        return 0;
    }

    my $sid = $self->getBatchValue('SID');
    return $self->checkAdditionalInstanceNumber ($number, $self->{params}->{InstanceNumber}->{str}, $sid);
}


sub isSidChanged{
    my ($self, $newSid, $oldSid) = @_;
    if (!defined $newSid){
        $newSid = $self->getValue ('newSID');
    }
    if (!defined $oldSid){
        $oldSid = $self->getValue ('SID');
    }
    return ($oldSid ne $newSid) ? 1 : 0;
}



#-------------------------------------------------------------------------------
# Checks the target SID and changes initial passwords
#
# Returns int retCode

sub checknewSID {

    my ($self, $newSid) = @_;

    my $params     = $self->{params};
    my $oldSid     = $self->getValue ('SID');
    my $sidChanged = $self->isSidChanged ($newSid, $oldSid);

    if (!defined $self->{sapSys}) {
        my $sidInfo = ($sidChanged) ? "'$oldSid' or '$newSid'" : "'$oldSid'";
        $self->PushError ("$gProductNameSystem $sidInfo does not exist");
        return 0;
    }
    my $targetSid = $self->{sapSys}->getTargetSID();

    if (defined $targetSid && $newSid ne $targetSid) {

        $self->PushError ($params->{newSID}->{str} .
            " '$newSid' does not match target system ID '"
            . $self->{sapSys}->get_globalSidDir() . "'");
        return 0;
	}

	if ($sidChanged
	    && !$self->checkAdditionalSID($newSid, $params->{newSID}->{str})) {
	    return 0;
	}

    if ($sidChanged && !$self->isCheckOnly()) {
        $self->clearInstallationPath($newSid);
    }

    $self->{current_sid} = $newSid;

    my $rc = 1;
    if ($self->getOwnInstance()->hasLocalHost()) {
        $rc = $self->initTargetUserParams($newSid);
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# If new SID and local host are detected, the parameters of the new user
# can be initialized.

sub initTargetUserParams {

    my ($self, $newSID) = @_;

    my $params      = $self->{params};
    my $oldSID      = $self->getValue('SID');
    my $sidChanged  = ($oldSID ne $newSID) ? 1 : 0;
    my $newUser     = new SDB::Install::NewDBUser($newSID);
    my $newUserName = $newUser->getSidAdmName();

    if (!$self->tryHandleExistingGroupAndUser($newUserName,
                                              $self->{_remoteUserExists},
                                              $sidChanged)) {
        return 0;
    }

    if (exists $params->{InitUser}) {
        $params->{InitUser}->{str} =
            sprintf($params->{InitUser}->{str_templ}, $newUserName);
    }

    if (exists $params->{InitUserHomeDir}) {
        $params->{InitUserHomeDir}->{str} =
            sprintf($params->{InitUserHomeDir}->{str_templ}, $newUserName);
    }

    if (!$sidChanged && $newUser->exists()) {
        # <sid>adm user already exists --> skip user parameters

        foreach my $param_id (@{$self->getUserParams()}){
            $self->setSkip($param_id);
        }
    }
    else{
       $params->{Password}->{value} = undef;
    }

    if (!$sidChanged && $self->{_remoteUserExists}) {
        # <sid>adm user is deleted due to unregistered local host
        # <sid>adm users exit at remote hosts
        # --> create <sid>adm used with the existing user values
        return $self->handleSlaveInitUserParams();
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the source SID, the database instance and the administrator.
#
# Returns int retCode

sub checkSID {

    my ($self, $sid) = @_;

    delete $self->{handleInitParamsCalled};
    my $sapSys = $self->{sapSys};

    if (!defined $sid) {
        $self->PushError ("$self->{params}->{SID}->{str} is undefined");
        return 0;
    }

    $self->setDefault('newSID', $sid);

    if (defined $sapSys) {

        if ($sapSys->isPreInitWithTargetSID()) {

            my $usrSapSid = join ($path_separator, '', 'usr', 'sap', $sid);
            $sapSys->completeInitWithSourceSID($sid, $usrSapSid);
            $self->setDefault('newSID', $sapSys->getTargetSID());
            $self->{params}->{newSID}->{set_interactive} = 0;
        }
        elsif ($sid ne $sapSys->get_sid()){
            $sapSys = undef;
        }
    }

    if (!defined $sapSys) {
        $self->resetSAPSystemHashes();
        my $systems = $self->getCollectSAPSystems();

        if (defined $systems->{$sid}){
            $sapSys = $systems->{$sid};
        }
        else{
            my $sapmnt = $self->getValue ('Target');
            if ($sapmnt){
                my $sapmntSid = $sapmnt . $path_separator. $sid;
                if (-d $sapmntSid){
                    $sapSys = new SDB::Install::SAPSystem();
                    $sapSys->initWithGlobDir($sid, $sapmnt);
                }
            }
        }
        $self->{sapSys} = $sapSys;
    }

    my $instance = (defined $sapSys) ? $self->getOwnInstance() : undef;

    if (!defined $sapSys || !$sapSys->hasNewDB || !defined $instance) {
        $self->PushError ("$gProductNameSystem '$sid' does not exist");
        return 0;
    }

    $self->setDefault('InstanceNumber', $instance->get_nr());
    $self->setCurrentSID($sid);
    $self->{sapSys} = $sapSys;

    if ($self->isShowPending()) {
        return 1;
    }

    my $rc = $self->handleInitHostMap ($instance);

    if ($rc) {
        if ($instance->hasLocalHost()) {
            $rc = $self->handleInitParams();
        }
        else {
            $self->{params}->{HostMap}->{mandatory} = 1;
        }
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Checks validity of the system administrator password used to stop the system

sub checksrcPasswd {

    my ($self, $password) = @_;

    if (!$self->verifySidAdmPassword($self->{sapSys}->getUser()->getSidAdm(),
                                     $password,
                                     $self->{params}->{srcPasswd}->{str})) {
        return 0;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Collects information of remote hosts and checks SID, instance number,
# and sidadm properties. These values have to belong to this SAP HANA system
# or have to be free.

sub CollectOtherHostInfos {

    my ($self, $isNotVerbose) = @_;

    my $newSID    = $self->getValue('newSID');
    my $newSidAdm = $self->getSysAdminUserName($newSID);

    if (!$self->SUPER::CollectOtherHostInfos($isNotVerbose, $newSidAdm)) {
        return undef;
    }

    if (!$self->checkCollectedHostInfo($newSID,
                                       $self->getValue('InstanceNumber'))) {
        return undef;
    }

    return $self->tryInitGroupAndUserID();
}


#-------------------------------------------------------------------------------
# Sets the hash 'isSlave' and exeuctes 'handleSlaveInitParams'
# if a slave is detected.

sub detectSlave {

    my ($self, $hostDir) = @_;

    my $hdbregFile = $hostDir . $path_separator . $hdbregPersFilename;

    if ($self->pers_exists($hdbregFile)) {

        my $hdbregPersData = $self->pers_load($hdbregFile);

        if (defined $hdbregPersData && $hdbregPersData->{unregisterHost}) {
            # This file indicates that this instance of a distributed
            # system was unregistered while other instances still exist.
            # If the SID is not changed, hdbrename has to create the system
            # administrator used with the existing user values.

            $self->{_remoteUserExists} = 1;
        }
        if (!$self->isCheckOnly()) {
            # delete hdbreg pers-file
            # because rename slave is done instead of register slave
            #         or hdbreg file indicates 'unregistered host'
            $self->setUnregisterPersFile($hdbregFile);
        }
    }

    my $pers_data = undef;
    my $persFile  = $hostDir . $path_separator . $self->getPersistentFile();

    if ($self->pers_exists($persFile)) {
        $pers_data = $self->pers_load($persFile);
        $self->{isSlave}            = 1;
        $self->{pers_fn_reg_rename} = $persFile;
    }

    my $rc = 1;

    if ($self->{isSlave}) {
        $rc = $self->handleSlaveInitParams($pers_data);
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Used in derived classes to set the state of the 'BasePath' properties.

sub doSetBasePathParamState {
    return 1;
}


#-------------------------------------------------------------------------------
# Defines ignore values for the parameter '--ignore'.

sub getHiddenIgnoreValues{
    return [qw (scan_password_database check_hostname)];
}


#-------------------------------------------------------------------------------
sub getParamNewSID {

    my ($self, $order, $section) = @_;

    return {
        'order'     => $order,
        'opt'       => 'target_sid',
        'short_opt' => 'S',
        'opt_arg'   => '<SID>',
        'type'      => 'string',
        'section'   => $section,
        'value'     => undef,
        'default'   => undef,
        'str'       => 'Target System ID',
        'mandatory' => 1,
        'hostctrl_opt'      => 'TARGET_SID',
        'init_with_default' => 1,
        'may_be_interactive'=> 1,
    };
}


#-------------------------------------------------------------------------------
sub getParamPassword {

    my ($self, $order, $section) = @_;

    my $paramPassword = $self->SUPER::getParamPassword($order, $section);

    $paramPassword->{opt}       = 'target_password';
    $paramPassword->{short_opt} = 'P';
    $paramPassword->{type}      = 'initial_passwd';
    $paramPassword->{str}       = 'Target System Administrator Password';
    $paramPassword->{str_templ} = 'Target System Administrator (%s) Password';

    return $paramPassword;
}


#-------------------------------------------------------------------------------
sub getParamSrcPasswd {

    my ($self, $order, $section) = @_;

    return {
        'order'     => $order,
        'opt'       => 'source_password',
        'type'      => 'passwd',
        'opt_arg'   => '<password>',
        'section'   => $section,
        'value'     => undef,
        'default'   => undef,
        'str'       => 'Source System Administrator Password',
        'str_templ' => 'Source System Administrator (%s) Password',
        'log_value' => sub { '***'; },
        'skip'      => 1,
        'mandatory' => 1
    };
}


#-------------------------------------------------------------------------------
# Returns the current valid SID (source or target).

sub getSID {
    $_[0]->{_oldOrNewSID};
}


#-------------------------------------------------------------------------------
# This method is called from handleInitParams and initializes succeeding
# parameters (hash 'params') on a host where hdbreg/hdbrename is executed
# at first (this host can de differ from the master host of the database)
#
# Returns int retCode

sub handleMasterInitParams{
    my ($self) = @_;
    if (!$self->SUPER::handleMasterInitParams()){
        return 0;
    }

    my $params      = $self->{params};
    my $instance    = $self->getOwnInstance();
    my $oldUser     = $self->{sapSys}->getUser();
    my $oldUserName = $oldUser->getSidAdmName();

    if ($oldUser->exists() && $instance->sapstartsrvIsRunning ()){
        $self->setSkip('srcPasswd', 0);
        $params->{srcPasswd}->{str} =
                sprintf($params->{srcPasswd}->{str_templ}, $oldUserName);
    }
    else{
        $self->setSkip('srcPasswd');
    }

    if (exists $params->{KeepUser}) {
        $params->{KeepUser}->{str} =
                sprintf($params->{KeepUser}->{str_templ}, $oldUserName);
    }

    if (exists $params->{KeepUserHomeDir}) {
        $params->{KeepUserHomeDir}->{str} =
                sprintf($params->{KeepUserHomeDir}->{str_templ}, $oldUserName);
    }

    my $rc     = 1;
    my $newSID = $self->getValue('newSID');
    if (defined $newSID) {
        $rc = $self->initTargetUserParams($newSID);
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Initializes a volume path parameter or backup path parameter for rename.
#
# Returns int retCode

sub initBasePathParamsForRename {

    my ($self, $newSID, $newNr) = @_;

    my $instance = $self->getOwnInstance();

    my @paramIDs = ('BasePathDataVolumes',
                    'BasePathLogVolumes',
                    'BasePathDataBackup',
                    'BasePathLogBackup',
                    'BasePathCatalogBackup');

    my @pathList = ($instance->getDataPath(1),
                    $instance->getLogPath(1),
                    $instance->getDataBackupPath(1),
                    $instance->getLogBackupPath(1),
                    $instance->getCatalogBackupPath(1));

    my $activeRoles = $instance->getActiveHostRoles();
    if (!defined $activeRoles) {
        $self->setErrorMessage(undef, $instance->getErrMsgLst());
        return undef;
    }

    if (defined $activeRoles->{$gHostRoleAcceleratorWorker} ||
        defined $activeRoles->{$gHostRoleAcceleratorStandby}) {

        my $acceleratorDataPath = $instance->getAcceleratorDataPath(1);

        if (defined $acceleratorDataPath){
            push @pathList, $acceleratorDataPath;
            push @paramIDs, 'AcceleratorDataPath';
        }

        my $acceleratorLogPath = $instance->getAcceleratorLogPath(1);

        if (defined $acceleratorLogPath){
            push @pathList, $acceleratorLogPath;
            push @paramIDs, 'AcceleratorLogPath';
        }
    }

    if (defined $activeRoles->{$gHostRoleEsWorker} ||
        defined $activeRoles->{$gHostRoleEsStandby}) {

        my $esDataPath = $instance->getEsDataPath(1);

        if (defined $esDataPath){
            push @pathList, $esDataPath;
            push @paramIDs, 'EsDataPath';
        }

        my $esLogPath = $instance->getEsLogPath(1);

        if (defined $esLogPath){
            push @pathList, $esLogPath;
            push @paramIDs, 'EsLogPath';
        }
    }
    if (defined $activeRoles->{$gHostRoleRDSync}) {

        my $downloadPath = $instance->getRDSyncFileDownloadPath(1);

        if (defined $downloadPath) {
            push @pathList, $downloadPath;
            push @paramIDs, 'BasepathFileDownloadRDSync';
        }

        my $uploadPath = $instance->getRDSyncFileUploadPath(1);

        if (defined $uploadPath) {
            push @pathList, $uploadPath;
            push @paramIDs, 'BasepathFileUploadRDSync';
        }
    }

    foreach my $i (0 .. $#pathList) {
        my $param = $self->{params}->{$paramIDs[$i]};
        $param->{skip}        = 0;
        $param->{_originPath} = $pathList[$i];
        $self->setDefault($paramIDs[$i], $instance->getNewVolumePath($pathList[$i], $newSID, $newNr,
                                                                     undef, undef, $self->getValue('Target')));
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Returns the hdbrename specific filename in case of an nremaned
# slave, otherwise undef

sub pers_filename {
    return $_[0]->{pers_fn_reg_rename};
}


#-------------------------------------------------------------------------------
# Returns the keys that are inserted into the persistent file in order
# to complete this tool on a remote location.

sub pers_keys {
    return [qw (oldHost
                oldSID
                oldInstanceNumber
                isNoHostStart
                initUserHomeDir
                AutoStart
                isRenameAccelerator
                oldUidAccelerator
                oldGidAccelerator
                AcceleratorDataPath
                AcceleratorLogPath
                oldAcceleratorDataPath
                oldAcceleratorLogPath
                isRenameExtendedStorage
                oldUidEs
                oldGidEs
                EsDataPath
                EsLogPath
                oldEsDataPath
                oldEsLogPath
                isRenameRDSync
                isRenameXS2
                oldUidRDSync
                oldGidRDSync
                BasepathFileDownloadRDSync
                BasepathFileUploadRDSync
                oldPathFileDownloadRDSync
                oldPathFileUploadRDSync)];
}


#-------------------------------------------------------------------------------
# Returns the value of the specified property from the profile.

sub readFromProfileFile {
    my ( $self, $instPath, $sid, $property ) = @_;
    my $error;
    my $rc;

    require SDB::Install::SAPSystem;
    my $tempSapSys = new SDB::Install::SAPSystem ();
    $tempSapSys->preInitWithTargetSID ( $sid, $instPath );
    unless ( $tempSapSys ) {
        $error = "$gProductNameSystem '$sid' does not exist, (installation path: '$instPath')";
        $self->{'error_state'} = $error;
        $self->getErrMsgLst->addMessage ( $error );
        return undef;
    }

    # read HDB profile file
    require SDB::Install::SAPProfiles;
    my $profiles = new SDB::Install::SAPProfiles ( $tempSapSys->get_ProfileDir () );
    my $defaultProfile = $profiles->get_DefaultProfile ();

    require SDB::Install::IniFile;
    my $iniFile = new SDB::Install::IniFile ( $defaultProfile );
    $rc = $iniFile->read ();
    unless ( $rc ) {
        $error = "Cannot read the profile '$defaultProfile' of $gProductNameSystem";
        $self->{'error_state'} = $error;
        $self->getErrMsgLst->addMessage ( $error );
        return undef;
    }

    my $result = $iniFile->getValue ( undef, $property );
    unless ( $result ) {
        $error =
            "No $property property is specified in the profile '$defaultProfile' of $gProductNameSystem";
        $self->{'error_state'} = $error;
        $self->getErrMsgLst->addMessage ( $error );
        return undef;
    }

    return $result;
}


#-------------------------------------------------------------------------------
# Sets the new data backup path if old path could not be renamed automatically.

sub setBasePathDataBackup {

    my ($self, $path) = @_;

    my $param = $self->{params}->{BasePathDataBackup};

    if (!defined $param->{_originPath}) {
        $param->{_originPath} = $self->getOwnInstance()->getDataBackupPath(1);
    }
    return $self->setBasePathValue($path, $param, 1);
}

#-------------------------------------------------------------------------------
# Sets the new catalog backup path if old path could not be renamed automatically.

sub setBasePathCatalogBackup {

    my ($self, $path) = @_;

    my $param = $self->{params}->{BasePathCatalogBackup};

    if (!defined $param->{_originPath}) {
        $param->{_originPath} = $self->getOwnInstance()->getCatalogBackupPath(1);
    }
    return $self->setBasePathValue($path, $param, 1);
}


#-------------------------------------------------------------------------------
# Sets the new data path if the old path could not be renamed automatically.

sub setBasePathDataVolumes {

    my ($self, $path) = @_;

    my $param = $self->{params}->{BasePathDataVolumes};

    if (!defined $param->{_originPath}) {
        $param->{_originPath} = $self->getOwnInstance()->getDataPath(1);
    }
    return $self->setBasePathValue($path, $param);
}


#-------------------------------------------------------------------------------
# Sets the new log backup path if old path could not be renamed automatically.

sub setBasePathLogBackup {

    my ($self, $path) = @_;

    my $param = $self->{params}->{BasePathLogBackup};

    if (!defined $param->{_originPath}) {
        $param->{_originPath} = $self->getOwnInstance()->getLogBackupPath(1);
    }
    return $self->setBasePathValue($path, $param, 1);
}


#-------------------------------------------------------------------------------
# Sets the new log path if the old path could not be renamed automatically.

sub setBasePathLogVolumes {

    my ($self, $path) = @_;

    my $param = $self->{params}->{BasePathLogVolumes};

    if (!defined $param->{_originPath}) {
        $param->{_originPath} = $self->getOwnInstance()->getLogPath(1);
    }
    return $self->setBasePathValue($path, $param);
}



#-------------------------------------------------------------------------------
# Sets the new es log path if the old path could not be renamed automatically.

sub setEsLogPath {

    my ($self, $path) = @_;

    my $param = $self->{params}->{EsLogPath};

    if (!defined $param->{_originPath}) {
        $param->{_originPath} = $self->getOwnInstance()->getEsLogPath(1);
    }
    return $self->setBasePathValue($path, $param);
}

#-------------------------------------------------------------------------------
# Sets the new es data path if the old path could not be renamed automatically.

sub setEsDataPath {

    my ($self, $path) = @_;

    my $param = $self->{params}->{EsDataPath};

    if (!defined $param->{_originPath}) {
        $param->{_originPath} = $self->getOwnInstance()->getEsDataPath(1);
    }
    return $self->setBasePathValue($path, $param);
}



#-------------------------------------------------------------------------------
#  Sets the new path if the old path could not be renamed automatically.

sub setBasePathValue {

    my ($self, $path, $param, $isBackupParam) = @_;

	if (!$self->getOwnInstance()->isVolumePathStandardRename
					                      ($param->{_originPath},
					                       $path,
					                       $self->getValue('newSID'),
					                       $self->getValue('InstanceNumber'))) {

        my $errlst =  new SDB::Install::MsgLst ();
        my $saveCntxt = $self->setMsgLstContext ([undef, $errlst]);
        my $rc = $self->checkBasePath ($path);
        $self->setMsgLstContext ($saveCntxt);
        if (!$rc){
            $self->appendErrorMessage ("Cannot set paramater '$param->{str}'", $errlst);
            return 0;
        }
        $param->{value} = $path;
        $self->doSetBasePathParamState($param, $isBackupParam, 1);
    } else {
        $self->doSetBasePathParamState($param, $isBackupParam, 0);
    }
    return 1;
}

sub checkBasePath{
    my ($self, $basePath) = @_;
    if (!$self->checkExistingDir ($basePath)){
        return 0;
    }
    my $sapSys = $self->{sapSys};
    my $newSid = $self->getValue ('newSID');
    my $oldSid = $sapSys->get_sid ();
    if (!defined $newSid){
        return 1;
    }
    my $sidChanged = $self->isSidChanged ($newSid, $oldSid);
    if (!$sidChanged){
        return 1;
    }
    my $sapmntOldSid = $sapSys->get_globalSidDir();
    my $sapmntNewSid = $sapmntOldSid;
    $sapmntNewSid =~ s/$oldSid/$newSid/;

    my @statbuf = stat ($sapmntNewSid);
    if (-d _ && !isMountPoint($sapmntNewSid)){
        my $realpath = $self->getAbsPath ($basePath);
        while ($realpath && $realpath ne '/'){
            if (isSameFile ($realpath, $sapmntNewSid, \@statbuf)){
                $self->appendErrorMessage ("Existing path '$basePath' is on new HANA shared file system '$sapmntNewSid'");
                $self->appendErrorMessage ("It prevents the rename of the HANA shared file system '$sapmntOldSid' => '$sapmntNewSid'.");
                return 0;
            }
            $realpath = dirname ($realpath);
        }
    }
    return 1;
}

#-------------------------------------------------------------------------------
# Enables/disables the parameters of database passwords depending on the
# parameter '--change_system_user_password'

sub setChangeSystemPasswd {

    my ($self, $value) = @_;

    my $skipChanging = 1;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {
        $self->{params}->{ChangeSystemPasswd}->{value} = 1;
        $skipChanging = 0;
    } elsif (defined $value && ($value =~ /$bool_false_pattern/i)){
        $self->{params}->{ChangeSystemPasswd}->{value} = 0;
    } else {
        $self->{params}->{ChangeSystemPasswd}->{value} = undef;
    }

    foreach my $currID (qw (SystemUser SrcSQLSysPasswd TrgSQLSysPasswd)){
        $self->{params}->{$currID}->{skip} = $skipChanging;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sets the source SID or the target SID if the SID is already renamed.
#
# Parameter: $currSID string

sub setCurrentSID {

    my ($self, $currSID) = @_;

    $self->{_oldOrNewSID} = $currSID;
}

#-------------------------------------------------------------------------------
# Returns a hash containing entries of the parameter 'BasepathFileDownloadRDSync'.
# In case of slave rename the parameter value is read from the pers-file.

sub getParamBasepathFileDownloadRDSync {
    my ($self, $order, $section) = @_;
    return {
        'order'   => $order,
        'opt'     => 'rdsync_downloadpath',
        'type'    => 'path',
        'opt_arg' => '<path>',
        'section' => $section,
        'value'   => undef,
        'default' => undef,
        'str'     => "Location of $gShortProductNameRDSync file download directory",
        'skip'    => 1,
        'init_with_default' => 1,
        'set_interactive'   => 1,
    };
}



#-------------------------------------------------------------------------------
# Returns a hash containing entries of the parameter 'BasepathFileUploadRDSync'.
# In case of slave rename the parameter value is read from the pers-file.

sub getParamBasepathFileUploadRDSync {
    my ($self, $order, $section) = @_;
    return {
        'order'   => $order,
        'opt'     => 'rdsync_uploadpath',
        'type'    => 'path',
        'opt_arg' => '<path>',
        'section' => $section,
        'value'   => undef,
        'default' => undef,
        'str'     => "Location of $gShortProductNameRDSync file upload directory",
        'skip'    => 1,
        'init_with_default' => 1,
        'set_interactive'   => 1,
    };
}

#-------------------------------------------------------------------------------
# Sets the new instance number and initializes the path parameters with
# the instance number formatted by SUPER::setInstanceNumber.
#
# Returns int retCode

sub setInstanceNumber {

    my ($self, $newNr) = @_;

    my $rc = $self->SUPER::setInstanceNumber($newNr);

    if ($rc) {
        $self->initBasePathParamsForRename($self->getValue('newSID'),
                                           $self->getValue('InstanceNumber'));
    }
    return $rc;
}

sub getIgnoreValues { return ['check_platform']; }


#===============================================================================

# define local check functions

sub checkBasePathLogVolumes;
*checkBasePathLogVolumes = \&checkBasePath;

sub checkBasePathDataVolumes;
*checkBasePathDataVolumes = \&checkBasePath;

# import check functions from SDB::Install::Configuration::NewDB

sub checkSystemRequirements;
*checkSystemRequirements = \&SDB::Install::Configuration::NewDB::checkSystemRequirements;

1;
