package SDB::Install::Configuration::Client;

use strict;
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::SysVars;
use SAPDB::Install::Hostname;
use SDB::Install::Globals qw ($gProductNameClient $gSapmntDirUx $gShortProductNameClient);
use SDB::Install::NewDBUser;
use SDB::Install::Installation::Client;
use SDB::Install::DebugUtilities;
use SDB::Install::System::Registration::ODBC;

our @ISA = qw (SDB::Install::Configuration::AnyConfig);
our $section = 'Client';

sub enumExistingInstallations{
	# simulate absence of client installations
	#return {};
	
	my ($self) = @_;
	my $errlst = new SDB::Install::MsgLst();
	my $installations = SDB::Install::Installation::Client::EnumClientInstallations ($errlst);
	if (!defined $installations){
		$self->AddError (undef, $errlst);
		return undef;
	}	
	my ($installation, $version,$manifest, $objMode, $platform);
	my $result = {};
	foreach my $path (keys (%$installations)){
		$installation = new SDB::Install::Installation::Client ($path); 
		$version = $installation->getVersionByManifest();
		if (!$version){
			next;
		}
		if ($isWin || $isLinux){
			$objMode = $installation->is64bit() ? '64bit' : '32bit';
		}
		
		$result->{$path} = {
			'version' => (defined $version ? $version : $installations->{$path}),
			'mode' => $objMode,
			'canUpdate' => $self->{kit}->canUpdate ($installation, $self)
		};		
	}
	return $result;
}

sub isInstallationSelected{
	return $_[0]->isValuePreset ('PATH');
}

sub InitDefaults{
	my ($self, $kit) = @_;
	if (defined $kit){
		$self->{kit} = $kit;
	}	
	$self->{_kitversion} = $kit->GetVersion;
	$self->{_kitplatform} = $kit->getPlatform ();
}

sub getProductName{
	return $gProductNameClient;
}


sub getProductVersion {
	return $_[0]->{_kitversion};
}

sub getShortProductName{
	return $gShortProductNameClient;
}

sub new{
	my $self = shift->SUPER::new (@_);
	my $order = 0;

	my $progpath = $self->getDefaultProgPath();
	
	if ($isWin){
		$progpath .= '\sap\hdbclient';
	}
	else{
		$progpath .= '/sap/hdbclient';
		if (!$installerIs64bit){
			$progpath .= '32';
		}
	}
	$self->{params} = {
        'SID' => {
            'order' => $order++,
            'opt' => 'sid',
            'short_opt' => 's',
            'type' => 'string',
            'section' => $section,
            'value' => undef,
            'str' => 'System ID',
            'opt_arg' => '<SID>',
            'init_with_default' => 0,
            'mandatory' => 0,
            'set_interactive' => 0
        },
    ($isWin ? (
        'SkipVCRedist' => $self->getParamSkipVCRedist ($order++)
    ): (
        'Target' => {
            'order' => $order++,
            'opt' => 'sapmnt',
            'alias_opts' => ['target'],
            'type' => 'path',
            'opt_arg' => '<installation_path>',
            'section' => $section,
            'value' => undef,
            'default' => $gSapmntDirUx,
            'set_interactive' => 0,
            'str' => 'Mount point for shared client installations'
        }
        )),
		 'PATH' => $self->getParamPATH ($order++, $section, $progpath),
    (!$isWin ? (): (
         'ODBCDriverName' => {
            'order' => $order++,
            'opt' => 'odbc_driver_name',
            'type' => 'string',
            'section' => $section,
            'value' => undef,
            'default' => undef,,
            'str' => 'ODBC Driver Name',
            'init_with_default' => 0,
            'set_interactive' => 0,
            'mandatory' => 0
        }
        )),
        'HostName' => {
            'order' => $order++,
            'opt' => 'hostname',
            'short_opt' => 'H',
            'type' => 'string',
            'section' => $section,
            'value' => undef,
            'default' => lc (hostname ()),
            'str' => 'Host Name',
            'init_with_default' => 0,
            'set_interactive' => 0,
            'persist_for_upgrade' => 1
        }
	};
	return $self;
}

# Override
sub getTimeoutValues{
    return undef;
}

sub setSID{
    my (
        $self,
        $value
    ) = @_;
    if (!$self->checkSID($value)) {
        return 0;
    }
    $self->{'params'}->{'SID'}->{'value'} = $value;
    return 1;
}

sub checkTarget{
    my (
        $self,
        $value
    ) = @_;
    my $properties = $self->{'params'}->{'Target'};
    my $sid = $self->{params}->{SID}->{value};

    if (!-d $value){
        $self->PushError ("$properties->{str} has to be an existing directory");
        return 0;
    }
    if (defined $sid){
        $self->setValue('PATH', join ($path_separator, $value, $sid, 'exe',
        $self->{_kitplatform}, "HDB_CLIENT_$self->{_kitversion}"));
    }
    else{
        $self->setValue('PATH', join ($path_separator, $value, "HDB_CLIENT_$self->{_kitversion}"));
    }
   return 1;
}

sub checkPATH{
	my ($self,$value) = @_;
	if (!$value) {
		$self->PushError("Installation path is empty.");
		return 0;
	}
	my $inst;
	if (-d $value){
		$inst = new SDB::Install::Installation ($value);
		if (!$inst->ErrorState () && $inst->isClient()){
			bless ($inst, 'SDB::Install::Installation::Client');
			my $errlst = new SDB::Install::MsgLst();
			if (!$self->{kit}->canUpdate($inst, $self, $errlst)){
				$self->PushError (undef, $errlst);
				return 0;			
			}
			$self->AddMessage (undef, $errlst);
		}
		else{
			$inst = undef;
		}
	}
	if(exists $self->{'params'}->{'ODBCDriverName'}) {
        my $isUpdate = 0;
        if (defined $inst){
            if (defined $inst->GenPackageList ()){
                $isUpdate = defined $inst->GetPackageById ('odbc');
            }
            $inst->FreePackages ();
        }
        if(!$isUpdate) {
            # we are a new installation with regard to the odbc package.
            if(not defined $self->{'params'}->{'ODBCDriverName'}->{'value'}) {
               $self->{'params'}->{'ODBCDriverName'}->{'value'} = $self->genODBCDrivername();
               $self->AddMessage("Parameter \"ODBCDriverName\" is ".
                                  "set automatically to $self->{'params'}->{'ODBCDriverName'}->{'value'}");
            }
        }
        elsif(defined $self->{'params'}->{'ODBCDriverName'}->{'value'}) {
            # we are a sw update, but user has set the cmdl parameter.
            # we ignore that and tell them in the log file.
           $self->{'params'}->{'ODBCDriverName'}->{'value'} = undef;
           $self->AddMessage("Parameter \"ODBCDriverName\" is ".
                             "ignored: no change of existing driver registrations on update.");
        }
	}
	return 1;
}

sub checkSID{
    my (
        $self,
        $value
    ) = @_;

    if ($isWin){
	return 1;
    }

    my $properties = $self->{'params'}->{'SID'};
    my $user = new SDB::Install::NewDBUser ($value);
    
    if (!$user->exists ()){
        $self->PushError("No such SID: $value", $user);
        return 0;
    }
    $self->{'ClientInstUID'} = $user->uid ();
    $self->{'ClientInstGID'} = $user->gid ();
    return 1;
}

sub setHostname{
    my (
        $self,
        $value
    ) = @_;
    $self->{'params'}->{'HostName'}->{'value'} = lc($value);
    return 1;
}


sub getIgnoreValues{
    my ($self) = @_;
    my @ignoreVals = (
        'check_platform',
        'check_diskspace'
    );
    if (!$self->isCalledBySHA()) {
        push(@ignoreVals, 'check_version');
    }
    return \@ignoreVals;
}

sub setODBCDrivername{
    my (
        $self,
        $value
    ) = @_;
    $self->{'params'}->{'ODBCDrivername'}->{'value'} = $value;
    return 1;
}

sub genODBCDrivername {
    my (
        $self
    ) = @_;
    my $driverName = undef;
    if($isWin) {
        my $driverNamePrefix  = 'HDBODBC';
        if(!$installerIs64bit) {
            $driverNamePrefix = 'HDBODBC32';
        }
        my $odbcObj = new SDB::Install::System::Registration::ODBC();
        my $drivernames  = $odbcObj->getDriverNames();
        my $max = 0;
        my $foundFirstinSequence = 0;
        for my $name (@{$drivernames}) {
            if($name =~ /^$driverNamePrefix/) {
                if($name =~ /^$driverNamePrefix$/) {
                    #                         ^^^
                    # the first installed driver is called 'HDBODBC' instead of 'HDBODBC_001':
                    $foundFirstinSequence = 1;
                }
                else {
                    # the subsequently installed drivers have names which contain serial numbers,
                    # starting with 'HDBODBC_002'. the next serial is just 'max(existing serials)+1';
                    # gaps in the sequence of previously installed serials are not reused. 
                    my ($serial) = ($name =~ /^${driverNamePrefix}_(.*)$/);
                    $serial =~ s/^0*([^0]*)/$1/; # trim leading zeroes, if any
                    if($serial =~ /^[0-9]+$/) {
                        $max = ($serial > $max) ? $serial : $max;
                    }
                }
            }
        }
        if($max == 0) { # ie we dont have numbered drivernames
            if($foundFirstinSequence) {
                # we install the 2nd driver on this machine
                $driverName = $driverNamePrefix.'_002';
            }
            else {
                # we install the 1st driver on this machine
                $driverName = $driverNamePrefix;
            }
        }
        else {
            $driverName = $driverNamePrefix.'_'.sprintf("%03d", $max+1);
        }
    }
    return $driverName;
}

sub isAdminRequired{
    return 0;
}

1;
