package SDB::Install::Configuration::ExternalProgramConfiguration;

use strict;

use base 'SDB::Install::Configuration';
use LCM::ProcessExecutor;
use SDB::Install::Configuration::XMLConfigParser;
use SDB::Install::Tools qw (printTableToArrayOfLines);
use File::stat;

our @cmdln_options = ('--dump_config_xml', '--batch');

sub new{
    my $self = shift->SUPER::new ();
    ($self->{program}, $self->{args}) = @_;
    return $self;
}

sub init{
    my ($self,$program, $args) = @_;
    if (!defined $program){
        $program = $self->{program};
    }
    if (!defined $args){
        $args = $self->{args};
    }

    if (!defined $program){
        $self->setErrorMessage ("no program defined");
        return undef;
    }

    my $st = File::stat::stat ($program);

    if (!defined $st){
        $self->setErrorMessage ("Cannot access file '$program': $!");
        return undef;
    }

    if (!-x $st){
        $self->setErrorMessage ("'$program' is not an executable");
        return undef;
    }
    my @args;
    if (defined $args){
         @args = @$args;
    }
    push @args, @cmdln_options;
    local %ENV = %ENV;
    delete $ENV{'HDB_INSTALLER_TRACE_FILE'};
    $ENV{LANG} = 'en_US.UTF-8';
    $ENV{LC_MESSAGES} = 'en_US.UTF-8';
    my $exer = new LCM::ProcessExecutor ($program,\@args);
    my $msg = $self->getMsgLst ()->addMessage ("Getting external program configuration");
    my $rc = $exer->executeProgram ();
    my $msglst = $msg->getSubMsgLst ();
    if (!defined $rc){
        $self->setErrorMessage ("Cannot execute external program", $exer->getErrMsgLst ());
        return undef;
    }
    my $output = $exer->getOutputLines ();
    if (!defined $output){
        $output = [];
    }
    if ($rc){
        foreach my $line (@$output){
            if ($line =~ /wrong usage/i){
                $msglst->addMessage ('Dynamic command line option detection is not supported.');
                return 0;
            }
        }
        $self->setErrorMessage ("Cannot get external program configuration", $exer->getMsgLst ());
        return undef;
    }
    my $parser = new SDB::Install::Configuration::XMLConfigParser ();

    if (!defined $parser->parse (join ("\n", @$output))){
        $self->setErrorMessage ("Cannot parse external program configuration: $@", $exer->getMsgLst ());
        return undef;
    }
    my $table = [
        ['Installer Version:', $parser->getConfigInstallerVersion ()],
        ['Component Version:', $parser->getConfigComponentVersion ()],
        ['Configuration Class:', $parser->getConfigClass ()],
    ];
    my $lines = printTableToArrayOfLines ($table, '  ');
    foreach my $line (@$lines){
        $msglst->addMessage ($line);
    }
    $self->{params} = $parser->getParams ();
    $self->{timeouts} = $parser->getTimeoutValues ();
    $self->{ignore_values} = [$parser->getIgnoreValues ()];
    return 1;
}

sub getTimeoutValues{
    return $_[0]->{timeouts};
}


sub getIgnoreValues{
    return $_[0]->{ignore_values}->[0];
}


sub getHiddenIgnoreValues{
    return $_[0]->{ignore_values}->[1];
}

1;
