package SDB::Install::Configuration::GenericServerPlugin;

use strict;
use SDB::Install::Configuration qw ($bool_false_pattern $bool_true_pattern);
use SDB::Install::Configuration::Generic;
use SDB::Install::SysVars;
use SDB::Install::SAPSystem;
use SDB::Install::Globals qw ($gProductNameSystem $gProductNameEngine);
use SDB::Install::DebugUtilities qw(dumpThings);

our $installationClass = 'SDB::Install::Installation::GenericServerPlugin';
our $kitClass = 'SDB::Install::Kit::GenericServerPlugin';

our $SQL_ROLE_ONLINE_REGISTRATION = 'AFLPM_ONLINE_REGISTRATION_EXECUTE';

our @ISA = qw (SDB::Install::Configuration::Generic);

sub getInstallationClass{
    return $installationClass;
}

sub getKitClass{
    return $kitClass;
}

sub new{
	my $self = shift->SUPER::new (@_);
	$self->defineInstallParams();
	return $self;
}

sub isUpdate {
    return $_[0]->{_isPluginUpdate};
}

sub InitDefaults{
    my $self = shift;
    my $rc = $self->SUPER::InitDefaults (@_);
    if (!$isWin && $> != 0){
        my $name = getpwuid ($>);
        my ($sid) = ($name =~ /^(\w{3})adm$/);
        if (!$sid){
            $self->PushError ("Wrong user: please start upgrade as root or <sid>adm user");
            return undef;
        }
        $sid = uc ($sid);
        if (!$self->setValue ('SID', $sid)){
            return undef;
        }
    }
    return $rc;
}


sub defineInstallParams{
    my ($self) = @_;
    my $order = 0;
    $self->{params} = {
        'CustomPluginsPath' => $self->getParamCustomPluginsPath($order++),
        'SID' => {
            'order' => $order++,
            'opt' => 'sid',
            'short_opt' => 's',
            'type' => 'string',
            'value' => undef,
            'default' => undef,
            'str' => "$gProductNameSystem ID",
            'opt_arg' => '<SID>',
            'mandatory' => 1
        },
        'SystemIsOffline'  => $self->getParamSystemIsOffline ($order++),
        'NoStart' => {
            'order' => $order++,
            'opt' => 'nostart',
            'type' => 'boolean',
            'value' => undef,
            'default' => 0,
            'str' => 'Do not start the instance after installation',
            'console_omit_word_Enter' => 1,
            'init_with_default' => 1,
            'set_interactive' => 0,
            'hidden' => 0
        },
        'SystemUser'   => $self->getParamSystemUser  ($order++, undef, 1),           # (order, section, skip)
        'SQLSysPasswd' => $self->getParamSQLSysPasswd($order++, undef, 'passwd', 1), # (order, section, type, skip),
        'UseHttp'      => $self->getParamUseHttp ($order++),
        'SSOCertificate' => $self->getParamSSOCertificate ($order++),
        'Password' => {
            'order' => $order++,
            'opt' => 'password',
            'short_opt' => 'p',
            'type' => 'passwd',
            'value' => undef,
            'default' => undef,
            'str' => 'System Administrator Password',
            'log_value' => sub {'***';},
            'mandatory' => 1
        }
    };
    return 1;
}


sub checkSID{
	my ($self, $value) = @_;
	my $properties = $self->{params}->{SID};

	if ($value !~ /^[A-Z][A-Z0-9][A-Z0-9]$/){
		$self->AddError ("Invalid $properties->{str}");
		return 0;
	}

	my $systems = CollectSAPSystems ();

	if (!exists $systems->{$value}){
		$self->AddError ("$properties->{str} \"$value\" " .
							"not found");
		return 0;
	}

	my $system = $systems->{$value};

	if (!$system->hasNewDB()){
			if ($system->hasTrex()){
				my $path = join ($path_separator,$system->{_usrSapSid},qw (SYS global trex install bin hdbuninst));
				if ($isWin){
					$path .= '.exe';
				}
				$self->AddError ("Cannot upgrade old NewDB instance (incompatible file system changes).");
				$self->PushError ("Please uninstall it using '$path' and reinstall it afterwards!");
			}
			else{
				$self->AddError ("$properties->{str} \"$value\" " .
					"has no $gProductNameEngine instance");
			}
			return 0;
	}

	$self->{params}->{Password}->{skip} = 0;
	$self->setSkip ('SSOCertificate', 0);
	if (!$self->{kit}->checkComponentDependency ($system)){
		# if plugin won't be activated, then no restart is required
		# => skipping password parameter
		$self->{params}->{Password}->{skip} = 1;
		$self->setSkip ('SSOCertificate');
		$self->{'_skipSystemUser'} = 1;
		$self->{_canActivate} = 0;
	}
	else{
		$self->{_canActivate} = 1;
	}
	my $instance;
	if (!$self->{params}->{Password}->{skip} && !$isWin){
		$instance = $system->getNewDBInstances()->[0];
		my $otherHosts = $instance->get_hosts ();
		if (!defined $otherHosts || @$otherHosts == 0){
			# skip sidadm password for single host systems on Linux
			$self->{params}->{Password}->{skip} = 1;
			$self->setSkip ('SSOCertificate');
		}
	}
	
	#
	# checking if there is a new installation, which isn't finished yet
	#

	my $origin_class = ref ($self);
	require SDB::Install::Configuration::NewDB;
	bless ($self, 'SDB::Install::Configuration::NewDB');
	$self->{current_sid} = $value;
	my $newinstallation_pending = $self->pers_exists ();
	bless ($self,$origin_class);
	if ($newinstallation_pending){
		$self->AddError ("The initial installation of $gProductNameEngine $value is pending. (" . $self->pers_date_string () . ")");
		$self->PushError ("Please resume server installation.");
		return 0;
	}

	my $parser = $self->{configparser};
	my $doTheImport = $parser->getImportContent();
    if($doTheImport =~ /$bool_true_pattern/i) {
        my $nostart = $self->getValue ('NoStart');
        if (defined $nostart && $nostart  !~ /$bool_true_pattern/i) {
            if(not $self->{'_skipSystemUser'}) {
                $self->{params}->{SystemUser}->{skip} = 0;
                $self->{params}->{SQLSysPasswd}->{skip} = 0;
            }
            else {
                $self->{params}->{SystemUser}->{skip} = 1;
                $self->{params}->{SQLSysPasswd}->{skip} = 1;
            }
        }
    }
	if (!$self->{kit}->canInstall ($system, $self)){
		$self->{params}->{SID}->{no_retry} = 1;
		$self->AddError (undef, $self->{kit});
		return 0;
	}

	$instance = $instance // $system->getNewDBInstances()->[0];
	my $installedPlugin = $instance->getPluginInstallation ($self->{productKey});

	$self->{_isPluginUpdate} = defined $installedPlugin ? 1 : 0;
	if ($self->{_isPluginUpdate}){
		$self->{_isInstalledPluginOnlineUpdatable} = $installedPlugin->getManifest()->isServerPluginOnlineUpdatable();
	}
	else{
		$self->{_isInstalledPluginOnlineUpdatable} = 0;
	}
	return 1;
}

#-------------------------------------------------------------------------------
# Checks whether --system_is_offline can be used

sub checkSystemIsOffline{
    my ($self, $value) = @_;
    my $param = $self->{params}->{SystemIsOffline};
    if (defined $value && $value  =~ /$bool_true_pattern/i){
        my $paramIdNoStart = 'NoStart';
        my $nostart = $self->assignBoolValue($paramIdNoStart, $self->getBatchValue($paramIdNoStart));
        if (!$nostart){
             $self->appendErrorMessage ("--$param->{opt} requires " . $self->getOpt($paramIdNoStart));
            return 0;
        }
        $self->setSkip('Password');
    }
    return 1;
}

sub checkNoStart{
	my ($self, $value) = @_;
	if (defined $value && $value  !~ /$bool_true_pattern/i){
	    my $parser = $self->{configparser};
    	my $doTheImport = $parser->getImportContent();
        if($doTheImport =~ /$bool_true_pattern/i) {
            if(not $self->{'_skipSystemUser'}) {
                $self->{params}->{SystemUser}->{skip} = 0;
                $self->{params}->{SQLSysPasswd}->{skip} = 0;
            }
        }
    }

    if ($self->isSkipped ('SystemUser')  && $self->{_canActivate} &&
        $self->isOnlineUpdatable ()){

        if ($self->isDBRunning ()){
            $self->setSkip ('SystemUser', 0);
            $self->setSkip ('SQLSysPasswd', 0);
        }
    }
	return 1;
}

sub checkPassword{
    my ($self, $value) = @_;

    my $sapsys = $self->getSystem ();

    if (!defined $sapsys){
        $self->getMsgLst()->addMessage ("Skipping password check due to unknown sid");
        return 1;
    }

    my $user = $sapsys->getUser ()->getSidAdm();

    if (!defined $user || !$user->exists){
        return 1;
    }

    my $rc = $user->verifyPassword ($value);

    if (!defined $rc){
        $self->AddWarning ("Cannot check password", $user);
        $self->AddMessage ("=> pass check");
        return 1;
    }

    if (!$rc){
        $self->AddError ('Unknown user password combination');
        return 0;
    }
    return 1;
}


sub isOnlineUpdatable{
    my ($self) = @_;
    if (!defined $self->{_onlineUpdatable}){
        my $msg = $self->getMsgLst()->addMessage("Checking whether online update is supported");
        my $msglst = $msg->getSubMsgLst ();
        if ($self->isUpdate()){
            $msglst->addMessage ("It's an update.");
            my $mf = $self->{kit}->getManifest ();
            if (!defined $mf){
                $self->appendErrorMessage ("Installation kit has no manifest");
                return undef;
            }
            if (!$mf->isServerPluginOnlineUpdatable ()){
                $msglst->addMessage ("New server plugin doesn't support online update.");
                $self->{_onlineUpdatable} = 0;
            }
            else{
                $msglst->addMessage ("New server plugin supports online update.");
                if (!$self->{_isInstalledPluginOnlineUpdatable}){
                    $msglst->addMessage ("Installed server plugin doesn't support online update.");
                    $self->{_onlineUpdatable} = 0;
                }
                else{
                    $msglst->addMessage ("Already installed server plugin supports online update as well.");
                    $self->{_onlineUpdatable} = 1;
                }
            }
            $self->{_onlineUpdatable} = $mf->isServerPluginOnlineUpdatable () && $self->{_isInstalledPluginOnlineUpdatable};
        }
        else{
            # disable online registration for new installations
            $msglst->addMessage ("It's a new installation.");
            $self->{_onlineUpdatable} = 0;
        }
        if ($self->{_onlineUpdatable}){
            $msglst->addMessage ("=> Online update is supported.");
        }
        else{
            $msglst->addMessage ("=> Online update isn't supported.");
        }
    }
    return $self->{_onlineUpdatable};
}


sub isDBRunning{
    my ($self) = @_;
    if ($self->getBatchValue('SystemIsOffline') =~ /$bool_true_pattern/i){
        return 0;
    }
    my $sapsys = $self->getSystem ();
    if (!defined $sapsys){
        return undef;
    }
    my $userName = $sapsys->getUser ()->getSidAdmName();
    if ($sapsys->_isRunning ($self, $userName, $self->getValue ('Password'))){
        return 1;
    }
    return 0;
}



sub checkSQLSysPasswd{
    my ($self, $value) = @_;
    my $sql;
    my $rc = $self->SUPER::checkSQLSysPasswd ($value, undef, undef, \$sql);

    if (!$rc){
        return $rc;
    }

    if (defined $rc && $rc && defined $sql && $self->{_canActivate} &&
        $self->isOnlineUpdatable ()){
        my $sqlUser = $self->getValue ('SystemUser');
        if (uc ($sqlUser) ne 'SYSTEM'){
            if (!defined $sql->execute ("SELECT ROLE_NAME from EFFECTIVE_ROLES WHERE USER_NAME = CURRENT_USER AND ROLE_NAME = '$SQL_ROLE_ONLINE_REGISTRATION'")){
                $self->setErrorMessage ("Cannot check prilileges of sql user '$sqlUser'", $sql->getErrMsgLst ());
                return undef;
            }
            my $resultSet = $sql->getResultSet ();
            if (!defined $resultSet){
                $self->setErrorMessage ('Cannot get result set', $sql->getErrMsgLst ());
                return undef;
            }
            if (!@$resultSet){
                $self->setErrorMessage ("Database user '$sqlUser' requires role '$SQL_ROLE_ONLINE_REGISTRATION'");
                return undef;
            }
        }
        $self->setSkip ('Password');
        $self->setSkip ('SSOCertificate');
        $self->{_onlineActivationSQLConnection} = $sql;
    }
    return 1;
}

sub getOnlineActivationSQLConnection{
    return $_[0]->{_onlineActivationSQLConnection};
}

sub getSystem{
	my $self = $_[0];
	my $sid = $self->getValue ('SID');
	if (!defined $sid){
		$sid = $self->{current_sid};
	}
	my $systems = CollectSAPSystems ();
	my $system = $systems->{$sid};
	return $system;
}

sub getTimeoutValues{
    return [qw (start_service stop_service start_instance stop_instance)];
}

sub getPluginPath{
    # returns <hanamnt>/<SID>/exe/<platform>/plugins
    my $self = $_[0];
    my $customPluginsPath = $self->getBatchValue ('CustomPluginsPath');
    if (defined $customPluginsPath){
        return $customPluginsPath;
    }
    my $system = $self->getSystem();
    return $system->get_pluginsDir();
}

sub get_InstallationPath{
	# returns installation path for this plugin:
	# <hanamnt>/<SID>/exe/<platform>/plugins/<plugin_name>_<version>
	my $self = $_[0];
	my $path;
	if ($isWin){
		$path = $self->getPluginPath() . '_new' . $path_separator . $self->{productKey};
	}else{
		$path = $self->getPluginPath() . $path_separator . $self->{productKey} . '_' . $self->{kit}->getVersionIdString ();
	}
	return $path;
}

sub isAdminRequired{
    return $isWin;
}

sub setUpdateParams{
	return 1;
}

sub enumExistingInstallations{
	return {};
}

1;
