package SDB::Install::Configuration::HdbHostctrl;

use strict;

use SDB::Install::Configuration::AnyConfig;
use SDB::Install::Configuration::AnyMultiHostConfig qw ($validHostRoles);
use SDB::Install::Configuration::AnyRegisterRename;
use SDB::Install::Configuration qw ($bool_false_pattern $bool_true_pattern);
use SDB::Install::DebugUtilities;
use SDB::Install::Globals qw ($gHostRoleAcceleratorWorker
                              $gOperationAddHost
                              $gOperationCollectHostInfo
                              $gOperationDBUpgradeHost
                              $gOperationModifyHost
                              $gOperationRegisterHost
                              $gOperationRemoveHost
                              $gOperationRenameHost
                              $gOperationUninstallInstance
                              $gOperationUnregisterHost
                              $gProductName
                              $gProductNameSystem);
use SDB::Install::NewDBUser;
use SDB::Install::RemoteHosts;
use SDB::Install::SAPHostControl;
use SDB::Install::SysVars qw ($isWin);

require SDB::Install::Configuration::AddHost;
require SDB::Install::Configuration::AnyRename;

our @ISA = qw (SDB::Install::Configuration::AnyConfig);

#-------------------------------------------------------------------------------
# Constructor
sub new {

    my $self = shift->SUPER::new (@_);

    my $order         = 0;
    my $section       = 'hostctrl';
    my $addHostOnly   = 'Valid for addhost';
    my $renameOnly    = 'Valid for rename';
    my $keepUserValid = '(Valid for removehost, rename, uninstall, unregister)';
    my $acceleratorOnly = "Valid for addhost with $gHostRoleAcceleratorWorker/standby only";
    my $collectHostOnly = 'Valid for collect_host_info';

    $self->{params} = {
        'OperationName'       => $self->getParamOperationName      ($order++, $section),
        'ArgStartOnly'        => $self->getParamStartOnly          ($order++, $section),
        'ArgStopOnly'         => $self->getParamStopOnly           ($order++, $section),
        'ArgCheckOnly'        => $self->getParamArgCheckOnly       ($order++, $section),
        'Target'              => $self->getParamTarget             ($order++, $section),
        'SID'                 => $self->getParamSID                ($order++, $section),
        'RemoteHostList'      => $self->getParamRemoteHostList     ($order++, $section),
        'RemoteHost'          => $self->getParamRemoteHost         ($order++, $section),
        'HostagentUserName'   => $self->getParamHostagentUserName  ($order++, $section, undef, 0, 0),
        'HostagentPassword'   => $self->getParamHostagentPassword  ($order++, $section),
        'Password'            => $self->getParamPassword           ($order++, $section),
        'ArgHostname'         => $self->getParamArgHostname        ($order++, $section, $addHostOnly),
        'ArgHostRole'         => $self->getParamArgHostRole        ($order++, $section, $addHostOnly),
        'ArgHostFailoverGroup'=> $self->getParamArgHostFailoverGroup($order++,$section, $addHostOnly),
        'ArgStoragePartition' => $self->getParamArgStoragePartition($order++, $section, $addHostOnly),
        'ArgWorkerGroup'      => $self->getParamWorkerGroup($order++, $section, $addHostOnly),
        'ArgTargetSID'        => $self->getParamArgTargetSID       ($order++, $section, $renameOnly),
        'ArgTargetNum'        => $self->getParamArgTargetNumber    ($order++, $section, $renameOnly),
        'ArgTargetPassword'   => $self->getParamArgTargetPassword  ($order++, $section, $renameOnly),
        'ArgSystemUser'       => $self->getParamSystemUser         ($order++, $section, 1, $acceleratorOnly),
        'ArgSQLSysPasswd'     => $self->getParamSQLSysPasswd       ($order++, $section, 'passwd', 1, undef, $acceleratorOnly),
        'ArgAcceleratorUser'  => $self->getParamAcceleratorUser    ($order++, $section, $acceleratorOnly),
        'ArgAcceleratorPw'    => $self->getParamAcceleratorPassword($order++, $section, $acceleratorOnly),
        'ArgKeepUser'         => $self->getParamArgKeepUser        ($order++, $section, $keepUserValid),
        'ArgKeepUserHomeDir'  => $self->getParamArgKeepUserHomeDir ($order++, $section, $keepUserValid),
        'ArgKeepXsUsers'      => $self->getParamArgKeepXsUsers     ($order++, $section, $keepUserValid),
        'ArgSkipModifySudoers'=> $self->getParamArgSkipModifySudoers($order++, $section, $keepUserValid),
        'ArgNoStart'          => $self->getParamArgNoStart         ($order++, $section),
        'ArgSkipCollect'      => $self->getParamArgSkipCollect     ($order++, $section, $collectHostOnly),
        'ArgSkipCollectCons'  => $self->getParamArgSkipCollectCons ($order++, $section, $collectHostOnly),
        'ArgEnableParams'     => $self->getParamArgEnableParams    ($order++, $section, $collectHostOnly),
        'ArgSapadm'           => $self->getParamArgSapadm          ($order++, $section, $collectHostOnly),
        'ArgSidadm'           => $self->getParamArgSidadm          ($order++, $section, $collectHostOnly),
        'ArgUserid'           => $self->getParamArgUserid          ($order++, $section, $collectHostOnly),
        'ArgGroupid'          => $self->getParamArgGroupid         ($order++, $section, $collectHostOnly),
        'ArgDataPath'         => $self->getParamArgDataPath        ($order++, $section, $collectHostOnly),
        'ArgLogPath'          => $self->getParamArgLogPath         ($order++, $section, $collectHostOnly),
        'ArgIgnore'           => $self->getParamArgIgnore          ($order++, $section),
        'UseHttp'             => $self->getParamUseHttp            ($order++, $section, 0, 0),
        'DumpHostctrlResult'  => $self->getParamDumpHostctrlResult ($order++, $section),
        'EnableHostctrlTrace' => $self->getParamEnableHostctrlTrace($order++, $section),
    };

    my $params = $self->{params};

    # Target and SID are set by SBD::Install::Installer::getInstallation
    $params->{SID}->{hidden}    = 1;
    $params->{Target}->{hidden} = 1;

    # ArgStartOnly and ArgStopOnl are set via corresponding operation name
    $params->{ArgStartOnly}->{hidden} = 1;
    $params->{ArgStopOnly}->{hidden}  = 1;

    $self->setSkip('Password');

    $params->{SID   }->{hostctrl_opt} = 'SID';
    $params->{Target}->{hostctrl_opt} = 'SAPMNT';

    $params->{HostagentPassword}->{type} = 'passwd';

    if (!$isWin) {
        $params->{HostagentUserName}->{set_interactive} = 0;
    }

    foreach my $id (keys (%$params)) {
        delete $params->{$id}->{short_opt};
    }

    $self->setSummaryOrder(['OperationName',
                            'Target',
                            'SID',
                            'RemoteHostList',
                            'RemoteHost',
                            'HostagentUserName',
                            'ArgTargetSID',        # rename   only
                            'ArgTargetNum',        # rename   only
                            'ArgHostname',         # addhost  only
                            'ArgHostRole',         # addhost  only
                            'ArgHostFailoverGroup',# addhost  only
                            'ArgWorkerGroup', # addhost  only
                            'ArgStoragePartition', # addhost  only
                            'ArgSystemUser',       # addhost  only
                            'ArgAcceleratorUser',  # addhost  only
                            'ArgCheckOnly',
                            'ArgKeepUser',
                            'ArgKeepUserHomeDir',
                            'ArgKeepXsUsers',
                            'ArgSkipModifySudoers',
                            'ArgNoStart',
                            'ArgSkipCollect',
                            'ArgSapadm',
                            'ArgSidadm',
                            'ArgUserid',
                            'ArgGroupid',
                            'ArgDataPath',
                            'ArgLogPath',
                            'UseHttp',
                            'DumpHostctrlResult',
                            'EnableHostctrlTrace']);

	return $self;
}


#-------------------------------------------------------------------------------
# Checks the parameter '--check_only'.

sub checkArgCheckOnly {

    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {

        if ($self->getValue('OperationName') eq $gOperationRenameHost) {
            $self->setSkip('ArgTargetSID', 0);
            $self->setSkip('ArgTargetNum', 0);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the multiple parameter '--role=<role>'

sub checkArgHostRole {

    my ($self, $roleArray) = @_;

    foreach my $currRole (@$roleArray) {

        if (!exists $validHostRoles->{$currRole}) {
            $self->PushError ("Unkown host role '$currRole'");
            return 0;
        }

        if ($validHostRoles->{$currRole}->{isAccelerator}) {
            $self->setSkip('ArgSystemUser',      0);
            $self->setSkip('ArgSQLSysPasswd',    0);
            $self->setSkip('ArgAcceleratorUser', 0);
            $self->setSkip('ArgAcceleratorPw',   0);
        }

        if (defined $validHostRoles->{$currRole}->{defaultGroup}) {
            $self->setSkip('ArgHostFailoverGroup', 0);
            my $paramGroup         = $self->{params}->{ArgGroup};
            my $targetDefault = (defined $self->getDefault('ArgGroup'))
                                     ? $validHostRoles->{worker   }->{defaultGroup}
                                     : $validHostRoles->{$currRole}->{defaultGroup};
            $self->setDefault('ArgGroup', $targetDefault);
            $paramGroup->{init_with_default} = 1;
        }

        if ($validHostRoles->{$currRole}->{hasPartition}) {
            $self->setSkip('ArgStoragePartition', 0);
        }
        if ($validHostRoles->{$currRole}->{hasWorkerGroup}) {
            $self->setSkip('ArgWorkerGroup', 0);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the parameter '--check_only'.

sub checkArgSkipCollect {

    my ($self, $value) = @_;

    if (defined $value) {
        $self->setSkip('ArgSkipCollectCons');
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the SID.

sub checkSID {

    my ($self, $sid) = @_;

    my $params = $self->{params};

    if ($sid !~ /^[A-Z][A-Z0-9][A-Z0-9]$/){
        $self->PushError("Invalid $params->{ArgSID}->{str} '$sid'");
        return 0;
    }

    my $sapSys = $self->getSAPSystem();
    my $user   = (defined $sapSys) ? $sapSys->getUser() : undef;

    if (defined $user) {
        $self->{_sidAdmUser} = $user->getSidAdm();
        $self->{_sidAdmName} = $user->getSidAdmName();
    }

    my $operationName = $self->getValue('OperationName');

    if (defined $self->{_sidAdmName}) {

        my $userExists = $user->exists();

        $params->{Password}->{type} = 'passwd' if ($userExists);
        $params->{Password}->{str}  =
            sprintf($params->{Password}->{str_templ}, $self->{_sidAdmName});

        if (($operationName eq $gOperationRenameHost)
            && !$self->getValue('ArgCheckOnly')) {

            $params->{ArgTargetPassword}->{type} = 'passwd' if ($userExists);
            $params->{ArgTargetPassword}->{str}  =
                    sprintf($params->{ArgTargetPassword}->{str_templ},
                            $self->{_sidAdmName});
        }

        if (($operationName eq $gOperationDBUpgradeHost)
            ||
            ($operationName eq $gOperationModifyHost)
            ||
            (($operationName eq $gOperationRegisterHost)
              && (defined $self->getValue('ArgStartOnly') ||
                  defined $self->getValue('ArgStopOnly')))) {

            $self->setDefault('HostagentUserName', $self->{_sidAdmName}, 1);
            $params->{HostagentUserName}->{set_interactive} = 0;
        }
    }

    if ($operationName eq $gOperationCollectHostInfo) {

        $self->setDefault('ArgSidadm', $self->{_sidAdmName});

        if (defined $user) {
            $self->setDefault('ArgUserid', $user->uid());
            $self->setDefault('ArgGroupid', $user->gid());
        }

        my $instance = $self->getOwnInstance();

        if (defined $instance) {
            $self->setDefault('ArgDataPath', $instance->getDataPath(1));
            $self->setDefault('ArgLogPath', $instance->getLogPath(1));
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the parameter '--start_only'.

sub checkArgStopOnly {

    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {
        $self->setSkip('ArgCheckOnly');
        $self->{params}->{ArgNoStart}->{str} =
                "Keep sapstartsrv running (parameter '--nostart')";
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the target instance number.

sub checkArgTargetNumber {

    my ($self, $value) = @_;
    return $self->checkInstanceNumber($value);
}


#-------------------------------------------------------------------------------
# Checks the parameter 'ArgTargetPassword'

sub checkArgTargetPassword{

    my ($self, $password) = @_;

    return $self->checkPasswordOrTargetPassword
            ($self->{params}->{ArgTargetPassword}, $password);
}


#-------------------------------------------------------------------------------
# Checks the remote host name and sets the default hostname for addhost.

sub checkRemoteHost {

    my ($self, $value) = @_;

    if ($value =~ /^[a-zA-Z]/) {
        $self->setDefault('ArgHostname', $value);
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the list of remote host names.

sub checkRemoteHostList {

    my ($self, $value) = @_;
    $self->setSkip('RemoteHost');
    return 1;
}


#-------------------------------------------------------------------------------
# Adds the hostagent user name to the password string.

sub checkHostagentUserName {

    my ($self, $name) = @_;

    if ($name eq $self->{_sidAdmName}) {
        $self->setSkip('Password', 0);
    }
    else {
        $self->setSkip('HostagentPassword', 0);
        $self->{params}->{HostagentPassword}->{str} =
            sprintf($self->{params}->{HostagentPassword}->{str_templ}, $name);
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks the parameter 'Password'

sub checkPassword{

    my ($self, $password) = @_;

    return $self->checkPasswordOrTargetPassword($self->{params}->{Password},
                                                $password);
}


#-------------------------------------------------------------------------------
# Checks the password of the system administrator

sub checkPasswordOrTargetPassword {

    my ($self, $param, $password) = @_;

    if (defined $self->{_sidAdmUser} && ($param->{type} eq 'passwd')) {

        if (!$self->verifySidAdmPassword($self->{_sidAdmUser},
                                         $password,
                                         $param->{str})) {
            return 0;
        }
    }
    elsif (!defined $password || length ($password) < 8){
        $self->PushError ("$param->{str} must contain at least 8 characters");
        return 0;
    }

    return 1;
}

#-------------------------------------------------------------------------------
# Dumps the specified structure if 'DumpHostctrlResult' is set.

sub dumpStructure {

    my ($self, $struct, $headline) = @_;

    if ($self->getValue('DumpHostctrlResult')) {

        my $delimiterLine = "\n" . ('=' x length($headline)) . "\n";

        print($delimiterLine);
        print("$headline\n"); dumpThings($struct, 8, 2);
        print("$delimiterLine\n");
    }
}


#-------------------------------------------------------------------------------
# Executes the selected operation via hostctrl.

sub executeHostctrlOperation {

    my ($self, $host) = @_;

    my $operation = $self->getValue('OperationName');
    my $user      = $self->getValue('HostagentUserName');
    my $userPw    = ($user eq $self->{_sidAdmName})
                    ? $self->getValue('Password')
                    : $self->getValue('HostagentPassword');
    my $useHttp   = $self->getValue('UseHttp');
    my $hostctrl;

    if ($self->getValue('EnableHostctrlTrace')) {

        require SDB::Install::Configuration::HdbHostctrlTrace;
        $hostctrl = new SDB::Install::Configuration::HdbHostctrlTrace
                                              ($host, $user, $userPw, $useHttp);
    }
    else {
        $hostctrl = new SDB::Install::SAPHostControl
                                              ($host, $user, $userPw, $useHttp);
    }

    if ($operation eq 'ListInstances') {

        my $result = $hostctrl->ListInstances();
        $self->getMsgLst()->appendMsgLst($hostctrl->getMsgLst());
        $self->dumpStructure($result, 'Result of ListInstances:');
        if (!defined $result) {
            $self->setErrorMessage("Hostctrl failed!", $hostctrl->getErrMsgLst());
            return undef;
        }

        $self->showResultListInstances($result);
        return 1;
    }

    my $arguments;
    my $paramIDs = ($operation eq $gOperationRenameHost)
                   ? ['ArgTargetPassword']
                   : ($operation eq $gOperationCollectHostInfo) ? []
                                                                : ['Password'];

    if ($operation eq $gOperationAddHost) {

        my $roleArray       = $self->getValue('ArgHostRole');
        $arguments->{ROLES} = join(',', @$roleArray);
        $self->setSkip('ArgHostRole');
        push @$paramIDs, 'ArgSQLSysPasswd';
        push @$paramIDs, 'ArgAcceleratorPw';
    }
    elsif ($operation eq $gOperationCollectHostInfo) {
        my $paramSkipCollect = $self->{params}->{ArgSkipCollect};
        if (!defined $paramSkipCollect->{value}) {
            $paramSkipCollect->{value} = 'users,groups';
        }
        elsif ($paramSkipCollect->{value} ne 'all') {
            $paramSkipCollect->{value} .= ',users,groups';
        }
    }

    foreach my $id (keys (%{$self->{params}})) {

        my $param = $self->{params}->{$id};
        my $arg   = $param->{hostctrl_opt};

        if (defined $arg && !$param->{skip}) {

            my $value = $self->getValue($id);

            if (defined $value) {

                if ($param->{type} =~ /bool/) {
                    $arguments->{$arg} = ($value) ? 'on' : 'off';
                }
                else {
                    $arguments->{$arg} = $value;
                }
            }
        }
    }

    my $xml = $self->getXmlPasswordStream($paramIDs);
    # getXmlPasswordStream() returns undef, or a ref to an array containing a single entry
    $arguments->{XMLINPUT} = $xml->[0] if (defined $xml);

    $self->dumpStructure($arguments,
                                  "Arguments of ExecuteOperation on '$host':");

    my $options = {'mTimeout' => -1};  # synchronous execution

    my $result = $hostctrl->ExecuteOperation($operation,
                                             $arguments,
                                             $options);

    $self->dumpStructure($result, "Result of host '$host':");

    $self->getMsgLst()->appendMsgLst($hostctrl->getMsgLst());

    my $errorInfo =
            "Executing hostctrl operation '$operation' failed on host '$host'!";

    if (!defined $result) {
        my $error = ($hostctrl->errorState()) ? $hostctrl->getErrMsgLst()
                                              : $hostctrl->getMsgLst();
        $self->setErrorMessage($errorInfo, $error);
        return undef;
    }

    my $lines = $hostctrl->getLastResultOutputlines();

    if (defined $lines && @$lines) {

        my $outputMsgLst = $self->getMsgLst();
        $outputMsgLst->addProgressMessage('');
        $outputMsgLst->addProgressMessage
                                    ("   ----- Output from host '$host' -----");
        $outputMsgLst->addProgressMessage('  |');
        my $isFirst = 1;

        foreach my $currLine (@$lines) {
            if (!$isFirst || ((length($currLine) > 0) && ($currLine !~ /^\s*$/))) {
                $outputMsgLst->addProgressMessage('  |  ' . $currLine);
                $isFirst = 0;
            }
        }
        $outputMsgLst->addProgressMessage('');
    }

    my $faultcode = $hostctrl->getLastResult_faultcode();
    my $exitcode  = $hostctrl->getLastResult_exitcode();

    if ((defined $exitcode && ($exitcode != 0)) || defined $faultcode) {
        $self->setErrorMessage($errorInfo . ': '
                                  . $hostctrl->getLastResult_faultstring());
        return undef;
    }

    $self->AddProgressMessage
        ("Hostctrl operation '$operation' done on host '$host'");

    return 1;
}


#-------------------------------------------------------------------------------
# Executes the selected operation via hostctrl on all specified host.

sub executeRemoteOperations {

    my ($self) = @_;

    my $hosts = $self->getHostsFromParam();

    foreach my $currHost (@$hosts) {

        my $rc = $self->executeHostctrlOperation($currHost);
        if (!$rc) {
            return $rc;
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing the host names for hostctrl.

sub getHostsFromParam {

    my ($self, $order, $section) = @_;

    my $host = $self->getValue('RemoteHost');

    if (defined $host) {
        return [$host];
    }

    my $hostList = $self->getValue('RemoteHostList');

    if (!defined $hostList) {
        return [];
    }

    my @hosts;
    foreach my $hostName (split (',', $hostList)) {
        push @hosts, $hostName;
    }

    return \@hosts;
}


#-------------------------------------------------------------------------------
# Parameter '--check_only'

sub getParamArgCheckOnly {

    my ($self, $order, $section) = @_;

    my $param = $self->getParamCheckOnly($order, $section);

    $param->{desc}            = $param->{str};
    $param->{str}             = 'Check Only';
    $param->{additional_desc} = '(Valid for register, rename, unregister, upgrade)';
    $param->{skip}            = 1;
    $param->{set_interactive} = 1;
    $param->{console_omit_word_Enter} = 1,

    return $param;
}


#-------------------------------------------------------------------------------

sub getParamArgDataPath {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'dataPath',
            'opt_arg'           => '<path>',
            'hostctrl_opt'      => 'DATA_PATH',
            'type'              => 'path',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Check Existence of Data Path',
            'desc'              => 'Checks the existence of this data path',
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgEnableParams {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'enable_params',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'default'           => 0,
            'str'               => 'Do you want to enter additional parameters',
            'set_interactive'   => 1,
            'skip'              => 1,
            'hidden'            => 1,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgGroupid {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'groupid',
            'opt_arg'           => '<id>',
            'hostctrl_opt'      => 'GROUPID',
            'type'              => 'number',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Collect Name of Group ID',
            'desc'              => 'Collects the name of this group id',
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgHostFailoverGroup {

    my ($self, $order, $section, $constraint) = @_;

    my $param = $self->getParamHostFailoverGroup($order, $section);

    $param->{hostctrl_opt} = 'GROUP';
    $param->{constraint}   = $constraint;
    $param->{desc} = 'Specifies the failover group of the additional host';
    $param->{skip} = 1;

    return $param;
}


#-------------------------------------------------------------------------------

sub getParamArgHostname {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'hostname',
            'opt_arg'           => '<name>',
            'hostctrl_opt'      => 'HOSTNAME',
            'type'              => 'string',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Host Name of the Additional Host',
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgIgnore {

    my ($self, $order, $section) = @_;

    return {'order'           => $order++,
            'opt'             => 'ignore',
            'opt_arg'         => '<check1>[,<check2>]...',
            'hostctrl_opt'    => 'IGNORE',
            'type'            => 'string',
            'section'         => $section,
            'value'           => undef,
            'default'         => undef,
            'str'             => 'Ignore failing prerequisite checks',
            'set_interactive' => 0,
           };
}


#-------------------------------------------------------------------------------

sub getParamArgKeepUser {

    my ($self, $order, $section, $additionalDesc) = @_;

    my $param = $self->getParamKeepUser($order, $section);

    $param->{additional_desc} = $additionalDesc;
    $param->{set_interactive} = 1;
    $param->{skip}            = 1;

    return $param;
}

#-------------------------------------------------------------------------------

sub getParamArgKeepXsUsers {
    my ($self, $order, $section, $additionalDesc) = @_;
    my $param = $self->getParamKeepXsUsers($order, $section);

    $param->{additional_desc} = $additionalDesc;
    $param->{set_interactive} = 1;
    $param->{skip}            = 1;

    return $param;
}

#-------------------------------------------------------------------------------

sub getParamArgSkipModifySudoers {
    my ($self, $order, $section, $additionalDesc) = @_;
    my $param = $self->getParamSkipModifySudoers($order, $section);

    $param->{additional_desc} = $additionalDesc;
    $param->{set_interactive} = 1;
    $param->{skip}            = 1;

    return $param;
}

#-------------------------------------------------------------------------------

sub getParamArgKeepUserHomeDir {

    my ($self, $order, $section, $additionalDesc) = @_;

    my $param = $self->getParamKeepUserHomeDir($order, $section);

    $param->{additional_desc} = $additionalDesc;
    $param->{set_interactive} = 1;
    $param->{skip}            = 1;

    return $param;
}


#-------------------------------------------------------------------------------

sub getParamArgLogPath {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'logPath',
            'opt_arg'           => '<path>',
            'hostctrl_opt'      => 'LOG_PATH',
            'type'              => 'path',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Check Existence of Log Path',
            'desc'              => 'Checks the existence of this log path',
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgNoStart {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'nostart',
            'hostctrl_opt'      => 'NOSTART',
            'type'              => 'boolean',
            'section'           => $section,
            'value'             => undef,
            'default'           => 0,
            'str'               => "Start sapstartsrv only (parameter '--nostart')",
            'desc'              => 'Does not start the instance, but starts the service (sapstartsrv)',
            'additional_desc'   => '(Valid for addhost, register, rename, start_instance)',
            'skip'              => 1,
            'constraint'        => $constraint,
            'init_with_default' => 1,
            'set_interactive'   => 1,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgHostRole {

    my ($self, $order, $section, $constraint) = @_;

    my $param = $self->getParamHostRole($order, $section);

    $param->{constraint} = $constraint;
    $param->{desc}       = 'Specifies a role of the additional host';
    $param->{skip}       = 1;

    return $param;
}


#-------------------------------------------------------------------------------

sub getParamArgSapadm {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'sapadm',
            'opt_arg'           => '<name>',
            'hostctrl_opt'      => 'SAPADM',
            'type'              => 'string',
            'section'           => $section,
            'value'             => undef,
            'default'           => 'sapadm',
            'str'               => 'Collect Properties of the Host Agent User',
            'desc'              => 'Collects the properties of the user which runs the SAP Host Agent',
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgSidadm {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'sidadm',
            'opt_arg'           => '<name>',
            'hostctrl_opt'      => 'SIDADM',
            'type'              => 'string',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Collect Properties of the Administrator',
            'desc'              => 'Collects the user properties of this administrator',
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------
# The xml tags 'groups' and 'users' are skipped by default (security).

sub getParamArgSkipCollect {

    my ($self, $order, $section, $constraint) = @_;

    my @xmlTags = ('all',
                   'certificateHostname',
                   'networkAddresses',
                   'saphostagentVersion',
                   'sapsysid',
                   'SAPSystems');

    return {
        'order'             => $order,
        'opt'               => 'skip_collect',
        'hostctrl_opt'      => 'SKIP_COLLECT',
        'type'              => 'csv', # comma-separated value
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'valid_values'      => \@xmlTags,
        'str'               => 'Skip XML Tags',
        'desc'              => 'Does not collect these xml tags',
        'constraint'        => $constraint,
        'init_with_default' => 0,
        'set_interactive'   => 0,
        'skip'              => 1,
    };
}


#-------------------------------------------------------------------------------
# Interactive selection of ArgSkipCollect.

sub getParamArgSkipCollectCons {

    my ($self, $order, $section, $constraint) = @_;

    my $param = $self->getParamArgSkipCollect($order, $section, $constraint);

    my @xmlTags;
    foreach my $currTag (@{$param->{valid_values}}) {
        push @xmlTags, $currTag
    }

    push @xmlTags, '-- do not skip xml tags --';

    $param->{hidden}          = 1;
    $param->{hostctrl_opt}    = undef;
    $param->{valid_values}    = \@xmlTags;
    $param->{default}         = $xmlTags[(scalar @xmlTags) -1];
    $param->{set_interactive} = 1;
    $param->{console_text}    = "Select the xml tags which should be skipped:\n";
    $param->{str}             = "XML Tags";
    $param->{interactive_str} = 'comma-separated list of the selected indices';
    $param->{interactive_index_selection} = 1;

    return $param;
};


#-------------------------------------------------------------------------------

sub getParamArgStoragePartition {

    my ($self, $order, $section, $constraint) = @_;

    my $param = $self->getParamStoragePartitionNumber($order, $section);

    $param->{hostctrl_opt}    = 'STORAGE_PARTITION';
    $param->{constraint}      = $constraint;
    $param->{set_interactive} = 1;
    $param->{desc} = 'Specifies the storage partition number of the additional host';
    $param->{skip} = 1;

    return $param;
}

#-------------------------------------------------------------------------------

sub getParamArgWorkerGroup {

    my ($self, $order, $section, $constraint) = @_;

    my $param = $self->getParamWorkerGroup($order, $section);

    $param->{hostctrl_opt}    = 'WG';
    $param->{constraint}      = $constraint;
    $param->{set_interactive} = 1;
    $param->{desc} = 'Specifies the worker group of the additional host';
    $param->{skip} = 1;

    return $param;
}



#-------------------------------------------------------------------------------

sub getParamArgTargetNumber {

    my ($self, $order, $section, $constraint) = @_;

    my $param = $self->getParamInstanceNumber($order,
                                              $section,
                                              'Target Instance Number');

    $param->{default}           = undef;
    $param->{opt_arg}           = '<instance_number>';
    $param->{init_with_default} = 0;
    $param->{constraint}        = $constraint;
    $param->{skip}              = 1;

    return $param;
}


#-------------------------------------------------------------------------------

sub getParamArgTargetPassword {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'     => $order,
            'opt'       => 'target_password',
            'type'      => 'initial_passwd',
            'opt_arg'   => '<password>',
            'section'   => $section,
            'value'     => undef,
            'default'   => undef,
            'str'       => 'Target System Administrator Password',
            'str_templ' => 'Target System Administrator (%s) Password',
            'log_value' => sub {'***';},
            'mandatory' => 1,
            'skip'      => 1,
            'constraint'=> $constraint,
    };
}


#-------------------------------------------------------------------------------

sub getParamArgTargetSID {

    my ($self, $order, $section, $constraint) = @_;

    my $param = $self->getParamNewSID($order, $section);

    $param->{constraint} = $constraint;
    $param->{hidden}     = 0;
    $param->{skip}       = 1;

    return $param;
}


#-------------------------------------------------------------------------------

sub getParamArgUserid {

    my ($self, $order, $section, $constraint) = @_;

    return {'order'             => $order,
            'opt'               => 'userid',
            'opt_arg'           => '<id>',
            'hostctrl_opt'      => 'USERID',
            'type'              => 'number',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Collect Properties of User ID',
            'desc'              => 'Collects the properties of this user id',
            'set_interactive'   => 1,
            'skip'              => 1,
            'constraint'        => $constraint,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------
# Parameter '--dump_hostctrl_result'

sub getParamDumpHostctrlResult {

    my ($self, $order, $section) = @_;

    return {'order'           => $order,
            'opt'             => 'dump_hostctrl_result',
            'type'            => 'boolean',
            'section'         => $section,
            'value'           => undef,
            'default'         => 0,
            'str'             => 'Dump Internal Hostctrl Result',
            'desc'            => 'Dumps the internal result of hostctrl',
            'set_interactive' => 0,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------
# Parameter '--enable_hostctrl_trace'

sub getParamEnableHostctrlTrace {

    my ($self, $order, $section) = @_;

    return {'order'           => $order,
            'opt'             => 'enable_hostctrl_trace',
            'type'            => 'boolean',
            'section'         => $section,
            'value'           => undef,
            'default'         => 0,
            'str'             => 'Enable Hostctrl Trace',
            'desc'            => 'Enables the trace of hostctrl',
            'set_interactive' => 0,
            'console_omit_word_Enter' => 1,
    };
}


#-------------------------------------------------------------------------------
# Parameter '--host=<hostname>'

sub getParamRemoteHost {

    my ($self, $order, $section) = @_;

    return {'order'             => $order,
            'opt'               => 'host',
            'opt_arg'           => '<name>',
            'type'              => 'string',
            'section'           => $section,
            'value'             => undef,
            'default'           => 'localhost',
            'str'               => 'Host Name or IP Address for Hostctrl',
            'desc'              => 'Host name or IP address for hostctrl',
            'init_with_default' => 1,
            'set_interactive'   => 1,
    };
}


#-------------------------------------------------------------------------------
# Parameter '--hostlist=<hostname1>[,<hostname2>]'

sub getParamRemoteHostList {

    my ($self, $order, $section) = @_;

    return {'order'             => $order,
            'opt'               => 'hostlist',
            'opt_arg'           => '<name1>[,<name2>]...',
            'type'              => 'string',
            'section'           => $section,
            'value'             => undef,
            'default'           => undef,
            'str'               => 'Host Names or IP Addresses for Hostctrl',
            'desc'              => 'Host names or IP addresses for multiple execution of hostctrl',
            'set_interactive'   => 0,
    };
}


#-------------------------------------------------------------------------------
# Parameter '--operation_name=<operation>'

sub getParamOperationName {

    my ($self, $order, $section) = @_;

    my $opNames = ['addhost',
                   'collect_host_info',
                   'modify',
                   'register',
                   'removehost',
                   'rename',
                   'start_instance',
                   'stop_instance',
                   'uninstall',
                   'unregister',
                   'upgrade',
                   'list'];

    my $opTools = [$gOperationAddHost,
                   $gOperationCollectHostInfo,
                   $gOperationModifyHost . ' (change listen interface)',
                   $gOperationRegisterHost,
                   $gOperationRemoveHost,
                   $gOperationRenameHost,
                   $gOperationRegisterHost . ' (start sapstartsrv and instance)',
                   $gOperationRegisterHost . ' (stop instance and sapstartsrv)',
                   $gOperationUninstallInstance,
                   $gOperationUnregisterHost,
                   $gOperationDBUpgradeHost. ' (update operating system integration)',
                   'List Instances'];

    my $maxLen = 0;

    foreach my $name (@$opNames) {
        my $len = length($name);
        $maxLen = $len if ($len > $maxLen);
    }

    my $desc = '';

    foreach my $i (0..(scalar(@$opNames)-1)) {
        my $currOp = $opNames->[$i];
        $desc .= '   ' . $currOp . (' ' x ($maxLen-length($currOp)))
              .  ' ' . $opTools->[$i] . "\n";
    }

    return {
        'order'             => $order,
        'opt'               => 'operation',
        'opt_arg'           => '<name>',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'valid_values'      => $opNames,
        'ui_values'         => $opTools,
        'ui_column_header'  => 'Hostctrl Operation',
        'str'               => 'Operation',
        'additional_desc'   => $desc,
        'init_with_default' => 0,
        'set_interactive'   => 1,
        'mandatory'         => 1,
        'interactive_index_selection' => 1,
    };
}


#-------------------------------------------------------------------------------
# Enables the operation specific parameters

sub setOperationName {

    my ($self, $value) = @_;

    my $params = $self->{params};

    if ($value eq 'addhost') {

        $params->{OperationName}->{value} = $gOperationAddHost;
        $self->setSkip('ArgHostname', 0);
        $self->setSkip('ArgNoStart',  0);
        $self->setSkip('ArgHostRole', 0);
        $self->setSkip('Password',    0);
        $self->setSkip('ArgSkipModifySudoers', 0);
    }
    elsif ($value eq 'collect_host_info') {

        $params->{OperationName}->{value} = $gOperationCollectHostInfo;
        $self->setSkip('ArgEnableParams',    0);
        $self->setSkip('ArgDataPath',        0);
        $self->setSkip('ArgGroupid',         0);
        $self->setSkip('ArgLogPath',         0);
        $self->setSkip('ArgSapadm',          0);
        $self->setSkip('ArgSidadm',          0);
        $self->setSkip('ArgSkipCollect',     0);
        $self->setSkip('ArgSkipCollectCons', 0);
        $self->setSkip('ArgUserid',          0);

    }
    elsif ($value eq 'modify') {

        $params->{OperationName}->{value} = $gOperationModifyHost;
        $self->setSkip('Password', 0);
    }
    elsif ($value eq 'register') {

        $params->{OperationName}->{value} = $gOperationRegisterHost;
        $self->setSkip('ArgCheckOnly', 0);
        $self->setSkip('ArgNoStart',   0);
        $self->setSkip('Password',     0);
        $self->setSkip('ArgSkipModifySudoers', 0);
        $params->{Password}->{type} = 'initial_passwd';
    }
    elsif ($value eq 'removehost') {

        $params->{OperationName     }->{value} = $gOperationRemoveHost;
        $self->setSkip('ArgKeepUser',        0);
        $self->setSkip('ArgKeepUserHomeDir', 0);
        $self->setSkip('ArgKeepXsUsers',     0);
        $self->setSkip('ArgSkipModifySudoers', 0);
    }
    elsif ($value eq 'rename') {

        $params->{OperationName     }->{value} = $gOperationRenameHost;
        $self->setSkip('ArgCheckOnly',       0);
        $self->setSkip('ArgKeepUser',        0);
        $self->setSkip('ArgKeepUserHomeDir', 0);
        $self->setSkip('ArgKeepXsUsers',     0);
        $self->setSkip('ArgSkipModifySudoers', 0);
        $self->setSkip('ArgNoStart',         0);
        $self->setSkip('ArgTargetPassword',  0);
    }
    elsif ($value eq 'start_instance') {

        $params->{OperationName}->{value} = $gOperationRegisterHost;
        $self->setSkip('ArgStartOnly', 0);
        $self->setSkip('ArgNoStart', 0);
        $params->{ArgStartOnly}->{hidden} = 0;
        $params->{ArgStartOnly}->{value}  = 'instance';
        $params->{ArgNoStart}->{str} =
                "Start sapstartsrv only (parameter '--nostart')";
    }
    elsif ($value eq 'stop_instance') {

        $params->{OperationName}->{value} = $gOperationRegisterHost;
        $self->setSkip('ArgStopOnly', 0);
        $self->setSkip('Password',    0);
        $params->{ArgStopOnly}->{hidden} = 0;
        $params->{ArgStopOnly}->{value}  = 'instance';
    }
    elsif ($value eq 'upgrade') {

        $params->{OperationName}->{value} = $gOperationDBUpgradeHost;
        $self->setSkip('ArgCheckOnly', 0);
        $self->setSkip('Password',     0);
        $self->setSkip('ArgSkipModifySudoers', 0);
    }
    elsif ($value eq 'uninstall') {

        $params->{OperationName     }->{value} = $gOperationUninstallInstance;
        $self->setSkip('ArgKeepUser',        0);
        $self->setSkip('ArgKeepUserHomeDir', 0);
        $self->setSkip('ArgKeepXsUsers',     0);
        $self->setSkip('ArgSkipModifySudoers', 0);
    }
    elsif ($value eq 'unregister') {

        $params->{OperationName     }->{value} = $gOperationUnregisterHost;
        $self->setSkip('ArgCheckOnly',       0);
        $self->setSkip('ArgKeepUser',        0);
        $self->setSkip('ArgKeepUserHomeDir', 0);
        $self->setSkip('ArgKeepXsUsers',     0);
        $self->setSkip('ArgSkipModifySudoers', 0);
        $self->setSkip('Password',           0);
    }
    elsif ($value eq 'list') {

        $params->{OperationName}->{value} = 'ListInstances';
        $self->setSkip('ArgIgnore');
        $self->setSkip('SID');
        $self->setSkip('Target');
    }
    else {
        $self->PushError("Unknown operation name '$value'.");
        return 0;
    }

    return 1;
}


#-------------------------------------------------------------------------------

sub setArgEnableParams {

    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_false_pattern/i)) {
        $self->setSkip('ArgDataPath');
        $self->setSkip('ArgGroupid');
        $self->setSkip('ArgLogPath');
        $self->setSkip('ArgSapadm');
        $self->setSkip('ArgSidadm');
        $self->setSkip('ArgUserid');
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the interactive input and sets the parameter '--collect_only'.

sub setArgSkipCollectCons {

    my ($self, $value) = @_;

    if (defined $value) {

        my $params    = $self->{params};
        my $doNothing = $self->getDefault('ArgSkipCollectCons');

        if ($value ne $doNothing) {
            $params->{ArgSkipCollect}->{value} = $value;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Checks and set the target SID.

sub setArgTargetSID {

    my ($self, $newSID) = @_;

    my $params = $self->{params};

    if ($newSID !~ /^[A-Z][A-Z0-9][A-Z0-9]$/){
        $self->PushError("Invalid $params->{ArgTargetSID}->{str} '$newSID'");
        return 0;
    }

    require SDB::Install::NewDBUser;

    if ($self->getValue('ArgCheckOnly')) {
        $params->{ArgTargetSID}->{value} = $newSID;
    }
    else {
        $self->setSkip('ArgSID', 0);
        $params->{ArgSID}->{value} = $newSID;
        $params->{ArgSID}->{str}   = $params->{ArgTargetSID}->{str};

        $params->{ArgTargetSID}->{value} = undef;
    }

    $params->{ArgTargetPassword}->{str} =
            sprintf($params->{ArgTargetPassword}->{str_templ},
                    $self->getSysAdminUserName($newSID));

    return 1;
}



#-------------------------------------------------------------------------------
# Returns undef (timeout parameters for operations not yet implemented)

sub showResultListInstances {

    my ($self, $result) = @_;

    my $info      = '';
    my $instances = {};

    if (defined $result && (ref($result) =~ /HASH/)) {

        foreach my $currKey (keys (%$result)) {

            my $item = $result->{$currKey};

            if (ref($item) =~ /HASH/) {
                my $sid = $item->{mSid};
                if (defined $sid) {
                    $instances->{$sid} = $item;
                }
            }
            elsif (ref($item) =~ /ARRAY/) {

                foreach my $currInstance (@$item) {

                    if (ref($currInstance) =~ /HASH/) {
                        my $sid = $currInstance->{mSid};
                        if (defined $sid) {
                            $instances->{$sid} = $currInstance;
                        }
                    }
                }
            }
        }
    }

    if (!%$instances) {
        $info = 'SAP Instance not found';
    }
    else {
        foreach my $sid (sort keys %$instances) {
            my $inst = $instances->{$sid};
            $info .= "\n  $sid ($inst->{mSystemNumber})\n"
                   . "    Version:  $inst->{mSapVersionInfo}\n"
                   . "    Hostname: $inst->{mHostname}\n";
        }
    }

    $self->getMsgLst()->addProgressMessage($info);
    return 1;
}


#===============================================================================
# import subroutines

sub getParamAcceleratorPassword;
* getParamAcceleratorPassword =
        \&SDB::Install::Configuration::AnyConfig::getParamAcceleratorPassword;

sub getParamAcceleratorUser;
* getParamAcceleratorUser =
        \&SDB::Install::Configuration::AnyConfig::getParamAcceleratorUser;

sub getHostRoleDescription;
* getHostRoleDescription =
        \&SDB::Install::Configuration::AnyMultiHostConfig::getHostRoleDescription;

sub getParamHostFailoverGroup;
* getParamHostFailoverGroup =
        \&SDB::Install::Configuration::AddHost::getParamHostFailoverGroup;

sub getParamNewSID;
*getParamNewSID = \&SDB::Install::Configuration::AnyRename::getParamNewSID;

sub getParamHostRole;
* getParamHostRole = \&SDB::Install::Configuration::AddHost::getParamHostRole;

sub getParamStartOnly;
* getParamStartOnly =
        \&SDB::Install::Configuration::AnyRegisterRename::getParamStartOnly;

sub getParamStopOnly;
* getParamStopOnly =
        \&SDB::Install::Configuration::AnyRegisterRename::getParamStopOnly;

sub getParamStoragePartitionNumber;
* getParamStoragePartitionNumber =
        \&SDB::Install::Configuration::AddHost::getParamStoragePartitionNumber;

sub getParamWorkerGroup;
* getParamWorkerGroup =
        \&SDB::Install::Configuration::NewServerConfig::getParamWorkerGroup;

1;
