# hdbmodify uses the 3 different instances of the class RemoteHosts.
#
# The method tryRemoteKeyAuthorization is executed for existing hosts (including
# host to be deleted) plus hosts that should be added.
# For these purpose the local hash '_currRemoteSSHHosts' contains an array
# of all hosts in order to return these hosts via getRemoteHostObjects.
# If a root password is not required the hash '_currRemoteSSHHosts' can be
# deleted, i.e. subsequent calls of getRemoteHostObjects must not return
# additional hosts, otherwise changing listen interface of existing hosts
# will fail.
#
# If a root password is required the method checkRootPassword has to work
# with the same hosts like checkRootUser, i.e. uses '_currRemoteSSHHosts'.
# After checking the root password, the hash '_currRemoteSSHHosts' has to be
# deleted.
#
# getRemoteHosts
# --------------
# Returns an instance of RemoteHosts containing all existing hosts.
#
#   - This list is created implicitly when detecting remote hosts during check SID
#
#   - If hosts are removed, these host names are deleted from the list
#
#   - The list is used to change the listen interface on remote hosts before
#     any host is added. This is done via:
#     $self-> executeRemoteParallel
#
#       -- executeRemoteParallel of AnyConfig calls:
#          $self->getRemoteHosts()->executeParallel/executeHostctrlParallel
#
# getRemovingRemoteHosts
# ----------------------
# Returns an instance of RemoteHosts containing all hosts of '--removehosts'
#
#   - hdbremovehost commands are executed before the listen interface is changed:
#     $self-> getRemovingRemoteHosts()->executeParallel/executeHostctrlParallel
#
#   - After successful execution of the hdbremovehost commands, the hosts are
#     removed from $self->getRemoteHosts()
#
# getAdditionalRemoteHosts
# ------------------------
# Returns an instance of RemoteHosts containing all hosts of '--addhosts'
#
#   - hdbaddhost is executed after the listen interface was changed:
#     $self-> getAdditionalremoteHosts()->executeParallel/executeHostctrlParallel


package SDB::Install::Configuration::HdbModify;

use strict;

use SDB::Install::Configuration qw ($bool_true_pattern $bool_false_pattern);
use SDB::Install::Configuration::AnyModifyConfig;
use SDB::Install::Configuration::AnyMultiHostConfig qw ($validHostRoles);
use SDB::Install::Globals qw ($gHostRoleAcceleratorWorker
                              $gHostRoleEsStandby
                              $gHostRoleEsWorker
                              $gHostRoleRDSync
                              $gHostRoleStreaming
                              $gHostRoleXS2Standby
                              $gHostRoleXS2Worker
                              $gLogDir
                              $gOperationModifyHost
                              $gOperationModifyHostRoles
                              $gOperationRemoveHost
                              $gProductNameSystem
                              $gShortProductNameXS2
                              $gProductNameEngine
                              $gXSParametersAddHostOrRoleConstraint
                              $gXSParametersRemoveHostOrRolesConstraint
                              $gDirNameAccelerator
                              $gDirNameEs
                              $gDirNameRDSync
                              $gDirNameStreaming
                              GetAutoInitializeServicesRolesComponentMap
                              GetHostRoleProperties);
use SDB::Install::RemoteHosts;
use SDB::Install::SAPControl;
use SDB::Install::System  qw (copySelectedFiles
                              getDirectoryFilenames
                              removeEmptyDirs);
use SDB::Install::SysVars qw ($isWin $path_separator);
use SDB::Install::Configuration::AutoInitFamilyServicesParamHandler;
use SDB::Install::Configuration::TenantUserListener;
use SDB::Install::Configuration::AddHostsParser;
use experimental qw (smartmatch);

our @ISA = qw (SDB::Install::Configuration::AnyModifyConfig);


#-------------------------------------------------------------------------------
# Constructor
sub new{
    my $self  = shift->SUPER::new (@_);

    my $order           = 0;
    my $section         = 'HdbModify';
    my $addHostsOnly    = "Valid with '--addhosts' only";
    my $removeHostsOnly = "Valid with '--removehosts' only";
    my $acceleratorOnly = "Valid with $gHostRoleAcceleratorWorker/standby only";
    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();


    $self->{params} = {
        ($isWin
            ? ('Drive'       => $self->getParamDrive             ($order++, $section),
               'Domain'      => $self->getParamDomain            ($order++, $section),
              )
            : ('Target'      => $self->getParamTarget            ($order++, $section), )
         ),
        'ScopeModify'        => $self->getParamScopeModify       ($order++, $section),
        'RemoteExecution'    => $self->getParamRemoteExecution   ($order++, $section),
        'UseHttp'            => $self->getParamUseHttp           ($order++, $section),
        'InstallHostagent'   => $self->getParamInstallHostagent  ($order++, $section, $addHostsOnly),
        'SID'                => $self->getParamSID               ($order++, $section),
        'AutoInitializeServices' => $self->getParamAutoInitializeServices($order++, $section, undef, $tenantCredentialsConstraint),
        'SSOCertificate'     => $self->getParamSSOCertificate    ($order++, $section),
        'AddRoles'           => $self->getParamAddRoles          ($order++, $section),
        'AddLocalRoles'      => $self->getParamAddLocalRoles     ($order++, $section),
        'RemoveRoles'        => $self->getParamRemoveRoles       ($order++, $section),
        'RemoveLocalRoles'   => $self->getParamRemoveLocalRoles  ($order++, $section),
        'RemoveHosts'        => $self->getParamRemoveHosts       ($order++, $section),
        'AutoAddXS2Roles'    => $self->getParamAutoAddXS2Roles   ($order++, $section, undef, 0),
        'AddHosts'           => $self->getParamAddHosts          ($order++, $section, 1),
        ($isWin
            ? ('HostagentUserName' => $self->getParamHostagentUserName($order++, $section),
              )
            : ('InstallSSHKey' => $self->getParamInstallSSHKey   ($order++, $section),
               'RootUser'      => $self->getParamRootUser        ($order++, $section),
               'RootPassword'  => $self->getParamRootPassword    ($order++, $section),
              )
        ),
        'SkipHostagentPw'    => $self->getParamSkipHostagentPw   ($order++, $section, $addHostsOnly),
        'HostagentPassword'  => $self->getParamHostagentPassword ($order++, $section, $addHostsOnly),
        'ListenInterface'    => $self->getParamListenInterface   ($order++, $section, 1),
        'InternalNetwork'    => $self->getParamInternalNetwork   ($order++, $section, 1),
        'KeepUser'           => $self->getParamKeepUser          ($order++, $section, $removeHostsOnly),
        'KeepUserHomeDir'    => $self->getParamKeepUserHomeDir   ($order++, $section, $removeHostsOnly),
        'KeepXsUsers'        => $self->getParamKeepXsUsers       ($order++, $section, $gXSParametersRemoveHostOrRolesConstraint),
        'SkipModifySudoers'  => $self->getParamSkipModifySudoers ($order++, $section),
        'Force'              => $self->getParamForce             ($order++, $section, $removeHostsOnly, 1),
        'NoStart'            => $self->getParamNoStart           ($order++, $section,
                                "Do not start added hosts and do not start $gProductNameSystem",
                                "Does not start added hosts and does not start $gProductNameSystem",
                                0), # not hidden
        'ImportContentXS2'   => $self->getParamImportContentXS2  ($order++,$section),
        'LoadInitialContentXS2' => $self->getParamLoadInitialContentXS2($order++, $section),
        'SystemUser'         => $self->getParamSystemUser        ($order++, $section, 1, $acceleratorOnly),
        'SQLSysPasswd'       => $self->getParamSQLSysPasswd      ($order++, $section, 'passwd', 1, undef, $acceleratorOnly),
        'AcceleratorUser'    => $self->getParamAcceleratorUser   ($order++, $section, $acceleratorOnly),
        'AcceleratorPassword'=> $self->getParamAcceleratorPassword($order++,$section, $acceleratorOnly, 'passwd'),
        'OrgManagerUser'     => $self->getParamOrgManagerUser    ($order++, $section, $gXSParametersAddHostOrRoleConstraint),
        'OrgManagerPassword' => $self->getParamOrgManagerPassword($order++, $section, $gXSParametersAddHostOrRoleConstraint),
        'ISCMode'            => $self->getParamISCMode            ($order++, $section),
        'Password'           => $self->getParamPassword           ($order++, $section),
        'TenantUser'            => $self->getParamTenantUser    ($order++, $section,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++,$section,1,$tenantCredentialsConstraint),
 
    };

    my $params = $self->{params};

    # Target and SID are set by SBD::Install::Installer::getInstallation
    $params->{SID}->{hidden}    = 1;
    $params->{Target}->{hidden} = 1;

    my $addRolesRestriction =
        "(Cannot be applied together with '"
        . $self->getOpt('AddHosts')        . "', '"
        . $self->getOpt('ListenInterface') . "', '"
        . $self->getOpt('RemoveHosts')     . "', '"
        . $self->getOpt('RemoveRoles')     . "')";

    $params->{AddRoles}->{additional_desc} .= $addRolesRestriction;
    $params->{AddLocalRoles}->{additional_desc} =
            "Valid roles: see '" .  $self->getOpt('AddRoles') . "'\n"
            . $addRolesRestriction;

    my $removeRolesRestriction =
        "(Cannot be applied together with '"
        . $self->getOpt('AddHosts')        . "', '"
        . $self->getOpt('AddRoles')        . "', '"
        . $self->getOpt('ListenInterface') . "', '"
        . $self->getOpt('RemoveRoles')     . "')";

    $params->{RemoveRoles}->{additional_desc} .= $removeRolesRestriction;
    $params->{RemoveLocalRoles}->{additional_desc} =
            "Valid roles: see '" .  $self->getOpt('RemoveRoles') . "'\n"
            . $removeRolesRestriction;

    my $withoutAddRemove =
        "(Cannot be applied together with '"
        . $self->getOpt('AddHosts')    . "', '"
        . $self->getOpt('RemoveHosts') . "', '"
        . $self->getOpt('AddRoles')    . "', '"
        . $self->getOpt('RemoveRoles') . "')";

    $params->{ScopeModify}->{additional_desc}           = $withoutAddRemove;
    $params->{LoadInitialContentXS2}->{additional_desc} = $withoutAddRemove;

    $params->{RootUser}->{init_with_default} = 1;
    $params->{RootUser}->{set_interactive}   = 1;
    
    $self->addParameterListener('AddHosts',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('AddHosts'));
    $self->addParameterListener('RemoveHosts',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('RemoveHosts'));
    $self->addParameterListener('AddRoles',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('AddRoles'));
    $self->addParameterListener('RemoveRoles',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('AddRoles'));
    $self->addParameterListener('AddLocalRoles',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('AddOrRemoveLocalRoles'));
    $self->addParameterListener('RemoveLocalRoles',SDB::Install::Configuration::AutoInitFamilyServicesParamHandler->new('AddOrRemoveLocalRoles'));
    $self->addParameterListener('TenantUser',SDB::Install::Configuration::TenantUserListener->new());
    
    $self->setSkip('Password');
    $self->setSkip('KeepUser');
    $self->setSkip('KeepUserHomeDir');

    $self->setSummaryOrder(['Target',
                            'SID',
                            'RemoveHosts',
                            'AddHosts',
                            'AddRoles',
                            'AddLocalRoles',
                            'AutoAddXS2Roles',
                            'ImportContentXS2',
                            'LoadInitialContentXS2',
                            'KeepUser',
                            'KeepUserHomeDir',
                            'ListenInterface',
                            'InternalNetwork',
                            'InstallHostagent',
                            'RemoteExecution',
                            'RootUser',
                            'HostagentUserName',
                            'SystemUser',
                            'AcceleratorUser',
                            'OrgManagerUser',
                            'TenantUser']);

	return $self;
}


#-------------------------------------------------------------------------------
# Adds additional local roles except worker and standby.
#
# Hash '_mapLocalSpecialRoles' does not contain accelerator and xs2

sub addLocalRoles {

    my ($self) = @_;

    my $roleArray = $self->{addRemoveLocalRoles};

    if (!defined $roleArray || !$self->{_isAddRoles}) {
        return 1;
    }

    my $sapSys   = $self->getSAPSystem();
    my $instance = $self->getOwnInstance();

    if (defined $self->{_roleOfLocalAccelerator}
        && !$self->tryCreateAcceleratorPaths($sapSys,
                                             $instance,
                                             $self)) {   # msglst
        return undef;
    }

    if (defined $self->{_mapLocalSpecialRoles}) {

        if (($self->{_mapLocalSpecialRoles}->{$gHostRoleEsWorker} ||
             $self->{_mapLocalSpecialRoles}->{$gHostRoleEsStandby})
             && !$self->tryCreateExtendedStoragePaths($sapSys,
                                                      $instance,
                                                      $self)) {   # msglst
            return undef;
        }

        if ($self->{_mapLocalSpecialRoles}->{$gHostRoleRDSync}
            && !$self->tryCreateRDSyncDownUploadPaths($sapSys,
                                                      $instance,
                                                      $self)) {   # msglst
            return undef;
        }
    }

    my $roleInfo = (scalar @$roleArray > 1) ? "s '" : " '";
    $roleInfo   .= join("', '", @$roleArray) . "'";
    my $msg      = $self->AddProgressMessage('Adding role'
         . $roleInfo . " on local host '" . $instance->get_host() . "'...");

    my $existingLogFiles = getDirectoryFilenames($gLogDir);

    $sapSys->SetProgressHandler  ($self->getMsgLst()->getProgressHandler());
    $instance->SetProgressHandler($self->getMsgLst()->getProgressHandler());

    my $rc = 1;

    if ($rc && defined $self->{_mapLocalSpecialRoles}) {
        $rc = $instance->writeSpecialHostServersIntoDaemonIni
                                          (0, $self->{_mapLocalSpecialRoles});
    }

    if ($rc) {
        $rc = $instance->registerAdditionalHostRoles($roleArray, $self);
    }

    if ($rc && defined $self->{_mapLocalSpecialRoles}) {

        $rc = $instance->addSpecialHosts($instance->get_host(),
                                         $self->{_mapLocalSpecialRoles},
                                         $self,
                                         $sapSys);
    }

    if ($rc && defined $self->{_roleOfLocalAccelerator}) {

        $rc = $instance->addAcceleratorHosts($instance->get_host(),
                                             $self->{_roleOfLocalAccelerator},
                                             $self);
    }

    $self->AddSubMsgLst($msg, $instance);

    if (defined $self->{options}->{action_id}) {

        copySelectedFiles($gLogDir,
                     $instance->get_hostNameTraceDir(),
                     $self->getMsgLst(),
                     $self->{options}->{action_id},
                     $existingLogFiles,
                     $sapSys->getUID(),
                     $sapSys->getGID());
    }

    my $requiresRestart = $self->_doRolesRequireRestart($roleArray);
    if ($rc && $self->{_isLocalWorkerXS2} && !$self->getValue('NoStart') && !$requiresRestart) {
        $rc = $self->waitForStartXS2($sapSys, $instance);
    }
    
    if ($rc) {
        $self->{_reconfigured} = 1;
        $self->AddProgressMessage('Role' . $roleInfo . ' added');
    }
    else {
        $self->AddError('Adding role' . $roleInfo . ' failed', $instance);
        return undef;
    }

    if ($requiresRestart) {
        my $resCodeEnable;
        my $resCodeDisable = $instance->disableAutoFailover($self->getMsgLst());
        eval {
            $rc = $self->_restartInstance($sapSys, $instance);
        };
        if($resCodeDisable){
            $resCodeEnable = $instance->enableAutoFailover($self->getMsgLst());
            if(!$resCodeEnable){
                my $sid = $self->getValue('SID');
                $self->AddError ("The role has been added successfully, but Auto-Failover couldn't be enabled. " .
                    "You need to do it manually by restoring the value of '[distributed_watchdog]/interval' " .
                    "from '[distributed_watchdog]/interval_backup' in '/usr/sap/$sid/SYS/global/hdb/custom/config/nameserver.ini'. " .
                    "If [distributed_watchdog]/interval_backup' not exists, remove '[distributed_watchdog]/interval' from the ini file " .
                    "to leave its default value active. Then execute hdbnsutil -reconfig.");
            }
        }
        if ($@){
            $self->AddError ("Cannot restart instance: $@");
            return undef;
        }
        return undef if $resCodeDisable && !$resCodeEnable;
        return if !$rc;
    }

    if ($self->{_isLocalWorkerXS2}
        && $self->getValue('ImportContentXS2') && !$self->getValue('NoStart')
        && $instance->hasXS2Controller()) {

        if (!$self->loadInitialContentXS2()) {
            return undef;
        }
    }

    return 1;
}

sub _restartInstance {
    my ($self, $sapSys, $instance) = @_;

    require SDB::Install::SAPSystemUtilities;
    my $util = new SDB::Install::SAPSystemUtilities($instance);

    if (!$self->stopSystemForModify($util, $sapSys, $instance, 1)) {
        return undef;
    }

    # restart sapstartsrv with updated daemon.ini
    if (!$self->startSapstartsrvForModify($util, $sapSys, $instance)) {
        return undef;
    }

    if (!$self->getValue('NoStart')) {
        $sapSys->resetMsgLstContext();

        my $passwordForWindows = ($isWin) ? $self->getValue('Password')
                                          : undef;

        my $rc = $util->startLocalInstance($sapSys,
                                        $instance,
                                        $self->getSID(),
                                        $passwordForWindows,
                                        $self);

        $self->getMsgLst->appendMsgLst($sapSys->getMsgLst());
        if (!$rc){
            $self->setErrorMessage('Cannot start system',
                                   $sapSys->getErrMsgLst());
            return undef;
        }
    }

    return 1;
}
sub _doRolesRequireRestart {
    my ($self, $aRoles) = @_;
    my $validHostRoles = GetHostRoleProperties();
    my $rolesRequiringRestart = grep {$validHostRoles->{$_}->{requiresRestart}} @$aRoles;
    return $rolesRequiringRestart;
}


#-------------------------------------------------------------------------------
sub checkAddHosts {
    my ($self, $value, $msglst_arg, $notHandleListenInterfaceParam) = @_;

    return 1 if(! defined($value) );
    return 0 if(!$self->SUPER::checkAddHosts($value, $msglst_arg, $notHandleListenInterfaceParam));

    $self->unskipLoadInitialContentParamsIfNecessary($value);
    return 1;
}

#-------------------------------------------------------------------------------
# Checks additional local host roles without using multiple-host parameters
# and disables all other functions of hdbmodify.

sub checkAddLocalRoles {

    my ($self, $roleList) = @_;
    $self->{_isAddRoles} = 1;
    return $self->checkAddRemoveLocalRoles([split (',', $roleList)], 'AddRoles');
}


#-------------------------------------------------------------------------------
# Checks additional local host roles without using multiple-host parameters
# and disables all other functions of hdbmodify.

sub checkAddRemoveLocalRoles {

    my ($self,
        $roleArray,
        $rolesParamID,
       ) = @_;

    my $localRolesParamID = ($rolesParamID eq 'AddRoles') ? 'AddLocalRoles'
                                                          : 'RemoveLocalRoles';

    if (!$self->disableNonRolesParams($localRolesParamID)) {
        return 0;
    }

    my @addRemoveRoles;
    my %mapAddRemoveRoles;

    if (defined $self->getBatchValue($rolesParamID)) {
        if (defined $self->{addRemoveLocalRoles}) {
            @addRemoveRoles = @{$self->{addRemoveLocalRoles}};
        }
        if (defined $self->{_mapAddRemoveLocalRoles}) {
            %mapAddRemoveRoles = %{$self->{_mapAddRemoveLocalRoles}};
        }
    }
    else {
        delete $self->{addRemoveLocalRoles};
        delete $self->{_mapAddRemoveLocalRoles};
        delete $self->{_roleOfLocalAccelerator};
        delete $self->{_isLocalWorkerXS2};
        delete $self->{_mapLocalSpecialRoles};
    }

    foreach my $currRole (@$roleArray) {

        if (($localRolesParamID eq 'RemoveLocalRoles')
            && !$self->checkIsRemovable($currRole)) {
            return 0;
        }

        if ($mapAddRemoveRoles{$currRole}) {
            $self->PushError("Duplicate local host role '$currRole'");
            return 0;
        }

        if (!exists $validHostRoles->{$currRole}) {
            $self->PushError("Unknown host role '$currRole'");
            return 0;
        }

        if ($validHostRoles->{$currRole}->{isColumnStore}) {
            $self->PushError("Local host role '$currRole' is not allowed");
            return 0;
        }

        if (($localRolesParamID eq 'AddLocalRoles')
            && !$self->isSpecialComponentInstalled($currRole)) {
            $self->appendErrorMessage("Role '$currRole' is invalid: required component is not installed");
            return 0;
        }

        if ($validHostRoles->{$currRole}->{isAccelerator}) {
            $self->{_roleOfLocalAccelerator} = $currRole;
            $self->setSkip('SystemUser',          0);
            $self->setSkip('SQLSysPasswd',        0);
            $self->setSkip('AcceleratorUser',     0);
            $self->setSkip('AcceleratorPassword', 0);
        }
        elsif ($validHostRoles->{$currRole}->{isXS2}) {
            if ($validHostRoles->{$currRole}->{isStandby}) {
                $self->{_isLocalStandbyXS2} = 1;
            } else {
                $self->{_isLocalWorkerXS2} = 1;
            }
        }
        else {
            if (defined $self->{_mapLocalSpecialRoles}) {
                $self->{_mapLocalSpecialRoles}->{$currRole} = 1;
            } else {
                $self->{_mapLocalSpecialRoles} = {$currRole => 1};
            }
        }

        push @addRemoveRoles, $currRole;
        $mapAddRemoveRoles{$currRole} = 1;
    }

    if (@addRemoveRoles) {
        $self->{addRemoveLocalRoles}     = \@addRemoveRoles;
        $self->{_mapAddRemoveLocalRoles} = \%mapAddRemoveRoles;
        my $isDistributed = $self->isDistributedSystem();
        if (!defined $isDistributed || $isDistributed){
            $self->setSkip('Password', 0);
        }
        my $instance    = $self->getOwnInstance();
        my $activeRoles = $instance->getHostRolesByIniFile();

        if (!defined $activeRoles) {
            $self->setErrorMessage ('Cannot get existing host roles',
                                    $instance->getErrMsgLst());
            return undef;
        }
        if ($localRolesParamID eq 'AddLocalRoles') {
            foreach my $currActive (@$activeRoles) {
                if ($mapAddRemoveRoles{$currActive}) {
                    $self->PushError("Local host role '$currActive' is already installed");
                    return 0;
                }
            }
        }
        else {
            foreach my $currRemove (@addRemoveRoles) {
               if (!($currRemove ~~ @$activeRoles)) {
                    $self->PushError("Host role '$currRemove' is not installed on local host");
                    return 0;
                }
            }
        }
    }
    my $csvValue = join(',', @$roleArray);
    $self->unskipLoadInitialContentParamsIfNecessary($csvValue) if ($localRolesParamID eq 'AddLocalRoles' && @addRemoveRoles);
    return 1;
}


#-------------------------------------------------------------------------------
# Checks additional host roles and disables all other functions of hdbmodify.
#

sub checkAddRoles {
    my ($self, $value) = @_;

    $self->{_isAddRoles} = 1;
    return $self->checkAddRemoveRoles($value, 'AddRoles');
}


#-------------------------------------------------------------------------------
# Checks host roles and disables all other functions of hdbmodify.

sub checkAddRemoveRoles {
    my ($self, $value, $addRemoveRolesParamID) = @_;

    if (!$self->disableNonRolesParams($addRemoveRolesParamID)) {
        return 0;
    }

    delete $self->{addRemoveLocalRoles};
    delete $self->{_mapAddRemoveLocalRoles};
    delete $self->{_roleOfLocalAccelerator};
    delete $self->{_isLocalWorkerXS2};
    delete $self->{_mapLocalSpecialRoles};

    # the class RemoteHosts has been created before by checkSID

    if (!$self->handleAddOrRemoveHosts($value, $addRemoveRolesParamID, 1)) {
        return 0;
    }

    my $localHost   = $self->getLocalHostOfAddRemoveRoles();
    my $remoteHosts = $self->getRemoteHostsOfAddRemoveRoles();

    if (!defined $localHost && !defined $remoteHosts) {
        $self->PushError($self->{params}->{$addRemoveRolesParamID}->{str}
                         . ': Roles are not defined');
        return 0;
    }

    if (defined $localHost) {

        my $roles = $self->getAddHostsParser()->getRoles($localHost);

        if (!$self->checkAddRemoveLocalRoles($roles,
                                             $addRemoveRolesParamID)) {
            return 0;
        }
    }

    if (defined $remoteHosts) {

        my $parser = $self->getAddHostsParser();

        foreach my $currHost (@{$remoteHosts->getHostNames()}) {

            my $currRoles = $parser->getRoles($currHost);

            if (!defined $currRoles) {
                $self->AddError("Host role not defined for host '$currHost'");
                return 0;
            }

            if ($addRemoveRolesParamID eq 'RemoveRoles') {
                foreach my $currRole (@$currRoles) {
                    if (!$self->checkIsRemovable($currRole)) {
                        return 0;
                    }
                }
            }
        }

        $self->setSkip('Password', 0);

        if (!$self->initRemoteHosts()) {
            return undef;
        }

        $self->enableMultipleHostParameters(1); # 1 - $useSidadmForHostctrl
    }

    $self->unskipLoadInitialContentParamsIfNecessary($value) if ($self->{_isAddRoles});
    return 1;
}

#-------------------------------------------------------------------------------
# Checks the option 'internal_network'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Returns int retCode

sub checkInternalNetwork {

    my ($self, $value) = @_;

    if (!defined $value) {
        return 1;
    }

    my $instance        = $self->getOwnInstance();
    my $wantedInterface = $self->getValue('ListenInterface');

    if (!defined $wantedInterface && $instance->isUsingLocalInterface()) {
        $self->{params}->{ListenInterface}->{value} = 'internal';
    }

    $self->setSkip('Password', 0);

    return $self->checkInternalNetworkOnly($value);
}


#-------------------------------------------------------------------------------
# Checks the inter service communication mode

sub checkISCMode() {
    my ($self, $value) = @_;

    my $doChange = $self->handleGlobalIniISCMode($value, 1); # 1 - skip changing

    if (!defined $doChange) {
        return undef; # any error occurred
    }
    elsif ($doChange) {
        $self->{_changeISCMode} = 1;
        $self->setSkip('Password', 0);
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks if the specified host role can be removed.

sub checkIsRemovable {
    my ($self, $role) = @_;

    if (!defined $validHostRoles->{$role}
        || !$validHostRoles->{$role}->{removable}) {

        $self->PushError("Remove role '$role' not supported");
        return 0;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the option 'listen_interface'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.
#
# Returns int retCode

sub checkListenInterface {
    my ($self, $wantedInterface) = @_;

    my $startStopRequired= 1;
    my $existingType     = $self->getListenInterfaceType();
    my $newType          = $self->detectNewListenInterfaceType($wantedInterface,
                                                               $existingType);

    if ($self->isSystemRestartRequired()) {
        $self->setSkip('Password', 0);
    }
    return $self->checkListenInterfaceOnly($newType);
}


#-------------------------------------------------------------------------------
sub checkRemoveHosts {
    my ($self, $value, $msglst_arg, $doNotFailOnLocalHost) = @_;

    return 1 if(! defined($value) );

    return $self->SUPER::checkRemoveHosts($value, $msglst_arg, $doNotFailOnLocalHost);
}


#-------------------------------------------------------------------------------
# Checks local roles to be removed without using multiple-host parameters
# and disables all other functions of hdbmodify.

sub checkRemoveLocalRoles {

    my ($self, $roleList) = @_;
    return $self->checkAddRemoveLocalRoles([split (',', $roleList)],
                                           'RemoveRoles');
}


#-------------------------------------------------------------------------------
# Checks host roles to be removed and disables all other functions of hdbmodify.

sub checkRemoveRoles {
    my ($self, $value) = @_;

    return $self->checkAddRemoveRoles($value, 'RemoveRoles');
}


#-------------------------------------------------------------------------------
# Checks the password of the root user in case of a distributed system.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkRootPassword {

    my ($self, $password) = @_;

    my $rc = $self->SUPER::checkRootPassword($password);
    if($rc){
        delete $self->{_currRemoteSSHHosts};
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Checks the root user name for existing remote hosts and additional hosts.
# Therefore the internal array '_currRemoteSSHHosts' can contains 2 instances of
# the class RemoteHosts, one for existing renote hosts and one for
# additional hosts.
# Refer to the description at the beginning of this source file.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkRootUser {

    my ($self, $remoteRootUser) = @_;

    my $remoteHostsObjects = $self->SUPER::getRemoteHostsObjects ();
    my $additionalHosts    = $self->getAdditionalRemoteHosts();

    if (defined $additionalHosts){
        push @$remoteHostsObjects, $additionalHosts;
    }

    $self->{_currRemoteSSHHosts} = $remoteHostsObjects;

    my $rc = $self->SUPER::checkRootUser($remoteRootUser);

    if (!$rc || $self->{params}->{RootPassword}->{skip}) {
        delete $self->{_currRemoteSSHHosts};
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Disables remote execution in case of scope 'instance'.

sub checkScopeModify {

    my ($self, $value) = @_;

    if ($value eq 'instance') {

        for my $currID ('AddHosts', 'AddRoles', 'RemoveHosts') {
            if (defined $self->getBatchValue($currID)) {

                $self->PushError("'" . $self->getOpt('ScopeModify')
                               . "=$value' cannot be applied together with '"
                               . $self->getOpt($currID) . "'");
                return 0;
            }
        }

        $self->setSkip('InstallHostagent');
        $self->setSkip('RemoteExecution');
        $self->setSkip('RemoveHosts');
        $self->setSkip('UseHttp');
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Checks the specified SID
#
# Parameter: String $sid
#
# Returns int retCode

sub checkSID {
	my ($self, $sid, $msglst_arg) = @_;

	my $rc = $self->SUPER::checkSID($sid, $msglst_arg);

	if (!$rc) {
        return $rc;
	}

	$self->setDefault('ListenInterface', $self->getListenInterfaceType());

    my $scope    = $self->getValue('ScopeModify');
    my $instance = $self->getOwnInstance();

    if (!defined $instance) {
        $self->PushError ("SAP System '$sid' has no HDB instance");
        return 0;
    }

    if (!defined $scope
        ||
        (($scope eq 'instance') && !$instance->exists_remote_host())) {
        $scope = 'system';
        $self->{params}->{ScopeModify}->{value} = $scope;
    }

    if ($scope eq 'system') {
        $self->{params}->{ScopeModify}->{hidden} = 1; # suppress summary output
    }

	return 1;
}


#-------------------------------------------------------------------------------
# Checks if a host with a xs2 controller exists.

sub checkLoadInitialContentXS2 {
    my ($self, $value) = @_;
    return 1 if (!defined $value || $value =~ /$bool_false_pattern/i);

    my $errInfo = "Cannot load initial content of $gShortProductNameXS2";

    delete ($self->{_xscontrollerHost});
    if (defined $self->{addRemoveLocalRoles}            ||
        defined $self->getRemoteHostsOfAddRemoveRoles() ||
        defined $self->getAdditionalRemoteHosts() ||
        defined $self->getRemovingRemoteHosts()) {
        $self->setErrorMessage ("'" . $self->getOpt('LoadInitialContentXS2')
            . "' cannot be applied together with '"
            . $self->getOpt('AddRoles')    . "' , '"
            . $self->getOpt('AddHosts')    . "' , '"
            . $self->getOpt('RemoveHosts') . "'");
        return 0;
    }

    my $instance    = $self->getOwnInstance();
    my $activeRoles = $instance->getActiveHostRoles();
    if (!defined $activeRoles) {
        $self->setErrorMessage(undef, $instance->getErrMsgLst());
        return undef;
    }
    if (!defined $activeRoles->{$gHostRoleXS2Worker}) {
        $self->setErrorMessage
                ("$errInfo ('$gHostRoleXS2Worker' is not configured)");
        return 0;
    }

    $self->{_xscontrollerHost} = $self->getXS2ControllerHost($instance,
                                                             $activeRoles);
    if (!defined $self->{_xscontrollerHost}) {
        $self->setErrorMessage("$errInfo (xscontroller not found)'");
        return 0;
    }

    if ($self->{_xscontrollerHost} ne $instance->get_host()) {

        if (!defined $self->getRemoteHosts()) {
            $self->setErrorMessage("$errInfo (cannot connect to host '"
                                   . $self->{_xscontrollerHost} ."'");
            return 0;
        }

        my @updHosts   = ($self->{_xscontrollerHost});
        my $remoteHost = undef;

        if ($self->isUseSAPHostagent()) {
            $remoteHost = new SDB::Install::RemoteHostctrlHosts(@updHosts);
        }
        else {
            $remoteHost = new SDB::Install::RemoteHosts(@updHosts);
            if (!$remoteHost->copyConnections($self->getRemoteHosts(),$self)){
                return 0;
            }
        }

        $remoteHost->SetProgressHandler($self->GetProgressHandler());
        $self->{_remoteHostForLoadingContentXS2} = $remoteHost;
    }
    
    $self->unskipLoadInitialContentParams();
    return 1;
}


#-------------------------------------------------------------------------------
# Does not collect any information of remote hosts.

sub CollectOtherHostInfos {
    my ($self) = @_;

    # provisional enabling of hostagent password
    my $additionalHosts = $self->getAdditionalRemoteHosts();
    if (defined $additionalHosts && $self->getValue('InstallHostagent')) {
        $self->setSkip('HostagentUserName', 0);
        $self->setSkip('HostagentPassword', 0);
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Disables parameters that are not used for adding/removing roles.

sub disableNonRolesParams {
    my ($self, $addRemoveRolesParamID) = @_;

    my @notAllowed = ('AddHosts',
                      'InternalNetwork',
                      'ListenInterface',
                      'RemoveHosts',
                     );

    if ($addRemoveRolesParamID =~ /^Add/) {
        push @notAllowed, 'RemoveLocalRoles';
        push @notAllowed, 'RemoveRoles';
    }
    else {
        push @notAllowed, 'AddLocalRoles';
        push @notAllowed, 'AddRoles';
    }

    # Add custom handling for LoadInitialContentXS2 because of bug 103170
    my $loadInitialContentValue = $self->getBatchValue('LoadInitialContentXS2');
    if($loadInitialContentValue =~ /$bool_true_pattern/){
        push(@notAllowed, 'LoadInitialContentXS2');
    }

    for my $currParamID (@notAllowed) {
        $self->setSkip($currParamID);

        if (defined($self->getBatchValue($currParamID))) {
            my $messageTemplate = "'%s' cannot be applied together with '%s'";
            $self->PushError(sprintf($messageTemplate, $self->getOpt($addRemoveRolesParamID), $self->getOpt($currParamID)));
            return 0;
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Writes a summary of the parameters onto the console and into the log file.
#
# Hides the parameter RemoteExecution in case of local work only.

sub displayParameterSummary {

    my ($self) = @_;

    if (!defined $self->getRemoteHosts() &&
        !defined $self->getAdditionalRemoteHosts() &&
        !defined $self->getRemovingRemoteHosts()) {

        $self->{params}->{RemoteExecution}->{hidden} = 1; # hide summary output
    }

    $self->SUPER::displayParameterSummary();
}


#-------------------------------------------------------------------------------
sub getParamRemoveLocalRoles {
    my ($self, $order, $section) = @_;

    return {
        'order'           => $order,
        'opt'             => 'remove_local_roles',
        'hostctrl_opt'    => 'REMOVE_LOCAL_ROLES',
        'opt_arg'         => '<role1>[,<role2>]...',
        'type'            => 'string',
        'section'         => $section,
        'value'           => undef,
        'default'         => undef,
        'str'             => 'Remove Local Roles',
        'desc'            => 'Removes roles from this host only',
        'set_interactive' => 0,
    };
}


#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--scope'.

sub getParamScopeModify {

    my ($self, $order, $section) = @_;

    my $desc = 'Changes the listen interface on all hosts (system) '
                . 'or manually execution on local instance';

    my $param = $self->getParamScope($order, $section, $desc);

    $param->{default}           = 'system';
    $param->{init_with_default} = 1;
    $param->{skip}              = 0;

    return $param;
}


#-------------------------------------------------------------------------------
# In case of root user check an array is returned containing 2 RemoteHosts
# instances (existing remote hosts and additional hosts).
# Refer to the description at the beginning of this source file.

sub getRemoteHostsObjects {

    my ($self) = @_;

    return (defined $self->{_currRemoteSSHHosts})
           ? $self->{_currRemoteSSHHosts}
           : $self->SUPER::getRemoteHostsObjects();
}


#-------------------------------------------------------------------------------
# Returns 1 if the system has been reconfigured.

sub isReconfigured {
    my ($self) = @_;
    return ($self->{_reconfigured}) ? 1 : 0;
}


#-------------------------------------------------------------------------------
# Reconfigures the listen interface.

sub reconfigureListenInterface {

    my ($self, $wantedType, $isChangingInternalNet) = @_;

    my $rc       = 1;
    my $msg      = $self->AddMessage("Setting listen interface to '$wantedType'...");
    my $instance = $self->getOwnInstance();

    if ($isChangingInternalNet) {
        $rc = $instance->changeInternalNetAndWriteGlobalIni($self);
    }
    else {
        $rc = $instance->reconfigureTrexNet($self);
    }

    if ($rc) {
        $self->{_reconfigured} = 1;
    }
    else {
        $self->AddError("Cannot set listen interface to '$wantedType'", $instance);
    }
    $self->AddSubMsgLst($msg, $instance);
    return $rc;
}


#-------------------------------------------------------------------------------
# Removes local roles except worker and standby.
#
# Hash '_mapLocalSpecialRoles' does not contain accelerator and xs2

sub removeLocalRoles {

    my ($self) = @_;

    my $roleArray = $self->{addRemoveLocalRoles};

    if (!defined $roleArray || $self->{_isAddRoles}) {
        return 1;
    }

    my $sapSys   = $self->getSAPSystem();
    my $instance = $self->getOwnInstance();

    my $roleInfo = (scalar @$roleArray > 1) ? "s '" : " '";
    $roleInfo   .= join("', '", @$roleArray) . "'";
    my $msg      = $self->AddProgressMessage('Removing role'
           . $roleInfo . " from local host '" . $instance->get_host() . "'...");
    my $existingLogFiles = getDirectoryFilenames($gLogDir);

    $sapSys->SetProgressHandler  ($self->getMsgLst()->getProgressHandler());
    $instance->setMsgLstContext([$msg->getSubMsgLst ()]);

    my $rc      = 1;
    my $roleMap = $self->{_mapLocalSpecialRoles};

    if (defined $self->{_roleOfLocalAccelerator}){
        if (!defined $roleMap){
            $roleMap = {};
        }
        $roleMap->{$self->{_roleOfLocalAccelerator}} = 1;
    }

    if ($roleMap && $rc) {
        $rc = $instance->removeSpecialHostRoles
                (0, # force
                 $instance->get_host(),
                 $self, # instconfig
                 $roleMap->{$gHostRoleEsWorker} || $roleMap->{$gHostRoleEsStandby},
                 defined $self->{_roleOfLocalAccelerator}, # is accelerator
                 $roleMap->{$gHostRoleStreaming},
                 $roleMap->{$gHostRoleRDSync});
    }

    if ($rc){
        foreach my $specialRole (sort (keys (%$roleMap))){
            # calls hdbnsutil -removeHostRole
            $rc = $instance->unregisterHostRole (undef, $specialRole, $self->getValue ('Force'), $self);
            if (!$rc){
                last;
            }
        }
    }

    if ($rc && $self->{_isLocalWorkerXS2}) {
        $rc =  $instance->removeHostRoleXS2Worker($self);
    }
    if ($rc && $self->{_isLocalStandbyXS2}) {
        $rc =  $instance->removeHostRoleXS2Standby($self);
    }
    if (defined $self->{options}->{action_id}) {

        copySelectedFiles($gLogDir,
                     $instance->get_hostNameTraceDir(),
                     $self->getMsgLst(),
                     $self->{options}->{action_id},
                     $existingLogFiles,
                     $sapSys->getUID(),
                     $sapSys->getGID());
    }

    if ($rc) {
        $self->{_reconfigured} = 1;
        $self->AddProgressMessage('Role' . $roleInfo . ' removed');
    }
    else {
        $self->AddError('Removing role' . $roleInfo . ' failed:', $instance);
        return undef;
    }

    if ($rc) {
        $self->{_reconfigured} = 1;
    }
    else {
        return $rc;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sends 'hdbaddhost' commands to remote hosts specified by '--addhosts'.

sub sendAddRemoteHostCommands {

    my ($self) = @_;
    
    my $additionalHosts = $self->getAdditionalRemoteHosts();

    if (!defined $additionalHosts){
        $self->AddMessage('Hosts to be added are not specified');
        return 1;
    }
    $self->{_reconfigured} = 1;
    return $self->sendAddHostCommands($self,
                                      $self->getSAPSystem(),
                                      $self->getOwnInstance());
}


#-------------------------------------------------------------------------------
# Sends 'hdbmodify --add_local_role'/'--remove_local_role' commands
# to remote hosts contained in '--add_roles'/'--remove_roles'.

sub sendAddRemoveRolesCommands {

    my ($self) = @_;

    my $remoteHosts = $self->getRemoteHostsOfAddRemoveRoles();
    my $parser      = $self->getAddHostsParser();

    if (!defined $remoteHosts || !defined $parser) {
        return 1;
    }

    if (!$self->isUseSAPHostagent()) {
        if (!$remoteHosts->copyConnections($self->getRemoteHosts(), $self)) {
            return 0;
        }
    } elsif ($self->getRemoteHosts()->isUseSidadm()) {
        $remoteHosts->useSidadm();
    }

    $self->{_reconfigured} = 1;

    my $hostctrlParamMap;
    my @templateParamRepl;
    my $actionProgressive;
    my $actionDone;
    my $paramID;

    if ($self->{_isAddRoles}) {
        $paramID           = 'AddLocalRoles';
        $actionProgressive = 'Adding roles to';
        $actionDone        = 'modified (roles added)';
    }
    else {
        $paramID           = 'RemoveLocalRoles';
        $actionProgressive = 'Removing roles from';
        $actionDone        = 'modified (roles removed)';
    }

    foreach my $currHost (@{$remoteHosts->getHostNames()}) {

        my $currRoles = $parser->getRoles($currHost);

        if (!defined $currRoles) {
            $self->AddError("Host role not defined for host '$currHost'");
            return undef;
        }

        my $currRoleList = join (',', @$currRoles);
        my $key          = $self->getHostctrlOpt($paramID);
        $hostctrlParamMap->{$currHost}->{$key} = $currRoleList;
        push @templateParamRepl, [$currRoleList];
    }

    my $additionalCmdLineParam = $self->getOpt($paramID) . '=%s';
    my $paramIDs = ['SystemUser', 'AcceleratorUser', 'OrgManagerUser', 'ImportContentXS2', 'KeepXsUsers', 'SkipModifySudoers', 'TenantUser', 'AutoInitializeServices'];
    my $loadInitialContentOption = $self->getHostctrlOpt('LoadInitialContentXS2');
    my $passwordKeys = [
        'HostagentPassword',
        'Password',
        'SQLSysPasswd',
        'AcceleratorPassword',
        'OrgManagerPassword',
        'SQLTenantUserPassword',
    ];

    if (!$self->excuteRemoteParallel($actionProgressive,
                                     $actionDone,
                                     'hdbmodify',                # executable
                                     'HdbModifyHostRoles',       # main program
                                     $gOperationModifyHostRoles, # hostctrl operation
                                     $passwordKeys,
                                     $paramIDs,
                                     $additionalCmdLineParam,
                                     $remoteHosts,
                                     undef,
                                     $hostctrlParamMap,
                                     \@templateParamRepl,
                                     undef,
                                     { $loadInitialContentOption => 'off' } # Explicitly disable LoadInitialContentXS2 - See Bug 103170
                                     )) {
        return undef;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Sends 'hdbremovehost' commands to remote hosts specified by '--removehosts'.

sub sendRemoveHostCommands {

    my ($self) = @_;

    my $removeHosts = $self->getRemovingRemoteHosts();

    if (!defined $removeHosts){
        $self->AddMessage('Hosts to be removed are not specified');
        return 1;
    }

    $self->{_reconfigured} = 1;

    require SDB::Install::Configuration::RemoveHost;
    my $removeHostInstconfig   = new SDB::Install::Configuration::RemoveHost();
    my $additionalCmdLineParam = '--sid=' . $self->getSID();

    my $passwordKeys = ['HostagentPassword',
                        'Password',
                        'SQLSysPasswd',
                        'SQLTenantUserPassword',
                        'AcceleratorPassword'];

    my $paramIDs = ['Force',
                    'KeepUser',
                    'KeepUserHomeDir',
                    'KeepXsUsers',
                    'SkipModifySudoers',
                    'SystemUser',
                    'TenantUser',
                    'AcceleratorUser',
                    'AutoInitializeServices'];

    my $optionMap = undef;
    if ($self->isSystemInCompatibilityMode()) {
        $optionMap->{'ACM'} = 'on';
        $additionalCmdLineParam = ' --addhost_compatiblity_mode=1';
    }
    if (!$self->excuteRemoteParallel('Removing',
                                     'removed',
                                     'hdbremovehost', # executable
                                     undef,           # main program
                                     $gOperationRemoveHost, # hostctrl operation
                                     $passwordKeys,
                                     $paramIDs,
                                     $additionalCmdLineParam,
                                     $removeHosts,
                                     $removeHostInstconfig,
                                     undef,
                                     undef,
                                     undef,
                                     $optionMap)) {
        return undef;
    }

    $self->resetSAPSystemHashes();
    delete $self->{ownInstance};
    my $instance = $self->getOwnInstance();
    my $retcode  = 1;

    foreach my $hostName (@{$removeHosts->getHostNames()}) {

        my $hostDir = $instance->get_hostNameDir($hostName);

        if (-d $hostDir) {
            removeEmptyDirs($hostDir);
        }
        if (-d $hostDir && !rmdir($hostDir)) {
            my @filenames;
            if (opendir (DH, $hostDir)) {
                @filenames = readdir(DH);
                closedir (DH);
            }
            foreach my $name (@filenames) {
                if (($name ne '.') && ($name ne '..')) {
                    $self->AddMessage("Directory '$hostDir' contains '$name'");
                }
            }
            $self->setErrorMessage("Directory '$hostDir' not removed");
            $retcode = undef;
        }
    }

    if (!$retcode) {
        return $retcode;
    }

    my $allHosts = $self->getRemoteHosts();
    $allHosts->removeRemoteHosts($removeHosts->getHostNames());

    if ($allHosts->getNumberOfHosts() == 0) {
        $self->clearRemoteHosts();
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Starts sapstartsrv.

sub startSapstartsrvForModify {

    my ($self,
        $util,     # SDB::Install::SAPSystemUtilities
        $sapSys,   # SDB::Install::SAPSystem
        $instance, # SDB::Install::SAPInstance::TrexInstance
       ) = @_;

    $sapSys->setMsgLstContext([$self->getMsgLst ()]);

    my $rc = $util->startSapstartsrv($sapSys, $instance, $self);

    if (!$rc){
        $self->setErrorMessage('Cannot start sapstartsrv',
                               $sapSys->getErrMsgLst());
        return undef;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Changes the inter service communication mode if necessary.

sub tryChangeISCMode() {
    my ($self) = @_;

    if (!$self->{_changeISCMode}) {
        return 1; # nothing to do
    }

    require SDB::Install::SAPSystemUtilities;
    my $util     = new SDB::Install::SAPSystemUtilities();
    my $sapSys   = $self->getSAPSystem();
    my $instance = $self->getOwnInstance();

    if (!$self->stopSystemForModify($util, $sapSys, $instance, 0, 1)) {
        return undef;
    }

    if (!defined $self->handleGlobalIniISCMode($self->getValue('ISCMode'))) {
        return undef;
    }
    $self->{_reconfigured} = 1;

    if (!$self->getValue('NoStart')) {
        if (!$self->_startSystem()) {
            return undef;
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Changes the listen interface if necessary.

sub tryChangeListenInterface {
    my ($self) = @_;

    if(!$self->hasValue('InternalNetwork') && !$self->hasValue('ListenInterface')){
        $self->AddMessage('Inter-Service Communication properties are not specified');
        return 1;
    }

    my $instance     = $self->getOwnInstance();
    my $existingType = $self->getListenInterfaceType();
    my $paramValue   = $self->getValue('ListenInterface');
    my $wantedType   = $self->detectNewListenInterfaceType($paramValue,
                                                           $existingType);

    my $wantedNetPrefix = $self->getValue('InternalNetwork');

    if (!defined($wantedNetPrefix) && $wantedType eq 'internal' && !$instance->getInternalNetworkPrefix()) {
        $wantedType = 'global'; # internal network prefix not specified
    }

    my $existingNetPrefix = $instance->getInternalNetworkPrefix();

    if (($wantedType eq $existingType) && !$self->isSystemRestartRequired()) {
        $self->AddMessage("Listen interface '$existingType' not changed");
        return 1;
    }
    if (!defined $paramValue || ($paramValue ne $wantedType)) {

        $self->{params}->{ListenInterface}->{value} = $wantedType;
        $self->addLogValueMsg('ListenInterface', $wantedType,
                              'is changed by hdbmodify, value =');
    }

    my $scope = $self->getValue('ScopeModify');
    if (! $self->isSystemRestartRequired()) {
        my $rc = $self->reconfigureListenInterface($wantedType);
        if ($rc && !$self->getValue('NoStart') && ($scope eq 'system')) {
            return $self->_startSystem();
        }
        return $rc;
    }
    # otherwise --> stop/start required

    require SDB::Install::SAPSystemUtilities;
    my $util   = new SDB::Install::SAPSystemUtilities();
    my $sapSys = $self->getSAPSystem();

    if (!$self->stopSystemForModify($util, $sapSys, $instance)) {
        return undef;
    }

    my $changingInternalNet = (defined $existingNetPrefix ||
                               defined $wantedNetPrefix) ? 1 : 0;

    if (!$self->reconfigureListenInterface($wantedType, $changingInternalNet)) {
        return undef;
    }

    if (!$self->startSapstartsrvForModify($util, $sapSys, $instance)) {
        return undef;
    }

    if (defined $self->getRemoteHosts()) {

        my $isRemoteReconfigWanted =
                ($instance->isUsingInternalInterface()
                 ||
                 ($instance->isUsingGlobalInterface() &&
                  defined $instance->getInternalNetworkPrefix())) ? 1 : 0;

        if (!$self->excuteRemoteSerialOrParallel
                            ('Changing listen interface of',
                             'reconfigured',
                             'hdbreg',        # executable
                             'HdbModifyHost', # main program
                             $gOperationModifyHost, # hostctrl operation
                             ['HostagentPassword', 'Password'],
                             undef, # $paramIDs
                             undef, # $cmdLineParams
                             undef, # $remoteHosts
                             undef, # $remoteInstconfig
                             undef, # $hostctrlParamMap
                             undef, # $templateParamRepl
                             undef, # $executeOnlyOnHosts
                             $isRemoteReconfigWanted, # $isSerialExecution
                            )) {
            return undef;
        }
    }

    if (!$self->getValue('NoStart') && ($scope eq 'system')) {
        if (!$self->_startSystem()) {
            return undef;
        }
    }

    if (($scope eq 'instance') && $instance->exists_remote_host() && $self->isSystemRestartRequired()) {
        my $hosts    = $instance->get_hosts();
        my $hostList = join (', ', @$hosts);
        $self->AddProgressMessage('To complete this action,'
            . " run 'hdbreg' on $hostList"
            . " and start $gProductNameSystem manually.");
    }

    return 1;
}

sub isSystemRestartRequired {
    my ($self) = @_;

    my $instance = $self->getOwnInstance();
    my $existingNetPrefix = $instance->getInternalNetworkPrefix();
    my $targetNetPrefix = $self->getValue('InternalNetwork');
    my $isChangingToNewInternalNetwork = (defined $targetNetPrefix
                                        &&
                                        (!defined $existingNetPrefix ||
                                         ($targetNetPrefix ne $existingNetPrefix))) ? 1 : 0;

    my $isResettingInternalNetwork = (!defined $targetNetPrefix && defined $existingNetPrefix) ? 1 : 0;

    return $isChangingToNewInternalNetwork || $isResettingInternalNetwork;
}

sub _startSystem {
    my ($self) = @_;
    require SDB::Install::SAPSystemUtilities;
    my $util   = new SDB::Install::SAPSystemUtilities();
    my $sapSys = $self->getSAPSystem();
    $sapSys->resetMsgLstContext();
    my $rc = $util->startSystem($sapSys,
                                $self,
                                $self->getValue('Password'),
                                'Reconfigure');
    $self->getMsgLst->appendMsgLst($sapSys->getMsgLst());
    if (!$rc) {
        $self->setErrorMessage('Cannot start system', $sapSys->getErrMsgLst());
        return undef;
    }
    return 1;
}

sub initRemoteHostsForInstance {
	my $self = shift();
	my $isAddingRoles = defined($self->getBatchValue('AddRoles')) || defined($self->getBatchValue('AddLocalRoles'));
	my $isRemovingRoles = defined($self->getBatchValue('RemoveRoles'));

	return $isAddingRoles || $isRemovingRoles ? 1 : $self->SUPER::initRemoteHostsForInstance(@_);
}

#-------------------------------------------------------------------------------
# Updates the content of XS2 on the local host or sends an update command
# to the host where the xscontroller resides.

sub tryLoadInitialContentXS2 {

    my ($self) = @_;

    if (!defined $self->{_xscontrollerHost}) {
        return 1; # nothing to do
    }

    my $instance = $self->getOwnInstance();

    if ($self->{_xscontrollerHost} eq $instance->get_host()) {

        # update content on local host
        if (!$self->loadInitialContentXS2()) {
            return undef;
        }
        $self->{_reconfigured} = 1;
    }
    elsif (!$self->{isSlave}) {

        $self->{_reconfigured} = 1;
        my $remoteHost   = $self->{_remoteHostForLoadingContentXS2};
        my $paramIDs     = ['LoadInitialContentXS2', 'SystemUser', 'OrgManagerUser','TenantUser'];
        my $passwordKeys = ['SQLSysPasswd','OrgManagerPassword','SQLTenantUserPassword'];
        if (!$self->excuteRemoteParallel("Loading initial content of $gShortProductNameXS2 on",
                                         "initial content of $gShortProductNameXS2 loaded",
                                         'hdbmodify',                # executable
                                         'HdbModifyHostRoles',       # main program
                                         $gOperationModifyHostRoles, # hostctrl operation
                                         $passwordKeys,
                                         $paramIDs,
                                         undef,                      # cmd line params
                                         $remoteHost)) {
            return undef;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Waits until 'xsexecagent', 'xscontroller', and 'xsuaaserver' are started.

sub waitForStartXS2 {

    my ($self, $sapSys, $instance) = @_;

    my $timeout = $self->getTimeout ('start_instance');

    if (!defined $timeout){
            $timeout = 7200;
    }

    my $waitDelay   = 2;
    my $waitRetries = int ($timeout / $waitDelay);
    my $host        = $instance->get_host();
    my $nr          = $instance->get_nr();
    my $sapcontrol  = new SDB::Install::SAPControl(undef,$nr);
    $instance->setMsgLstContext($self->getMsgLstContext());
    $instance->takeTraceDirSnapshot($host);
    $sapcontrol->setMsgLstContext([$self->getMsgLst()]);
    $sapcontrol->set_callback($instance->{_f_callback});

    if (!$sapcontrol->WaitforStarted ($waitRetries, $waitDelay)) {
        $self->getErrMsgLst()->addError(undef, $sapcontrol);
        $instance->analyzeTraceDirDeltas($host);
        return undef;
    }
    return 1;
}

sub isAutoInitializeServicesApplicableParameterCallback {
    my($self, $paramId, $valKey, $newValue) = @_;
    my $parser = new SDB::Install::Configuration::AddHostsParser();
    if($paramId eq 'AddRoles' || $paramId eq 'AddHosts') {
        $parser->parse ($newValue, $paramId, 0);
        my $hosts = $parser->getHosts();
        my $isAutoInitSupported = 0;
        foreach my $host(@$hosts){
            my $newRoles = $parser->getRoles($host);
            $isAutoInitSupported |= $self->_rolesSupportAutoInit($newRoles);
        }
        return $isAutoInitSupported;
    }
    if ($paramId eq 'AddOrRemoveLocalRoles') {
        my @rolesArr = split(',', $newValue);
        return 1 if ($self->_rolesSupportAutoInit(\@rolesArr));
    }
    if ($paramId eq 'RemoveHosts') {
        return $self->isRemoveHostsValueApplicableForAutoInitialize($paramId, undef, $newValue);
    }
   return 0;
}

sub isRemoveHostsValueApplicableForAutoInitialize {
    my($self,$paramId,$valKey,$value) = @_;
    my $hdbInstance = $self->getOwnInstance();
    return 0 if (!defined $hdbInstance);

    my @hostsToRemove = split(',', $value);
    my $isAutoInitSupported = 0;
    for my $host (@hostsToRemove) {
        my $hostRoles = $hdbInstance->getHostRolesByIniFile($host);
        $isAutoInitSupported |= $self->_rolesSupportAutoInit($hostRoles);
    }
    return $isAutoInitSupported;
}

sub _rolesSupportAutoInit {
    my ($self, $rolesArr) = @_;
    my $hdbInstance = $self->getOwnInstance();
    return 0 if (!defined $hdbInstance);

    my $validAutoInitializeServicesRolesComponentMap = GetAutoInitializeServicesRolesComponentMap();
    my $isAutoInitSupported = 0;
    foreach my $role (@$rolesArr){
        my $validComponetForRole = $validAutoInitializeServicesRolesComponentMap->{$role};
        next if(!defined $validComponetForRole);

        my $componentDir = undef;
        my $componentDirName = undef;
        my $validHostRoles       = GetHostRoleProperties();
        if ($validHostRoles->{$role}->{isStreaming}) {
            $componentDir = $hdbInstance->getStreamingLcmDir();
            $componentDirName = $gDirNameStreaming;
        }
        elsif ($validHostRoles->{$role}->{isExtendedStorage}) {
            $componentDir = $hdbInstance->getExtendedStorageLcmDir();
            $componentDirName = $gDirNameEs;
        }
        elsif ($validHostRoles->{$role}->{isAccelerator} && !$validHostRoles->{$role}->{isStandby}) {
            $componentDir = $hdbInstance->getAcceleratorLcmDir();
            $componentDirName = $gDirNameAccelerator;
        }
        elsif ($validHostRoles->{$role}->{isRDSync}) {
            $componentDir = $hdbInstance->getRemoteDataSyncLcmDir();
            $componentDirName = $gDirNameRDSync;
        }
        if(defined $componentDir && -d $componentDir){
            my $manifest = SDB::Install::Manifest->new($hdbInstance->getInstalledComponentManifestPath($componentDirName));
            if($manifest->isAutoInitializeServiceSupported()){
                $self->getMsgLst()->addMessage ("Component directory $componentDirName found, which requires auto initializing of services.");
                if(!$validHostRoles->{$role}->{isExtendedStorage}){
                    $self->{_requiresTenantUserCredentials} = 1;
                }
                $isAutoInitSupported = 1;
            }
        }
    }
    return $isAutoInitSupported;
}

sub checkRequiresTenantUserCredentials{
    my ($self) = @_;
    return $self->{_requiresTenantUserCredentials};
}

sub shouldWarnIfCalledStandalone{
    my ($self) = @_;
    return $self->_calledStandaloneCriterion();
}

sub getExeName{
    return "hdbmodify";
}

sub getResidentHdblcmPath{
    my ($self) = @_;
    return $self->_constructResidentHdblcmPath();
}

1;
