package SDB::Install::Configuration::HdbReg;

use strict;

use SDB::Install::Configuration qw ($bool_true_pattern);
use SDB::Install::Configuration::AnyRegisterRename;
use SDB::Install::Globals   qw($gProductName $gProductNameSystem $gSapmntDirUx $gXSParametersConstraint $gXSParametersGenericConstraint);
use SDB::Install::SysVars   qw($isWin $path_separator);
use LCM::Configuration::ParametersCreator;
#use SDB::Install::DebugUtilities qw(dumpThings);

require SDB::Install::Configuration::NewDB;

our @ISA = qw (SDB::Install::Configuration::AnyRegisterRename);

our $section        = 'HdbReg';
our $registerOnly   = 'Valid for normal registration';
our $sysPasswdOnly  = "Valid for changing system user password";
our $unregisterOnly = "Valid with '--unregister' only";

our $peristentFileName = 'hdbreg';
sub getPersistentFile {return $peristentFileName}

# Hash keys of this class: 'localHostName' (string)
#                          'isSlave'       (bool)
#                          'ownInstance'   (SDB::Install::SAPInstance::TrexInstance)
#                          'params'        (hash)
#                          'sapSys'        (SDB::Install::SAPSystem)


#-------------------------------------------------------------------------------
# Constructor
#
sub new {
    my $self = shift->SUPER::new (@_);
    my $order = 0;

    $self->{params} = {
        'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $section),
        'Unregister'       => $self->getParamUnregister      ($order++, $section, 'unregister', 'Unregisters'),
        'StartOnly'        => $self->getParamStartOnly       ($order++, $section),
        'StopOnly'         => $self->getParamStopOnly        ($order++, $section),
        'ChangeSystemPasswdOnly' => $self->getParamChangeSystemPasswd($order++, $section, 0, 1),
        'CheckOnly'        => $self->getParamCheckOnly       ($order++, $section),
        'ShowPending'      => $self->getParamShowPending     ($order++, $section, undef, $registerOnly),
        'InitUser'         => $self->getParamInitUser        ($order++, $section,        $registerOnly),
        ($isWin
            ? ('Drive'     => $self->getParamDrive           ($order++, $section),
               'Domain'    => $self->getParamDomain          ($order++, $section),
              )
            : ('Target'    => $self->getParamTarget          ($order++, $section), )
         ),
        'RemoteExecution'  => $self->getParamRemoteExecution ($order++, $section),
        'UseHttp'          => $self->getParamUseHttp         ($order++, $section),
        'SAPControlUseHttp'  => $self->getParamSAPControlUseHttp  ($order++, $section, 1, 0),
        'InstallHostagent' => $self->getParamInstallHostagent($order++, $section, $registerOnly),
        'SID'              => $self->getParamSID             ($order++, $section),
        'SSOCertificate'   => $self->getParamSSOCertificate  ($order++, $section),
        'InstanceNumber'   => $self->getParamInstanceNumber  ($order++, $section, 'Instance Number', 1),
        'HostMap'          => $self->getParamHostMap         ($order++, $section, $registerOnly),
        'SystemUsage'      => $self->getParamSystemUsage     ($order++, $section, $registerOnly),
        'Scope'            => $self->getParamScope           ($order++, $section, undef, $registerOnly),
        'ScopeInteractive' => $self->getParamScopeInteractive($order++, $section, undef, $registerOnly),
        'SystemIsOffline'  => $self->getParamSystemIsOffline ($order++, $section),
        'Password'         => $self->getParamPassword        ($order++, $section),
        ($isWin
            ? ('HostagentUserName' => $self->getParamHostagentUserName($order++, $section, $registerOnly)
              )
            : ('InstallSSHKey'     => $self->getParamInstallSSHKey    ($order++, $section),
               'RootUser'          => $self->getParamRootUser         ($order++, $section),
               'RootPassword'      => $self->getParamRootPassword     ($order++, $section),
            )
        ),
        'HostagentPassword' => $self->getParamHostagentPassword($order++, $section,    $registerOnly),
        'ListenInterface'   => $self->getParamListenInterface  ($order++, $section, 1, $registerOnly),
        'InternalNetwork'   => $self->getParamInternalNetwork  ($order++, $section, 1, $registerOnly),
        'KeepUser'          => $self->getParamKeepUser         ($order++, $section,    $unregisterOnly),
        ($isWin
            ? ()
            : ('KeepUserHomeDir'   => $self->getParamKeepUserHomeDir($order++, $section,             $unregisterOnly),
               'InitUserHomeDir'   => $self->getParamInitUserHomeDir($order++, $section,             $registerOnly),
               'HomeDir'           => $self->getParamHomeDir($order++, $section,'/usr/sap/$SID/home',$registerOnly),
               'Shell'             => $self->getParamShell  ($order++, $section,                     $registerOnly),
               'UID'               => $self->getParamUID    ($order++, $section,                     $registerOnly),
               'GID'               => $self->getParamGID    ($order++, $section, undef, 1,           $registerOnly),
               'XSSpaceUserIdSAP'  => GetParamXSSpaceUserIdSAP($order++, $section, $gXSParametersConstraint),
               'XSSpaceUserIdProd' => GetParamXSSpaceUserIdProd($order++, $section, $gXSParametersConstraint),
               'SkipModifySudoers' => $self->getParamSkipModifySudoers($order++, $section),
            )
        ),
        'KeepXsUsers'       => $self->getParamKeepXsUsers      ($order++, $section, $gXSParametersGenericConstraint),
        'ChangeSystemPasswd'=> $self->getParamChangeSystemPasswd($order++,$section,1,0,$registerOnly),
        'SystemUser'        => $self->getParamSystemUser       ($order++, $section, 1, $sysPasswdOnly),
        'SrcSQLSysPasswd'   => $self->getParamSrcSQLSysPasswd  ($order++, $section,    $sysPasswdOnly),
        'TrgSQLSysPasswd'   => $self->getParamTrgSQLSysPasswd  ($order++, $section,    $sysPasswdOnly),
        'RevertToSnapshot'  => $self->getParamRevertToSnapshot ($order++, $section,    $registerOnly),
        'AutoStart'         => $self->getParamAutoStart        ($order++, $section, 1, $registerOnly),
        'NoStart'           => $self->getParamNoStart          ($order++, $section,
                               "Do not start $gProductNameSystem but start service (sapstartsrv) instead",
                               "Does not start $gProductNameSystem but starts service (sapstartsrv) instead",
                               0), # not hidden
    };

    # Target and SID are set by SBD::Install::Installer::getInstallation
    $self->{params}->{SID}->{hidden}    = 1;
    $self->{params}->{Target}->{hidden} = 1;

    $self->setSummaryOrder(['Target',
                            'SID',
                            'InstanceNumber',
                            'HostMap',
                            'SystemUsage',
                            'InitUser',
                            'HomeDir',
                            'InitUserHomeDir',
                            'Shell',
                            'UID',
                            'GID',
                            'KeepUser',
                            'KeepUserHomeDir',
                            'ListenInterface',
                            'InternalNetwork',
                            'InstallHostagent',
                            'Scope',
                            'RemoteExecution',
                            'RootUser',
                            'HostagentUserName',
                            'CheckOnly',
                            'AutoStart',
                            'NoStart',
                           ]);

    #print("\n\n======= ".'SDB::Install::Configuration::HdbReg::new()'." =========\n\n");
    #print('$self->{params}: '."\n"); dumpThings($self->{params}, 3, 2);
    #print("\n\n================\n\n");

    return $self;
}

#-------------------------------------------------------------------------------
# Checks the existence of the base path for data volumes.
#
# Parameter: String $path
# Returns int retCode

sub checkBasePathDataVolumes {
    my ($self, $path) = @_;
    return $self->existsPath($path, 'BasePathDataVolumes');
}


#-------------------------------------------------------------------------------
# Checks the existence of the base path for log volumes.
#
# Parameter: String $path
# Returns int retCode

sub checkBasePathLogVolumes {
    my ($self, $path) = @_;
    return $self->existsPath($path, 'BasePathLogVolumes');
}


#-------------------------------------------------------------------------------
sub checkChangeSystemPasswd {
    my ($self, $value) = @_;

    if (defined $value && ($value =~ /$bool_true_pattern/i)) {

       $self->setSkip('SystemUser',      0);
       $self->setSkip('SrcSQLSysPasswd', 0);
       $self->setSkip('TrgSQLSysPasswd', 0);
    }
    return 1;
}


#-------------------------------------------------------------------------------
sub checkChangeSystemPasswdOnly {
    my ($self, $value) = @_;

    return $self->checkChangeSystemPasswd($value);
}


#-------------------------------------------------------------------------------
sub checkInternalNetwork {

    my ($self, $value) = @_;
    return $self->checkInternalNetworkOnly($value);
}


#-------------------------------------------------------------------------------
sub checkListenInterface {

    my ($self, $value) = @_;
    return $self->checkListenInterfaceOnly($value);
}


#-------------------------------------------------------------------------------
# Checks the specified SID
#
# Parameter: String $sid
#
# Returns int retCode 

sub checkSID {

    my ($self, $sid) = @_;

    delete $self->{handleInitParamsCalled};
    $self->{current_sid} = $sid;
    my $sapSys     = $self->{sapSys};
    my $unregister = $self->getValue('Unregister');

    if (!defined $sapSys) {

        my $systems = $self->getCollectSAPSystems();
        $sapSys     = $systems->{$sid};

        if (!defined $sapSys) {

            $self->resetSAPSystemHashes();

            my $installPath = $self->getValue ('Target');

            if (!$installPath) {

                if (!defined $unregister) {

                    $self->PushError ($self->{params}->{Target}->{str}
                                                               ." not defined");
                    return 0;
                }
            }
            else {

                my $installPathSID = $installPath . $path_separator . $sid;

                if (-d $installPathSID) {
                    $sapSys = new SDB::Install::SAPSystem();
                    $sapSys->initWithGlobDir($sid, $installPath);
                }
                elsif (!defined $unregister) {
                    # $installPathSID directory not found

                    $self->PushError ($self->{params}->{Target}->{str}
                                          ." '$installPathSID' does not exist");
                    return 0;
                }
            }
        }

        if (!defined $unregister &&
            !$self->checkExistingSID
                          ($sid, $self->{params}->{SID}->{str}, $sapSys)) {
            return 0;
        }

        if (defined $sapSys) {
            $self->{sapSys} = $sapSys;
        }
    }

    if ($self->isShowPending()) {
        return 1;
    }

    my $instance = (defined $sapSys) ? $self->getOwnInstance() : undef;

    if (!defined $sapSys || !$sapSys->hasNewDB || !defined $instance) {
        $self->PushError ("SAP System '$sid' has no HDB instance");
        return 0;
    }

    if (defined $unregister && !-d $sapSys->getUsrSapSid()) {
        $self->PushError ("Directory '" . $sapSys->getUsrSapSid() .
                          "' does not exist");
        return 0;
    }

    my $paramInstNr                 = $self->{params}->{InstanceNumber};
    $self->setDefault('InstanceNumber', $instance->get_nr());
    $paramInstNr->{set_interactive} = 0 if (defined $self->getDefault('InstanceNumber'));

    my $rc        = 1;
    my $startOnly = $self->getValue('StartOnly');
    my $userName  = $sapSys->getUser()->getSidAdmName();

    if (defined $unregister || defined $startOnly
                            || defined $self->getValue('StopOnly')) {

        # In case of remote hosts, parameter 'Scope' has to be enabled
        # in order to assign 'batchValue' to 'value'
        # and to create the class RemoteHosts in case of value='system'

        $self->tryEnableScopeParams();
        $self->setSkip('ScopeInteractive');

        if (!$isWin
            && defined $startOnly
            && (($startOnly eq 'instance')
                ||
                (($startOnly eq 'all_instances')
                                             && !$self->isUseSAPHostagent()))) {
            $self->setSkip('Password');
        }
        elsif (defined $sapSys) {
            $self->changePasswordStr($userName);
        }

        my $hostDir = $instance->get_hostNameDir();

        if (defined $hostDir && -d $hostDir) {
            $self->{pers_fn_reg_rename} =
                $hostDir . $path_separator . $self->getPersistentFile();
        }

        $self->showSystemProperties();

        if (defined $unregister) {

            if (exists $self->{params}->{KeepUser}) {
                $self->{params}->{KeepUser}->{str} =
                    sprintf($self->{params}->{KeepUser}->{str_templ},$userName);
            }

            if (exists $self->{params}->{KeepUserHomeDir}) {
                $self->{params}->{KeepUserHomeDir}->{str} =
                    sprintf($self->{params}->{KeepUserHomeDir}->{str_templ},
                                                                     $userName);
            }
            if (!$isWin && !$self->isRemoteScope($unregister) &&
                $sapSys->getUser()->exists () &&
                !$instance->sapstartsrvIsRunning ()){

                $self->setSkip ('Password');
            }
        }
    }
    elsif (!$self->getValue('ChangeSystemPasswdOnly')) {

        $rc = $self->handleInitHostMap ($instance);

        if ($rc) {
            if ($instance->hasLocalHost()) {
                $rc = $self->handleInitParams();
            }
            else {
                $self->{params}->{HostMap}->{mandatory} = 1;
            }
        }

        if (exists $self->{params}->{InitUser}) {
            $self->{params}->{InitUser}->{str} =
              sprintf($self->{params}->{InitUser}->{str_templ}, $userName);
        }

        if (exists $self->{params}->{InitUserHomeDir}) {
            $self->{params}->{InitUserHomeDir}->{str} =
                sprintf($self->{params}->{InitUserHomeDir}->{str_templ},
                                                                     $userName);
        }

        my $noStart = $self->getBatchValue('NoStart');

        if (defined $noStart
            &&
            (($noStart eq '') || ($noStart =~ /$bool_true_pattern/i))) {

            $self->{params}->{ChangeSystemPasswd}->{set_interactive} = 0;
        }
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Collects information of remote hosts and checks SID, instance number,
# and user/group ID. These values have to belong to this SAP HANA system
# or have to be free.

sub CollectOtherHostInfos {
    my ($self, $isNotVerbose) = @_;

    if (defined $self->getValue('Unregister') ||
        defined $self->getValue('StartOnly')  ||
        defined $self->getValue('StopOnly')) {
        # skip collecting host info
        return 1;
    }

    my $sidadm = $self->{sapSys}->getUser()->getSidAdmName();

    if (!$self->SUPER::CollectOtherHostInfos($isNotVerbose, $sidadm)) {
        return undef;
    }

    if (!$self->checkCollectedHostInfo($self->getValue('SID'),
                                       $self->getValue('InstanceNumber'))) {
        return undef;
    }

    return $self->tryInitGroupAndUserID();
}


#-------------------------------------------------------------------------------
# Sets the hash 'isSlave' and exeuctes 'handleSlaveInitParams' in case of slave.

sub detectSlave {

    my ($self, $hostDir) = @_;

    my $isUnregHost = 0;
    my $pers_data   = undef;
    my $persFile    = $hostDir . $path_separator . $self->getPersistentFile();

    if ($self->pers_exists($persFile)) {

        $pers_data = $self->pers_load($persFile);

        if (defined $pers_data && $pers_data->{unregisterHost}) {
            # This file indicates that this instance of a distributed
            # system was unregistered while other instances still exist.
            # So hdbreg has to use the existing user values,
            # i.e. call handleSlaveInitUserParams.
            # In case of a master continue with master handling.

            $isUnregHost = 1;
            $pers_data   = undef;

            if (!$self->isCheckOnly()) {
                $self->setUnregisterPersFile($persFile);
            }
        }
        else {
            $self->{isSlave}            = 1;
            $self->{pers_fn_reg_rename} = $persFile;
        }
    }

    my $remoteUserExists = ($isUnregHost || $self->{isSlave}) ? 1 : 0;

    if (!$self->tryHandleExistingGroupAndUser
            ($self->{sapSys}->getUser()->getSidAdmName(), $remoteUserExists)) {
        return 0;
    }

    my $rc = 1;

    if ($remoteUserExists) {
        $rc = $self->handleSlaveInitUserParams();
    }

    if ($rc && $self->{isSlave}) {
        $rc = $self->handleSlaveInitParams($pers_data);
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Enables the multiple host parameters.
# sidadm user is used instead of hostagent user
# in case of hostctrl and start/stop system.
#
# A comment at the beginning of AnyConfig shows depending parameters.

sub enableMultipleHostParameters {

    my ($self) = @_;

    my $useSidadmForHostctrl = defined $self->getValue('StartOnly') ||
                               defined $self->getValue('StopOnly');

    return $self->SUPER::enableMultipleHostParameters($useSidadmForHostctrl);
}


#-------------------------------------------------------------------------------
# Checks the existence of the path.
#
# Parameter: String $path, String $paramID
#
# Returns int retCode 

sub existsPath {
    
    my ($self, $path, $paramID) = @_;
    
    if (!-d $path) {
        $self->PushError
                ("$self->{params}->{$paramID}->{str} '$path' does not exist");
        return 0;
    }    
    return 1;
}


#-------------------------------------------------------------------------------
# Initializes default values
#
# Returns int retCode

sub InitDefaults {
    
    my ($self) = @_;
    
    if (!defined $self->SUPER::InitDefaults()) {
        return undef;
    }
    
    if (!defined $self->getDefault('Target')) {
        $self->setDefault('Target', $gSapmntDirUx);
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Initializes the scope and skips unused parameters
# in case of changing system user password only

sub initChangeSystemPasswdOnly {

    my ($self) = @_;

    my $scope    = 'instance';
    my $optScope = $self->{params}->{Scope}->{opt};
    $self->{options}->{$optScope}          = $scope;
    $self->{params}->{Scope}->{batchValue} = $scope;
    $self->setSkip('CheckOnly');
    $self->setSkip('NoStart');
    $self->setSkip('Password');
    $self->setSkip('ShowPending');
    $self->skipRegisterOnlyParams();
    return 1;
}


#-------------------------------------------------------------------------------
# Defines ignore values for the parameter '--ignore'.

sub getHiddenIgnoreValues{
    return [qw (scan_password_database check_hostname)];
}


#-------------------------------------------------------------------------------
# This method is called from handleInitParams and initializes succeding
# parameters (hash 'params') on slave hosts.
#
# Parameter: hash $pers_data  *)
#
#    *) Contains the content of the persistent file created for hosts by
#       the master (filename: '.../HDB<nr>/<host>/<persistentFileName>').
#       $pers_data are undefined in case of DBUpgradeHost.
#
# Returns int retCode

sub handleSlaveInitParams {
    my ($self, $pers_data) = @_;

    my $rc = $self->SUPER::handleSlaveInitParams($pers_data);

    if (defined $pers_data) {

        $self->{oldHost} = $pers_data->{oldHost};
        $self->{params}->{AutoStart}->{value} = $pers_data->{AutoStart};

        if ($pers_data->{initUserHomeDir}) {
            $self->{params}->{InitUserHomeDir}->{value} = 1;
        }

        if ($pers_data->{isNoHostStart}) {
            $self->{isNoHostStart} = 1;
        }

        if ($pers_data->{isRenameAccelerator}) {
            $self->{isRenameAccelerator} = 1;
            $self->{oldUidAccelerator}   = $pers_data->{oldUidAccelerator};
            $self->{oldGidAccelerator}   = $pers_data->{oldGidAccelerator};
        }

        if ($pers_data->{isRenameExtendedStorage}) {
            $self->{isRenameExtendedStorage} = 1;
            $self->{oldUidEs} = $pers_data->{oldUidEs};
            $self->{oldGidEs} = $pers_data->{oldGidEs};
        }

        if ($pers_data->{isRenameRDSync}) {
            $self->{isRenameRDSync} = 1;
            $self->{oldUidRDSync}   = $pers_data->{oldUidRDSync};
            $self->{oldGidRDSync}   = $pers_data->{oldGidRDSync};
        }
        if ($pers_data->{isRenameXS2}) {
            $self->{isRenameXS2} = 1;
        }
    }

    my $instance = $self->getOwnInstance();
    my $newHost  = $instance->get_host ();
    my $oldHost  = (defined $self->{oldHost}) ? $self->{oldHost} : $newHost;

    $self->{params}->{HostMap}->{skip}  = 1;
    $self->{params}->{HostMap}->{value}->{lc($oldHost)} = lc($newHost);

    $self->AddProgressMessage("$self->{params}->{Target}->{str}: "
                             . $self->{sapSys}->get_target());
    return $rc;
}


#-------------------------------------------------------------------------------
# This method is called from handleInitParams and initializes succeding
# parameters (hash 'params') on a host where hdbreg/hdbrename is executed
# at first (this host can de differ from the master host of the database)
#
# Returns int retCode

sub handleMasterInitParams {

    my ($self) = @_;

    if (!$self->SUPER::handleMasterInitParams()){
        return 0;
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Overrides the function of SDB::Install::PersistencyMngr
# The first execution of hdbreg creates these files for all slave hosts.
# Each slave deletes its file at the end of (slave) hdbreg.
#
# Returns the filename if this host is an outstanding slave, otherwise undef

sub pers_filename{
    return $_[0]->{pers_fn_reg_rename};
}


#-------------------------------------------------------------------------------
# Returns the keys that are inserted into the persistent file in order
# to complete this tool on a remote location.

sub pers_keys{
    return [qw (oldHost
                oldSID
                oldInstanceNumber
                isNoHostStart
                initUserHomeDir
                AutoStart
                isRenameAccelerator
                oldUidAccelerator
                oldGidAccelerator
                isRenameExtendedStorage
                oldUidEs
                oldGidEs
                isRenameRDSync
                isRenameXS2
                oldUidRDSync
                oldGidRDSync
                unregisterHost)];
}


#-------------------------------------------------------------------------------
# Scans the command line options for '--unregister', '--start_only'
# or '--stop_only' and checks detected values.
#
# Parameter $paramID  string ('Unregister', 'startOnly', 'StopOnly')
#
# returns (retcode, scope)

sub scanParamUnregStartStop {

    my ($self, $paramID) = @_;

    my $params = $self->{params};
    my $value  = $self->getBatchValue($paramID);
    my $scope  = undef;

    if (defined $value) {

        $scope = ($value eq '') ? $self->getDefault($paramID) : lc($value);

        if (!$self->checkValue($paramID, $scope)) {
            return (undef, undef);
        }

        $self->setSkip($paramID, 0);

        $self->skipRegisterOnlyParams();
        $self->setSkip('ShowPending');

        if ($paramID eq 'Unregister') {

            $params->{Scope}->{hidden} = 1; # hide summary output
        }
        else {
            $params->{Scope}->{valid_values} =
                    $params->{$paramID}->{valid_values};

            $self->setSkip('CheckOnly');
            if (($scope eq 'system_only') || ($paramID eq 'StopOnly')) {
                $self->setSkip('NoStart');
            }
        }

        my $optScope                   = $params->{Scope}->{opt};
        $self->{options}->{$optScope}  = $scope;
        $params->{Scope}->{batchValue} = $scope;
    }
    return (1, $scope);
}


#-------------------------------------------------------------------------------
# Skips the remaining parameters in case of unregister or start/stop only.
#
# Returns int retCode

sub skipRegisterOnlyParams {

    my ($self, $setHidden) = @_;

    foreach my $id (keys (%{$self->{params}})) {
        my $param = $self->{params}->{$id};
        if (defined $param->{constraint} && $param->{constraint} eq $registerOnly) {
            $param->{skip}   = 1;
            $param->{hidden} = 1 if ($setHidden);
        }
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Hides parameters that are not provided for unregister.

sub hideRegisterAndStartStopParams {

    my ($self) = @_;
    my $params = $self->{params};

    $params->{StartOnly}->{hidden} = 1;
    $params->{StopOnly }->{hidden} = 1;
    $params->{NoStart  }->{hidden} = 1;
    $params->{StartOnly}->{hidden} = 1;

    $params->{KeepUser       }->{constraint} = undef;
    $params->{KeepUserHomeDir}->{constraint} = undef;

    $self->skipRegisterOnlyParams(1);

    return 1;
}


sub getIgnoreValues { return ['check_platform']; }


#===============================================================================
# import check functions from SDB::Install::Configuration::NewDB

sub checkSystemRequirements;
*checkSystemRequirements = \&SDB::Install::Configuration::NewDB::checkSystemRequirements;

sub shouldWarnIfCalledStandalone{
    my ($self) = @_;
    return $self->_calledStandaloneCriterion();
}

sub getExeName{
    return "hdbreg";
}

sub getResidentHdblcmPath{
    my ($self) = @_;
    return $self->_constructResidentHdblcmPath();
}

1;

