package SDB::Install::Configuration::NewDB;

use strict;
use SDB::Install::Configuration;
use SDB::Install::Configuration::NewServerConfig;
use SDB::Install::SysVars;
use SAPDB::Install::Hostname;
use SDB::Install::System;
use SAPDB::Install::System::Unix qw (sysinfo);

use SDB::Install::Globals qw ($gDriveWin
                              $gProductName
                              $gProductNameEngine
                              $gSapmntDirUx
                              $gSapsysGroupIdDefault
                              $gSapsysGroupName
                              $gShortProductNameEngine
                              $gMinMemGB);

use SAPDB::Install::System::Win32::API;
use SDB::Install::SAPSystem qw (CollectSAPSystems ExistsInstanceNumber
			$STEP_CREATE_INSTANCE $STEP_EXTRACT @STEP_NAMES FORBIDDEN_SIDS
			isForbiddenSID);
use SAPDB::Install::MachInfo;
our @ISA = qw (SDB::Install::Configuration::NewServerConfig);

our $section = 'Server';



our $volumeSubDirPerHost = 'mnt00001';

#
# reserve well known installation names
#


#our @user_params = $isWin ? ('Password','Domain') : ('Password','Shell',
#	'HomeDir', 'UID', 'GID');

sub InitDefaults{
    my ($self, $kit) = @_;
    my $physicalMemory = $self->getPhysicalMem();

	if (!defined $physicalMemory || ($physicalMemory < $gMinMemGB * 1024 * 1024 * 1024)) {
		$self->{params}->{RestrictMaxMem}->{skip} = 1;
		$self->{params}->{MaxMem}->{skip} = 1;
	} else {
		$self->setDefault('MaxMem', int($physicalMemory/(1024*1024)));
		$self->{params}->{MaxMem}->{minValue} = $gMinMemGB * 1024;
		$self->{params}->{MaxMem}->{maxValue} = int($physicalMemory/(1024*1024));
	}

    if (defined $kit){
        $self->{kit} = $kit;
        $self->{_kitversion} = $kit->GetVersion;
        $self->setValue ('versionId', $kit->getVersionIdString ());
        my $hw_checkscript = join ($path_separator, $kit->getArchiveDir (), 'HanaHwCheck.py');
        my $mf = $kit->getManifest();
        if (!defined $mf){
            $self->AddError (undef, $kit);
            return undef;
        }
        if (!$isWin && !$isPowerPC && ($mf->isCustomerReleaseBranch () ||
            exists $ENV{'HDB_FORCE_HANA_HARDWARE_CHECK'})){
            if (!-f $hw_checkscript){
                $self->AddWarning ("Hardware check script '$hw_checkscript' not found");
            }
            else{
                $self->{_hwCheckScript} = $hw_checkscript;
            }
        }
        $self->{kitDir} = $kit->getArchiveDir ();
    }

	my $msglst = new SDB::Install::MsgLst ();

	$self->setDefault('InstanceNumber', undef);
	my $defaultInstanceNumber = $self->getFreeInstanceNumber ();
	$self->setDefault ('InstanceNumber', $defaultInstanceNumber);
	$self->{params}->{InstanceNumber}->{valid_values} = $self->getFreeInstanceNumbers ();

	if (!$isWin){
		my $gid = getgrnam ($gSapsysGroupName);
		if (defined $gid){
			$self->{params}->{GID}->{value} = $gid;
			$self->{sapsysids}->{$gid} = [hostname()];
			$self->{localSapsysGID}    = [ $gid, hostname() ];
		}
	}
	else{
		my $drive = getSAPDrive ($self->getMsgLst ());
		if ($drive){
			$self->{params}->{Drive}->{value} = uc ($drive);
			$self->{params}->{Drive}->{isStaticValue} = 1;
			#$self->{params}->{Target}->{value} = $path;
		}
	}

	return 1;
}

#-------------------------------------------------------------------------------
# Constructor
#
# Parameters: $caller      (derived class of SDB::Install::App,  may be undef)
#             $options     (used by SDB::Install::Configuration, may be undef)
#             $config_file (used by SDB::Install::Configuration, may be undef)

sub new{
    my ($class, $caller, @args) = @_;

    my $self        = $class->SUPER::new(@args);
    my $order       = 0;
    my $is_hdbsetup = (defined $caller
                       && $caller->isa('SDB::Install::App::Gui::Installation'));
   $self->{params} = {
        ($isWin
            ? ()
            : ('UsePMem' => $self->getParamUsePMem($order++, $section))
        ),
        'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $section),
        ($is_hdbsetup
            ? ()
            : ('CheckOnly' => $self->getParamCheckOnly($order++, $section, 'install'), )
        ),
        'RemoteExecution'  => $self->getParamRemoteExecution ($order++, $section),
        'UseHttp'          => $self->getParamUseHttp         ($order++, $section),
        'InstallHostagent' => $self->getParamInstallHostagent($order++, $section),
        'HostName'         => $self->getParamHostName        ($order++, $section),
        'CreateInitialContainer' => $self->getParamCreateInitialContainer($order++, $section),
        ($isWin
            ? ()
               # today add host via hdbinst is implemented for Linux only
            : ('AddHosts'  => $self->getParamAddHosts      ($order++, $section, 1),
              )
        ),
        ($isWin
            ? ('HostagentUserName' => $self->getParamHostagentUserName($order++, $section),
              )
            : ('CheckMnt'      => $self->getParamCheckMnt      ($order++, $section),
               'Target'        => $self->getParamTarget        ($order++, $section),
               'InstallSSHKey' => $self->getParamInstallSSHKey ($order++, $section),
               'RootUser'      => $self->getParamRootUser      ($order++, $section), # calls collect host info
               'RootPassword'  => $self->getParamRootPassword  ($order++ ,$section), # calls collect host info
              )
         ),
        'HostagentPassword'=> $self->getParamHostagentPassword($order++,$section),   # calls collect host info
        'SID'              => $self->getParamSID             ($order++, $section),
        'InstanceNumber'   => $self->getParamInstanceNumber  ($order++, $section),
        'WorkerGroup'      => $self->getParamWorkerGroup     ($order++, $section,0 , $STEP_CREATE_INSTANCE),
        'DbMode'           => $self->getParamDbMode          ($order++, $section),
        'DbIsolation'      => $self->getParamDbIsolation     ($order++, $section),
        'CreateInitialTenant' => $self->getParamCreateInitialTenant ($order++, $section),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $section),
		'SystemUsage'      => $self->getParamSystemUsage     ($order++, $section),
        ($isWin
            ? ('DistributedSystem' => {'order'             => $order++,
                                       'opt'               => 'cfs',
                                       'type'              => 'boolean',
                                       'section'           => $section,
                                       'value'             => undef,
                                       'default'           => 0,
                                       'str'               => 'Multiple Host System',
                                       'set_interactive'   => 0,
                                       'init_with_default' => 1,
                                       'persStep'          => $STEP_EXTRACT },
               'Drive' => $self->getParamDrive ($order++, $section),
               'Domain' => $self->getParamDomain ($order++, $section),
              )
            : ()
        ),
        'Password'         => $self->getParamPassword        ($order++, $section),
        ($isWin
            ? ()
            : ('HomeDir'          => $self->getParamHomeDir         ($order++, $section, '/usr/sap/$SID/home'),
               'Shell'            => $self->getParamShell           ($order,   $section),
               'UID'              => $self->getParamUID             ($order++, $section),
               'GID'              => $self->getParamGID             ($order++, $section, $gSapsysGroupIdDefault, 1),
               'StorageConfigDir' => $self->getParamStorageConfigDir($order++, $section),
               'InternalNetwork'  => $self->getParamInternalNetwork ($order++, $section, 0, undef, 0, 1),
              )
        ),
        'BasePathDataVolumes'  => $self->getParamBasePathDataVolumes($order++, $section,
                                 ($isWin ? '$Drive\usr\sap\$SID\SYS\global\hdb\data'
                                         : '$Target/$SID/global/hdb/data')),
        'BasePathLogVolumes'   => $self->getParamBasePathLogVolumes ($order++, $section,
                                 ($isWin ? '$Drive\usr\sap\$SID\SYS\global\hdb\log'
                                         : '$Target/$SID/global/hdb/log')),

        ($isWin
            ? ()
            : ('BasePathPMem'   => $self->getParamBasePathPMem ($order++, $section))
        ),
        'CustomConfigDir' => $self->getParamCustomConfigDir($order++, $section),
        'ChangeInitialSSFSKey' => $self->getParamChangeInitialSSFSKey  ($order++, $section),
        'installType' => {
            'order' => $order++,
            'opt' => 'installType',
            'short_opt' => 'T',
            'type' => 'string',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'Installation Type',
            'init_with_default' => 1,
            'set_interactive' => 0,
            'hidden' => 1,
            'persist_for_upgrade' => 1,
            'DEPRECATED' => 1
        },
        'StripSymbols'       => $self->getParamStripSymbols      ($order++, $section),
		'NoStart'            => $self->getParamNoStart           ($order++, $section,
		                        'Do not start the instance after installation', 'Does not start the instance after installation', 1),
		'MachineUtilization' => $self->getParamMachineUtilization($order++, $section),
        'RestrictMaxMem'     => $self->getParamRestrictMaxMem    ($order++, $section),
        'MaxMem'             => $self->getParamMaxMem            ($order++, $section),
		'SQLSysPasswd'       => $self->getParamSQLSysPasswd      ($order++, $section,
		                                                          'initial_passwd', 0, 'SYSTEM'),
        'AutoStart'          => $self->getParamAutoStart         ($order++, $section),
        'Repository'         => $self->getParamRepository        ($order++, $section),
        'ISCMode'            => $self->getParamISCMode           ($order++, $section),
		'EnableXSEngine' => {
			'order' => $order++,
			'opt' => 'xs_engine',
			'alias_opts' => ['xsengine','execution_service'],
			'type' => 'boolean',
			'section' => $section,
			'value' => undef,
			'default' => 1,
			'str' => 'Enable XS Engine',
            'desc'=> 'Enables the XS engine',
			'console_omit_word_Enter' => 1,
			'init_with_default' => 1,
			'set_interactive' => 0,
			'persStep' => $STEP_CREATE_INSTANCE
		},
        'XSEngineHttpPort' => {
            'order' => $order++,
            'opt' => 'xs_engine_http_port',
            'alias_opts' => ['execution_service_http_port'],
            'type' => 'number',
            'opt_arg' => '<port>',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'XS Engine HTTP Port',
            'desc'=> 'Specifies the HTTP port of the XS engine',
            'init_with_default' => 0,
			'log_value' => sub {defined $_[0] ? $_[0] : '80' . $self->getValue('InstanceNumber');},
            'set_interactive' => 0,
            'skip' => 1
        },
        'XSEngineHttpsPort' => {
            'order' => $order++,
            'opt' => 'xs_engine_https_port',
            'alias_opts' => ['execution_service_https_port'],
            'type' => 'number',
            'opt_arg' => '<port>',
            'section' => $section,
            'value' => undef,
            'default' => undef,
            'str' => 'XS Engine HTTPS Port',
            'desc'=> 'Specifies the HTTPS port of the XS engine',
            'init_with_default' => 0,
			'log_value' => sub {defined $_[0] ? $_[0] : '43' . $self->getValue('InstanceNumber');},
            'set_interactive' => 0,
            'skip' => 1
        },
        'ImportContent' => $self->getParamImportContent ($order++, $section),
        # internal parameter to build versioned exe directory
        'versionId' => {
            'order' => $order++,
            'type' => 'string',
            'value' => undef,
            'default' => undef,
            'init_with_default' => 0,
            'set_interactive' => 0,
            'hidden' => 1,
            'persStep' => $STEP_EXTRACT
        },
        ($isWin
            ? ('RetryHostagentUserName' => $self->getParamRetryParam($order++, 'HostagentUserName'))
            : ('SkipModifySudoers' => $self->getParamSkipModifySudoers ($order++, $section))
        ),
        'RetryHostagentPassword'        => $self->getParamRetryParam($order++, 'HostagentPassword'),
	};

    $self->setSummaryOrder(['Target',
                            'SID',
                            'InstanceNumber',
                            'DbMode',
                            'DbIsolation',
                            'SystemUsage',
                            'HomeDir',
                            'Shell',
                            'UID',
                            'GID',
                            'BasePathDataVolumes',
                            'BasePathLogVolumes',
                            'HostName',
                            'AddHosts',
                            'InternalNetwork',
                            'ImportContent',
                            'InstallHostagent',
                            'RemoteExecution',
                            'RootUser',
                            'HostagentUserName',
                            'SkipModifySudoers',
                            'CheckOnly',
                           ]);
	return $self;
}


sub getParamCreateInitialContainer{
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'opt'               => 'create_initial_container',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 0,
        'str'               => 'Skip createSecureStore and initTopology ',
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'hidden'            => 1,
        'skip'              => 0
    };
}


sub getProductName{
	return $gProductNameEngine;
}

sub getProductVersion {
    my ($self) = @_;
    if (!defined $self->{kit}){
        return $self->SUPER::getProductVersion ();
    }
    return $self->{kit}->GetVersion();
}

sub getShortProductName{
	return $gShortProductNameEngine;
}


sub checkCreateInitialContainer{
    my ($self, $createInitialContainer) = @_;
    if ($createInitialContainer){
        return $self->setValue('NoStart', 1);
    }
    return 1;
}


sub checkNoStart{
	my ($self, $value) = @_;
	if ($value =~ /$bool_true_pattern/i){
        if (defined $self->getAddHostsParser()) {
            $self->PushError ("Parameter '" . $self->getOpt('NoStart')
                    . "' cannot be applied together with additional hosts");
            return 0;
        }
		$self->{params}->{SQLSysPasswd}->{skip} = 1;
		$self->setValue('ImportContent', 'false');
	}
	return 1;
}


#-------------------------------------------------------------------------------
# Checks the root user name for additional hosts.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkRootUser{
    my ($self, $value) = @_;

    if (defined $self->getRemoteHosts()
        && !$self->tryRemoteKeyAuthorization($value)) {

        return undef;
    }
    return 1;
}

sub validatePersistency{
	my ($self, $pers) = @_;
	if (!defined $pers){
		$pers = $self->pers_load();
	}
	
	if (!defined $pers){
		return undef;
	}
	if ($pers->{_kitversion} ne $self->{_kitversion}){
		$self->AddError ("Cannot resume broken installation of '$self->{current_sid}': wrong version ($self->{_kitversion})");
		$self->PushError ("$gProductNameEngine installation kit '$pers->{_kitversion}' required");
		
		return undef;
	}
	if (!defined $self->SUPER::validatePersistency ($pers)){
		return undef;
	}
	
    my $pendingMsg = "Resuming broken installation at step '"
                   . $self->pers_getstepname($pers->{step}) . "'.";

    $self->AddProgressMessage ($pendingMsg);
	return 1;
}


sub getHWCheckScript{
    return $_[0]->{_hwCheckScript};
}


sub runHWCheckScript{
    my ($self) = @_;
    return 1; # Skip hardware check since SP12.
              # Complete code removal is planned for the next SP due to
              # mount && chown for storage connector api.
    my $result = 1;
    my $checkscript = $self->getHWCheckScript();
    if (defined $checkscript){
        require SDB::Install::PythonExecutor;
        my $python = new SDB::Install::PythonExecutor ();
        if (!$python->hasPythonExe()){
            $self->PushError ("Cannot perform HANA platform check: python interpreter not found");
        }
        else{
            my @args;
            if (defined $self->{_basepath_datavolumes}){
                push @args, '-datapath='.$self->{_basepath_datavolumes};
            }
            if (defined $self->{_basepath_logvolumes}){
                push @args, '-logpath='.$self->{_basepath_logvolumes};
            }
            if (defined $self->getValue ('MachineUtilization')){
                push @args, '-machine_utilization='.$self->getValue ('MachineUtilization');
            }
            my $msg = $self->AddMessage ("Performing HANA platform check...");
            $python->setScript ($checkscript);
            $python->runScript (\@args);
            $self->AddSubMsgLst ($msg, $python);
        }
    }
    return $result;
}


sub checkBasePathDataVolumes{
    my ($self, $value) = @_;
    if (!$value || $self->getDefault ('BasePathDataVolumes') eq $value){
        return 1;
    }
    if (defined $self->getValue ('StorageConfigDir')){
        return 1;
    }
    if (!-e $value){
        $self->PushError ("Path '$value' does not exist");
        return 0;
    }

    if (!-d _){
        $self->PushError ("Path '$value' is no directory");
        return 0;
    }
    # checking r-x of patent directories
    return $self->check_path_access ($value,0);
}

sub checkBasePathLogVolumes{
    my ($self, $value) = @_;

    if (!$value || $self->getDefault ('BasePathLogVolumes') eq $value){
        return 1;
    }
    if (defined $self->getValue ('StorageConfigDir')){
        return 1;
    }
    if (!-e $value){
        $self->PushError ("Path '$value' does not exist");
        return 0;
    }

    if (!-d _){
        $self->PushError ("Path '$value' is no directory");
        return 0;
    }

    # checking r-x of patent directories
    return $self->check_path_access ($value,0);
}


sub checkMinMem{
    my ($self) = @_;
    my $mem = $self->getPhysicalMem();
	if (defined $mem) {
		if ($mem < $gMinMemGB * 1024 * 1024 * 1024) {
			my $errorText = "Physical memory on this host is " . sprintf("%.3f", $mem / (1024 * 1024 * 1024)) . " GB. Minimum required: 24 GB.";
			my $errorMsg = $self->getMsgLst()->addError($errorText);
			if ($self->getIgnore('check_min_mem')){
				$self->getMsgLst()->addMessage("Ignoring error due to command line switch '--ignore=check_min_mem'");
			}
			else{
				$self->getErrMsgLst()->appendMsg($errorMsg);
				$self->appendErrorMessage('Override memory check with command line option --ignore=check_min_mem');
                return undef;
			}
		}
	} else {
		$self->AddWarning('Cannot check physical memory: GetMachInfo() failed');
	}
	return 1;
}


sub checkSID{
    my ($self, $value) = @_;
    my $rc = $self->SUPER::checkSID ($value);

    if (!$rc){
        return $rc;
    }

    if (exists $self->{params}->{UID}
        && !defined $self->getDefault('UID')
        && !defined $self->getBatchValue('UID')) {

        $self->setDefault('UID', $self->getFreeUserID());
    }

    #
    # checking whether there is an unregistered or to another host assigned
    # HANA system
    #

    my $target = $self->getValue ('Target');
    if (defined $target && !$self->pers_exists ()){
        my $sapSys = new SDB::Install::SAPSystem ();
        $sapSys->initWithGlobDir($value, $target);
        if ($sapSys->ErrorState ()){
            return 1;
        }
        if($sapSys->hasNewDB){
            my $msglst = new SDB::Install::MsgLst ();
            my $hdbInstance = $sapSys->getNewDBInstances()->[0];
            $sapSys->asString ($msglst);
            if (defined $hdbInstance->get_host()){
                # assigned to own host
                $self->setErrorMessage ("$self->{params}->{SID}->{str} '$value' is already"
                                . " in use by an un-registered $gProductNameEngine System at $target on this host:", $msglst);
            }
            else{
                # not assigned to own host
                $self->setErrorMessage ("$self->{params}->{SID}->{str} '$value' is already"
                                . " in use by a $gProductNameEngine System at $target on another host:", $msglst);
            }
            return 0;
        }
    }
    return 1;
}

sub checkRepository{
    my ($self, $value) = @_;
    foreach my $repositoryParam ('EnableXSEngine', 'ImportContent'){
        $self->setDefault ($repositoryParam, ($value =~ /$bool_true_pattern/i) ? 1 : 0);
    }
    return 1;
}

sub checkSystemRequirementsAfterConfig{
    my ($self, $sysinfo, $manifest) = @_;
    if ($isWin){
        return 1;
    }
    my $datapath = $self->getValue ('BasePathDataVolumes');
    if (!$datapath){
        $datapath = $self->getDefault ('BasePathDataVolumes');
    }

    my $logpath = $self->getValue ('BasePathLogVolumes');
    if (!$logpath){
        $logpath = $self->getDefault ('BasePathLogVolumes');
    }

    my $mounter;

    my $storage_ini = $self->getValue ('StorageConfigDir');
    my ($uid,$gid);
    if (defined $storage_ini){
        require SDB::Install::HdbMounter;
        $mounter = new SDB::Install::HdbMounter ($self->{kitDir});
        $mounter->setParams ($storage_ini,
            $datapath . $path_separator . $volumeSubDirPerHost,
            $logpath . $path_separator . $volumeSubDirPerHost,
            $self->getValue ('SID'), 1);
        my $msg = $self->AddMessage ("Mounting storage devices...");
        $self->{mounter} = $mounter;
        if (!$mounter->mount()){
            $self->AddError ("Cannot mount storage devices", $mounter);
            $self->AddSubMsgLst($msg, $mounter);
            return undef;
        }
        $self->AddSubMsgLst($msg, $mounter);
        $uid = $self->getValue ('UID');
        if (defined $uid){
            $uid = int ($uid);
        }
        $gid = $self->getValue ('GID');
        if (defined $gid){
            $gid = int ($gid);
        }
    }
    if ($datapath){
        if (!-e $datapath){
            my $path = '';
            my $tmp;
            foreach my $node (split ($path_separator, $datapath)){
                if (!$node){
                    next;
                }
                $tmp = $path . $path_separator . $node;
                if (!-e $tmp){
                    if (!$path){
                        $path = $path_separator;
                    }
                    last;
                }
                $path = $tmp;
            }
            $datapath = $path;
        }
        elsif (-e $datapath . $path_separator . $volumeSubDirPerHost){
            $datapath .= $path_separator . $volumeSubDirPerHost;
            if (defined $uid && defined $gid){
                if (chown ($uid,$gid, $datapath)){
                    $self->getMsgLst()->addMessage ("Owner of $datapath changed (uid = $uid, gid = $gid)");
                }
                else{
                    $self->getMsgLst()->addWarning ("Cannot change owner of $datapath: $!");
                }
            }
        }
    }

    if ($logpath){
        if (!-e $logpath){
            my $path = '';
            my $tmp;
            foreach my $node (split ($path_separator, $logpath)){
                if (!$node){
                    next;
                }
                $tmp = $path . $path_separator . $node;
                if (!-e $tmp){
                    if (!$path){
                        $path = $path_separator;
                    }
                    last;
                }
                $path = $tmp;
            }
            $logpath = $path;
        }
        elsif (-e $logpath . $path_separator . $volumeSubDirPerHost){
            $logpath .= $path_separator . $volumeSubDirPerHost;
            if (defined $uid && defined $gid){
                if (chown ($uid,$gid, $logpath)){
                    $self->getMsgLst()->addMessage ("Owner of $logpath changed (uid = $uid, gid = $gid)");
                }
                else{
                    $self->getMsgLst()->addWarning ("Cannot change owner of $logpath: $!");
                }
            }
        }
    }

    $self->{_basepath_datavolumes} = $datapath;
    $self->{_basepath_logvolumes} = $logpath;
    my $hardwareCheckRc = $self->runHWCheckScript ();
    if (!defined $hardwareCheckRc && $self->getIgnore ('check_diskspace')){
        $self->getMsgLst ()->addMessage ("Ignoring error due to command line switch --ignore=check_diskspace");
        $self->resetError ();
        return 1;
    }
    return $hardwareCheckRc;
}

sub checkSystemRequirements{
	my ($self, $sysinfo, $manifest) = @_;
    $self->{sysinfo} = $sysinfo;
	my $result = 1;
	my $required_instruction_set = 'sse4.2';
	my $required_instruction_set_pattern = quotemeta ($required_instruction_set);

	if (defined $sysinfo->{processor_features} &&
		($sysinfo->{processor_features} !~ /$required_instruction_set_pattern/)){

		$self->appendErrorMessage ("CPU is not supported. Instruction set '$required_instruction_set' is missing.");
        if ($self->getIgnore ('check_platform')){
            $self->resetError ();
            $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
        }
        else{
            $result = undef;
        }
	}

    if ($sysinfo->isSles()) {
		$sysinfo->setMsgLstContext ($self->getMsgLstContext ());
        if (!$sysinfo->isSles12SpSupported()){
            if ($self->getIgnore ('check_platform')){
                $self->resetError ();
                $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
            }
            else{
                $result = undef;
            }
        }
    }
    elsif($sysinfo->isRedHat()) {
        $sysinfo->setMsgLstContext ($self->getMsgLstContext ());
        if (!$sysinfo->isRhel7SpSupported()){
            if ($self->getIgnore ('check_platform')){
                $self->resetError ();
                $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
            }
            else{
                $result = undef;
            }
        }
    }
    if (!defined $sysinfo->_checkGccRuntime($self, $manifest)){
        $result = undef;
    }

    if (defined $self->getHWCheckScript() && !defined $self->runHWCheckScript ()){
        $result = undef;
    }
	if (!$isWin){
		return $result;
	}
	
	if (!defined $sysinfo->{version}){
		$self->AddWarning ("SysInfo.version is unknown: skipping platform check");
		return $result;
	}
		
	my ($major,$minor) = ($sysinfo->{version} =~ /NT\s+(\d+)\.(\d+)/);
	
	if (!defined $major){
		$self->AddWarning ("Cannot parse version string '$sysinfo->{version}': skipping platform check");
		return $result;
	}			
	
	if (int ($major) < 6 ||
		int ($major) == 6 && int ($minor) < 1){
		$self->PushError ("At least MS Windows 7 / Windows Server 2008 R2 is required");
        if ($self->getIgnore ('check_platform')){
            $self->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
        }
        else{
            $result = undef;
        }
	}
	return $result;
}


sub getIgnoreValues{
	return [qw (check_diskspace check_min_mem)];
}

sub getHiddenIgnoreValues{
	return [qw (check_hardware check_platform scan_password_database check_hostname check_pmem)];
}

sub getTimeoutValues{
    return [qw (start_instance start_service)];
}

#-------------------------------------------------------------------------------
# Collect the SAP HANA systems and fills the specified array/hash references
# and returns an array of installed SIDs.

sub getNewDBInstallations {
    my ($self,
        $installedSystems,    # hash  reference to insert SAPSystem classes for update
        $pendingInstallations # array reference to insert SAPSystem classes
       ) = @_;

    my $savedSID    = $self->{current_sid};
    my $savedTarget = $self->{params}->{Target}->{value};

    my @installedSIDs = ();
    my $allSystems    = CollectSAPSystems();

    foreach my $currSID (sort keys %{$allSystems}) {

        my $currSystem = $allSystems->{$currSID};
        $self->{current_sid}               = $currSID;
        $self->{params}->{Target}->{value} = $currSystem->get_target();

        if ($self->pers_exists()) {
            push(@$pendingInstallations, $currSystem);
        }
        elsif ($currSystem->hasNewDB()) {
            push(@installedSIDs, $currSID);
            $installedSystems->{$currSID} = $currSystem;
        }
    }

    $self->{current_sid}               = $savedSID;
    $self->{params}->{Target}->{value} = $savedTarget;

    return @installedSIDs
}

#-------------------------------------------------------------------------------
# Writes a summary of the parameters onto the console and into the log file.
#
# Hides the parameter RemoteExecution in case of local work only.

sub displayParameterSummary {

    my ($self) = @_;

    if (!defined $self->getRemoteHosts()) {
        $self->{params}->{RemoteExecution}->{hidden} = 1; # hide summary output
    }

    $self->SUPER::displayParameterSummary();
}

sub shouldWarnIfCalledStandalone{
    my ($self) = @_;
    return $self->_calledStandaloneCriterion();
}

sub getExeName{
    return "hdbinst";
}

sub getResidentHdblcmPath{
    my ($self) = @_;
    return $self->_constructResidentHdblcmPath();
}

1;
