package SDB::Install::Configuration::NewServerConfig;

use strict;

use SAPDB::Install::Hostname;

use SAPDB::Install::System::Win32::API;
use SDB::Install::Configuration::AnyMultiHostConfig;
use SDB::Install::Configuration qw ($bool_false_pattern $bool_true_pattern);
use SDB::Install::Globals;
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::System;
use SDB::Install::System::Mounts;
use SDB::Install::SysVars;
use SDB::Install::SAPSystem qw (ExistsInstanceNumber
                $STEP_CREATE_INSTANCE $STEP_EXTRACT @STEP_NAMES);
use File::Basename qw(dirname);
use File::Spec;
use File::stat;
use IO::Dir;

our @ISA = qw (SDB::Install::Configuration::AnyMultiHostConfig Exporter);
our @EXPORT_OK = qw(checkShell checkHostName);

our $existing_drives = [SAPDB::Install::System::Win32::API::GetHDDrives()];
our $volumeSubDirPerHost = 'mnt00001';
our $statusFileExtension = 'hdbinstall';

sub getParamDrive{
    my ($self, $order, $section) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'drive',
        'type'                      => 'string',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => $gDriveWin,
        'str'                       => 'Windows Drive Letter',
        'set_interactive'           => 1,
        'init_with_default'         => 1,
        'valid_values'              => $existing_drives,
        'f_get_console_description' => sub {''}
    };
}

sub getParamTarget{
    my ($self, $order, $section) = @_;
    my $paramTarget = $self->SUPER::getParamTarget ($order, $section);
    $paramTarget->{default} = $gSapmntDirUx;
    $paramTarget->{may_be_interactive} = 1;
    $paramTarget->{persStep} = $STEP_EXTRACT;
    return $paramTarget;
}

sub getParamCheckMnt{
    my ($self, $order, $section) = @_;
    return {
        'order' => $order,
        'opt' => 'checkmnt',
        'type' => 'path',
        'section' => $section,
        'value' => undef,
        'default' => undef,
        'str' => 'Non-standard Shared File System',
        'desc' => "Specifies a shared path (e.g. '<installation_path>/<SID>'),"
                . ' which can be accessed by all hosts to install a multiple host system'
                . ' if the installation path is not shared',
        'opt_arg' => '<path>',
        'init_with_default' => 0,
        'set_interactive' => 0,
        'hidden' => 0
    };
}

sub getParamInstanceNumber{
    my ($self, $order, $section) = @_;
    my $paramInstanceNumber = $self->SUPER::getParamInstanceNumber ($order, $section, 'Instance Number', 0);
    
    $paramInstanceNumber->{f_get_console_description}   = sub {''};
    $paramInstanceNumber->{mandatory}                   = 1;
    $paramInstanceNumber->{persStep}                    = $STEP_CREATE_INSTANCE;
    $paramInstanceNumber->{leaveEmptyInConfigDump}      = 1;
    return $paramInstanceNumber;
}

#-------------------------------------------------------------------------------
# Returns the hash entries of the parameter 'root_user'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamRootUser{
    my ($self, $order, $section) = @_;
    my $paramRootUser = $self->SUPER::getParamRootUser ($order, $section);
    $paramRootUser->{init_with_default} = 1;
    $paramRootUser->{set_interactive} = 1;
    $paramRootUser->{desc} = "Root user if adding hosts to $gProductNameSystem";
    return $paramRootUser;
}


#-------------------------------------------------------------------------------
# Returns a hash entries of the parameter 'root_password'.
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub getParamRootPassword{
    my ($self, $order, $section) = @_;
    my $paramRootPassword = $self->SUPER::getParamRootPassword ($order, $section);
    $paramRootPassword->{opt_arg} = '<password>';
    return $paramRootPassword;
}

sub getParamDomain{
    my ($self, $order, $section) = @_;
    my $param = $self->SUPER::getParamDomain($order, $section);
    $param->{persStep} = $STEP_EXTRACT;
    return $param;
}

sub getParamStorageConfigDir{
    my ($self, $order, $section) = @_;
    return {
    'order'                 => $order,
        'opt'               => 'storage_cfg',
        'type'              => 'path',
        'opt_arg'           => '<path>',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Directory containing a storage configuration',
        'init_with_default' => 0,
        'hidden'            => 0,
        'set_interactive'   => 0
    };
}


our @dbModeValidValues = qw (multidb);
our @dbModeUIValues = ('The system contains one system database and 1..n tenant databases');

sub getParamDbMode{
    my ($self, $order, $section, $constraint) = @_;
    return {
    'order'                         => $order,
        'opt'                       => 'db_mode',
        'type'                      => 'string',
        'valid_values'              => \@dbModeValidValues,
        'visible_alias_values'      => ['multiple_containers'],
        'ui_values'                 => \@dbModeUIValues,
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 'multiple_containers',
        'str'                       => 'Database Mode',
        'interactive_index_selection' => 1,
        #'console_omit_word_Enter'   => 1,
        'init_with_default'         => 1,
        'constraint'                => $constraint,
        'hidden'                    => 1,
        'set_interactive'           => 0
    };
}


our @dbIsolationValidValues = qw (low high);
our @dbIsolationUIValues = ('All databases are owned by the sidadm user', 'Each database has its own operating system user');

sub getParamDbIsolation{
    my ($self, $order, $section, $constraint) = @_;
    return {
    'order'                         => $order,
        'opt'                       => 'db_isolation',
        'type'                      => 'string',
        'valid_values'              => \@dbIsolationValidValues,
        'visible_alias_values'      => ['low', 'high'],
        'ui_values'                 => \@dbIsolationUIValues,
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 'low',
        'str'                       => 'Database Isolation',
        'interactive_index_selection' => 1,
        #'console_omit_word_Enter'   => 1,
        'init_with_default'         => 1,
        'constraint'                => $constraint,
        'hidden'                    => 0,
        'set_interactive'           => 0
    };
}


sub getParamBasePathDataVolumes{
    my ($self, $order, $section, $default) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'datapath',
        'type'                      => 'path',
        'opt_arg'                   => '<path>',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => $default,
        'str'                       => 'Location of Data Volumes',
        'init_with_default'         => 1,
        'set_interactive'           => 1,
        'footprint_str'             => 'data volumes',
        'persStep'                  => $STEP_CREATE_INSTANCE,
        'diskspace_check_subdir'    => $volumeSubDirPerHost,
        'required_disk_space'       => 0x80000000 # 2gb
    };
}


sub getParamBasePathLogVolumes{
    my ($self, $order, $section, $default) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'logpath',
        'type'                      => 'path',
        'opt_arg'                   => '<path>',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => $default,
        'str'                       => 'Location of Log Volumes',
        'init_with_default'         => 1,
        'set_interactive'           => 1,
        'footprint_str'             => 'log volumes',
        'persStep'                  => $STEP_CREATE_INSTANCE,
        'diskspace_check_subdir'    => $volumeSubDirPerHost,
        'required_disk_space'       => 0xc0000000 # 3gb
    };
}

sub getParamBasePathPMem{
    my ($self, $order, $section) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'pmempath',
        'type'                      => 'path',
        'opt_arg'                   => '<path>',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => '/hana/pmem/$SID',
        'str'                       => 'Location of Persistent Memory Volumes',
        'init_with_default'         => 1,
        'set_interactive'           => 1,
        'skip'                      => 1,
        'persStep'                  => $STEP_CREATE_INSTANCE
    };
}


sub getParamRestrictMaxMem{
    my ($self, $order, $section) = @_;
    return {
        'order'                     => $order,
        'opt'                       => 'restrict_max_mem',
        'type'                      => 'boolean',
        'section'                   => $section,
        'value'                     => undef,
        'default'                   => 0,
        'str'                       => 'Restrict maximum memory allocation?',
        'desc'                      => 'Restricts maximum memory allocation',
        'console_omit_word_Enter'   => 1,
        'init_with_default'         => 1,
        'set_interactive'           => 1,
        'leaveEmptyInConfigDump'    => 1
    };
}


sub getParamMaxMem {

    my ($self, $order, $section) = @_;

    return {
            'order'                     => $order,
            'opt'                       => 'max_mem',
            'type'                      => 'number',
            'section'                   => $section,
            'value'                     => undef,
            'minValue'                  => undef,
            'maxValue'                  => undef,
            'default'                   => 0,
            'str'                       => 'Maximum Memory Allocation in MB',
            'desc'                      => 'Maximum memory allocation in MB',
            'set_interactive'           => 1,
            'may_be_interactive'        => 1,
            'init_with_default'         => 1,
            'leaveEmptyInConfigDump'    => 1
    };
}

sub getParamCustomConfigDir{
    my ($self, $order, $section) = @_;
    return {
        'order'             => $order,
        'opt'               => 'custom_cfg',
        'type'              => 'path',
        'opt_arg'           => '<path>',
        'section'           => $section,
        'value'             => undef,
        'default'           => undef,
        'str'               => 'Directory containing custom configurations',
        'init_with_default' => 0,
        'hidden'            => 0,
        'set_interactive'   => 0
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--workergroup'.

sub getParamWorkerGroup {
    my ($self, $order, $section, $skip, $persStep) = @_;

    return {
        'order'             => $order++,
        'opt'               => 'workergroup',
        'type'              => 'string',
        'section'           => $section,
        'value'             => undef,
        'default'           => 'default',
        'str'               => 'Worker Group',
        'desc'              => 'Specifies the Worker goup',
        'init_with_default' => 1,
        'set_interactive'   => 1,
        'skip'              => $skip,
        'persStep'          => $persStep
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--repository'.

sub getParamRepository {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order++,
        'opt'               => 'repository',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 1,
        'str'               => 'Enable HANA repository',
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'skip'              => 0
    };
}

#-------------------------------------------------------------------------------
# Returns a hash containing the entries of the parameter '--use_pmem'.

sub getParamUsePMem {
    my ($self, $order, $section) = @_;

    return {
        'order'             => $order,
        'opt'               => 'use_pmem',
        'type'              => 'boolean',
        'section'           => $section,
        'value'             => undef,
        'default'           => 0,
        'str'               => 'Enable usage of persistent memory',
        'init_with_default' => 1,
        'set_interactive'   => 0,
        'skip'              => 0,
        'persStep'          => $STEP_CREATE_INSTANCE
    };
}


sub InitDefaultsForDistributedSystem{
    my ($self) = @_;

    if (defined $self->{params}->{InstanceNumber}){
        my @vacant_numbers;
        $self->setDefault('InstanceNumber', undef);
        foreach my $instance ('00','01','02','03','04','05','06','07', '08', '09', 10..97){
            if (!defined SDB::Install::SAPSystem::ExistsInstanceNumber ($instance)){
                if (!defined $self->{remoteInstances}->{$instance}){
                    if (!defined $self->getDefault('InstanceNumber')){
                        $self->setDefault('InstanceNumber', $instance);
                    }
                    push @vacant_numbers, $instance;
                }
            }
        }
        $self->{params}->{InstanceNumber}->{valid_values} = \@vacant_numbers;
    }

    return $self->tryInitGroupAndUserID();
}


#-------------------------------------------------------------------------------
# Checks the option '--addhosts' and creates the hash '_remoteHosts'
#
# A comment at the beginning of AnyConfig shows the correct order of this
# parameter according to the depending parameters.

sub checkAddHosts{

    my ($self, $value) = @_;

    my $rc = $self->handleAddOrRemoveHosts($value, 'AddHosts');

    if ($rc) {
        my $remoteHosts = $self->getAdditionalRemoteHosts();

        if (defined $remoteHosts) {
            $self->{_remoteHosts} = $remoteHosts;
            $self->setSkip('InternalNetwork', 0);
        }
    }
    return $rc;
}

sub getRemoteInstallerExecutable{
    my ($self) = @_;
    return $self->getRemoteInstallerExecutableFromKit ();
}

sub checkHostName{
    my ($self, $value, $isRemoteHost) = @_;
    if (!$self->verifyNewHostName ($value, $isRemoteHost)){
        if ($self->getIgnore ('check_hostname')){
            $self->getMsgLst()->addMessage ("Ignoring error due to command line switch '--ignore=check_hostname'");
            $self->resetError ();
        }
        else{
            return 0;
        }
    }
    return 1;
}

sub setHostName{
	my ($self,$value) = @_;
	if (!$self->checkHostName ($value)){
		return 0;
	}
	$self->{params}->{HostName}->{value} = lc ($value);
	return 1;
}

sub checkCheckMnt{
    my ($self, $value) = @_;
    if (!-e $value){
        $self->PushError ("Check mount '$value' is not accessible: $!");
        return 0;
    }
    if (!-d _){
        $self->PushError ("Check mount '$value' is not a directory");
        return 0;
    }
    if (!-w _){
        $self->PushError ("Check mount '$value' is not writable");
        return 0;
    }
    return 1;
}

sub checkRestrictMaxMem{
	my ($self, $value) = @_;
	if ($value =~ /$bool_false_pattern/i){
		$self->setValue('MaxMem', 0);
		$self->{params}->{MaxMem}->{skip} = 1;
	}
	return 1;
}

sub checkMaxMem{
	my ($self, $value) = @_;
	if ($value == 0) {
		return 1;
	}
	my $physicalMemoryMB = int($self->getPhysicalMem()/(1024*1024));
	my $gMinAllcationLimitMB = $gMinAllcationLimitGB * 1024;
	if ($value < $gMinAllcationLimitMB) {
		my $errorText = "Physical memory allocated to system (--max_mem=$value) is too low. Minimum is ".
						"$gMinAllcationLimitMB ($gMinAllcationLimitGB GB).";
		my $errMsg = $self->getMsgLst()->addError($errorText);
		if ($self->getIgnore ('check_min_mem')) {
			$self->getMsgLst()->addMessage ("Ignoring error due to command line switch '--ignore=check_min_mem'");
			return 1;
		} else {
			$self->getErrMsgLst()->appendMsg($errMsg);
			return 0;
		}
	} elsif ($value > $physicalMemoryMB) {
		$self->PushError ("Cannot allocate more than the available physical memory.");
		return 0;
	}
	return 1;
}

sub checkDatabaseUserPassword {
    my ($self, $user, $password, $paramId) = @_;
    if ($password eq 'manager') {
        return 1;
    }

    my $properties = $paramId ? $self->{params}->{$paramId} : $self->{params}->{SQLSysPasswd};
    my $msglst = new SDB::Install::MsgLst ();
    my $rc = $self->complyDefaultSqlPasswordPolicy ($password, $msglst, $properties->{str});
    if (!$rc){
        $self->PushError ("$properties->{str} is invalid", $msglst);
    }
    return $rc;
}

sub checkSID{
    my ($self, $value) = @_;

    my $properties = $self->{params}->{SID};

    my $rc = 1;

    my $errlist = new SDB::Install::MsgLst ();

    if ($self->isForbiddenSID ($value, $errlist)) {
        $rc = 0;
    }

    if (length ($value) != 3){
        $errlist->addError ('The value has to be composed of three characters.');
        $rc = 0;
    }

    if ($value !~ /^[A-Z]/){
        $errlist->addError ('The first character has to be an upper case letter.');
        $rc = 0;
    }

    if ($value !~ /^.[A-Z0-9]/){
        $errlist->addError ('The second character has to be an upper case letter or a decimal digit.');
        $rc = 0;
    }

    if ($value !~ /^..[A-Z0-9]/){
        $errlist->addError ('The third character has to be an upper case letter or a decimal digit.');
        $rc = 0;
    }

    if (!$rc){
        $self->PushError ("Invalid $properties->{str}", $errlist);
        return 0;
    }

    $self->{current_sid} = $value;

    my $systems = $self->getCollectSAPSystems();

    if (defined $self->{remoteSystems} && defined $self->{remoteSystems}->{$value}){
        $self->PushError ("$properties->{str} \"$value\" " .
                    "is already in use on host(s) " . join (',', @{$self->{remoteSystems}->{$value}}));
        $rc = 0;
    }

    if (exists $systems->{$value}){
        if ( $systems->{$value}->hasTrex ()){
            $self->PushError ("$properties->{str} \"$value\" " .
                    "is already in use by a Trex instance");
            $self->PushError ("Using a Trex instance and a $gProductNameEngine instance in one system is not allowed.");
            return 0;
        }

        my $rc = $self->pers_exists ();
        if ($rc){
            if (!defined $self->validatePersistency){
                return undef;
            }
        }
        else{
            # checking for deprecated status file in /usr/sap
            my $pers = $self->validateOldLocalPersistencyFile ();
            if (defined $pers){
                if (!defined $self->validatePersistency ($pers)){
                    return undef;
                }
            }
            elsif (defined $systems->{$value} && $systems->{$value}->hasNewDB()){
                $self->PushError ("$properties->{str} '$value' is already"
                                . " in use by a $gProductNameEngine instance!");
                return 0;
            }
        }
    }

    my $user = new SDB::Install::NewDBUser ($value);
	return 0 if ( ! $self->validateSidBasedOnRemoteSidadms($value) );

    $self->{username} = $user->getSidAdmName();

    my $warnings_text = "";
    my $passwordProperties = $self->{params}->{Password};
    $self->changePasswordStr($self->{username});
    if ($isWin) {
        if ($user->exists()){
            $self->{params}->{Domain}->{value} = $user->getSidAdm()->{bsname}->{first};
            if (!$self->pers_exists()) {
                $warnings_text = "The defined user '$self->{username}' already exists on the system. Neither the password, nor any other attribute of the user will be changed.\nVerify that the user is correct.";
            }
            $passwordProperties->{type} = 'passwd';
        }
        else{
            $passwordProperties->{type} = 'initial_passwd';
        }
        $passwordProperties->{skip} = 0;
    }
    else {
        if ($user->exists()){
            if (!$passwordProperties->{skip}){
                $self->{save_user_settings} = {
                    'Shell' => $self->{params}->{Shell}->{value},
                    'HomeDir' => $self->{params}->{HomeDir}->{value},
                    'UID' => $self->{params}->{UID}->{value}
                };
            }
            $self->{params}->{Shell}->{value} = $user->getSidAdm()->{shell};
            $self->{params}->{HomeDir}->{value} = $user->getSidAdm()->{home};
            $self->{params}->{UID}->{value} = $user->uidUx();
            if (defined $self->{_remoteHosts}){
                $passwordProperties->{skip} = 0;
            }
            else{
                $passwordProperties->{skip} = 1;
            }
            if (!$self->pers_exists()) {
                $warnings_text = "The defined user '$self->{username}' already exists on the system. Neither the password, nor any other attribute of the user will be changed.\nVerify that the user is correct.";
            }
        }
        else {
            if (defined $self->{save_user_settings} && $passwordProperties->{skip}){
                $self->{params}->{Shell}->{value} = $self->{save_user_settings}->{Shell};
                $self->{params}->{HomeDir}->{value} = $self->{save_user_settings}->{HomeDir};
                $self->{params}->{UID}->{value} = $self->{save_user_settings}->{UID};
            }
            $passwordProperties->{type} = 'initial_passwd';
            $passwordProperties->{skip} = 0;
        }
    }
# Workaround for bug '81911' -> remove any not needed warnings that might have been previously added
# TODO: migrate to parameter warnings in SPS11, because usage of general warnings prooves to be problematic
    if(defined($self->{warnings})){
        $self->{warnings} = [ grep { $_ !~ /^The defined user/} @{$self->{warnings}} ];
    }
    if ($warnings_text) {
        #
        # TODO: remove when configuration is changed to "check values only once"
        #
        push @{$self->{warnings}}, $warnings_text;
        if (!defined $self->{user_warning} || !exists $self->{user_warning}->{$value}){
            if (!defined $self->{user_warning}){
                $self->{user_warning} = {};	
            }
            $self->{user_warning}->{$value} = 1;
            foreach my $line(split("\n",$warnings_text)) {
                $self->AddProgressMessage($line);
            }
        }
    }
    $self->clearInstallationPath($value);
    $self->enableHostagentPasswordIfRequired();
    return 1;
}

sub setTarget{
	my ($self, $value) = @_;
	my $param = $self->{params}->{Target};
    trim_path (\$value);
    if (!$self->checkTarget ($value)){
		return 0;
	}
	$param->{value} = $value;
	return 1;
}


sub check_path_access{
    my ($self,$path, $want,$uid,$gid) = @_;
    if ($isWin){
        return 1; # skip on windows
    }
    $gid = getgrnam ($gSapsysGroupName) if (!defined $gid);
    if (!defined $uid && defined $self->{params}->{SID}->{value}){
        my $user = new SDB::Install::NewDBUser ($self->{params}->{SID}->{value});
        if ($user->exists()){
            $uid = $user->uid ();
        }
    }
    my $errlst = new SDB::Install::MsgLst();
    if (!access_check ($path, $want, $uid, $gid, $errlst)){
        if (is_nfsidmap_required ($path)){
            my $msglst = $self->getMsgLst();
            $msglst->addError ("Directory '$path' is not accessible", $errlst);
            $msglst->addMessage ("Ignoring error due to nfs4 setup.");
        }
        else{
            $self->appendErrorMessage ("Directory '$path' is not accessible", $errlst);
            return 0;
        }
    }
    return 1;
}

sub checkBasePathAgainstSapmnt {
# If the installation path matches Data and/or Log basepaths, setPermissions (TrexInstance)
# is being called recuresively on it and many files and dirs get wrong ownership and permissions.
    my ($self, $basePath) = @_;
    my $sid = $self->getSID();
    my $target = $self->getValue('Target');
    my $sidDir = File::Spec->catfile($target, $sid);
    if ($basePath eq $target || $basePath eq $sidDir) {
        return 0;
    }
    return 1;
}

sub checkBasePath{
    my ($self, $value) = @_;

    if (!$self->checkBasePathAgainstSapmnt($value)) {
        $self->appendErrorMessage("Data and Log paths can't match the <sapmnt> and <sapmnt>/<SID> directories.");
        return undef;
    }

    if (defined $self->getValue ('StorageConfigDir')){
        # skip path check, when storage connector is used
        return 1;
    }

    if (defined $self->getAddHostsParser() && defined $self->{_remoteHosts} && $self->{_remoteHosts}->authOk() == 0) {
        my $cfg;


        my $hanaHosts = $self->getAddHostsParser ()->getHanaHosts ();

        if (!defined $hanaHosts || !@$hanaHosts){
            # no further hana hosts found
            return 1;
        }

        if (! -e $value){

            #
            # create the directory locally just for the remote host check
            # it will succeed if the volume path is shared
            #

            $cfg = {'mode' => 0750, 'firstCreated' => undef};
            if (!defined makedir ($value, $cfg)){
                $self->setErrorMessage ("Error preparing the remote host volume path check", $cfg);
                return undef;
            }
        }

        my $rc =  $self->isDirectoryExistsOnRemoteHosts($value, $hanaHosts);

        #
        # remove the created directory after the check
        # check result doesn't matter
        #
        if (defined $cfg && defined $cfg->{firstCreated}){
            removeEmptyDirs ($cfg->{firstCreated});
        }
        return $rc;
    }
    return 1;
}

sub checkBasePathDataVolumes;
*checkBasePathDataVolumes = \&checkBasePath;

sub checkBasePathLogVolumes;
*checkBasePathLogVolumes = \&checkBasePath;




sub _checkDaxFileSystem{
    my ($self, $path, $errorMessage) = @_;
    my $stat = File::stat::stat($path);
    if (!defined $stat){
        $self->appendErrorMessage("$errorMessage: Cannot access '$path': $!");
        return 0;
    }
    if (!-d $stat){
        $self->appendErrorMessage("$errorMessage: '$path' is not a directory");
        return 0;
    }
    my $nvmMounts = $self->{_nvmMounts};
    my ($mountPoint,$rc,$error) = getMountPoint ($path);
    if (!grep {$_->{mnt_dir} eq $mountPoint} @$nvmMounts){
        my $errMsg = $self->getMsgLst()->addError("$errorMessage: Filesystem '$mountPoint' of '$path' is not a dax filesystem");
        if (!$self->getIgnore('check_pmem')){
            $self->getErrMsgLst()->appendMsg ($errMsg);
            return 0;
        }
        $self->getMsgLst()->addMessage ('Ignoring error due to --ignore=check_pmem');
    }
    return 1;
}


sub checkBasePathPMem{
    my ($self,$basePathPMem) = @_;
    my $dir = new IO::Dir();
    if (!$dir->open($basePathPMem)){
        $self->appendErrorMessage("Cannot open directory '$basePathPMem': $!");
        return undef;
    }
    my $param = $self->{params}->{'BasePathPMem'};
    my $errorMessage = "Error checking '$param->{str}'";
    my $rc = 1;
    my @pmemLst;
    while (my $ent = $dir->read()){
        if ($ent eq '.' || $ent eq '..'){
            next;
        }
        my $path = $basePathPMem . $path_separator . $ent;
        if ($self->_checkDaxFileSystem($path, $errorMessage)){
            push @pmemLst, $ent;
            next;
        }
        $rc = 0;
    }
    if (@pmemLst){
        $self->{_pmemLst} = \@pmemLst;
    }
    else{
        $self->appendErrorMessage("$errorMessage: No pmem filesystem found in '$basePathPMem'");
        $rc = 0;
    }
    return $rc;
}

sub getPMemLst{
    return $_[0]->{_pmemLst};
}


sub checkDbMode{
    my ($self, $value) = @_;
    my $isSingleDb = $value eq 'singledb';
    $self->setSkip ('DbIsolation', $isSingleDb);
    $self->setSkip ('CreateInitialTenant', $isSingleDb);
    if (!$isSingleDb){
        if(defined $self->{params}->{SQLSysPasswd}){
            $self->{params}->{SQLSysPasswd}->{str} =~ s/(?<!System\s)Database/System Database/g;
        }
        if(defined $self->{params}->{SystemUser}){
            $self->{params}->{SystemUser}->{str} =~ s/(?<!System\s)Database/System Database/g;
        }
    }
    return 1;
}

sub checkDbIsolation{
    my ($self, $value) = @_;
    if ($value eq 'high'){
        my $batchValueCreateInitialTenant = $self->getBatchValue ('CreateInitialTenant');
        if (defined $batchValueCreateInitialTenant){
            $batchValueCreateInitialTenant = $self->assignBoolValue ('CreateInitialTenant', $batchValueCreateInitialTenant);
            if ($batchValueCreateInitialTenant){
                $self->appendErrorMessage ("'--create_initial_tenant' is not supported in high isolation mode!");
                return undef;
            }
        }
        $self->setSkip ('CreateInitialTenant', 1);
    }
    return 1;
}

sub _checkNVMMounts{
    my ($self, $errorMessage) = @_;
    my $mounts = new SDB::Install::System::Mounts();
    my $nvmMounts = $mounts->getNVMMounts();
    if (!defined  $nvmMounts){
        my $errMsg = $self->getMsgLst()->addError("$errorMessage: Cannot detect pmem filesystems", $mounts->getErrMsgLst());
        if (!$self->getIgnore('check_pmem')){
            $self->getErrMsgLst()->appendMsg ($errMsg);
            return undef;
        }
        $self->getMsgLst()->addMessage ('Ignoring error due to --ignore=check_pmem');
    }
    elsif (@$nvmMounts == 0){
        my $errMsg = $self->getMsgLst()->addError("$errorMessage: No pmem filesystem found");
        if (!$self->getIgnore('check_pmem')){
            $self->getErrMsgLst()->appendMsg ($errMsg);
            return undef;
        }
        $self->getMsgLst()->addMessage ('Ignoring error due to --ignore=check_pmem');
    }
    $self->{_nvmMounts} = $nvmMounts;
    return 1;
}



sub checkUsePMem{
    my ($self, $valueUsePMem) = @_;
    if ($valueUsePMem){
        my $param = $self->{params}->{'UsePMem'};
        my $errorMessage = "Error checking '$param->{str}'";
        if (!$self->_checkNVMMounts($errorMessage)){
            return 0;
        }
    }
    $self->setSkip('BasePathPMem', !$valueUsePMem);
    return 1;
}

sub isDirectoryExistsOnRemoteHosts{
    my ($self, $path, $hosts) = @_;

    if ($self->{_remoteHosts}->existsFile($path, 1, undef, $hosts)){
	 $self->PushError ("Directory '$path' is not accessible on all hosts", $self->{_remoteHosts});
	 return 0;
    }
    
    return 1;
}

sub trim_path_win{
    ${$_[0]} =~ s/([^:])\\+$/$1/;
}

sub trim_path_ux{
    if (${$_[0]} eq '/'){
        return;
    }
    ${$_[0]} =~ s/\/\/+/\//g;
    ${$_[0]} =~ s/\/$//;
}

sub trim_path;
*trim_path = $isWin ? \&trim_path_win : \&trim_path_ux;


sub setBasePathDataVolumes{
    my ($self, $value) = @_;
    my $param = $self->{params}->{BasePathDataVolumes};
    trim_path (\$value);
    if (!$self->checkBasePathDataVolumes ($value)){
        return 0;
    }
    $param->{value} = $value;
    return 1;
}

sub setBasePathLogVolumes{
    my ($self, $value) = @_;
    my $param = $self->{params}->{BasePathLogVolumes};
    trim_path (\$value);
    if (!$self->checkBasePathLogVolumes ($value)){
        return 0;
    }
    $param->{value} = $value;
    return 1;
}


sub checkInstanceNumber{
    my ($self, $value) = @_;
    if (!$self->SUPER::checkInstanceNumber ($value)){
        return 0;
    }

    my $ownSid = $self->getBatchValue ('SID');
    if (!$self->checkLeftoverInstanceSymlinks($value ,$ownSid)) {
        return 0;
    }

    return $self->checkAdditionalInstanceNumber ($value, $self->{params}->{InstanceNumber}->{str}, $ownSid);
}

sub checkPassword{
    my ($self, $value) = @_;
    my $properties = $self->{params}->{Password};

    if ($isWin && $properties->{type} eq 'initial_passwd'){
        require SAPDB::Install::System::Win32::API;
        my $domain = $self->getBatchValue ('Domain');
        if (!defined $domain){
            $domain = '';
        }
        if (lc ($domain) eq lc (hostname())){
            # local host
            $domain = '';
        }
        my ($rc, $errmsg) = SAPDB::Install::System::Win32::API::validatePasswordPolicy ($domain, $value);
        if (!defined $rc){
            $self->getMsgLst()->AddWarning ("Cannot validate os password policy due to runtime error: $errmsg");
        }
        elsif($rc){
            $self->PushError ("$properties->{str} doesn't meet the operating system password policy: $errmsg");
            return 0;
        }
    }

    if (!$value || length ($value) < 8){
        $self->PushError ("$properties->{str} must contain at least 8 characters");
        return 0;
    }
    return 1;
}

sub checkLeftoverInstanceSymlinks {
    my ($self, $instanceNumber, $sid) = @_;
    my $sidDir = File::Spec->catfile($gSAPLocation, $sid);
    my $sidDirObject = File::stat::stat($sidDir);
    my $dirContents = $sidDirObject && -d $sidDirObject ? listDir($sidDir) : [];
    my @symlinkCandidates = grep { $_ =~ /HDB\d\d/ && $_ ne sprintf("HDB%02d", $instanceNumber) } @{$dirContents};

    if (@symlinkCandidates) {
        my $msgList = SDB::Install::MsgLst->new();
        $msgList->addMessage(File::Spec->catfile($sidDir, $_)) for @symlinkCandidates;
        $self->setErrorMessage("The following files need to be removed before the installation can continue:", $msgList);
        return 0;
    }

    return 1;
}

sub checkDrive{
    my ($self, $value) = @_;
    if ($value =~ /^[a-z]:?$/i){
        return 1;
    }
    $self->PushError ("Invalid drive letter '$value'");
    return 0;
}


our @shells;

sub checkShell{
    my ($self, $value) = @_;
    if (!-x $value){
        $self->PushError ("Invalid shell: Cannot find executable file '$value'");
        return 0;
    }


    if (!@shells){
        if (-r '/etc/shells'){
            if (!open (FD, '/etc/shells')){
                return 1;
            }
            @shells = grep {chomp $_} <FD>;
            close (FD);
        }
        else{
            return 1;
        }
    }

    foreach my $shell (@shells){
        if ($shell eq $value){
            return 1;
        }
    }
    $self->PushError ("Invalid login shell '$value': Please see /etc/shells");
    return 0;
}


sub setDrive{
    my ($self, $value) = @_;
    $value = uc ($value);
    if (length ($value) == 1){
        $value .= ':';
    }
    $self->{params}->{Drive}->{value} = $value;
}

sub checkTarget{
    my ($self, $value) = @_;
    my $param = $self->{params}->{Target};
    if (!$isWin && $value eq '/usr/sap' ||
        $isWin && ($value =~ /\\usr\\sap$/)){
            if (defined $self->getAddHostsParser()){
                $self->PushError ("$param->{str} '$value' is not allowed for a multiple host system");
                return 0;
            }
            elsif ($self->isa('LCM::Configuration::GenericStackInstallConfiguration')) {
                $self->PushError ("$param->{str} '$value' is not allowed");
                return 0;
            }
            return 1;
    }
    else{
        if (defined $value && !-d $value){

            if ($value eq $gSapmntDirUx) {

                my $rc        = 1;
                my ($rootDir) = $value =~ /\/([a-zA-Z0-9]+)\//;
                $rootDir      = '/'.$rootDir;

                if (!-d $rootDir) {
                    $rc = mkdir($rootDir);
                }

                if ($rc) {
                    chmod(0755, $rootDir); # Try to ensure correct permissions for the newly created directory
                    $rc = mkdir($value);
                }

                if ($rc) {
                    chmod(0755, $value); # Try to ensure correct permissions for the newly created directory
                    $self->AddMessage("Installation directory '$value' created");
                }
                else {
                    $self->PushError("Cannot create installation directory '$value': $!");
                    return 0;
                }
            }
            else {
                $self->PushError ("Installation directory '$value' does not exist");
                return 0;
            }
        }
    }
    if (0 && defined $self->getAddHostsParser() && defined $self->{_remoteHosts} &&
        $self->{_remoteHosts}->authOk() == 0){
        #
        # create check file
        #
        my $check_file = $value . $path_separator . sprintf ('hdbinst_sapmnt_check_%s_%d', hostname(), $$);
        if (!open (TMP,'>' . $check_file)){
            $self->PushError ("Cannot create file '$check_file': $!");
            return 0;
        }
        close (TMP);

        #
        # check if the file is visable on all hosts as well
        #

        if ($self->{_remoteHosts}->existsFile($check_file) != 0){
            $self->PushError ("Directory '$value' is not accessible on all hosts", $self->{_remoteHosts});
            unlink ($check_file);
            return 0;
        }
         unlink ($check_file);
    }

    # checking r-x
    return $self->check_path_access ($value, R_OK | X_OK);
}



sub checkStorageConfigDir{
    my ($self, $value) = @_;
    my $properties = $self->{params}->{StorageConfigDir};
    if (!-d $value){
        $self->PushError ("$properties->{str} '$value' has to be an existing directory");
        return 0;
    }
    my $global_ini = $value . $path_separator .'global.ini';

    if (!-f $global_ini){
        $self->PushError ("$properties->{str} '$value' doesn't contain file 'global.ini'");
        return 0;
    }

    my $ini = new SDB::Install::IniFile ($global_ini);
    if (!defined $ini->getValue ('storage','ha_provider')){
        $self->PushError ("$properties->{str} '$value' is invalid: 'global.ini' doesn't contain a 'ha_provider' in 'storage' section");
        return 0;
    }
    my $log = $ini->getValue ('persistence', 'basepath_logvolumes');
    my $data = $ini->getValue ('persistence', 'basepath_datavolumes');
    my $rc = 1;
    if ($data){
         if (!$self->setValue ('BasePathDataVolumes', $data)){
            $self->PushError ("Storage configuration failure in '$global_ini' (basepath_datavolumes)");
            $rc = 0;
        }
    }
    if ($log){
        if (!$self->setValue ('BasePathLogVolumes', $log)){
            $self->PushError ("Storage configuration failure in '$global_ini' (basepath_logvolumes)");
            $rc = 0;
        }
    }
    return $rc;
}

sub checkCustomConfigDir{
    my ($self, $path) = @_;
    my $customdirFound = 0;

    # check relative to caller
    my $abspath = $self->getAbsPath($path);
    $customdirFound = -d $abspath;
    if (!$customdirFound) {
        # check relative to installation kit
        my $installer = new SDB::Install::Installer();
        $abspath = dirname($installer->{runtimedir}) . $path_separator . $path;
        $customdirFound = -d $abspath;
    }

    if (!$customdirFound) {
        $self->appendErrorMessage ("$self->{params}->{CustomConfigDir}->{str} '$abspath$' (computed from $path) has to be an existing directory");
        return undef;
    }

    # check if there is at least one readable *.ini file in the directory
    $! = 0;
    if( opendir(my $DIRH, $abspath) ) {
        my @allfiles = readdir($DIRH);
        my ($errno, $errtxt) = (int $!, "$!");
        closedir($DIRH);
        if(@allfiles>=2) {
            # we could read directory contents
            my @inifiles = grep(-f "$abspath$path_separator$_" && /\.ini$/  && -r _, @allfiles);
            if(@inifiles < 1) {
                $self->appendErrorMessage ("$self->{params}->{CustomConfigDir}->{str} '$abspath$' (computed from $path) does not contain any database configuration files");
                return undef;
            }
        }
        else {
            # no files found at all, not even . and ..
            $self->appendErrorMessage ("$self->{params}->{CustomConfigDir}->{str} '$abspath$' (computed from $path) cannot be read ($errno: $errtxt)");
            return undef;
        }
    }
    else {
        # opendir failed
        $self->appendErrorMessage ("$self->{params}->{CustomConfigDir}->{str} '$abspath$' (computed from $path) cannot be opened ($!)");
        return undef;
        
    }
    return $abspath;
}

sub setCustomConfigDir{
    my ($self, $path) = @_;
    my $abspath = $self->checkCustomConfigDir ($path);
    if (defined $abspath) {
        $self->{params}->{CustomConfigDir}->{value} = $abspath;
        return 1;
    }

    return 0;
}

#-------------------------------------------------------------------------------
# Collects information of remote hosts and checks SID, instance number,
# and user/group ID. These values have to belong to this SAP HANA system
# or have to be free.

sub CollectOtherHostInfos {
    my ($self, $isNotVerbose) = @_;

    my $sid    = $self->getBatchValue('SID');
    my $sidadm = (defined $sid) ? $self->getSysAdminUserName($sid) : undef;

    if (!$self->SUPER::CollectOtherHostInfos($isNotVerbose, $sidadm)) {
        return undef;
    }
    return $self->InitDefaultsForDistributedSystem();
}


#-------------------------------------------------------------------------------
# Returns filename.
# File contains the last step of an aborted installation.

sub pers_filename {
    return $_[0]->getPersFilenameInstUpd ($statusFileExtension, 1);
}

sub getOldLocalPersFilename{
    return $_[0]->getOldLocalPersFilenameInstUpd ($statusFileExtension);
}

#-------------------------------------------------------------------------------
# Returns name of the step where installation is pending

sub pers_getstepname{
    return $STEP_NAMES[defined $_[1] ? $_[1] : $_[0]->{step}];
}

sub pers_remove{
    my $self = shift;
    my $oldStatusFile = $self->getOldLocalPersFilenameInstUpd ($statusFileExtension);
    if ($oldStatusFile){
        $self->SUPER::pers_remove ($oldStatusFile);
    }
    return $self->SUPER::pers_remove (@_);
}

sub isSystemInCompatibilityMode{
    my $self = shift;
    return 0 if ($self->getValue('DbMode') eq 'singledb');
    return $self->getValue('CreateInitialTenant');
}

sub setCreateInitialTenant{
    my($self,$value) = @_;
    return 0 if(!$self->checkValue('CreateInitialTenant',$value,$self->getMsgLst()));
    $self->{params}->{'CreateInitialTenant'}->{value}=$value;
    if($value && exists $self->{params}->{'TenantUser'}){
        $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'TenantUser'});
    }
    if($value && exists $self->{params}->{'SQLTenantUserPassword'}){
        $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'SQLTenantUserPassword'});
    }
    return 1;
}
1;

