package SDB::Install::ContainerOSConfig;

use base SDB::Install::Base;

use SDB::Install::SysVars;
use SDB::Install::Globals;
use SAPDB::Install::System::Unix;
use strict;

sub new{
    my $self = shift->SUPER::new ();
    return $self;
}
    # /proc/sys/kernel/shmmax                >= 1073741824
    # /proc/sys/kernel/shmmni                >= 524288
    # /proc/sys/kernel/shmall                >= 8388608
    # /proc/sys/fs/file-max                  >= 20000000
    # /proc/sys/fs/aio-max-nr                 = 18446744073709551615
    # /proc/sys/vm/memory_failure_early_kill  = 1
    # /proc/sys/vm/max_map_count              = 2^31-1 = 2147483647
    # /proc/sys/net/ipv4/ip_local_port_range >= 40000
    # $(ulimit -n)                           >= 1048576
sub checkSystemConfig{
    my (
        $self,
        $SIDadm,
        $assumedMemoryGigas,
        $ChecksToFailCSVString
    ) = @_;

    if ($isWin){
        return 1;
    }

    if($assumedMemoryGigas){
        $self->{ram} = 0x40000000 * $assumedMemoryGigas;
        $self->{gb_ram} = $assumedMemoryGigas;
        $self->{memory_assumption} = 1;
        $self->getMsgLst()->addMessage ("Assuming $assumedMemoryGigas GB of memory.");
    }

    my $msg = $self->getMsgLst()->addMessage ("Checking os configuration for $gProductName");
    $self->setMsgLstContext ([$msg->getSubMsgLst()]);

    $self->{checkHash} = {};
    $self->{failAll} = 0;
    for my $checkToFail (split(/\s*,\s*/, $ChecksToFailCSVString)) {
        if ('all' eq $checkToFail) {
            $self->{failAll} = 1;
            last;
        }
        elsif ('none' eq $checkToFail) {
            $self->{checkHash} = {};
            last;
        }
        else {
            $self->{checkHash}->{$checkToFail} = $checkToFail;
        }
    }

    my $checkResult;
    my $retval = 1;

    $checkResult = $self->fileDescriptorsPerUser ($SIDadm);
    $retval = $self->_checkResult('ulimit.nofile', $checkResult, $retval);

    if( $isPowerPC ) {
        $checkResult = $self->stackPerUser ($SIDadm);
        $retval = $self->_checkResult('ulimit.stack', $checkResult, $retval);
    }
    
    $checkResult = $self->fileDescriptorsPerHost ();
    $retval = $self->_checkResult('fs.file-max', $checkResult, $retval);

    $checkResult = $self->checkAsyncIoRequests ();
    $retval = $self->_checkResult('fs.aio-max-nr', $checkResult, $retval);


    $checkResult = $self->checkShmmax ();
    $retval = $self->_checkResult('kernel.shmmax', $checkResult, $retval);

    $checkResult = $self->checkShmni ();
    $retval = $self->_checkResult('kernel.shmni', $checkResult, $retval);

    $checkResult = $self->checkShmall ();
    $retval = $self->_checkResult('kernel.shmall', $checkResult, $retval);


    $checkResult = $self->checkMaxMapCount ();
    $retval = $self->_checkResult('vm.max_map_count', $checkResult, $retval);

    $checkResult = $self->checkPortRange ();
    $retval = $self->_checkResult('net.ipv4.ip_local_port_range', $checkResult, $retval);

    $checkResult = $self->checkMemoryFailureEarlyKill();
    $retval = $self->_checkResult('vm.memory_failure_early_kill', $checkResult, $retval);

    $msg->endMessage (0, "Check os configuration");

    return $retval;
}


sub _checkResult{
    my ($self, $checkName, $checkResult, $retval) = @_;
    return $retval && ($checkResult || !($self->{checkHash}->{$checkName} || $self->{failAll}) );
}

sub _contained{
    my (
        $self,
        $text,
        $name,
    ) = @_;
    my $pattern = quotemeta ($text);
    if (!open (FD, $name)){
        $self->appendErrorMessage ("Cannot open file '$name': $!");
        return undef;
    }
    my @lines = <FD>;
    close (FD);
    my @s;
    foreach my $l (@lines){
        if ($l =~ /$pattern/){
            @s = split($pattern,$l);
            if ($s[0] !~ /#/){
                return 1;
            }
        }
    }
    return 0;
}

sub _checkContainer {
    my (
        $self,
        $procFilePath,
        $sysctlKey,
        $targetValue
    ) = @_;

    my $msg = $self->getMsgLst()->addMessage ("Checking $sysctlKey");
    my $value;
    if(open (FD, $procFilePath)) {
        $value = <FD>;
        $value = int ($value);
        close (FD);
    }
    if(!defined $value) {
        $self->appendErrorMessage("Sysctl '$sysctlKey' not found. It has to be set to (at least) $targetValue.");
        return 0;
    }
    if(int ($value) < $targetValue) {
        $self->appendErrorMessage ("Sysctl '$sysctlKey' has to be increased from $value to $targetValue.");
        return 0;
    }
    $self->getMsgLst()->addMessage ("Sysctl '$sysctlKey' is up-to-date (value=$value).");
    return 1;
}


our $SOFTLIMIT_NOFILE = 1048576;

sub fileDescriptorsPerUser{
    my ($self,$SIDadm) = @_;
    # allow the SIDadm user to have up to 8192 open file descriptors
    # by adding this to /etc/security/limits.conf
    # the dash in linea means to set both limits, soft and hard, in one line
    my $file   = '/etc/security/limits.conf';
    my $line1  = "$SIDadm          -       nofile          $SOFTLIMIT_NOFILE";
    #my $line2 = "$SIDadm          hard    nofile          8000\n";
    $self->getMsgLst()->addMessage ("Checking maximal number of open files per user");
    my $checkResult = 1;
    if (!$self->_contained ($line1, $file)){
        $self->appendErrorMessage ("'$file' needs line '$line1'");
        $checkResult = 0;
    }
    my ($rc,$soft,$hard) = getrlimit (RLIMIT_NOFILE);

    if ($rc){
        $self->getMsgLst()->addWarning ("getrlimit (RLIMIT_NOFILE) failed: errno = $rc");
        return $checkResult;
    }

    $self->getMsgLst()->addMessage ("nofile max = $hard");
    $self->getMsgLst()->addMessage ("nofile cur = $soft");

    return $checkResult;
}

our $SOFTLIMIT_STACK = 16384;

sub stackPerUser{
    my ($self,$SIDadm) = @_;
    my $file   = '/etc/security/limits.conf';
    my $line1  = "$SIDadm          soft       stack          $SOFTLIMIT_STACK";
    my $msg = $self->getMsgLst()->addMessage("Checking maximal stack per user");
    my $checkResult = 1;
    if (!$self->_contained ($line1, $file)){
        $self->appendErrorMessage ("'$file' needs line '$line1'");
        $checkResult = 0;
    }
    return $checkResult;
}

our $file_max = 20000000;
our $sysctl_file_max = 'fs.file-max';
our $proc_file_max = '/proc/sys/fs/file-max';

sub fileDescriptorsPerHost{
    my ($self) = @_;
    my $checkResult = $self->_checkContainer($proc_file_max, $sysctl_file_max, $file_max);
    return $checkResult;
}

sub checkKernelParameter{
    my ($self,$paramList) = @_;
    my $procval;
    foreach my $param (@$paramList){
        if (!open (FD, "<" . $param->[1])){
            $self->appendErrorMessage ("Cannot open file '$param->[1]': $!");
            return undef;
        }
        $procval = <FD>;
        chomp ($procval);
        $procval = int ($procval);

        close (FD);

        if ($procval < $param->[3]){
            $self->appendErrorMessage ("$param->[2] should be increased from $procval to $param->[3]");
            return 0;
        }
        else{
            $self->getMsgLst()->addMessage ("$param->[2] is up-to-date ($procval), minimal value is $param->[3]");
        }
    }
    return 1;
}

sub getRamSize{
    my ($self) = @_;
    if (!defined $self->{ram}){
        my $info = sysinfo ();
        my $ram = 0;
        if (defined $info && defined $info->{totalram} && $info->{mem_unit}){
            $ram = $info->{totalram} * $info->{mem_unit};
        }
        $self->{ram} = $ram;
    }
    return $self->{ram};
}


sub getGBofRam{
    my ($self) = @_;
    if (!defined $self->{gb_ram}){
        my $ram = $self->getRamSize ();
        my $gb = 0x40000000;
        my $gb_ram = 0;
        if ($ram){
            $gb_ram = int ($ram/$gb) + 1;
            $self->getMsgLst()->addMessage ("$gb_ram gb ram detected");
            $self->{gb_ram} = $gb_ram;
        }
    }
    return $self->{gb_ram};
}


#
# max value 2147483647 of signed int
#

sub checkMaxMapCount{
    my ($self) = @_;
    my $max_map_count = 2147483647;

    my @paramLst = (
         ['max_map_count', '/proc/sys/vm/max_map_count', 'vm.max_map_count', $max_map_count]
    );
    return $self->checkKernelParameter (\@paramLst);
}


sub checkShmmax{
    my ($self) = @_;
    #max segment size in bytes
    my $CONST_SHMMAX = 0x40000000; # 1gb, independent of the # of installations
    if(!$self->{memory_assumption}) {
        $CONST_SHMMAX = 1073741824;
    }
    my @shmparams = ( ['SHMMAX', '/proc/sys/kernel/shmmax','kernel.shmmax', $CONST_SHMMAX] );
    return $self->checkKernelParameter (\@shmparams);
}

sub checkShmni{
    my ($self) = @_;
    my $gb_ram = $self->getGBofRam ();

    # system-wide max number of segment identifiers
    #Size – shmni value – physical Memory
    #S(mall) – 4096  < 64 GB
    #M(edium) – 65536 < 256 GB
    #L(arge)- 524288 > 256 GB
    my $CONST_SHMMNI = 4096;
    if ($gb_ram >= 256){
        $CONST_SHMMNI = 524288;
    }
    elsif($gb_ram >= 64){
        $CONST_SHMMNI = 65536;
    }
    if(!$self->{memory_assumption}) {
        $CONST_SHMMNI = 524288;
    }
    my @shmparams = ( ['SHMMNI','/proc/sys/kernel/shmmni','kernel.shmmni', $CONST_SHMMNI] );
    return $self->checkKernelParameter (\@shmparams);
}

sub checkShmall{
    my ($self) = @_;
    my $ram = $self->getRamSize ();
    my $gb_ram = $self->getGBofRam ();

    #max segment size in bytes
    my $CONST_SHMMAX = 0x40000000; # 1gb, independent of the # of installations

    # system-wide max number of segment identifiers
    #Size – shmni value – physical Memory
    #S(mall) – 4096  < 64 GB
    #M(edium) – 65536 < 256 GB
    #L(arge)- 524288 > 256 GB

    my $CONST_SHMMNI = 4096;
    if ($gb_ram >= 256){
        $CONST_SHMMNI = 524288;
    }
    elsif($gb_ram >= 64){
        $CONST_SHMMNI = 65536;
    }
    #system-wide total shared memory size in OS pages (4k)
    my $CONST_SHMALL = int ($CONST_SHMMAX / 4096 * ($CONST_SHMMNI/16));

    # set upper limit to 90% of ram
    my $upper_shmall_limit = int ($ram * 9 / 40960);
    if ($CONST_SHMALL > $upper_shmall_limit){
        $CONST_SHMALL = $upper_shmall_limit;
    }

    if(!$self->{memory_assumption}) {
        $CONST_SHMMAX = 1073741824;
        $CONST_SHMMNI = 524288;
        $CONST_SHMALL = 8388608;
    }
    my @shmparams = ( ['SHMALL','/proc/sys/kernel/shmall','kernel.shmall', $CONST_SHMALL] );
    return $self->checkKernelParameter (\@shmparams);
}

sub checkPortRange{
    my ($self) = @_;
    my $fn = '/proc/sys/net/ipv4/ip_local_port_range';

    if (!open (FD, $fn)){
        $self->getMsgLst()->addMessage("WARNING: Unable to access /proc/sys/net/ipv4/ip_local_port_range (skipping check)");
        return 1;
    }
    my $line = <FD>;
    chomp ($line);
    my $checkResult = 1;
    my ($port_low,$port_high) = split (/\s+/,$line);
    my $limit_low = 32768; # was 40000
    if (int ($port_low) < $limit_low && int ($port_high) > $limit_low){
        $self->appendErrorMessage ("'/proc/sys/net/ipv4/ip_local_port_range' is not set correctly");
        $self->appendErrorMessage ("expected lower limit is '$limit_low'");
        $checkResult = 0;
    }
    close (FD);
    return $checkResult;
}

our $proc_aio_max_nr = '/proc/sys/fs/aio-max-nr';

sub checkAsyncIoRequests{
    my ($self) = @_;
    my $AIO_MAX_NR = 18446744073709551615; # ULONG_MAX = 2^64-1, Bug 169992
    return $self->_checkContainer($proc_aio_max_nr, 'fs.aio-max-nr', $AIO_MAX_NR);
}


sub checkMemoryFailureEarlyKill{
    my ($self) = @_;
    if (!-e '/proc/sys/vm/memory_failure_early_kill'){
        $self->getMsgLst()->addMessage ("Skipping 'vm.memory_failure_early_kill': not supported by linux kernel");
        return 1;
    }
    return $self->_checkContainer('/proc/sys/vm/memory_failure_early_kill', 'vm.memory_failure_early_kill', 1);
}

1;
