#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: manage MaxDB features 


package SDB::Install::FeatureManager;

use SDB::Install::PackageManager;

our @ISA = qw (SDB::Install::PackageManager);

my $i = 0;

our %featuregroups = (
	'server' => {'name' => 'Database Server', 'collapse' => 1, 'desc' => 'SAP MaxDB Database Kernel and extensions.', 'seq' => $i++},
	'gui' => {'name' => 'Administration and SQL Query GUI', 'collapse' => 1, 'desc' => 'SAP MaxDB Gui frontend tools.', 'seq' => $i++},
	'interfaces' => {'name' => 'SQL Connnectivity', 'collapse' => 1, 'desc' => 'SAP MaxDB SQL interfaces for various programming languages.', 'seq' => $i++},
	'addons' => {'name' => 'AddOns', 'collapse' => 1, 'desc' => 'SAP MaxDB tools.', 'seq' => $i++},
	'test' => {'name' => 'Test Tools', 'collapse' => 1 , 'desc' => 'SAP MaxDB components required for test purposes.', 'seq' => $i++}
);


#
# feature list
#
#  $self-->{features}+-->{$feature_id}+-->{selected} = 0|1
#                    |                |          
#                    .                +-->{packages}+--->{$package_id}               
#                    .                              |
#                    .                              .	
#                    .                              .
#                    .                              .

#
# feature list
#
#  $self-->{features}+-->{$feature_id}+-->{selected} = 0|1
#                    |                |          
#                    .                +-->{packages}+--->{$package_id}
#	                 .                |             |
#                    .                +-->{desc}    .
#                    .                              .	
#                    .                              .




#
# feature groups
#
#  $self-->{feature_groups}+-->{$group_id}+-->{selected} = 0|1|2
#                    |                    |          
#                    .                    +-->{tree_node}
#                    .                    |          
#                    .                    +-->{desc}          	
#                    .                              
#                    .                              



#
# feature select counter
#
# $self->{feature_counter}+-->{$package_id} = $counter
#                         |
#                         .
#                         .
#                         .


#
# feature tree
#
# $self->{feature_tree}}+-->{groups}+-->{$group}+-->{groups}...
#                       |           |           |
#                       |           |           +-->{features}...
#                       |           .
#                       |           .
#                       |           .
#                       |              
#                       +-->{freatures} = {$feature1 => undef ,$feature2 => undef...]
#                          




#----------------------------------------------------------------------------

sub InitFeatures{
	my ($self) = @_;

	if (!defined $self->{packages}){
		$self->AddError ('No packages defined');
		return undef;
	}
	
	if (defined $self->{features}){
		return $self->{features};
	}
	
	my %features;
	my %feature_counter;
	my %tree = %featuregroups;

	$self->{feature_tree}->{groups} = \%tree;

	$self->{features} = \%features;
	$self->{feature_counter} = \%feature_counter;	

	foreach my $id (keys %{$self->{packages}}){
		$feature_counter{$id} = 0;
		my $package = $self->{packages}->{$id};
		my $flist = $package->Provides ();
		if (defined $flist && %$flist){
			foreach my $feature (keys %$flist){
				my $feature_group = $flist->{$feature}->{group};
				
				if (exists $features{$feature}){
					$features{$feature}->{packages}->{$id} = undef;
					$features{$feature}->{selected} &&=  $package->{selected};	
				}	
				else{
					$features{$feature} = {'packages' => {$id => undef},
								'selected' => $package->{selected}};
					$features{$feature}->{details} = $flist->{$feature};
				}
				if (defined $feature_group){
					if (! exists $self->{feature_groups}->{$feature_group}){
						$self->{feature_groups}->{$feature_group}->{Name} = $feature_group;
						$self->{feature_groups}->{$feature_group}->{selected} = 0;
					}
					
					if (defined $tree{$feature_group}->{features}){
						$tree{$feature_group}->{features}->{$feature} = undef;
					}
					else{
						$tree{$feature_group}->{features} = {$feature => undef};
					}
					$self->{feature_groups}->{$feature_group}->{tree_node} = $tree{$feature_group};
				}
				else{
					if (defined $self->{feature_tree}->{features}){
						$self->{feature_tree}->{features}->{$feature} = undef;
					}
					else{
						$self->{feature_tree}->{features} = {$feature => undef};
					}
					
				}
			}
		}
	}

	$self->{features} = \%features;
	
	$self->ResetFeatureCounter ();		
	
	return \%features;
}

#----------------------------------------------------------------------------

sub SyncPackageSelection2Features{
	my ($self) =@_;
	my $features = $self->{features};
	
	my $selected;
	
	foreach my $feature (keys (%$features)){
		$selected = 1;
		foreach my $package (keys (%{$features->{$feature}->{packages}})){
			if (!$self->{packages}->{$package}->{selected}){
				$selected = 0;
				last;
			}
		}				
		$features->{$feature}->{selected} = $selected;
	}
	$self->ResetFeatureCounter ();	
}


#----------------------------------------------------------------------------

sub ResetFeatureCounter{
	my ($self) = @_;
	
	my $counter = $self->{feature_counter};
	foreach my $package (keys %$counter){
		$counter->{$package} = 0;
	}	
	
	my $features = $self->{features};
	
	foreach my $feature (keys (%$features)){
		if ($features->{$feature}->{selected}){
			foreach my $package (keys (%{$features->{$feature}->{packages}})){
				$self->IncreaseFeatureCounter ($self->{packages}->{$package});
			}				
		}
	}
	
}	

#----------------------------------------------------------------------------

sub GetFeatures{
	my ($self,$tree) = @_;
	if (!defined $tree){
		$tree = $self->{feature_tree};
	}
	
	if (exists $tree->{features}){
		return [sort keys %{$tree->{features}}];
	}
	return [];
}

sub GetFeatureGroups{
	my ($self,$tree) = @_;
	
	if (!defined $tree){
		$tree = $self->{feature_tree};
	}
	else{
		return {};
	}

	if (exists $tree->{groups}){
		return $tree->{groups};
	}
	return {}; 
}

#----------------------------------------------------------------------------

sub SelectFeatureByName{
	my ($self,$name) = @_;
	
	foreach my $id (keys %{$self->{features}}){
		if ($self->{features}->{$id}->{details}->{name} eq $name){
			return $self->SelectFeature ($id);
		}
	}	

	foreach my $id (keys %featuregroups){
		if ($featuregroups{$id}->{name} eq $name){
			return $self->SelectFeatureGroup ($id);
		}
	}
	return 0;
}


#----------------------------------------------------------------------------

sub SelectFeature{
	my ($self,$feature_id) = @_;
	
	if (! exists $self->{features}->{$feature_id}){
		$self->AddError ('Feature is not defined');
		return undef;	
	}

	my $feature = $self->{features}->{$feature_id};

	if ($feature->{selected}){
		return 1;
	}
	
	$feature->{selected} = 1;
	
	my $groups =  $self->FindContainingFeatureGroups($feature_id);
	
	foreach my $group (@$groups){
		if (exists $self->{feature_saved_selection}->{$group}){
			delete $self->{feature_saved_selection}->{$group};	
		}
	}	
	
	foreach my $package_id (keys %{$feature->{packages}}){
		my $package = $self->{packages}->{$package_id};
		if (!$self->{feature_counter}->{$package_id}){
			$self->SelectPackage ($package);
		}
		$self->IncreaseFeatureCounter ($package)
	}	
	
	return 1;
}


#----------------------------------------------------------------------------

sub SelectFeatureGroup{
	my ($self, $id) = @_;
	
	if (! exists $self->{feature_groups}->{$id}){
		$self->AddError ('Feature group is not defined');
		return undef;	
	}
	my $group = $self->{feature_groups}->{$id};

	my $state = $self->FeatureGroupState ($id);

	my $saved_selection;

	if ($state == 1){
		return 1;
	 }
	
	if ($state == 2){
		
		#
		#	save selection
		#
		
		my $features = $self->GetFeatureGroupFeatures ($id);
		my %feature_selections;
		$saved_selection = {$id => \%feature_selections};
			
		foreach my $feature (@$features){
			if (!exists $feature_selections{$feature}){
				$feature_selections{$feature} = $self->{features}->{$feature}->{selected};
			}
			if (!$self->{features}->{$feature}->{selected}){
				$self->SelectFeature($feature);
			}
		}
		$self->{feature_saved_selection} = $saved_selection;
		return 1;	 
	 }
	
	 if ($state == 0){
		if (!$self->RestoreFeatureGroup($id)){
			my $features = $self->GetFeatureGroupFeatures ($id);
			foreach my $feature (@$features){
				$self->SelectFeature($feature);
			}
		}
	 }
	
		 
	return 1;
}

#----------------------------------------------------------------------------

sub RestoreFeatureGroup{
	my ($self,$id) = @_;
	if (exists $self->{feature_saved_selection}->{$id}){
		my $saved_selection = $self->{feature_saved_selection}->{$id};
		foreach my $feature (keys %$saved_selection){
			if ($saved_selection->{$feature}){
				$self->SelectFeature ($feature);
			}
			else{
					$self->UnselectFeature ($feature);
			}
		}
		return 1;
	
	}
	return 0;
}


#----------------------------------------------------------------------------

sub FindContainingFeatureGroups{
	my ($self,$feature,$node) = @_;
	
	if (!defined $node){
		$node = $self->{feature_tree};
	}
	if (exists $node->{features}->{$feature}){
		return ([]);
	}
	
	foreach my $group (keys %{$node->{groups}}){
		if (!exists $self->{feature_groups}->{$group}){
			next;
		}
		my $rc = $self->FindContainingFeatureGroups($feature,$self->{feature_groups}->{$group}->{tree_node});
		if (defined $rc){
			push @$rc, $group;
			return $rc;
		}
	}
	return undef;
}


#----------------------------------------------------------------------------

sub UnselectFeature{
	my ($self,$feature_id) = @_;
	
	if (! exists $self->{features}->{$feature_id}){
		$self->AddError ('Feature is not defined');
		return undef;	
	}

	my $feature = $self->{features}->{$feature_id};

	if (!$feature->{selected}){
		return 1;
	}
	
	$feature->{selected} = 0;
	
	my $groups =  $self->FindContainingFeatureGroups($feature_id);
	
	foreach my $group (@$groups){
		if (exists $self->{feature_saved_selection}->{$group}){
			delete $self->{feature_saved_selection}->{$group};	
		}
	}	

	foreach my $package_id (keys %{$feature->{packages}}){
		my $package = $self->{packages}->{$package_id};
		$self->DecreaseFeatureCounter ($package);
	}	
	
	return 1;
}

#----------------------------------------------------------------------------

sub UnselectFeatureGroup{
	my ($self,$id) = @_;
	
	if (! exists $self->{feature_groups}->{$id}){
		$self->AddError ('Feature group is not defined');
		return undef;	
	}
	my $tree = $self->{feature_groups}->{$id}->{tree_node};

	my $state = $self->FeatureGroupState ($id);

	if ($state == 0){
		return 1;
	} 
	
	my $saved_selection;
	
	if (exists $self->{feature_saved_selection}->{$id}){
		$saved_selection = $self->{feature_saved_selection}->{$id}; 
	}
	
	foreach my $group (keys %{$tree->{groups}}){
		$self->UnselectFeatureGroup ($group);
	}
	
	foreach my $feature (keys %{$tree->{features}}){
		$self->UnselectFeature ($feature);
	}
	
	if (defined $saved_selection){
		$self->{feature_saved_selection}->{$id} = $saved_selection;
	}
		
	return 1;
}


#----------------------------------------------------------------------------

sub IncreaseFeatureCounter{
	my ($self,$package) = @_;
	my $counter = $self->{feature_counter};
	
	if (!defined $counter->{$package->{id}}){
		$counter->{$package->{id}} = 0;
	}	
	
	$counter->{$package->{id}}++;
	my $packages = $self->GetRequiredPackages ($package);	
	if (defined $packages){
		foreach my $required_package (@$packages){
			$self->IncreaseFeatureCounter ($required_package)
		}
	}	
	
	return 1;
}


#----------------------------------------------------------------------------

sub DecreaseFeatureCounter{
	my ($self,$package) = @_;

	if (!$self->{feature_counter}->{$package->{id}}){
		return undef;
	}

	$self->{feature_counter}->{$package->{id}}--;
	
	if (!$self->{feature_counter}->{$package->{id}}){
		$self->UnselectPackage($package);
	}

	my $packages = $self->GetRequiredPackages ($package);	
	if (defined $packages){
		foreach my $required_package (@$packages){
			$self->DecreaseFeatureCounter ($required_package)
		}
	}	
	
	return 1;
}

sub FeatureGroupHasFeatures{
	my ($self,$feature_group) = @_;
	if (exists $self->{feature_groups}->{$feature_group}){
		my $tree_node = $self->{feature_groups}->{$feature_group}->{tree_node};
		if (exists $tree_node->{features} && %{$tree_node->{features}}){
			return 1;
		} 
		foreach my $group (keys %{$tree_node->{groups}}){
			if ($self->FeatureGroupHasFeatures ($group)){
				return 1;
			}
		}
		return 0;
	}
	return undef;
}

sub GetFeatureGroupFeatures{
	my ($self,$feature_group) = @_;
	my @result;
	if (exists $self->{feature_groups}->{$feature_group}){
		my $tree_node = $self->{feature_groups}->{$feature_group}->{tree_node};
		if (exists $tree_node->{features} && %{$tree_node->{features}}){
			push @result, keys (%{$tree_node->{features}});
		} 
		foreach my $group (keys %{$tree_node->{groups}}){
			my $list = $self->GetFeatureGroupFeatures($group);
			if (defined $list && @$list){
				push @result,@$list;
			}
		}
		return \@result;
	}
	return undef;
}



sub GetFeatureName{
	my ($self,$id) = @_;
	$self->{features}->{$id}->{details}->{name};
}

sub GetFeatureDescription{
	my ($self,$id) = @_;
	$self->{features}->{$id}->{details}->{desc};
}

sub GetFeatureGroupName{
	my ($self,$id) = @_;
	$featuregroups{$id}->{name};
}

sub GetFeatureGroupDescription{
	my ($self,$id) = @_;
	$featuregroups{$id}->{desc};
}


sub GetFeatureGroupOrderNum{
	my ($self,$id) = @_;
	$featuregroups{$id}->{seq};
}



sub FeatureGroupExpand{
	my ($self,$id) = @_;
	(defined $featuregroups{$id}->{collapse} && $featuregroups{$id}->{collapse}) ? 0 : 1;
}



#	
#	returns state of feature group:
#   undef => group has no features
#	0 => all features are unselected
#   1 => all features are selected
#	2 => state is undetermined (has selected and unselected features)
#

sub FeatureGroupState{
	my ($self,$feature_group) = @_;
	my $all_state;
	if (exists $self->{feature_groups}->{$feature_group}){
		my $tree_node = $self->{feature_groups}->{$feature_group}->{tree_node};

		if (exists $tree_node->{features}){
			foreach my $feature (keys (%{$tree_node->{features}})){
				if (!defined $all_state){
					$all_state = $self->{features}->{$feature}->{selected};
				}
				else{
					if ($all_state != $self->{features}->{$feature}->{selected}){
						return 2;
					}  
				}
			}
		}
			
		foreach my $group (keys %{$tree_node->{groups}}){
			my $group_state = $self->FeatureGroupState($group);
			if (defined $group_state){
				if ($group_state == 2){
					return 2;
				}
				if (!defined $all_state){
					$all_state = $group_state;
				}
				else{
					if ($all_state != $group_state){
						return 2;
					}  
				}
			}
		}
		return $all_state;
	}
	return undef;
}

#	
#	returns state of feature:
#   undef => feature is unknown
#	0 => feature is unselected
#   1 => feature is selected
#

sub FeatureState ($$){
	my ($self,$feature) = @_;
	if (exists $self->{features}->{$feature}){
		return $self->{features}->{$feature}->{selected};		
	} 
	return undef;
}
1;
