#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: Configuration dialog => gui of SDB::Install::Configuration classes

package SDB::Install::Gui::Dialogs::GenericParameter;

use Wx qw (
	wxCENTRE
	wxVERTICAL
	wxHORIZONTAL
	wxEXPAND
	wxALL
	wxLEFT
	wxALIGN_CENTER_VERTICAL
	wxALIGN_CENTER_HORIZONTAL 
	wxALIGN_RIGHT
	wxTE_MULTILINE
	wxOK
	wxTOP
	wxBOTTOM
	wxICON_EXCLAMATION
	wxID_OK
	wxDD_NEW_DIR_BUTTON
	wxBITMAP_TYPE_PNG
	wxALIGN_LEFT
	wxTE_PASSWORD
	wxCB_READONLY
	wxSP_ARROW_KEYS
	wxSL_LABELS
	wxSL_HORIZONTAL
	);

use Wx::Event qw( EVT_BUTTON EVT_CHECKBOX EVT_TEXT);

use SAPDB::Install::Wizard::Dialog::Base;
use SAPDB::Install::Control::PathCtrl;
use SDB::Install::Gui::Design;
use SDB::Install::Globals qw ($gProductName $gProductNameEngine);
use SDB::Install::Gui::Control::Label;
use strict;

our @ISA = qw (SAPDB::Install::Wizard::Dialog::Base);

sub new{
	my($self ) = shift->SUPER::new(@_);
	$self->{id} = 'GenericParams';
	$self->{label_text} = "Specify the Installation Properties";
	$self->{name} = 'Define Installation Properties';
	return $self;
}


sub getParamKeys{
	my ($self, $instconfig) = @_;
	
	return $instconfig->getParamIds ();
}

sub getAlreadySetParamKeys{
	return [];
}

sub addControls(){return 1;}

sub init {
	my ($self) = @_;
	return undef           if (!defined $self->checkParams());
	return {errstate => 1} if (!defined $self->setBatchValueOfRequiredParams());

	return $self->createDialog();
}

sub checkParams {
	my ($self) = @_;
	my $app = Wx::wxTheApp();
	my $instconfig = $app->getInstconfig();

	my $msg = $app->getMsgLst ()->addMessage ("Checking configuration of installation (dialog $self->{name})");
	$msg->setEndTag ('Configuration Check');
	$instconfig->setMsgLstContext([$msg->getSubMsgLst(),undef]);
	my $rc = $instconfig->CheckParams($app->{batch_mode});
	if($self->{batch_mode} && !$rc){
		if (defined $rc){
			$app->setErrorMessage ("running in batch mode", $instconfig->getErrMsgLst());
			$app->PrintUsage ();
		} else {
			$app->setErrorMessage (undef, $instconfig->getErrMsgLst());
		}
		$msg->getSubMsgLst()->appendMsgLst ($instconfig->getMsgLst());
		return undef;
	}
	return 1;
}

sub setBatchValueOfRequiredParams {
	my ($self) = @_;
	my $app = Wx::wxTheApp;
	my $instconfig = $app->getInstconfig();
	my $paramKeys = $self->getParamKeys ($instconfig);
	my $lastParamId = $paramKeys->[-1];
	return 1 if (!defined $lastParamId);

	my $rc = $instconfig->setBatchValueOfRequiredParams($lastParamId);
	if (!defined $rc) {
		$app->ShowErrorMsg ('Initialize configuration failed', $instconfig->getErrMsgLst ());
	}
	return $rc;
}

sub createDialog {
	my ($self) = @_;
	my $app = Wx::wxTheApp();
	my $instconfig = $app->getInstconfig();

	my $sizer = Wx::BoxSizer->new(wxVERTICAL);
	$self->{main_sizer} = $sizer;
	$sizer->AddSpacer (20);

	my $is_gtk1 =  Wx::wxGTK1();
	$self->{'titel_big'} = Wx::StaticText->new($self, -1, $self->{label_text});
	$self->{'titel_big'}->SetFont( SDBFONT_DialogTitle ) if (!$is_gtk1);
	$sizer->Add($self->{'titel_big'},0, wxLEFT , 20 );
	$self->{'grid_sizer'} = Wx::FlexGridSizer->new(6, 2, 5, 15);
	$sizer->Add ($self->{'grid_sizer'},0, wxEXPAND | (wxALL) & (~wxBOTTOM), 20);

	my %dynctrl;
	my $setParamKeys = $self->getAlreadySetParamKeys ();
	my %setParamKeys;
	if (defined $setParamKeys &&  @$setParamKeys){
		@setParamKeys{@$setParamKeys} = ();
	}
	$self->resetSkippedParams();

	my $param_count = 0;
	my $paramKeys = $self->getParamKeys ($instconfig);
	foreach my $param_id (@$paramKeys){
		if (!$param_id){
			$self->{'grid_sizer'}->AddSpacer (16);
			$self->{'grid_sizer'}->AddSpacer (16);
			next;
		}
		my $param = $instconfig->{params}->{$param_id};
		
		if ($param->{skip}){
			next;
		}
		if (defined $param->{set_interactive} && !$param->{set_interactive}){
			next;
		}
		my $label = $dynctrl{$param_id . '_label'} = $param->{type} eq 'boolean' ?
			new Wx::StaticText ($self, -1, '') : 
			SDB::Install::Gui::Control::Label->new
				( $self, -1, $param->{'str'},
					$param->{mandatory});
		
		my $value = (defined $param->{value} ? $param->{value} :
			$instconfig->getDefault ($param_id));
		$value //= '';

		my $ctrl = $dynctrl{$param_id . '_ctrl'} =
			($param->{type} eq 'number' && defined $param->{minValue} && defined $param->{maxValue}) ?
				new Wx::Slider ($self, -1, $value, $param->{minValue} ,$param->{maxValue}, [-1,-1], [-1,-1], wxSL_HORIZONTAL | wxSL_LABELS) :
			($param->{type} eq 'number' && not defined $param->{valid_values}) ?
				new Wx::SpinCtrl ($self, -1, $value, [-1,-1], [-1,-1], wxSP_ARROW_KEYS(), 0, 99999) :
			$param->{type} eq 'passwd' ?
				new Wx::TextCtrl ($self, -1, $value,[-1,-1], [-1,-1],wxTE_PASSWORD ) :
			$param->{type} eq 'initial_passwd' ?
				new Wx::TextCtrl ($self, -1, $value,[-1,-1], [-1,-1],wxTE_PASSWORD ) :
			$param->{type} eq 'path' ?
				new SAPDB::Install::Control::PathCtrl ($self,-1,$value, [-1,-1], [-1,-1]) :
			$param->{type} eq 'boolean' ?
				new Wx::CheckBox ($self, -1,$param->{'str'}, [-1,-1], [-1,-1]) :
			defined $param->{valid_values} ? $self->_createComboBox($instconfig, $param_id, $value) :
				new Wx::TextCtrl ( $self, -1, $value,[-1,-1], [-1,-1]);
		if ($param->{type} eq 'boolean'){
			$ctrl->SetValue ($value);
		}
		if (defined $param->{persFileValue} || exists $setParamKeys{$param_id} || $param->{isStaticValue}){
			$ctrl->Enable (0);
		}

		my $input_sizer = Wx::BoxSizer->new(wxVERTICAL);
		$input_sizer->Add ($ctrl, 1, wxEXPAND, 20);
		if(defined $param->{recommendation}) {
			my $recommendation_text = $dynctrl{$param_id . '_recommendation_text'} = Wx::StaticText->new($self, -1, $param->{recommendation});
			$input_sizer->Add ($recommendation_text, 0, wxEXPAND, 8);
		}

		if (defined $param->{desc}){
			$label->SetToolTip ($param->{desc});
			$ctrl->SetToolTip ($param->{desc});
		}
		$self->{'grid_sizer'}->Add($label,0, wxTOP, 8);
		$self->{'grid_sizer'}->Add($input_sizer, 1, wxEXPAND);

		$param_count++;
	
	}
	
	$self->{dynctrl} = \%dynctrl;
	$self->{'grid_sizer'}->AddGrowableCol( 1 );
	
	
	$self->addControls ($sizer);
	$self->AddAdditionalParams($instconfig) if ($self->can('AddAdditionalParams'));
	$self->SetSizerAndFit($sizer);
	$self->SetAutoLayout(1);
	$self->Layout();
	$self->Show(1);

	if ($instconfig->ErrorState){
		$self->GetWizard ()->LayoutDlg ();
		if (defined $SDB::Wx::Splash::Window){
			$SDB::Wx::Splash::Window->Close();
			undef $SDB::Wx::Splash::Window;
		}
		$app->ShowErrorMsg ('Initialize configuration failed', $instconfig->getErrMsgLst ());
		return {'errstate' => 1};
	}
	if ($param_count == 0 || $app->{batch_mode}){
		return {'auto_next' => 1};
	}
	return {};
}

sub addControlToGridSizer {
	my ($self, $control, $options) = @_;
	$self->{'grid_sizer'}->Add($control,@{$options->{controlOptions}});
	my $parameterId = $options->{parameterId};
	my $controlType = $options->{controlType};
	$self->{dynctrl}->{"$parameterId"."_"."$controlType"} = $control;
}

sub resetSkippedParams {
	return 1;
}

sub _createComboBox {
	my ( $self, $instconfig, $param_id, $value ) = @_;
	my $param = $instconfig->{params}->{$param_id};
	my ( $defaultValue, $arrayWithValues );
	
	if ( defined $param->{ui_values} ) {
		($defaultValue, $arrayWithValues) = $param->{interactive_index_selection} ? $self->_crateMixedValues($param, $value) : ($instconfig->getUIStringFromValue($param_id, $value), $param->{ui_values});
	} elsif (exists $param->{visible_alias_values}) {
		($defaultValue, $arrayWithValues) = ($value,$param->{visible_alias_values});
	} else {
		($defaultValue, $arrayWithValues) = ($value,$param->{valid_values});
	}
	
	return new Wx::ComboBox ( $self, -1, $instconfig->getUIStringFromValue($param_id, $defaultValue), [-1,-1],[-1,-1], $arrayWithValues, wxCB_READONLY);
}

sub _crateMixedValues {
	my ( $self,$param, $value ) = @_;
	my @values = ();
    my $text;
    my $validValues = (exists $param->{visible_alias_values})
                      ? $param->{visible_alias_values}
                      : $param->{valid_values};
    
    for ( 0 .. (@$validValues - 1) ) {

        $text = $validValues->[$_] . " - " . $param->{ui_values}->[$_];
    	push ( @values, $text);
    	
        if (($value eq $param->{valid_values}->[$_])
            ||
            (exists $param->{visible_alias_values}
             && ($value eq $param->{visible_alias_values}->[$_]))) {

            $value = $validValues->[$_] . " - " . $param->{ui_values}->[$_];
    	}
    }
    
    return ($value, \@values);
}

sub Process{
	my ($self,$config) = @_;

    #
    #   check and set values
    #
    
    my ($instconfig) = (
           Wx::wxTheApp->{instconfig},  # systeminstallation
        );

	$instconfig->resetError ();
	
	my ($param, $value, $ctrl);
	
	
	my $msglst = $instconfig->getMsgLst();

	my ($old_value, $logvalue, $defaultvalue);

	my $isUpgrade = $instconfig->isa ('SDB::Install::Configuration::Upgrade');

	my $setParamKeys = $self->getAlreadySetParamKeys ();
	my %setParamKeys;
	if (defined $setParamKeys &&  @$setParamKeys){
		@setParamKeys{@$setParamKeys} = ();
	}

    # check for mandatory parameters which were left empty by the user before we do the full-fledged check using the Configuration class:
	foreach my $param_id (@{$self->getParamKeys ($instconfig)}){
		$param = $instconfig->{params}->{$param_id};
		if($param->{skip}) {
			next;
		}
		if(exists $setParamKeys{$param_id}) {
				next;
		}
		if(defined $param->{set_interactive} && !$param->{set_interactive}) {
			next;
		}
		if(not $param->{mandatory}) {
			next;
		}
		if($param->{type} =~ /bool/i) {
			next;
		}
		$ctrl = $self->getParamCtrl ($param_id);
		if(defined $ctrl) {
			if( ref $ctrl eq "Wx::ComboBox" && $param->{interactive_index_selection} ) {
				next;
			}
			else {
                $value = $instconfig->getValueFromUIString ($param_id, $ctrl->GetValue ());
    			if((not defined $value) || ($value eq '')) {
    				Wx::MessageBox("Checking $param->{str} failed.\nValue is empty", "Configuration Error", wxOK|wxCENTRE|wxICON_EXCLAMATION, $self);
    				$ctrl->SetFocus();
    				return undef;
    			}
			}
		}
	}
	# now use the Configuration ($instconfig) to perform the more elaborate checks:
	foreach my $param_id (@{$self->getParamKeys ($instconfig)}){

		$param = $instconfig->{params}->{$param_id};
		
		if ($param->{skip}){
			next;
		}

		if (exists $setParamKeys{$param_id}){
				next;
		}

		if (defined $param->{set_interactive} && !$param->{set_interactive}){
			next;
		}

		$ctrl = $self->getParamCtrl ($param_id);
		
		if (defined $ctrl){
			if ( ref $ctrl eq "Wx::ComboBox" && $param->{interactive_index_selection} ) {
				$value = $self->_getValueFromMixedComboBox( $ctrl );
				$value = $instconfig->getValueFromUIString( $param_id, $value );
			} else {
                $value = $instconfig->getValueFromUIString ($param_id, $ctrl->GetValue ());
			}
			if ($param->{type} =~ /bool/i){
				$value = $value ? 1 : 0;
			}
			$old_value = $param->{value};
			if (!$instconfig->setValue ($param_id, $value)){
				Wx::wxTheApp->ShowErrorMsg("Configuration Error", $instconfig->getErrMsgLst());
				$ctrl->SetFocus();
				return undef;
			}
			$logvalue = $instconfig->getLogValue ($param_id, $value);
			$defaultvalue = $instconfig->getDefault ($param_id);
			if (!defined $old_value || $old_value ne $param->{value}){
				if (defined $defaultvalue && ($param->{type} eq 'number' &&
					int ($defaultvalue) == int ($value) ||
					$defaultvalue eq $value)){
					$msglst->addMessage ( $self->_getUserConfirmationMessagePattern($param, $param_id, $logvalue));
				}
				else{
					$msglst->addMessage ( $self->_getIsSetByUserMessagePattern($param, $param_id, $logvalue));
				}
			}
			elsif (!defined $param->{value}){
				$msglst->addMessage ($self->_getSetByUserMessagePattern($param));
			}
			
		}
	}
	return {};
}

sub _getValueFromMixedComboBox {
	my ($self, $ctrl) = @_;
	my $value = $ctrl->GetValue();
	my @splitedValue = split ("-", $value);
	my $elementIndex = @splitedValue;
	$value = $splitedValue[$elementIndex-1];
	$value =~ s/^\s*(.*)\s*$/$1/;
	return $value; 
}

sub Cleanup{
	my ($self) = @_;
    #
    #   Destroy Sizer
    #
    if(defined $self->GetSizer){
        $self->GetSizer->Clear (1);
    }

    #
    #   Destroy elements
    #
   delete $self->{dynctrl};
   delete $self->{title_big};
}

sub getParamLabel{
    my ($self,$paramId) = @_;
    if (!defined $self->{dynctrl}){
        return undef;
    }
    return $self->{dynctrl}->{$paramId . '_label'};
}

sub getParamCtrl{
    my ($self,$paramId) = @_;
    if (!defined $self->{dynctrl}){
        return undef;
    }
    return $self->{dynctrl}->{$paramId . '_ctrl'};
}

sub getParamRecommendationText{
    my ($self,$paramId) = @_;
    if (!defined $self->{dynctrl}){
        return undef;
    }
    return $self->{dynctrl}->{$paramId . '_recommendation_text'};
}

sub Destroy{
	my ($self) = @_;
	$self->Cleanup();
	$self->SUPER::Destroy;
}

sub Back {
	my ($self) = @_;

	my $sizer = $self->{'grid_sizer'};
	
	for my $control (keys %{$self->{dynctrl}}) {
		$sizer->Detach ($self->{dynctrl}->{$control});
		$self->{dynctrl}->{$control}->Destroy();
		delete $self->{dynctrl}->{$control};
	}
}

sub DESTROY{
	my ($self) = @_;
	print ">>>>> destroying $self\n"
		if $SAPDB::Install::DEBUG::DESTRUCT;
}

1;